/* CMSynergyVer %full_filespec: mt_cache.c;4:csrc:1 % */

/************************************************************************
 *
 *  mt_cache.c
 *
 *  cache specific functions
 *
 *
 ************************************************************************/
#include "mt_archdefs.h"
#include "mt_mips.h"
#include "mt_sysdefs.h"
#include "mt_cachedefs.h"


uint8   sys_64bit;

uint8   sys_l2cache_enabled;

// Use default values so that it will not be zero-ed during init
uint32 sys_icache_linesize = 0xFFFF;
uint32 sys_icache_lines = 0xFFFF;
uint32 sys_dcache_linesize = 0xFFFF;
uint32 sys_dcache_lines = 0xFFFF;

#ifdef SW_CACHE_CONFIG


#define BPW_SETTINGS	8
static uint32 bpw[BPW_SETTINGS];

#define ASSOC_SETTINGS	8
static uint32 assoc[ASSOC_SETTINGS];

/************************************************************************
 *
 *                          sys_cpu_config
 *  Description :
 *  -------------
 *
 *  Configure CPU cache settings
 *
 *
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void sys_cpu_config( uint8 icache, uint8 dcache, 
			uint32 i_bpw, uint32 i_assoc, uint32 d_bpw, uint32 d_assoc )
{
    uint32 l, s, config1_reset;
  
    config1_reset = sys_cp0_read32( R_C0_Config, R_C0_SelConfig1 );

    if( icache )
    {
	  if( i_bpw == 0 )
	  {
	        l = 0;
	        s = (config1_reset & M_Config1IS) >> S_Config1IS;
	  }
	  else
	  {
	        l = (config1_reset & M_Config1IL) >> S_Config1IL;

	        for( s = 0;
	             CACHE_CALC_SPW(s) != i_bpw / CACHE_CALC_LS(l);
	             s++ );
	  }
      sys_cpu_icache_config( s, l, i_assoc - 1, config1_reset );
    }

    if( dcache )
    {
	  if( d_bpw == 0 )
	  {
	        l = 0;
	        s = (config1_reset & M_Config1DS) >> S_Config1DS;
	  }
	  else
	  {
	        l = (config1_reset & M_Config1DL) >> S_Config1DL;

	        for( s = 0;
	             CACHE_CALC_SPW(s) != d_bpw / CACHE_CALC_LS(l);
	             s++ );
	  }

      sys_cpu_dcache_config( s, l, d_assoc - 1, config1_reset );
    }
}


/************************************************************************
 *                          sys_cpu_cache_bpw
 *
 * parameter icache - TRUE -> icache, FALSE -> dcache
 *
 ************************************************************************/
void sys_cpu_cache_bpw( uint8 icache, t_sys_array  *sys_array )
{
    uint32 spw, spw_min, linesize;

    sys_array->array = bpw;

    /* Calc number of sets (lines) per way and determine linesize */      
    if( icache )
    {
        spw      = sys_icache_lines / sys_icache_assoc;
        linesize = sys_icache_linesize;
    }
    else
    {
        spw      = sys_dcache_lines / sys_dcache_assoc;
        linesize = sys_dcache_linesize;
    }

    spw_min = 64;  /* Minimum 64 sets per way */

 /* Init count of valid settings */
    sys_array->count = 0;

	 /*  Sets per way may be (64),128,256,512,1024,2048,4096.
	  *  Available spw are calculated by starting with the
	  *  hardware default and stepping down.
	  *  The check on count should not be necessary.
	  *  64 sets per way is not allowed for 5Kc/5Kf.
	  */
	while( (spw >= spw_min) && (sys_array->count < BPW_SETTINGS - 1) )
   	{
	     bpw[sys_array->count] = spw * linesize;  /* Calc bytes per way */
	     sys_array->count++;
	     spw >>= 1;
    }

	 /* 0 bytes per way is also possible by setting line size to 0 */
	bpw[sys_array->count] = 0;
	sys_array->count++;

}


/************************************************************************
 *                          sys_cpu_cache_assoc
 * parameter icache - TRUE -> icache, FALSE -> dcache
 *
 ************************************************************************/
void sys_cpu_cache_assoc( uint8 icache,  t_sys_array  *sys_array )
{
    uint32 max, i, index;
    
    sys_array->array = assoc;

    /* Determine max associativity */      
    max = icache ? sys_icache_assoc : sys_dcache_assoc;

	for( index = 0, i = MIN( ASSOC_SETTINGS, max );
	     index < MIN( ASSOC_SETTINGS, max );
	     index++, i-- )
    {
	    assoc[index] = i;
	}
   
	sys_array->count = max;

}
#ifdef SDL_IGNORE_UNUSED_FUNCTION
/************************************************************************
 *
 *                          validate_cache
 *  Description :
 *  -------------
 *
 *  Validate cache setting
 *
 * 	parameter icache - TRUE -> icache, FALSE -> dcache
 *
 *  Return values :
 *  ---------------
 *
 *  TRUE -> OK, FALSE -> Failed
 *
 ************************************************************************/
static uint8 validate_cache( uint8 icache, uint32 bpw, uint32 assoc )
{
    t_sys_array sys_array;
    uint32	i;

    sys_cpu_cache_bpw( icache, &sys_array );

    for( i = 0; 
	     ( i < sys_array.count ) &&
	     ( bpw != sys_array.array[i] );
	 	i++ );
    
    if( i == sys_array.count ) return FALSE;

    sys_cpu_cache_assoc( icache, &sys_array );

    for( i = 0; 
	     ( i < sys_array.count ) &&
	     ( assoc != sys_array.array[i] );
	 i++ );
    
    if( i == sys_array.count ) return FALSE;

    return TRUE;
}


/************************************************************************
 *
 *                          validate_cpuconfig
 *  Description :
 *  -------------
 *
 *  Validate and decode cpuconfig environment variable of format :
 *
 *  Return values :
 *  ---------------
 *
 *  TRUE -> OK, FALSE -> Failed
 *
 ************************************************************************/
uint8 validate_cpuconfig(void)
{
    uint32 i_bpw, i_assoc, d_bpw, d_assoc;
    
/*
	sys_icache_lines = 1024;
	sys_icache_linesize = 16;
	sys_icache_assoc = 4;
	sys_dcache_lines = 512;
	sys_dcache_linesize = 16;
	sys_dcache_assoc = 4;
	i_bpw   = 4096; 
	i_assoc = 4;
	d_bpw   = 2048; 
	d_assoc = 4;
*/
	i_bpw   = sys_icache_lines/sys_icache_assoc*sys_icache_linesize; 
	i_assoc = sys_icache_assoc;
	d_bpw   = sys_dcache_lines/sys_dcache_assoc*sys_dcache_linesize; 
	d_assoc = sys_dcache_assoc;

	sys_cpu_config( TRUE, TRUE, i_bpw, i_assoc, d_bpw, d_assoc );

	if( !validate_cache( TRUE, i_bpw, i_assoc ) )
		return FALSE;

	if( !validate_cache( FALSE, d_bpw, d_assoc ) )
		return FALSE;
        
	return TRUE;
}

#endif // SDL_IGNORE_UNUSED_FUNCTION
#endif /* SW_CACHE_CONFIG */


/************************************************************************
 *
 *                          sys_dcache_flush_all
 *  Description :
 *  -------------
 *
 *  Flush entire DCACHE.
 *
 *
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void sys_dcache_flush_all( void )
{
    uint32 line;
    uint32 addr;

	sys_dcache_get_parameters();

    for( line = 0, addr = KSEG0(0);  line < sys_dcache_lines; 
	 	line++, addr += sys_dcache_linesize )
    {
        sys_dcache_flush_index( addr );
    }

}  


/************************************************************************
 *
 *                          sys_icache_invalidate_all
 *  Description :
 *  -------------
 *
 *  Invalidate entire ICACHE.
 *
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void sys_icache_invalidate_all( void )
{
    uint32 line;
    uint32 addr;

	sys_icache_get_parameters();


    for( line = 0, addr = KSEG0(0); line < sys_icache_lines; 
         line++, addr += sys_icache_linesize )
    {
        sys_icache_invalidate_index( addr );
    }

	// We saw that for some reason, this function enables the interrupts, and we can get an interrupt while in assert.
    //sys_flush_pipeline();
}


/************************************************************************
 *
 *                          sys_flush_caches
 *  Description :
 *  -------------
 *
 *  First flush entire DCACHE, then invalidate entire ICACHE
 *
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void sys_flush_caches( void )
{
    sys_dcache_flush_all();
    sys_icache_invalidate_all();
}


/************************************************************************
 *
 *                          sys_flush_cache_line
 *  Description :
 *  -------------
 *
 *  First flush DCACHE line, then invalidate ICACHE line
 *
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
void sys_flush_cache_line( void *addr )
{
    addr = (void *)((uint32)addr & ~0x3);

    sys_dcache_flush_addr( (uint32)addr );
    sys_icache_invalidate_addr( (uint32)addr, TRUE );
}


