
/************************************************************************
 *
 *  mt_cacheconf.s
 *
 *  cache configuration functions
 *
 ************************************************************************/


/************************************************************************
 *  Include files
 ************************************************************************/
#ifdef SW_CACHE_CONFIG

#include "mt_sysdefs.h"
#include "mt_mips.h"

	.set noreorder
			
/************************************************************************
 *
 *                          sys_cpu_icache_config
 *  Description :
 *  -------------
 *
 *  Configure icache
 *
 *  a0 holds the requested sets per way field
 *  a1 holds the requested line size (0 or hardware default) field
 *  a2 holds the requested associativity field
 *  a3 holds the reset value of CONFIG1
 *	
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
LEAF( sys_cpu_icache_config )

	/* Reserve space on stack */
	addiu	sp, -8*4

	/* Store return address, s0..s4 */
	sw	ra, 0(sp)
	sw	s0, 4(sp)
	sw	s1, 8(sp)
	sw	s2, 12(sp)
	sw	s3, 16(sp)
	sw	s4, 20(sp)
	sw	s5, 24(sp)
	sw	s6, 28(sp)

	/* Copy registers to preserved registers */
	move	s0, a0
	move	s1, a1
	move	s2, a2
	move	s3, a3
	MFC0(   s4, C0_Status )
	MFC0(   s5, C0_Config )
	MFC0_SEL_OPCODE( 22, R_C0_Config1, R_C0_SelConfig1 )     /* s6 = $22 */

	/* Disable interrupts */
	li	t0, ~M_StatusIE
	and	t0, s4, t0
	MTC0(   t0, C0_Status )
	
	/* Shift to uncached */
	la	t0, sys_cpu_icache_config_uncached
	KSEG1A( t0)
	jr	t0
	nop

sys_cpu_icache_config_uncached :
			       
	/* Enable write access to config1 */
	li	t0, C0_CONFIG_WC_BIT
	or	t0, s5, t0
	MTC0(	t0, C0_Config )

	/* Set Icache settings to reset values */
	li	t0, (M_Config1IS | M_Config1IL | M_Config1IA)
	and	t0, s3, t0
	li	t1, ~(M_Config1IS | M_Config1IL | M_Config1IA)
	and	t1, s6, t1
	or	t1, t0

	/* Write CONFIG1 register */
	MTC0_SEL_OPCODE( 9, R_C0_Config1, R_C0_SelConfig1 )   /* t1 = $9 */

	/* Empty pipeline */
	nop; nop; nop; nop; nop; nop

	/* Prepare for initialising cache */
	
	/* a1 := line size (bytes) */
	la      a1, sys_icache_linesize
	lw	a1, 0(a1)

	/* a0 := cache size (bytes) */
	la      a0, sys_icache_lines
	lw	a0, 0(a0)
	multu	a0, a1
	mflo    a0

	/* Initialise */
	la	t0, sys_init_icache
	KSEG1A( t0)		/* Make it uncached */
	jal	t0
	nop

	/* Set Icache settings to requested values */
	li	t0, ~(M_Config1IS | M_Config1IL | M_Config1IA)
	and	t0, s6, t0

	li	t1, S_Config1IS
	sllv	t1, s0, t1
	or	t0, t1

	li	t1, S_Config1IL
	sllv	t1, s1, t1
	or	t0, t1

	li	t1, S_Config1IA
	sllv	t1, s2, t1
	or	t0, t1

	/* Write CONFIG1 register */
	MTC0_SEL_OPCODE( 8, R_C0_Config1, R_C0_SelConfig1 )    /* t0 = $8 */

	/* Disable write access to config1 */
	MTC0(	s5, C0_Config )

	/* Empty pipeline */
	nop; nop; nop; nop; nop; nop

	/* Restore CP0 STATUS */
	MTC0(   s4, C0_Status )

	/* Return */
	lw	ra, 0(sp)
	lw	s0, 4(sp)
	lw	s1, 8(sp)
	lw	s2, 12(sp)
	lw	s3, 16(sp)
	lw	s4, 20(sp)
	lw	s5, 24(sp)
	lw	s6, 28(sp)

	jr	ra
	addiu	sp, 8*4

END( sys_cpu_icache_config )


		
/************************************************************************
 *
 *                          sys_cpu_dcache_config
 *  Description :
 *  -------------
 *
 *  Configure dcache
 *
 *  a0 holds the requested sets per way field
 *  a1 holds the requested line size (0 or hardware default) field
 *  a2 holds the requested associativity field
 *  a3 holds the reset value of CONFIG1
 *	
 *  Return values :
 *  ---------------
 *
 *  None
 *
 ************************************************************************/
LEAF( sys_cpu_dcache_config )

	/* Reserve space on stack */
	addiu	sp, -8*4

	sw	ra, 0(sp)
	sw	s0, 4(sp)
	sw	s1, 8(sp)
	sw	s2, 12(sp)
	sw	s3, 16(sp)
	sw	s4, 20(sp)
	sw	s5, 24(sp)
	sw	s6, 28(sp)

	/* Copy registers to preserved registers */
	move	s0, a0
	move	s1, a1
	move	s2, a2
	move	s3, a3
	MFC0(   s4, C0_Status )
	MFC0(   s5, C0_Config )
	MFC0_SEL_OPCODE( 22, R_C0_Config1, R_C0_SelConfig1 )     /* s6 = $22 */
	
	/* Disable interrupts */
	li	t0, ~M_StatusIE
	and	t0, s4, t0
	MTC0(   t0, C0_Status )
		
	/* Flush D-Cache */
	addiu	sp, -4*4
	la      t0, sys_dcache_flush_all
	jal	t0
	nop
	addiu	sp, 4*4

	/* Shift to uncached */
	la	t0, sys_cpu_dcache_config_uncached
	KSEG1A( t0)
	jr	t0
	nop

sys_cpu_dcache_config_uncached :

	/* Enable write access to config1 */
	li	t0, C0_CONFIG_WC_BIT
	or	t0, s5, t0
	MTC0(	t0, C0_Config )

	/* Set Dcache settings to reset values */
	li	t0, (M_Config1DS | M_Config1DL | M_Config1DA)
	and	t1, s3, t0
	li	t0, ~(M_Config1DS | M_Config1DL | M_Config1DA)
	and	t2, s6, t0
	or	t2, t1

	/* Write CONFIG1 register */
	MTC0_SEL_OPCODE( 10, R_C0_Config1, R_C0_SelConfig1 )  /* t2 = $10 */

	/* Empty pipeline */
	nop; nop; nop; nop; nop; nop

	/* Prepare for initialising cache */

	/* a2 := processor type */
	la	a2, sys_processor
	lw	a2, 0(a2)

	/* a1 := line size (bytes) */
	la      a1, sys_dcache_linesize
	lw	a1, 0(a1)

	/* a0 := cache size (bytes) */
	la      a0, sys_dcache_lines
	lw	a0, 0(a0)
	multu	a0, a1
	mflo    a0

	/* Initialise */
	la	t0, sys_init_dcache
	KSEG1A( t0)		/* Make it uncached */
	jal	t0
	nop
	
	/* Set Dcache settings to requested values */
	li	t0, ~(M_Config1DS | M_Config1DL | M_Config1DA)
	and	t0, s6, t0

	li	t1, S_Config1DS
	sllv	t1, s0, t1
	or	t0, t1

	li	t1, S_Config1DL
	sllv	t1, s1, t1
	or	t0, t1

	li	t1, S_Config1DA
	sllv	t1, s2, t1
	or	t0, t1

	/* Write CONFIG1 register */
	MTC0_SEL_OPCODE( 8, R_C0_Config1, R_C0_SelConfig1 )   /* t0 = $8 */

	/* Disable write access to config1 */
	MTC0(	s5, C0_Config )

	/* Empty pipeline */
	nop; nop; nop; nop; nop; nop

	/* Restore CP0 STATUS */
	MTC0(   s4, C0_Status )

	/* Return */
	lw	ra, 0(sp)
	lw	s0, 4(sp)
	lw	s1, 8(sp)
	lw	s2, 12(sp)
	lw	s3, 16(sp)
	lw	s4, 20(sp)
	lw	s5, 24(sp)
	lw	s6, 28(sp)

	jr	ra
	addiu	sp, 8*4
	
END( sys_cpu_dcache_config )
		

/************************************************************************
 *
 *                          sys_cpu_configurability
 *  Description :
 *  -------------
 *
 *  Determine cpu configurability
 *
 *  This feature is present specifically to support configuration
 *  testing of the core in a lead vehicle, and is not supported
 *  in any other environment.  Attempting to use this feature
 *  outside of the scope of a lead vehicle is a violation of the
 *  MIPS Architecture, and may cause unpredictable operation of
 *  the processor.
 *
 *  Return values :
 *  ---------------
 *
 *  v0 = mask with the following bit definitions :
 *
 *  Bit 0 : Set if MMU may be configured (TLD -> Fixed)
 *  Bit 1 : Set if cache may be downsized
 *
 ************************************************************************/
LEAF( sys_cpu_configurability )

	/* Reserve space on stack */
	addiu	sp, -2*4

	/* Store return address */
	sw	ra, 0(sp)

	/* Disable interrupts (store old value of CP0 STATUS) */
	MFC0(   t0, C0_Status )
	sw	t0, 4(sp)
	li	t1, ~M_StatusIE
	and	t1, t0, t1
	MTC0(   t1, C0_Status )
		
	/* Shift to uncached */
	la	t0, sys_cpu_configurability_uncached
	KSEG1A( t0)
	jr	t0
	nop

sys_cpu_configurability_uncached:

	/* Set default = Not configurable */
	move    v0, zero

	/* Check that Config0[WC] bit is cleared */
	MFC0(   t1, C0_Config )
	li	t2, C0_CONFIG_WC_BIT
	and	t3, t1, t2
	bne	t3, zero, done
	nop

	/* Set Config0[WC] bit */
	or      t1, t2
	MTC0(   t1, C0_Config )

	/* Check Config0[8] : MMU configurability */
	li	t2, (1 << 8)
	and	t3, t1, t2
	bne	t3, zero, 1f
	nop
	/*  Config0[8] == 0 => TLB.
	 *  Try to set bit and read it back.
	 */
	or	t3, t1, t2
	MTC0(   t3, C0_Config )
	MFC0(   t3, C0_Config )
	and	t3, t3, t2
	beq	t3, zero, 1f
	nop
	/*  Bit was set, so MMU may be configured.
	 *  Clear bit and set mask.
	 */	
	MTC0(   t1, C0_Config )
	ori	v0, 1

1:
	/* Determine cache configurability
	 *
	 * Read CONFIG1 register, which holds implementation data
	 * t9 = $25
	 */
	MFC0_SEL_OPCODE( 25, R_C0_Config1, R_C0_SelConfig1 )
#define config1	t9

	/*  Check IL (I-cache line size) field.
	 *  If 0, there is no (I)cache.
	 */
	li      t2, M_Config1IL
	and	t3, t9, t2
	beq	t3, zero, done
	nop

	/* Try to clear IL and read it back */
	li      t2, ~M_Config1IL
	and	t3, t9, t2
	MTC0_SEL_OPCODE( 11, R_C0_Config1, R_C0_SelConfig1 )  /* t3 = $11 */
	MFC0_SEL_OPCODE( 11, R_C0_Config1, R_C0_SelConfig1 )     
	li	t2, M_Config1IL
	and	t3, t2
	bne	t3, zero, done
	nop
	/*  IL was cleared, so CACHE may be configured.
	 *  Restore setting and set mask.
	 */
	MTC0_SEL_OPCODE( 25, R_C0_Config1, R_C0_SelConfig1 )
	ori     v0, 2

done:
	/* Clear WC bit */
	li	t2, ~C0_CONFIG_WC_BIT	
	and	t1, t2
	MTC0(   t1, C0_Config )

	/***** Reinitialise I-cache *****/

	/* a1 := line size (bytes) */
	la      a1, sys_icache_linesize
	lw	a1, 0(a1)

	/* a0 := cache size (bytes) */
	la      a0, sys_icache_lines
	lw	a0, 0(a0)
	multu	a0, a1
	mflo    a0

	/* Initialise */
	la	t0, sys_init_icache
	KSEG1A( t0)		/* Make it uncached */
	jal	t0
	nop
	
	/* Restore CP0 STATUS */
	lw	t0, 4(sp)
	MTC0(   t0, C0_Status )

	/* Return */
	lw	ra, 0(sp)
	jr	ra
	addiu	sp, 2*4

END( sys_cpu_configurability )
	
#endif /* SW_CACHE_CONFIG */

