
/************************************************************************
 *
 *  mt_cacheinit.s
 *
 *  Cache functions
 *
 *
 ************************************************************************/


/************************************************************************
 *  Include files
 ************************************************************************/

#include "mt_sysdefs.h"
#include "mt_mips.h"
		
	.set noreorder
	
/************************************************************************
 *
 *                          sys_init_cache
 *  Description :
 *  -------------
 *
 *  Invalidate I and D caches
 *
 *	
 *  Return values :
 *  ---------------
 *
 *  Always 0
 *
 ************************************************************************/

LEAF(sys_init_cache)

#define RA					t4
#define l2cache_size		t1
#define l2cache_linesize	t0
#define icache_size			t3
#define icache_linesize		t2
#define dcache_size			t1
#define dcache_linesize		t0

	move	RA, ra

	/* Initialise instruction cache */	
	li	a0, 0x4000
	li	a1, 0x10
	bal	sys_init_icache
	nop
	
	/* Initialise data cache */
/*
	li	a0, 0x2000
	li	a1, 0x10
	bal	sys_init_dcache
	move	a2, k1
*/
	/* Done */
	jr	RA
	move	v0, zero
//	nop


END(sys_init_cache)


/************************************************************************
 *
 *                          sys_init_icache
 *
 *  Description :
 *  -------------
 *  Invalidate entire ICACHE
 *
 *  Inputs : a0 = cache size (bytes)
 *           a1 = line size  (bytes)
 *	
 *  Return values :
 *  ---------------
 *  None
 *
 ************************************************************************/
/*
	.section ".base_inst", "ax"
	.globl	sys_init_icache
	.ent	sys_init_icache

sys_init_icache:
*/

LEAF( sys_init_icache )
	
#ifdef NO_CACHE

	jr	ra
	nop

#else
		
    /* Clear TagLo and  TagHi. */
    
//	MTC0(   zero, C0_TagLo )
//	MTC0(   zero, C0_TagHi )
	
	beq   a0, zero, 2f		 /* just in case cache size = 0 */

	/* Calc an address that will correspond to the first cache line */
	li	a2, KSEG0BASE

	/* Calc an address that will correspond to the last cache line  */
	addu	a3, a2, a0
	subu    a3, a1

	/* Loop through all lines, invalidating each of them */
1:	
SET_MIPS3()
//	cache	ICACHE_INDEX_STORE_TAG, 0(a2)	/* clear tag */
	cache	ICACHE_INDEX_INVALIDATE, 0(a2)
SET_MIPS0()
	bne	a2, a3, 1b
	addu	a2, a1
2:
	jr	ra
	nop

#endif
		
END( sys_init_icache )
		

/************************************************************************
 *
 *                          sys_init_dcache
 *
 *  Description :
 *  -------------
 *  Invalidate entire ICACHE
 *
 *  Inputs : a0 = cache size (bytes)
 *           a1 = line size  (bytes)
 *           a2 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  None
 *
 ************************************************************************/

LEAF( sys_init_dcache )

#ifdef NO_CACHE

	jr	ra
	nop

#else

//	MTC0( zero, C0_TagLo )
//	MTC0( zero, C0_TagHi )

	beq   a0, zero, 2f		 /* just in case cache size = 0 */

	/* Calc an address that will correspond to the first cache line */
	li	a2, KSEG0BASE

	/* Calc an address that will correspond to the last cache line  */
	addu	a3, a2, a0
	subu    a3, a1

	/* Loop through all lines, invalidating each of them */
1:	
SET_MIPS3()
//	cache	DCACHE_INDEX_STORE_TAG, 0(a2)	/* clear tag */
	cache	DCACHE_INDEX_WRITEBACK_INVALIDATE, 0(a2)
SET_MIPS0()
	bne	a2, a3, 1b
	addu	a2, a1
2:
	jr	ra
	nop

#endif
		
END( sys_init_dcache )


#ifdef SW_CACHE_CONFIG

.section ".text", "ax"

LEAF(sys_init_cache_org)

	/**** Determine cache settings ****/

#define RA					t4
#define l2cache_size		t1
#define l2cache_linesize	t0
#define icache_size			t3
#define icache_linesize		t2
#define dcache_size			t1
#define dcache_linesize		t0

	move	RA, ra

	/* Determine processor */
	MFC0(   k1, C0_PRId)
	li	t0, M_PRIdImp | M_PRIdCoID
	and     k1, t0

	move	a0, k1

        /* L2 cache */
	bal     sys_determine_l2cache_avail_flash
	nop
	beq	v1, zero, 1f
	nop

	bal	sys_determine_l2cache_linesize_flash
	nop
	move	l2cache_linesize, v0

	bal	sys_determine_l2cache_lines_flash
	nop
	multu	l2cache_linesize, v0
	mflo	l2cache_size

	beq	l2cache_size, zero, 1f
	nop

	/* Initialise L2 */
	move	a0, l2cache_size
	move	a1, l2cache_linesize
	move	a2, k1
	bal	sys_init_l2cache_flash
	nop
1:
	/* L1 cache */
	
	bal	sys_determine_icache_linesize_flash
	nop
	move	icache_linesize, v0

	bal	sys_determine_icache_lines_flash
	nop
	multu	icache_linesize, v0
	mflo	icache_size

	bal	sys_determine_dcache_linesize_flash
	nop
	move	dcache_linesize, v0

	bal	sys_determine_dcache_lines_flash
	nop
	multu	dcache_linesize, v0
	mflo	dcache_size

	beq	icache_size, zero, 1f
	nop
	beq	dcache_size, zero, 1f
	nop

	/* Initialise instruction cache */
	move	a0, icache_size
	move	a1, icache_linesize
	bal	sys_init_icache
	nop
	
	/* Initialise data cache */
	move	a0, dcache_size
	move	a1, dcache_linesize
	bal	sys_init_dcache
	move	a2, k1

	/* Done */
	jr	RA
	nop
/*	move	v0, zero */

	/* D-cache or I-cache has size 0,  set CPU uncached */
1:
	MFC0(   v0, C0_Config )
	and	v0, ~M_ConfigK0
	or	v0, K_CacheAttrU
	MTC0(   v0, C0_Config )

	/* Done */
	jr	RA
	nop
/*	move	v0, zero */

END(sys_init_cache_org)



/************************************************************************
 *
 *                          cache_params
 *  Description :
 *  -------------
 *
 *  Determine cache info and store it 
 *
 *	
 *  Return values :
 *  ---------------
 *
 *  Always 0
 *
 ************************************************************************/

LEAF(cache_params)

	/* Reserve space on stack */
	addiu	sp, -8*4

	/* Store return address */
	sw	ra, 0(sp)

	/* Determine processor */
	MFC0(   k1, C0_PRId)
	li	t0, M_PRIdImp | M_PRIdCoID
	and     k1, t0

	/* Redetermine cache info and store it */
	
	move	a0, k1

	/* L1 Cache */
	jal	sys_determine_icache_assoc_flash
	nop
	la	t0, sys_icache_assoc
	sw	v0, 0(t0)
	
	jal	sys_determine_icache_lines_flash
	nop
	la	t0, sys_icache_lines
	sw	v0, 0(t0)

	jal	sys_determine_icache_linesize_flash
	nop
	la	t0, sys_icache_linesize
	sw	v0, 0(t0)

	jal	sys_determine_dcache_assoc_flash
	nop
	la	t0, sys_dcache_assoc
	sw	v0, 0(t0)
				
	jal	sys_determine_dcache_lines_flash
	nop
	la	t0, sys_dcache_lines
	sw	v0, 0(t0)
		
	jal	sys_determine_dcache_linesize_flash
	nop
	la	t0, sys_dcache_linesize
	sw	v0, 0(t0)

	/* Return */
	lw	ra, 0(sp)

	jr	ra
	addiu	sp, 8*4

END(cache_params)

/************************************************************************
 *
 *                          sys_determine_l2cache_linesize_flash
 *
 *  Description :
 *  -------------
 *  Determine L2 CACHE linesize
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = L2 CACHE linesize in bytes
 *
 ************************************************************************/
LEAF(sys_determine_l2cache_linesize_flash)	

	move	v0, zero

	li	t9, MIPS_25Kf
	beq	a0, t9, l2cache_linesize_25kf
	nop
	jr	ra
	nop

l2cache_linesize_25kf:

	/* Read CONFIG2 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config2, R_C0_SelConfig2 )

	/* L2 cache line size */
	li	t8, M_Config2SL
	and	t9, t8
	beq	t9, zero, 1f
	nop

	li	t8, S_Config2SL
	srl	t9, t8
	li	t8, 0x2
	sll	v0, t8, t9
1:
	jr	ra
	nop

END(sys_determine_l2cache_linesize_flash)	


/************************************************************************
 *
 *                          sys_determine_l2cache_lines_flash
 *
 *  Description :
 *  -------------
 *  Determine number of L2 CACHE lines
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = number of L2 CACHE lines
 *
 ************************************************************************/
LEAF(sys_determine_l2cache_lines_flash)	

	li	t9, MIPS_25Kf
	beq	a0, t9, l2cache_lines_25kf
	nop
	jr	ra
	move	v0, zero

l2cache_lines_25kf:

	/* Read CONFIG2 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config2, R_C0_SelConfig2 )

	/* S-cache lines
	 * Calculated as associativity * sets per way
	 */
	li	t8, M_Config2SA
	and	t8, t9
	srl	t8, S_Config2SA
	addiu	t8, 1				/* t8 = associativity	*/

	li	t7, M_Config2SS
	and	t7, t9
	srl	t7, S_Config2SS
	li	t9, 0x40
	sll	t7, t9, t7			/* t7 = sets per way	*/

	multu	t8, t7
	mflo    v0

	jr	ra
	nop

END(sys_determine_l2cache_lines_flash)


/************************************************************************
 *
 *                          sys_determine_l2cache_assoc_flash
 *
 *  Description :
 *  -------------
 *  Determine L2 CACHE associativity
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = L2 CACHE associativity
 *
 ************************************************************************/
LEAF(sys_determine_l2cache_assoc_flash)	

	li	t9, MIPS_25Kf
	beq	a0, t9, l2cache_assoc_25kf
	nop
	jr	ra
	move	v0, zero

l2cache_assoc_25kf:

	/* Read CONFIG2 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config2, R_C0_SelConfig2 )

	/* L2-cache associativity */
	li	t8, M_Config2SA
	and	t9, t8
	srl	v0, t9, S_Config2SA
	jr	ra
	addiu	v0, 1

END(sys_determine_l2cache_assoc_flash)


/************************************************************************
 *
 *                          sys_determine_l2cache_avail_flash
 *
 *  Description :
 *  -------------
 *  Determine whether L2 CACHE is available and enabled
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = 1 if L2 CACHE is available, otherwise 0.
 *  v1 = 1 if L2 CACHE is enabled,   otherwise 0.
 *
 ************************************************************************/
LEAF(sys_determine_l2cache_avail_flash)	

	li	t9, MIPS_25Kf
	beq	a0, t9, l2cache_avail_25kf
	nop

	move	v0, zero
	jr	ra
	move	v1, zero

l2cache_avail_25kf:

	/* Read CONFIG2 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config2, R_C0_SelConfig2 )

	/* L2 cache line size */
	li	t8, M_Config2SL
	and	t9, t8
	beq	t9, zero, 1f
	move	v0, zero
	li	v0, 1
1:	
	jr	ra
	move	v1, v0 /* L2 can't be disabled */

END(sys_determine_l2cache_avail_flash)	


/************************************************************************
 *
 *                          sys_determine_icache_linesize_flash
 *
 *  Description :
 *  -------------
 *  Determine ICACHE linesize
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = ICACHE linesize in bytes
 *
 ************************************************************************/
LEAF(sys_determine_icache_linesize_flash)	

	/* Check if it is a MIPS32/64 processor */	
	srl	t9, a0, S_PRIdCoID
	and     t9, M_PRIdCoID >> S_PRIdCoID
	li	t8, C0_PRID_COMP_NOT_MIPS32_64
	bne	t9, t8, icache_linesize_mips32
	nop
	/* Unknown CPU */
icache_linesize_zero:
	jr	ra
	move	v0, zero

icache_linesize_mips32:
	
	/* Read CONFIG1 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config1, R_C0_SelConfig1 )

#define config1	t9

	/* I-cache line size */
	li	t8, M_Config1IL
	and	t8, config1
	beq	t8, zero, icache_linesize_zero
	li	t7, S_Config1IL
	srl	t8, t7
	li	t7, 0x2
	sll	v0, t7, t8

	jr	ra
	nop

END(sys_determine_icache_linesize_flash)	
	


/************************************************************************
 *
 *                          sys_determine_icache_lines_flash
 *  Description :
 *  -------------
 *  Determine number of ICACHE lines
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = number of ICACHE lines
 *
 ************************************************************************/
LEAF(sys_determine_icache_lines_flash)	

	/* Check if it is a MIPS32/64 processor */	
	srl	t9, a0, S_PRIdCoID
	and     t9, M_PRIdCoID >> S_PRIdCoID
	li	t8, C0_PRID_COMP_NOT_MIPS32_64
	bne	t9, t8, icache_lines_mips32
	nop
	/* Unknown CPU */
icache_lines_zero:
	jr	ra
	move	v0, zero

icache_lines_mips32:
	
	/* Read CONFIG1 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config1, R_C0_SelConfig1 )

#define config1	t9

	/* I-cache lines
	 * Calculated as associativity * sets per way
	 */
	li	t8, M_Config1IA
	and	t8, config1
	li	t7, S_Config1IA
	srl	t8, t7
	addiu	t8,1				/* t8 = associativity	*/

	li	t7, M_Config1IS
	and	t7, config1
	li	t9, S_Config1IS
	srl	t7, t9
	li	t9, 0x40
	sll	t7, t9, t7			/* t7 = sets per way	*/

	multu	t8, t7
	mflo    v0

	jr	ra
	nop
	
END(sys_determine_icache_lines_flash)


/************************************************************************
 *
 *                          sys_determine_icache_assoc_flash
 *
 *  Description :
 *  -------------
 *  Determine ICACHE associativity
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = ICACHE associativity
 *
 ************************************************************************/
LEAF(sys_determine_icache_assoc_flash)	

	/* Check if it is a MIPS32/64 processor */	
	srl	t9, a0, S_PRIdCoID
	and     t9, M_PRIdCoID >> S_PRIdCoID
	li	t8, C0_PRID_COMP_NOT_MIPS32_64
	bne	t9, t8, icache_assoc_mips32
	nop
	/* Unknown CPU */
icache_assoc_zero:
	jr	ra
	move	v0, zero

icache_assoc_mips32:
	
	/* Read CONFIG1 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config1, R_C0_SelConfig1 )

#define config1	t9

	/* I-cache associativity */
	li	t8, M_Config1IA
	and	t8, config1
	li	t7, S_Config1IA
	srl	t8, t7
	addiu	v0,t8,1				/* t8 = associativity	*/
	jr	ra
	nop

END(sys_determine_icache_assoc_flash)


/************************************************************************
 *
 *                          sys_determine_dcache_linesize_flash
 *
 *  Description :
 *  -------------
 *  Determine DCACHE linesize
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = DCACHE linesize in bytes
 *
 ************************************************************************/
LEAF(sys_determine_dcache_linesize_flash)	

	/* Check if it is a MIPS32/64 processor */	
	srl	t9, a0, S_PRIdCoID
	and     t9, M_PRIdCoID >> S_PRIdCoID
	li	t8, C0_PRID_COMP_NOT_MIPS32_64
	bne	t9, t8, dcache_linesize_mips32
	nop
	
	/* Unknown CPU */
dcache_linesize_zero:
	jr	ra
	move	v0, zero

dcache_linesize_mips32:
	
	/* Read CONFIG1 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config1, R_C0_SelConfig1 )

#define config1	t9

	/* D-cache line size */
	li	t8, M_Config1DL
	and	t8, config1
	beq	t8, zero, dcache_linesize_zero
	li	t7, S_Config1DL
	srl	t8, t7
	li	t7, 0x2
	sll	v0, t7, t8

	jr	ra
	nop

END(sys_determine_dcache_linesize_flash)	


/************************************************************************
 *
 *                          sys_determine_dcache_lines_flash
 *
 *  Description :
 *  -------------
 *  Determine number of DCACHE lines
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = number of DCACHE lines
 *
 ************************************************************************/
LEAF(sys_determine_dcache_lines_flash)	

	/* Check if it is a MIPS32/64 processor */	
	srl	t9, a0, S_PRIdCoID
	and     t9, M_PRIdCoID >> S_PRIdCoID
	li	t8, C0_PRID_COMP_NOT_MIPS32_64
	bne	t9, t8, dcache_lines_mips32
	nop
	
	/* Unknown CPU */
dcache_lines_zero:
	jr	ra
	move	v0, zero

dcache_lines_mips32:
	
	/* Read CONFIG1 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config1, R_C0_SelConfig1 )

#define config1	t9

	/* D-cache lines
	 * Calculated as associativity * sets per way
	 */
	li	t8, M_Config1DA
	and	t8, config1
	li	t7, S_Config1DA
	srl	t8, t7
	addiu	t8,1				/* t8 = associativity	*/

	li	t7, M_Config1DS
	and	t7, config1
	li	t9, S_Config1DS
	srl	t7, t9
	li	t9, 0x40
	sll	t7, t9, t7			/* t7 = sets per way	*/

	multu	t8, t7
	mflo    v0

	jr	ra
	nop
	
END(sys_determine_dcache_lines_flash)


	
/************************************************************************
 *
 *                          sys_determine_dcache_assoc_flash
 *
 *  Description :
 *  -------------
 *  Determine DCACHE associativity
 *
 *  input : a0 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  v0 = DCACHE associativity
 *
 ************************************************************************/
LEAF(sys_determine_dcache_assoc_flash)	

	/* Check if it is a MIPS32/64 processor */	
	srl	t9, a0, S_PRIdCoID
	and     t9, M_PRIdCoID >> S_PRIdCoID
	li	t8, C0_PRID_COMP_NOT_MIPS32_64
	bne	t9, t8, dcache_assoc_mips32
	nop
	
	/* Unknown CPU */
dcache_assoc_zero:
	jr	ra
	move	v0, zero

dcache_assoc_mips32:
	
	/* Read CONFIG1 register, which holds implementation data */
	MFC0_SEL_OPCODE( R_t9, R_C0_Config1, R_C0_SelConfig1 )

#define config1	t9

	/* I-cache associativity */
	li	t8, M_Config1DA
	and	t8, config1
	li	t7, S_Config1DA
	srl	t8, t7
	addiu	v0,t8,1				/* t8 = associativity	*/
	jr	ra
	nop
	
END(sys_determine_dcache_assoc_flash)


/************************************************************************
 *
 *                          sys_init_l2cache_flash
 *
 *  Description :
 *  -------------
 *  Invalidate entire L2 CACHE
 *
 *  Inputs : a0 = cache size (bytes)
 *           a1 = line size  (bytes)
 *	     a2 = processor ID
 *	
 *  Return values :
 *  ---------------
 *  None
 *
 ************************************************************************/
LEAF( sys_init_l2cache_flash )

	li	t0, MIPS_25Kf
	beq	t0, a2, 1f
	nop

	MTC0(   zero, C0_TagLo )
	MTC0(   zero, C0_TagHi )

	b	2f
	nop
	
1:
	/* 25Kf */
	MTC0_SEL_OPCODE( R_zero, R_C0_L23TagLo, R_C0_SelL23TagLo )
	MTC0_SEL_OPCODE( R_zero, R_C0_L23TagHi, R_C0_SelL23TagHi )

2:		
	beq     a0, zero, 2f		 /* just in case cache size = 0 */

	/* Calc an address that will correspond to the first cache line */
	li	t0, KSEG0BASE
	addu	t1, t0, a0
	subu	t1, a1		/* t1 = last line */

	/* Loop through all lines, invalidating each of them */
1:	
SET_MIPS3()
	cache	SCACHE_INDEX_STORE_TAG, 0(t0)	/* clear tag */
SET_MIPS0()
	bne	t0, t1, 1b
	addu	t0, a1
2:	
	jr	ra
	nop

END( sys_init_l2cache_flash )

	
/************************************************************************	
 *
 *                          sys_cpu_type
 *  Description :
 *  -------------
 *
 *  Determine whether we have :
 *
 *  a) 32 or 64 bit CPU
 *  b) MIPS32/MIPS64 CPU or other type.
 *  c) Support for MIPS16e ASE or not.
 *  d) Support for EJTAG or not.
 *  e) Support for FPU or not.
 *  f) Support for hardware cache coherency or not.
 *
 *  Parameters :
 *  ------------
 *
 *  None

 *  Return values :
 *  ---------------
 *
 *  Following global variables are set :
 *
 *  sys_64bit
 *  sys_mips32_64
 *  sys_mips16e
 *  sys_ejtag
 *  sys_arch_rev
 *  sys_fpu
 *  sys_cpu_cache_coherency
 *
 ************************************************************************/
LEAF(sys_cpu_type)

	/* Reserve space on stack */
	addiu	sp, -2*4

	/* Store return address */
	sw	ra, 0(sp)

	/* Determine processor */
	MFC0(   t0, C0_PRId)
	li	t1, M_PRIdImp | M_PRIdCoID
	and     t0, t1

	/* MIPS 20Kc and 25Kf have support for hw cache coherency */
	li	t1, MIPS_20Kc
	beq	t1, t0, 1f
	li	t2, 1
	li	t1, MIPS_25Kf
	beq	t1, t0, 1f
	nop
	move	t2, zero
1:
	la	t1, sys_cpu_cache_coherency
	sb	t2, 0(t1)
		
	/* Check if it is a MIPS32/64 processor */	
	li      t1, M_PRIdCoID
	and	t1, t0
	srl	t1, S_PRIdCoID
	li	t2, C0_PRID_COMP_NOT_MIPS32_64
	bne	t1, t2, mips32_64
	nop

	/* Not a MIPS32/64 processor */
	la      t1, sys_mips32_64
	sb	zero, 0(t1)
	la	t1, sys_mips16e
	sb	zero, 0(t1)
	la	t1, sys_ejtag
	sb	zero, 0(t1)
	la	t1, sys_arch_rev
	sb	zero, 0(t1)
	la	t1, sys_smallpage_support
	sb	zero, 0(t1)
	
	li	t1, QED_RM52XX
	beq	t1, t0, 1f
	li	t2, 1
	li	t1, QED_RM70XX
	beq	t1, t0, 1f
	nop
	move	t2, zero	/* 32 bit, no FPU by default */
1:
	la	t0, sys_64bit
	sb	t2, 0(t0)
	b	mips_fir_in_t1
	move	t1,t2

mips32_64:

	/* MIPS32/MIPS64 processor */
	la      t1, sys_mips32_64
	li	t0, 1
	sb	t0, 0(t1)

	MFC0(   t0, C0_Config )

	/* Determine architecture revision level */
	li	t1, M_ConfigAR
	and	t1, t0
	srl	t1, S_ConfigAR
	la	t2, sys_arch_rev
	sb	t1, 0(t2)
	
	/* Determine 32/64 based on AT field of CONFIG register */	
	li	t1, M_ConfigAT
	and	t0, t1
	srl	t0, S_ConfigAT
	li	t1, K_ConfigAT_MIPS32
	beq	t0, t1, 1f
	move	t0, zero
	li	t0, 1
1:
	la	t1, sys_64bit
	sb	t0, 0(t1)

	/* Read CONFIG1 register, which holds implementation data
	 * t0 = $8
	 */
	MFC0_SEL_OPCODE( 8, R_C0_Config1, R_C0_SelConfig1 )     

	/* Determine mips16e, ejtag, fpu support based on CONFIG1 */
	srl     t1, t0, S_Config1CA
	and	t1, 1
	la	t2, sys_mips16e
	sb	t1, 0(t2)

	srl     t1, t0, S_Config1EP
	and	t1, 1
	la	t2, sys_ejtag
	sb	t1, 0(t2)

	/* Determine small page support (requires CONFIG3 to be there) */
	sll	t1, t0, 31 - S_Config1M
	bgez	t1, 1f
	move	t1, zero
	MFC0_SEL_OPCODE( 10 /* t2 */, R_C0_Config2, R_C0_SelConfig2 )     
	sll	t1, t2, 31 - S_Config2M
	bgez	t1, 1f
	move	t1, zero
	MFC0_SEL_OPCODE( 10 /* t2 */, R_C0_Config3, R_C0_SelConfig3 )     
	srl	t1, t2, S_Config3SP
	and	t1, 1
1:
	la	t2, sys_smallpage_support
	sb	t1, 0(t2)

	/* FPU */
	srl     t1, t0, S_Config1FP
	and	t1, 1

mips_fir_in_t1:
	bne	t1, zero, 2f
	nop

	/* FPU not implemented: Clear CP0.status.FR bit */
	MFC0(   t0, C0_Status)
	li	t1, ~M_StatusFR
	and	t1, t0
	MTC0(   t1, C0_Status)
	b	1f
	move	t1, zero

2:
	/* FPU implemented: Read CP1 fir value */
	MFC0(   t0, C0_Status)
	li	t1, M_StatusCU1
	or	t1, t0
	MTC0(   t1, C0_Status)

	cfc1    t1, $0

	/* restore status reg */
	MTC0(   t0, C0_Status)
	bne	t1,zero, 1f
	nop
	or	t1, 1		/* make sure that fir != 0 */
1:
	la	t2, sys_fpu
	sw	t1, 0(t2)

	/* Return */
	lw	ra, 0(sp)
   	jr ra
	nop

END(sys_cpu_type)

#endif /* SW_CACHE_CONFIG */
