/*****************************************************************************
#*    
#*   MODULE NAME:
#*       MT_OsInt.s
#*   DESCRIPTION: 
#*       Mips Operating system interrupt handling assembler module
#*       and BootLoader.
#*   FUNCTION LIST: 
#*       boot_code
#*       Init
#*       Interrupt handler
#*       Exception handler
#*       handler
#*   AUTHOR:  
#*       Ziv Avital
#*   DATE: 
#*       24/04/2000
#*   COPYRIGHT: 
#*       (C) Metalink Ltd.
#*       All rights are strictly reserved. Reproduction or divulgence in any 
#*       form whatsoever is not permitted without written authority from the 
#*       copyright owner. Issued by Metalink Transmission Devices Ltd in 
#*       Israel - 11/94.
#*   REVISION HISTORY: 
#*       MetalinkRevision: 
*****************************************************************************/

/*****************************************************************************
#*
#*        Library Function Include Files
#*
#*****************************************************************************/
.include "mt_osmip.hs"
#include "mt_build_config.h"

/*****************************************************************************
#* Function Name:    boot_code
#* Description: 
#*   Boot program
#*   The CPU start at address 0xbfc00000 (in ROM mode)
#*   this is minimal code to jump to KSEG0
#* Input Parameters: 
#*   none
#* Output Parameters: 
#*   none
#* Return Value: 
#*   none
#* Algorithm: 
#*
#* Revision: 
#*
#*****************************************************************************/

/*#    .section ".boot", .text          # > 0xbfc00000   (In remark since it gives warning in Multi 3.5)*/
    .set noreorder

boot_code:
.if SIMULATOR
    mtc0  zero,$16                  /*# Config the Config register to cashed KSEG0*/
    lui   k0, %hi(Init)           /*#jump to Init*/
    addi  k0, k0,%lo(Init) 
    jr    k0
    nop   
.endif
                  
/*****************************************************************************
#* Function Name:    Init
#* Description: 
#*   Continue the Boot program, this code is i  KSEG0 (0x80000000)
#*   Configure Cause & Status register and jump to main C function
#* Input Parameters: 
#*   none
#* Output Parameters: 
#*   none
#* Return Value: 
#*   none
#* Algorithm: 
#*
#* Revision: 
#*
#*****************************************************************************/
.section ".base_inst", "ax"
.extern main              
/*# .extern label_main*/

/*# for Version number*/
 
    j Init
    nop
    nop
    nop
    
Init:

/*##--------------------------------------------------------------------
## Set the IV bit at the cause register -
## the Interrupt vector will be at address 0x80000200
## and the Exceptions vector at 0x80000180
##--------------------------------------------------------------------*/
.if 0
    mfc0    t0,C0_CAUSE      
    lui     t1,0x0080        
    or      t0,t1,t0         
    lui     t1,0xffbf          #Clear bit 22 - WP
    ori     t1,0xfcff          #clear bits 8-9 - Softwre INT
    and     t0,t1,t0
    mtc0    t0,C0_CAUSE         

.else

/*#    li    a0, 0x4000
#    li    a1, 0x10
#    bal    sys_init_cache
#    nop*/

/*############################### 1 ################################*/
start:    
    /*# Initialize the register file*/
    or    $1,$0, $0
    or    $2,$0, $0
    or    $3,$0, $0
    or    $4,$0, $0
    or    $5,$0, $0
    or    $6,$0, $0
    or    $7,$0, $0
    or    $8,$0, $0
    or    $9,$0, $0
    or    $10,$0, $0
    or    $11,$0, $0
    or    $12,$0, $0
    or    $13,$0, $0
    or    $14,$0, $0
    or    $15,$0, $0
    or    $16,$0, $0
    or    $17,$0, $0
    or    $18,$0, $0
    or    $19,$0, $0
    or    $20,$0, $0
    or    $21,$0, $0
    or    $22,$0, $0
    or    $23,$0, $0
    or    $24,$0, $0
    or    $25,$0, $0
    or    $26,$0, $0
    or    $27,$0, $0
    or    $28,$0, $0
    or    $29,$0, $0
    or    $30,$0, $0
    or    $31,$0, $0

    /*
    # this commands will write a new jump instruction to address 0x80000180(exception handler)
    # at physical address 0x0 so when TLB Exception occure we will jump to general Exception Handler.
    # *(uint32 *)(0xa0000000) = 0x8000060

    # 0x8000060 -> jump command to 0x80000180
    */

    li $10, 0x8000060
    li $11, 0xa0000000
    sw $10, 0($11)

#if defined (ENET_INC_UMAC)

    
	mfc0  t0,$16,0	/* enable bit 19 C0_Config=$16,0    */
	li    t1,0x3    /*change cache coherency to write back with write allocation BITS 0:2 */
	or    t0,t1,t0
	mtc0  t0,$16,0
	#endif
    jal    clean_cache
    nop
    jal shadowFileInit
    nop

/*################################ 1 ################################

################################ 3 ################################*/

    /*# Disable watch exceptions*/
    mtc0    $0, $18

    /*# Clear Watch Status bits*/
    li      $11, 0x3
    mtc0    $11, $19

    /*# Clear WP bit to avoid watch exception upon user code entry
    # Clear IV bit - Interrupts go to general exception vector
    # Clear software interrupts*/
    mtc0    $0, $13

/*################################ 3 ################################

################################ 5 ################################*/
    /*# Clear Count register*/
    mtc0    $0, $9

    /*# Set compare to -1 to delay 1st count=compare
    # Also, clears timer interrupt*/
    li    $10, -1
    mtc0    $10, $11
/*################################ 5 ################################
##--------------------------------------------------------------------*/
.if 0
    mtc0    $0,$28
    lui     $3,0x0000
    ori     $4,$0,0x0010 
    ori     $5,$0,0x4000 
    ori     $6,$0,0x2000 

    /* Clean ICache */
icloop:
    cache   0,0x0000($3)
    addu    $3,$3,$4
    bne     $3,$5,icloop
    nop


    /* Clean DCache */
    lui     $3,0x0000
dcloop: 
    cache   0x9,0x0000($3)
    addu    $3,$3,$4
    bne     $3,$6,dcloop
    nop

/*# Dcache forced loading
# Avri G: this is good only for FPGA for multiple runing without reset
#    lui        $10,0x8600
#    ori        $11,$10,0x4000
#    ori        $12,$0,0x0004
# dcache_loop:
#    lw        $13,0x0000($10)
#    addu    $10,$10,$12
#    bne        $10,$11,dcache_loop
#    nop*/

.endif  #remove this clean cache
    lui     t0,0x0080
    mtc0    t0,C0_CAUSE

.endif
/*##--------------------------------------------------------------------
## Initialize the GP register to end of Data Segment
## and the SP to the end of STACK segment       Alex
##--------------------------------------------------------------------*/

    lui      $gp, (%hi(__gp))    #0x7fff
    addiu    $gp, $gp, %lo(__gp)

/*#    lui      $gp, (%hi(__ghsbegin_sbss))    #0x7fff
#    addiu    $gp, $gp, %lo(__ghsbegin_sbss)
#    addiu    $gp, $gp, 0x4000    # Add 32K to $gp (The linker need that for varibales location)
#    addiu    $gp, $gp, 0x4000*/

    lui      $sp, %hi(__ghsend_stack)    
    addiu    $sp, $sp, %lo(__ghsend_stack)

/*#    lui      $sp, %hi(__ghsbegin_stack)    
#    addiu    $sp, $sp, %lo(__ghsbegin_stack)*/

    lui     $6, 0xa600
    lw      $7, 0x0f04($6)
    li      $8, 0x00000001
    bne     $7, $8, nnext
    nop

    lui      $sp, %hi(__ghsend_stack2)    
    addiu    $sp, $sp, %lo(__ghsend_stack2)

/*#    lui      $sp, %hi(__ghsbegin_stack2)    
#    addiu    $sp, $sp, %lo(__ghsbegin_stack2)*/


/*##--------------------------------------------------------------------
## Configure the Status register:
## Enable all coprocessors, and clear the BEV bit so the
## interrupts will be at 0x80000x00 ,
## Unmask all interrupts but dont enable IE bit. Work in Kernel mode.
##--------------------------------------------------------------------*/
nnext:
    lui  t0,0xf000             
    ori  t0,0x0000  #<E.Z> changed from 0xff00 - for EIC          
    mtc0 t0,C0_SR             

/*##--------------------------------------------------------------------
## Select the vectore spacing:
## Spacing of 0 is equivalent to having no vectored interrupts and 
## interrupts fall to 0x80000200
## Minimum vector spacing is 0x20       
##--------------------------------------------------------------------*/
    lui  t0,0x0000             
    ori  t0,0x0020 #<o.s> changed from "ori  t0,0x0,0x0020" OR with 0x0 has no meaning    #<E.Z> changed from 0x0000 - for EIC 
    mtc0 t0,$12,1

/*##--------------------------------------------------------------------
## Jump to main C function
##--------------------------------------------------------------------*/
.if SIMULATOR
    j _main
    nop
.else

/*# Use jr in order to jump from non-cached to cached memory*/
    li $10, label_main
    jr $10
    nop
.endif

/***********************************************************************
#*Function name: shadowFileInit 
#*Description: This code is on address 0x80000600 it's part of intialization process,
#*               it's called immediately after Initialize normal shadow register file is completed, 
#*               first we set shadow     register file 1 as PSS - so we can access it 
#*                and then reset all registers using wrpgpr
#************************************************************************/

.section ".shadow_file_init", .text    
shadowFileInit:
.set noreorder
    li        k0,0x40
    nop
    mtc0    k0,$12,2 /*#SRSCtrl */
    nop                

.set reorder
/*#now set 0 to all shadow file GPRs*/
    wrpgpr  $0,$0
    wrpgpr  $1,$0
    wrpgpr  $2,$0
    wrpgpr  $3,$0
    wrpgpr  $4,$0
    wrpgpr  $5,$0
    wrpgpr  $6,$0
    wrpgpr  $7,$0
    wrpgpr  $8,$0
    wrpgpr  $9,$0
    wrpgpr  $10,$0
    wrpgpr  $11,$0
    wrpgpr  $12,$0
    wrpgpr  $13,$0
    wrpgpr  $14,$0
    wrpgpr  $15,$0
    wrpgpr  $16,$0
    wrpgpr  $17,$0
    wrpgpr  $18,$0
    wrpgpr  $19,$0
    wrpgpr  $20,$0
    wrpgpr  $21,$0
    wrpgpr  $22,$0
    wrpgpr  $23,$0
    wrpgpr  $24,$0
    wrpgpr  $25,$0
    wrpgpr  $26,$0
    wrpgpr  $27,$0
    wrpgpr  $28,$0
    wrpgpr  $29,$0
    wrpgpr  $30,$0
    wrpgpr  $31,$0
    mtc0    $0,$12,2 
    jr      ra /*#return */
    or      k0,$0, $0    /*##takes advantage of jump hazards(next instruction executed)*/
    

/*****************************************************************************
#* Function Name:    Interrupt Vector
#* Description: 
#*   This is the code in the hardware intterupt vector (0x8000200)
#*   it's seperated to upper mips and lower mips because we have different interrupts
#*   according to the interrupt we use primary or secondary register file and 
#*   save relevant registers.
#*     All references to unexpected interrupts will lead to assertion.
#*****************************************************************************/

    .section ".vector", .text    /*# Put this code in the Int vector > 0x80000200*/
    .set noreorder
    .extern InterruptManagerIsrHandlers /*this is the ISR table "C"*/
    
default_handler:
/*####<E.Z> all unmapped interrupts fall here so just return from interrupt#####*/
    eret
    nop


/*################### Gen 5 LMAC Interrupt Vector #################################*/
#if defined(ENET_INC_LMAC)
	.section ".vector_irq1", .text    /*# Put this code in the Int vector > 0x80000220*/
handler_01:
	li		k0,InterruptManagerIsrHandlers
	lw		k1,0x4(k0)
	mfc0	k0,C0_EPC
	j		shadowSet_handler
	nop

	.section ".vector_irq2", .text    /*# Put this code in the Int vector > 0x80000240*/
handler_02:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x8(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop


    .section ".vector_irq3", .text    /*# Put this code in the Int vector > 0x80000260*/
handler_03:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0xc(k0)
	mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq4", .text    /*# Put this code in the Int vector > 0x80000280*/
handler_04:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x10(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq5", .text    /*# Put this code in the Int vector > 0x800002A0*/
handler_05:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x14(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq6", .text    /*# Put this code in the Int vector > 0x800002C0*/
handler_06:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x18(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq7", .text    /*# Put this code in the Int vector > 0x800002E0*/
handler_07:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x1c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop



    .section ".vector_irq8", .text    /*# Put this code in the Int vector > 0x80000300*/
handler_08:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x20(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq9", .text    /*# Put this code in the Int vector > 0x80000320*/
handler_09:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x24(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq10", .text    /*# Put this code in the Int vector > 0x80000340*/
handler_10:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x28(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq11", .text    /*# Put this code in the Int vector > 0x80000360*/
handler_11:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x2c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq12", .text    /*# Put this code in the Int vector > 0x80000380*/
handler_12:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x30(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
   

    .section ".vector_irq13", .text    /*# Put this code in the Int vector > 0x800003A0*/
handler_13:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x34(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
handler_14:
    .section ".vector_irq14", .text    /*# Put this code in the Int vector > 0x800003C0*/
	li      k0,InterruptManagerIsrHandlers
    lw      k1,0x38(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    .section ".vector_irq15", .text    /*# Put this code in the Int vector > 0x800003E0*/
handler_15:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x3c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq16", .text    /*# Put this code in the Int vector > 0x80000400*/
handler_16:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x40(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

handler_17:
    .section ".vector_irq17", .text    /*# Put this code in the Int vector > 0x80000420*/
unused17_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x44(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq18", .text    /*# Put this code in the Int vector > 0x80000440*/
handler_18:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x48(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq19", .text    /*# Put this code in the Int vector > 0x80000460*/
handler_19:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x4c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq20", .text    /*# Put this code in the Int vector > 0x80000480*/
handler_20:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x50(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq21", .text    /*# Put this code in the Int vector > 0x800004A0*/
handler_21:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x54(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq22", .text    /*# Put this code in the Int vector > 0x800004C0*/
handler_22:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x58(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq23", .text    /*# Put this code in the Int vector > 0x800004E0*/
handler_23:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x5c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq24", .text    /*# Put this code in the Int vector > 0x80000500*/
handler_24:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x60(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq25", .text    /*# Put this code in the Int vector > 0x80000520*/
handler_25:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x64(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq26", .text    /*# Put this code in the Int vector > 0x80000540*/
handler_26:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x68(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq27", .text    /*# Put this code in the Int vector > 0x80000560*/
handler_27:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x6c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq28", .text    /*# Put this code in the Int vector > 0x80000580*/
handler_28:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x70(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq29", .text    /*# Put this code in the Int vector > 0x800005A0*/
handler_29:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x74(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq30", .text    /*# Put this code in the Int vector > 0x800005C0*/
handler_30:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x78(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq31", .text    /*# Put this code in the Int vector > 0x800005E0*/
handler_31:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x7c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

/* #endif defined(ENET_INC_LMAC) */

#elif defined(ENET_INC_UMAC)

/*###<E.Z> All interrupts of UMAC use second register file since UMAC doesn't support nested Int's*/


/*    ################### UMAC Interrupt Vector #################################*/

    .section ".vector_irq1", .text    /*# Put this code in the Int vector > 0x80000220*/
uart_rx_ovrflw_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x04(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq2", .text    /*# Put this code in the Int vector > 0x80000240*/
ba_err_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x08(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq3", .text    /*# Put this code in the Int vector > 0x80000260*/
wep_crc_err_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x0c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq4", .text    /*# Put this code in the Int vector > 0x80000280*/
uart_tx_empty_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x10(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq5", .text    /*# Put this code in the Int vector > 0x800002A0*/
pac_tick_tmr_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x14(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop


    .section ".vector_irq6", .text    /*# Put this code in the Int vector > 0x800002C0*/
hrc_doorbell0_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x18(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    
    .section ".vector_irq7", .text    /*# Put this code in the Int vector > 0x800002E0*/
hrc_dma_cplt_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x1c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    /*ISR compiled as 32bit thus no change at ISAmode bit*/
    

    .section ".vector_irq8", .text    /*# Put this code in the Int vector > 0x80000300*/
hrc_mstr_abort_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x20(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    /*ISR compiled as 32bit thus no change at ISAmode bit*/
    

    .section ".vector_irq9", .text    /*# Put this code in the Int vector > 0x80000320*/
emw_cplt_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x24(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq10", .text    /*# Put this code in the Int vector > 0x80000340*/
lcpu_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x28(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq11", .text    /*# Put this code in the Int vector > 0x80000360*/
uart_tx_empty1_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x2c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop


    .section ".vector_irq12", .text    /*# Put this code in the Int vector > 0x80000380*/
agg_tmrs_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x30(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    


    .section ".vector_irq13", .text    /*# Put this code in the Int vector > 0x800003A0*/
wep_cplt_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x34(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    

    .section ".vector_irq14", .text    /*# Put this code in the Int vector > 0x800003C0*/
debra_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x38(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq15", .text    /*# Put this code in the Int vector > 0x800003E0*/
unused15_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x3c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq16", .text    /*# Put this code in the Int vector > 0x80000400*/
unused16_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x40(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq17", .text    /*# Put this code in the Int vector > 0x80000420*/
unused17_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x44(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq18", .text    /*# Put this code in the Int vector > 0x80000440*/
unused18_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x48(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq19", .text    /*# Put this code in the Int vector > 0x80000460*/
unused19_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x4c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq20", .text    /*# Put this code in the Int vector > 0x80000480*/
unused20_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x50(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq21", .text    /*# Put this code in the Int vector > 0x800004A0*/
unused21_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x54(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq22", .text    /*# Put this code in the Int vector > 0x800004C0*/
unused22_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x58(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    

    .section ".vector_irq23", .text    /*# Put this code in the Int vector > 0x800004E0*/
unused23_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x5c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq24", .text    /*# Put this code in the Int vector > 0x80000500*/
unused24_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x60(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq25", .text    /*# Put this code in the Int vector > 0x80000520*/
unused25_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x64(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq26", .text    /*# Put this code in the Int vector > 0x80000540*/
unused26_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x68(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    .section ".vector_irq27", .text    /*# Put this code in the Int vector > 0x80000560*/
unused27_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x6c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq28", .text    /*# Put this code in the Int vector > 0x80000580*/
unused28_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x70(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq29", .text    /*# Put this code in the Int vector > 0x800005A0*/
unused29_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x74(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq30", .text    /*# Put this code in the Int vector > 0x800005C0*/
unused30_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x78(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
    
    .section ".vector_irq31", .text    /*# Put this code in the Int vector > 0x800005E0*/
unused31_handler:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x7c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

#else /*single cpu*/
/*    ################### SINGLE CPU Interrupt Vector #################################*/

.section ".vector_irq1", .text    /*# Put this code in the Int vector > 0x80000220*/
handler_01:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x4(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq2", .text    /*# Put this code in the Int vector > 0x80000240*/
handler_02:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x8(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq3", .text    /*# Put this code in the Int vector > 0x80000260*/
handler_03:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0xc(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop


.section ".vector_irq4", .text    /*# Put this code in the Int vector > 0x80000280*/
handler_04:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x10(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq5", .text    /*# Put this code in the Int vector > 0x800002A0*/
handler_05:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x14(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq6", .text    /*# Put this code in the Int vector > 0x800002C0*/
handler_06:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x18(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
   
.section ".vector_irq7", .text    /*# Put this code in the Int vector > 0x800002E0*/
handler_07:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x1c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq8", .text    /*# Put this code in the Int vector > 0x80000300*/
handler_08:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x20(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq9", .text    /*# Put this code in the Int vector > 0x80000320*/
handler_09:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x24(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

    
.section ".vector_irq10", .text    /*# Put this code in the Int vector > 0x80000340*/
handler_10:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x28(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq11", .text    /*# Put this code in the Int vector > 0x80000360*/
handler_11:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x2c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
	
.section ".vector_irq12", .text    /*# Put this code in the Int vector > 0x80000380*/
handler_12:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x30(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
 

.section ".vector_irq13", .text    /*# Put this code in the Int vector > 0x800003A0*/
handler_13:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x34(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq14", .text    /*# Put this code in the Int vector > 0x800003C0*/
handler_14:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x38(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq15", .text    /*# Put this code in the Int vector > 0x800003E0*/
handler_15:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x3c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop

.section ".vector_irq16", .text    /*# Put this code in the Int vector > 0x80000400*/
handler_16:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x40(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop


.section ".vector_irq17", .text    /*# Put this code in the Int vector > 0x80000420*/
handler_17:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x44(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
   

.section ".vector_irq18", .text    /*# Put this code in the Int vector > 0x80000440*/
handler_18:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x48(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
   

.section ".vector_irq19", .text    /*# Put this code in the Int vector > 0x80000460*/
handler_19:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x4c(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
   

.section ".vector_irq20", .text    /*# Put this code in the Int vector > 0x80000480*/
handler_20:
    li      k0,InterruptManagerIsrHandlers
    lw      k1,0x50(k0)
    mfc0    k0,C0_EPC
    j       shadowSet_handler
    nop
  
/*delia - not in use*/
.section ".vector_irq21", .text    /*# Put this code in the Int vector > 0x800004A0*/
handler_21:

   wait

.section ".vector_irq22", .text    /*# Put this code in the Int vector > 0x800004C0*/
handler_22:

    wait    

.section ".vector_irq23", .text    /*# Put this code in the Int vector > 0x800004E0*/
handler_23:

    wait    
    

.section ".vector_irq24", .text    /*# Put this code in the Int vector > 0x80000500*/
handler_24:

    wait    
    

.section ".vector_irq25", .text    /*# Put this code in the Int vector > 0x80000520*/
handler_25:

    wait    
    

.section ".vector_irq26", .text    /*# Put this code in the Int vector > 0x80000540*/
handler_26:

    wait    
    

.section ".vector_irq27", .text    /*# Put this code in the Int vector > 0x80000560*/
handler_27:

    wait    
    

.section ".vector_irq28", .text    /*# Put this code in the Int vector > 0x80000580*/
handler_28:

    wait    
    

.section ".vector_irq29", .text    /*# Put this code in the Int vector > 0x800005A0*/
handler_29:

    wait    
    

.section ".vector_irq30", .text    /*# Put this code in the Int vector > 0x800005C0*/
handler_30:

    wait    
    

.section ".vector_irq31", .text    /*# Put this code in the Int vector > 0x800005E0*/
handler_31:

    wait    

     
#endif    /*single cpu*/

    
/*****************************************************************************
#* Function Name:    Exception handler
#* Description: 
#*   This is the code in the exceptions intterupt vector (0x8000180)
#*   all it does is to stay in loop so at debugging we can catch this exception
#*   and fix it.
#* Input Parameters: 
#*   none
#* Output Parameters: 
#*   none
#* Return Value: 
#*   none
#* Algorithm: 
#*
#* Revision: 
#*
#*****************************************************************************/

  .if SIMULATOR
    .section ".vector", .text       /*# Put this code in the Int vector > 0x80000200*/
  .else
    .section ".exception", .text    /*# Put this code in the Int vector > 0x80000180*/
  .endif
  .set noreorder

    /*
    # save old exception sp
    # save the sp
    */

    or       k0,sp,zero				    /* # save sp to k0 */

    /* # load the new sp for exception */
    lui      $sp, %hi(__ghsend_Exc_stack)    
    addiu    $sp, $sp, %lo(__ghsend_Exc_stack)
    subu     sp,4 
    sw       k0, 0(sp)                  /* # save old sp */
    
    or       k1, t7, zero               /* #save r15 */
    lui      t7, %hi(ErrorHandlerExceptionregs)
    addiu    t7, t7, %lo(ErrorHandlerExceptionregs)
    sw       k0,  0*4(t7)               /* # save old sp */
    sw       ra,  1*4(t7)               /* # save old ra */
    sw       v0,  2*4(t7)               /* # save v0 */
    sw       v1,  3*4(t7)               /* # save v1 */
    sw       a0,  4*4(t7)               /* # save a0 */
    sw       a1,  5*4(t7)               /* # save a1 */
    sw       a2,  6*4(t7)               /* # save a2 */
    sw       a3,  7*4(t7)               /* # save a3 */
    sw       s0,  8*4(t7)               /* # save s0 */
    sw       s1,  9*4(t7)               /* # save s1 */
    sw       s2, 10*4(t7)               /* # save s2 */
    sw       s3, 11*4(t7)               /* # save s3 */
    sw       s4, 12*4(t7)               /* # save s4 */
    sw       s5, 13*4(t7)               /* # save s5 */
    sw       s6, 14*4(t7)               /* # save s6 */
    sw       s7, 15*4(t7)               /* # save s7 */

    move     t7, k1
    
    subu     sp,24                      /* # minimum non-leaf stack frame so ... */
                                        /* # called function can store a0-a3 */
    j     	 ErrorHandler_Exception     /* # C Exception function */
    nop

.text


/*****************************************************************************
#* Function Name:    shadowSet_handler
#* Description: 
#*   Called from the interrupt vector.
#*   Saves  the registers in use for each int calls the ISR and restore registers
#*   this function uses the second register file and thus only special registers need to be 
#*     saved.
#* Parameters: k0 -C0_EPC, k1- ISR address
#*****************************************************************************/
    
.globl shadowSet_handler  

#ifdef ENET_INC_LMAC
shadowSet_handler:

    rdpgpr sp,sp        
    rdpgpr gp,gp        
    addiu  sp,sp,-C_SIZE2*4    //#allocate space on stack for registers


/*#save status and EPC  registers LO + HI*/    

    mflo    t0
    mfhi    t1    
    mfc0    t2,C0_SR
    sw      k0,C_EPC*4(sp)  
    sw      t0,C_LO2*4(sp) 
    sw      t1,C_HI2*4(sp)
    sw      t2,C_SR*4(sp)    

/*#.if ENET_INC_LMAC..... This code is only comments because currently we don't enable any nested interrupts
                    # when working with second register file....

#    ## <E.Z> in case of LMAC we enable delia_tgl as nested interrupt
#    li      k0,0xffff00f9
#    and     k1,k0,k1
#    ori     k1,k1,0x3c00 #this sets IPL value to 15 which means we enable delia(16) and higher
#    mtc0    k1,C0_SR
##endif*/
    jalr ra,k1
    nop
    /****************On return from ISR***************/

    lw      t0,C_SR*4(sp)   
    lw      k0,C_EPC*4(sp)           
    lw      t1,C_LO2*4(sp)        
    mtc0    t0,C0_SR        # takes up to 3 cycles to change Reg.
    lw      k1,C_HI2*4(sp)
    mtlo    t1
    mthi    k1
    mtc0    k0,C0_EPC    
    wrpgpr  gp,gp

    eret
    nop
    
    .end shadowSet_handler

#else /*ENET_INC_LMAC*/

shadowSet_handler:


    /* Save current registers */
    jal     _tx_thread_context_save
    nop

    /* Execute the IRQ handler */
    /*#jalr    ra,a0*/
	jalr	ra,k1
    nop
    /* Restore previous registers and return from interrupt */
    j         _tx_thread_context_restore
    nop
    
.end shadowSet_handler


#endif     

/*****************************************************************************
#* Function Name:    asm_error
#* Description: 
#*   Called from the interrupt vector on unexpected interrupt, output the interrupt
#*     value who generated the error to address 0x100 on shared Ram
#* Parameters: k1 - error value.
#*****************************************************************************/ 
.set noreorder
.globl asm_error
asm_error:
    mfc0  k0,$12,2
    j asm_error
    nop
.end asm_error
/*****************************************************************************
#* Function Name:    asm_print
#* Description: CURRENTLY NOT IN USE!!!!
#*   This function send parameters k0 and k1 to external C function for printing
#*****************************************************************************/ 
 .globl asm_print
asm_print:
    
    addiu sp,sp,-0x4
    sw      ra,0(sp)
    or      a0,k0,zero
//    jal      AsmIntLogEvent
    or      a1,k1,zero
    lw      ra,0(sp)
    addiu sp,sp,0x4
    jr ra
    nop
 .end asm_print

.set reorder
.set at    /*#end of assembler code now we may use AT*/
    .section .cacheclean

clean_cache:    
.set noreorder

        mfc0    $10, $16, 1            /* # .word 0x400a8001 */
        nop
        /* # Isolate I$ Line Size */
        sll     $11, $10, 10
        srl     $11, 29

        /* # Skip ahead if No I$ */
        beq     $11, $0, no_icache
        nop
        li      $14, 2
        sllv    $11, $14, $11           /* # Now have true I$ line size in bytes */

        sll     $12, $10, 7
        srl     $12, 29
        li      $14, 64
        sllv    $12, $14, $12           /* # I$ Sets per way */

        sll     $13, $10, 13
        srl     $13, 29                 /* # I$ Assoc (-1) */
        add     $13, 1
        mul     $12, $12, $13           /* # Total number of sets */

        lui     $14, 0x8000             /* # Get a KSeg0 address for cacheops */

        /* # Clear TagLo/TagHi registers */
        mtc0    $0, $28
        mtc0    $0, $29
        mtc0    $0, $28, 2
        mtc0    $0, $29, 2

        move    $15, $12

        /* # Index Store Tag Cache Op */
        /* # Will invalidate the tag entry, clear the lock bit, and clear the LRF bit */
ic_clean_loop:    
        cache   0x8, 0($14)
        add     $15, -1                 /* # Decrement set counter */
        add     $14, $11                /* # Get next line address */
        bne     $15, $0, ic_clean_loop
        nop
no_icache:
        /* # Isolate D$ Line Size */
        sll     $11, $10, 19
        srl     $11, 29

        /* # Skip ahead if No D$ */
        beq     $11, $0, no_dcache
        nop
        li      $14, 2
        sllv    $11, $14, $11           /* # Now have true D$ line size in bytes */

        sll     $12, $10, 16
        srl     $12, 29
        li      $14, 64
        sllv    $12, $14, $12           /* # D$ Sets per way */

        sll     $13, $10, 22
        srl     $13, 29                 /* # D$ Assoc (-1) */
        add     $13, 1

        mul     $12, $12, $13           /* # Get total number of sets */

        lui     $14, 0x8000             /* # Get a KSeg0 address for cacheops */

        /* # Clear TagLo/TagHi registers */
        mtc0    $0, $28
        mtc0    $0, $29

        move    $15, $12

       /* # Index Store Tag Cache Op */
       /* # Will invalidate the tag entry, clear the lock bit, and clear the LRF bit */
dc_clean_loop:        
        cache   0x9, 0($14)
        add     $15, -1                 /* # Decrement set counter */
        add     $14, $11                /* # Get next line address */
        bne     $15, $0, dc_clean_loop
        nop
no_dcache:

    jr ra
    nop
    
.set reorder



