//***************************************************************
// File: MT_NSSMemoryPool.c
// Class/Module:
// Purpose: 
// Description: 
// 
// Modification history:
//           27-jUL-2006   [author]  Created
//  
// Class Invariant: 
// Copy Semantics:
// ***************************************************************

/* Include Files */
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "OSAL_Api.h"

#include "MT_NSSMemoryPool.h"
#include "stringLibApi.h"
#if defined (ENET_INC_LMAC)
#include "lminfra.h"
#include "TpcClbrHndlr.h"
#include "lm.h"
#endif
#include "ErrorHandler_Api.h"

/******************************************/
/*        DO NOT REMOVE THIS LINE!        */
/******************************************/
#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_MEMORY_MGMT
#define LOG_LOCAL_FID 1


//******************************************************************************
//----------------------------------------------------------
//                NSSMemoryPool::Create
// 
// Purpose:  Creates a simple memory pool, allocated and
//           constructs its buffers
// Input  :  
//      uint32     inBufferSize    User data size of the buffer in bytes
//      uint32     inNBuffers      Number of buffers in this pool
// Output :  
//      none
// Precondition:  
//
// Returns: none
//----------------------------------------------------------
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif

void CreatePool(MemPool_T *inMemPool, void *inMemAllocated,uint32 inSizeOfPool, uint32 inBufferSize, uint32 inNBuffes)
{
    uint32 addr,i,total_mem_size;
    MemPoolHdr_T *hdr;
    
    /* Minimum number of block is 2 */
    DEBUG_ASSERT(inNBuffes > 1);
	
    //Make sure that buffer size is aligned with 4Byte
    inBufferSize = ((inBufferSize + 3)/4) * 4;
    //The buffer size id the user size + header size
    inBufferSize += sizeof(MemPoolHdr_T);

    //Calculate the toal memory size consumpted by this memory pool
    //                 Buffer memory size          Free list arry memory size
    total_mem_size = ((inBufferSize * inNBuffes) + (sizeof(MemPoolHdr_T*) * inNBuffes));

    //The first 32 bits in inMemAllocated should contain the block size thet inMemAllocated points to
    //This lone checks that the block size is big enought to contain this memory pool
    ASSERT(total_mem_size >= inSizeOfPool); 
  
    /* memory is not allocated */
    ASSERT(inMemPool != ((MemPool_T *)0));
  
    //Allocate memory for the buffer array
    inMemPool->BufferList = (MemPoolHdr_T **)inMemAllocated;
    
  
    //Addr initialized to the pointer of the first buffer in this pool 
    addr = ((uint32)inMemAllocated + (sizeof(MemPoolHdr_T ) * inNBuffes));
    inMemPool->PoolStart = (void*)addr;

  
  //Init the array of free buffers
  for(i = 0;i<inNBuffes;i++)
  {
    hdr = (MemPoolHdr_T*)addr;
    *hdr = FREE_NSSBUF;
    inMemPool->BufferList[i] = hdr;

    addr+= inBufferSize;
  }

  inMemPool->PoolSize = inNBuffes;           //Number of buffers that belongs to this pool
  inMemPool->BufferSize = inBufferSize;      //Actual buffer size (including its header)
  inMemPool->NBuffersInPool = i;             //Set the index of the buffer array to the last;

}

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif


//******************************************************************************
//----------------------------------------------------------
//                NSSMemoryPool::AllocBuf
// 
// Purpose:  Allocate a buffer from the memory pool
// Input  :  
//      none
// Output :  
//      none
// Precondition:  
//
// Returns: a void pointer to the memory allocated
//----------------------------------------------------------
void  *AllocBuf(MemPool_T *inMemPool)
{
	TX_INTERRUPT_SAVE_AREA;
	MemPoolHdr_T *hdr;
    void *ret_val;

    OSAL_DISABLE_INTERRUPTS(&interrupt_save);

    //Assert if the pool is empty
    ASSERT(inMemPool->NBuffersInPool);
  
    inMemPool->NBuffersInPool--;
    hdr = inMemPool->BufferList[inMemPool->NBuffersInPool];
    
    OSAL_ENABLE_INTERRUPTS(interrupt_save);

    //Assert if the header is marked as allocated
    ASSERT(FREE_NSSBUF == *hdr);
    
    *hdr = USED_NSSBUF;

    //Return a pointer to the user buffer (after the header)
    ret_val = (hdr+1);

    return ret_val;
}
    
//******************************************************************************
//----------------------------------------------------------
//                NSSMemoryPool::FreeBuf
// 
// Purpose:  Returns a memory buffer to its pool
// Input  :  
//      void *   inBuf   The address of a buffer to be freed
// Output :  
//      none
// Precondition:  
//
// Returns: none
//----------------------------------------------------------
void FreeBuf(MemPool_T *inMemPool, void  *inBuf)
{
    TX_INTERRUPT_SAVE_AREA;
	MemPoolHdr_T *hdr;
    uint32 pool_start_addr;
    int buffer_offset;

    //Assert if it's an attempt to free a null pointer
    DEBUG_ASSERT(inBuf);
    
    hdr = (MemPoolHdr_T *)inBuf;
    
    //Place hdr on the buffer header;
    hdr--;

    //Assert if it's an attempt to free already free buffer
    DEBUG_ASSERT(USED_NSSBUF == *hdr);
    
    //If the buffer is marked as used (as it should be)    
    //Mark the buffer as free and return it to pool
    *hdr = FREE_NSSBUF;
    pool_start_addr = (uint32)inMemPool->PoolStart;
    buffer_offset = (uint32)hdr - pool_start_addr;

    //Check if the buffer belongs to this pool address range(it doesn't check if it's already used)
    if ((buffer_offset < 0) ||
      ((buffer_offset % inMemPool->BufferSize) ||
      ((uint32)hdr >= (pool_start_addr + inMemPool->BufferSize*inMemPool->PoolSize  ))))
    {
        DEBUG_ASSERT(0);
    }

    //*****************************************************************
    //If it's re-entrant function: Critical section should start here  
    //*****************************************************************
    OSAL_DISABLE_INTERRUPTS(&interrupt_save);
    //Return the buffer to its pool
    inMemPool->BufferList[inMemPool->NBuffersInPool] = hdr;
    inMemPool->NBuffersInPool++;
    OSAL_ENABLE_INTERRUPTS(interrupt_save);
    //*****************************************************************
    //If it's re-entrant function: Critical section should end here    
    //*****************************************************************
}

//******************************************************************************
//----------------------------------------------------------
//                NSSMemoryPool::GetBufferById
// 
// Purpose:  Returns address of a buffer by its sequence number (e.g 0 is the
//           first buffer on the pool).
// Input  :  
//      uint32   theNum   The sequence number of the requested buffer
// Output :  
//      none
// Precondition:  
//
// Returns: A pointer to the requested buffer (buffer is NOT allocated)
//   
//----------------------------------------------------------
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif

void* GetBufferAddressById(MemPool_T *inMemPool, uint32 inNumber)
{
  MemPoolHdr_T *hdr;
  void *ret_val;
  uint32 start,addr;

  start = (uint32)inMemPool->PoolStart;
  addr = start + inNumber*(uint32)inMemPool->BufferSize;

  hdr = (MemPoolHdr_T*)addr;
  // return the pointer to the buffer (right after the header)
  ret_val = hdr+1;
  
  return ret_val;
}

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default 
#endif




