/***********************************************************************************
 File:			PacketDescriptors.c
 Module:			MemoryManagement
 Purpose:			Utility functions for managing a pool of Packet Descriptors
 Description:		Utility functions for managing a pool of Packet Descriptors
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/

#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "stringLibApi.h"
#include "PacketDescriptor.h"
#include "PacketDescriptorsPool_api.h"
#include "ShramPacketDescriptors.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_MEMORY_MGMT 
#define LOG_LOCAL_FID 2


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

InitializePdPool 

Description:
------------
	Initializes PD pool utility function  (linked list of packet descriptors) 
Input:
-----
	PacketDescriptorPool_t *  pdPoolToInitialize - PDs pool we want to initialize
	TxPd_t* PdPoolArray - The Buffer in SHRAM which holds the PDs
	uint32 numOfPdsInShramBuffer - The number of PDs in the shram buffer
	PdType_e pdType - The type of the PDs in this pool
		
Output:
-------
	Pool is initialized

Returns:
--------
	void - 
	
**********************************************************************************/

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif
void Pool_InitializePdPool(PacketDescriptorPool_t *  pPdPoolToInitialize,
					TxPd_t* pPdPoolArray,
					uint32 numOfPdsInBuffer,
					PdType_e pdType)
{
	uint32 i = 0;
	TxPd_t* pPd = pPdPoolArray;

	memset(pPdPoolArray, 0, (sizeof(TxPd_t)* numOfPdsInBuffer)); 
	
	// initialize pool head pointer
	pPdPoolToInitialize->pHead = pPdPoolArray;

	// create the linked list pool
	for (i=0 ; i < numOfPdsInBuffer ; i++)
	{
		// initialize PD fields
		pPd->pdType = pdType;
		if (pdType == PD_TYPE_BEACON)
		{
			pPd->mcUnicast = MULTICAST;
			pPd->txQGroupId = HW_TX_Q_TYPE_BEACON;
		}

		// if not last PD in pool then set next pointer to point
		// to next cell, else - last PD - set next pointer to NULL
		if (i < (numOfPdsInBuffer - 1))
		{
			
			// The next Pd ptr value is 16 bit long field that points to a 32 bit address.
			// The value is computed as : ShortPtr = (LongPtr - PdBaseAddress) >> 2 ; LongPtr = (ShortPtr << 2) + PdBaseAddress ;
			pPdPoolArray++;
			pPd->nextPdPointer = SET_NEXT_PD(pPdPoolArray);
			pPd = pPdPoolArray;
		}
		else // last PD in pool
		{
			pPd->nextPdPointer = NEXT_PD_NULL;
		}
	}

	pPdPoolToInitialize->numOfDescInPool =  numOfPdsInBuffer;
	pPdPoolToInitialize->maxDescInPool = numOfPdsInBuffer;
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif


/**********************************************************************************

Pool_ReleasePdList 

Description:
------------
	release a linked list of PDs back to the PDs Pool
	
Input:
-----
	PacketDescriptorPool_t *  pdPool - Pool to push the PDs List back into.
	TxPd_t* pHeadPdList - a list of PDs to be pushed back to the pool
		
Output:
-------
	void

Returns:
--------
	void  
	
**********************************************************************************/
void Pool_ReleasePdList(PacketDescriptorPool_t *  pPdPool, TxPd_t* pHeadPdList, TxPd_t* pTailPdList, uint16 numPdsInList)
{
	uint16 pdCounter = 1;
	uint32 pdAddress = 0;

	// save the current head pointer of the PD pool
	uint32 oldHeadPdPtr = ((uint32*)pPdPool->pHead) - PacketDescriptorsPtr;

	// Set the pd pool head pointer to point to the newly added released pds list
	pPdPool->pHead = pHeadPdList;

	if (pTailPdList)	
	{
		// if a tail pointer to the released PDs list is supplied then the last PD in the list should point
		// to the old head PD of the pool.
		// Number of PDs should be supplied as an input in this case.
		pTailPdList->nextPdPointer = oldHeadPdPtr;
		pdCounter = numPdsInList;
	}
	else
	{
		// if tail pointer is not supplied then we must loop on the released PD untill we get to the last PD
		while (pHeadPdList->nextPdPointer != NEXT_PD_NULL)
		{
			pdAddress = (uint32)PacketDescriptorsPtr + (uint32)(pHeadPdList->nextPdPointer << 2);
			pHeadPdList = (TxPd_t*)pdAddress;
			pdCounter++;
		}

		pHeadPdList->nextPdPointer = oldHeadPdPtr;
	}
	DEBUG_ASSERT((pPdPool->numOfDescInPool + pdCounter) <= pPdPool->maxDescInPool);
	pPdPool->numOfDescInPool += pdCounter;
}


/**********************************************************************************

Pool_PushPd 

Description:
------------
	Push a PD to the pool. 
	
Input:
-----
	PacketDescriptorPool_t *  pdPool  - Pool of packet descriptors
	TxPd_t* pd - a packet descriptor to be pushed back to the pool
		
Output:
-------
	Pool is updated

Returns:
--------
	void  
	
**********************************************************************************/
void Pool_PushPd(PacketDescriptorPool_t *  pPdPool, TxPd_t* pPd)
{
	pPd->nextPdPointer = ((uint32*)pPdPool->pHead) - PacketDescriptorsPtr;

	pPdPool->pHead = pPd;
	DEBUG_ASSERT(pPdPool->numOfDescInPool < pPdPool->maxDescInPool);
	pPdPool->numOfDescInPool++;
}


/**********************************************************************************

beaconHandler_PullBeaconPd 

Description:
------------
	Pulls a PD from the Beacon handler pool. Pulls form the head of the PD list
	
Input:
-----
	PacketDescriptorPool_t *  pdPool - Pool of packet descriptors
		
Output:
-------
	TxPd_t* pd - a packet descriptor pulled from the pool
	Pool is updated

Returns:
--------
	void - 
	
**********************************************************************************/
TxPd_t* Pool_PullPd(PacketDescriptorPool_t *  pdPool)
{
	TxPd_t* pPd = NULL;
	uint32 pdStartAddress;

	if (pdPool->numOfDescInPool > 0)
	{
		pPd = pdPool->pHead;

		pdStartAddress = (uint32)PacketDescriptorsPtr + (uint32)(pdPool->pHead->nextPdPointer << 2);
		pdPool->pHead = (TxPd_t*)pdStartAddress;

		pdPool->numOfDescInPool--;
	}

	return pPd;
}

