/***********************************************************************************
 File:		performanceCounters.c
 Module:		
 Purpose:		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "arc_reg.h"
#include "performanceCounters.h"
#include "performanceCounters_api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "System_CoreTimersApi.h"
#include "tx_application_Api.h"



/*---------------------------------------------------------------------------------
/						Defines
/----------------------------------------------------------------------------------*/

#define ONE_SECOND_IN_MS	1000

/*---------------------------------------------------------------------------------
/						Macros
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions
/----------------------------------------------------------------------------------*/

/**********************************************************************************

<function name>

Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/	


/**********************************************************************************
PerformanceCounters_SetConditionToCounter

Description:
------------
	Set one performance counter  to count a given condition

Input:
-----
	counter index, countable condition
	
Output:
-------

Returns:
--------

**********************************************************************************/
void PerformanceCounters_SetConditionToCounter(uint8 counterIndex, uint8 countableCondition)
{
	// Configure counter index 
	_sr(counterIndex, PCT_INDEX);
	
	// Set this counter to count the chosen condition
	_sr(countableCondition, PCT_CONFIG);
}

/**********************************************************************************
PerformanceCounters_SetCounterLimit

Description:
------------
	Set a limit to a single counter which will trigger an interrupt when reached.

Input:
-----
	counter index, countable condition
	
Output:
-------

Returns:
--------

**********************************************************************************/
void PerformanceCounters_SetCounterLimit(uint8 counterIndex, uint32 countLimitLow, uint32 countLimitHigh)
{
	uint32 enabledInterrupts;
	
	// Configure counter index 
	_sr(counterIndex, PCT_INDEX);

	// Configure low count limit (LSBs)
	_sr(countLimitLow, PCT_INT_CNTL);

	// Configure high count limit (MSBs)
	_sr(countLimitHigh, PCT_INT_CNTH);
	
	// Enable the interrupt for this counter
	enabledInterrupts = _lr(PCT_INT_CTRL);
	enabledInterrupts |= (0x1 << counterIndex);
	_sr(enabledInterrupts, PCT_INT_CTRL);
}


/**********************************************************************************
isr_PerformanceCounters_CountLimitReached

Description:
------------
	isr function for when one of the counters has reached the configured limit

Input:
-----
	
Output:
-------

Returns:
--------

**********************************************************************************/
void isr_PerformanceCounters_CountLimitReached(void)
{
	AuxRegPctControl_u pctControlReg;
	uint32 sleepCyclesLow;
	uint32 totalCycles;	
	

	// At this point we know that the cycles counter reached 1 sec. We want to take a snapshot of the sleep cycles counter and reset it.
	// Set index to counter 1 (sleep cycle count)
	_sr(PERFORMANCE_COUNTER_1, PCT_INDEX);

	// Take a snapshot of sleep cycle count.
	pctControlReg.val = 0;
	pctControlReg.bitFields.enable = 1;
	pctControlReg.bitFields.snapshot = 1;
	_sr(pctControlReg.val, PCT_CONTROL);

	// Read the snapshot of sleep cycles counter
	sleepCyclesLow = _lr(PCT_SNAPL);
	
	// Reset (set to 0) the sleep cycles counter (it's OK to do it now, since we are in ISR, and we don't count sleeps now).
	_sr(0, PCT_COUNTL);
	
	// Set index to counter 0 (total cycle count)
	_sr(PERFORMANCE_COUNTER_0, PCT_INDEX);

	// Take a snapshot of total cycle count.
	_sr(pctControlReg.val, PCT_CONTROL);

	// Read the snapshot of total cycles counter
	totalCycles = _lr(PCT_SNAPL);

	// Reduce 1sec from the total cycles counter (640,000,000 cycles since we have 640Mhz clock)
	totalCycles -= CONVERT_MS_TO_SYS_TICK(ONE_SECOND_IN_MS);
	_sr(totalCycles, PCT_COUNTL);

	// Update the CPU load array (each entry is CPU load of 1 second in %).
	// We shift right by 4, due to we have only 32 bits, and otherwise we will get overflow. The resolution is still very accurate.
	cpuLoadArray[cpuLoadArrayIndex] = (((sleepCyclesLow>>4)*100)/(CONVERT_MS_TO_SYS_TICK(ONE_SECOND_IN_MS)>>4)); 
	cpuLoadArrayIndex++; 
	if(cpuLoadArrayIndex == CPU_LOAD_ARRAY_SIZE)
	{
		cpuLoadArrayIndex = 0; 
	}

/*
	// Clear the performance counter interrupt, and set the trigger it again
	while (_lr(PCT_INT_ACT) != 0)
	{
		_sr(1, PCT_INT_ACT);
	}
	_sr(1, PCT_INT_CTRL);
*/	
}



/**********************************************************************************

performanceCounters_Init

Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void PerformanceCounters_Init(void)
{
//	uint32 cycleCountLimitLow;
//	uint32 cycleCountLimitHigh;
	AuxRegPctControl_u pctControlReg;


	pctControlReg.val = 0;
	pctControlReg.bitFields.enable = 1;
	
	//set the performance counter interrupt level to 0
	_sr(20, IRQ_INTERRUPT);
	_sr(0, IRQ_PRIORITY);
	// global enable
	_sr(pctControlReg.val, PCT_CONTROL);

	// We want to use 2 counters (0/1) to count %sleep (100*sleepCycles/totalCycles):

	// Configure counter 0 to count total cycles
	PerformanceCounters_SetConditionToCounter(PERFORMANCE_COUNTER_0, COUNTABLE_CONDITION_TOTAL_CYCLE_COUNT);

	// Configure counter 1 to count cycles while in sleep
	PerformanceCounters_SetConditionToCounter(PERFORMANCE_COUNTER_1, COUNTABLE_CONDITION_SLEEP_CYCLE_COUNT);

	// configure an interrupt every 1 sec (assuming 640Mhz clock)
//	cycleCountLimitLow = 640000000; // This results in 640,000,000 which is lower then 32 bits. If it's changed, we might need to use LimitHigh.
//	cycleCountLimitHigh = 0;
//	PerformanceCounters_SetCounterLimit(PERFORMANCE_COUNTER_0, cycleCountLimitLow, cycleCountLimitHigh);

	System_CoreTimersSetTimer(SYSTEM_CORE_TIMER_1, FALSE, ONE_SECOND_IN_MS, isr_PerformanceCounters_CountLimitReached);
	

}


