/***************************************************************
 File:		Mac2Index.c
 Module:	  	MAC 2 index
 Purpose: 	holds all the implementation of the MAC 2 index API functions
 Description:  This module contains the implementation of the MAC 2 index module 
 			which is responsible of converting MAC address to station id 
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "OSAL_Api.h"
#include "ProtDbRegs.h"
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ProtectedDbLock_Api.h"
#include "ProtectedDbLock.h"
#include "RegAccess_Api.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ShramStationDatabase.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PROTECTED_DB_LOCK
#define LOG_LOCAL_FID 0



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

void ProtectedDbLock_triggerHwCopy(uint32* destinationShramAddress, uint32 sizeInWords);



/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
void ProtectedDbLock_triggerHwCopy(uint32* destinationShramAddress, uint32 sizeInWords)
{	
	RegProtDbEntity0ProtStartCommand_u command;

	command.val = 0;  //reset all bits

	/* Write shram destination	address at the commandRegister registers */
	
	command.bitFields.entity0ProtDestBaseAddr = (uint32)CONVERT_BYTES_TO_WORDS(destinationShramAddress);
	command.bitFields.entity0ProtWordsNum = sizeInWords; //we already verified that length is word aligned
	RegAccess_Write(REG_PROT_DB_START_COMMAND_ADDRESS, command.val); 

	return;
}



/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

void ProtectedDbLock_ReadReq(uint32* destinationShramAddress, uint32 sizeInwords, uint32* scratchPadAddress)
{
	RegProtDbEntity0ProtIrq_u  protectedIrq;
	RegProtDbEntity0ProtClearIrq_u protectedIrqClear; 
	TX_INTERRUPT_SAVE_AREA;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// polling the HW status register 
	protectedIrqClear.val = 0;//reset all bits 
	protectedIrq.val = 0; //reset all bits (although its a read only parameter)
	do
	{
		RegAccess_Read(REG_PROT_DB_IRQ_ADDRESS, &protectedIrq.val); // sample interrupt bit 
	}	while(protectedIrq.val == 0x0); 

	protectedIrqClear.bitFields.entity0ProtClearIrqStb = 0x1; //Clear IRQ for next time 
	RegAccess_Write(REG_PROT_DB_CLEAR_IRQ_ADDRESS, protectedIrqClear.val);

	memcpy32((void*)scratchPadAddress,destinationShramAddress,sizeInwords); 	//copy data to scratchpad memory 

 	OSAL_ENABLE_INTERRUPTS(interrupt_save); //free resource for others 

}

void ProtectedDbLock_WriteReq(uint32* destShramAddr,void* sourceAddr, uint32 sizeInWords)
{
	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/*No need to poll - this function is called after Read REQ above was called*/		
	memcpy32((void*)PROT_DB_SHADOW_MEM_BASE_ADDR_VAL,sourceAddr,sizeInWords); 	//copy data to shadow memory 

	ProtectedDbLock_triggerHwCopy(destShramAddr, sizeInWords);

	OSAL_ENABLE_INTERRUPTS(interrupt_save); //free resource for others 
	return;

}

void ProtectedDbLock_CopyReq(uint32* destShramAddr,void* sourceAddr, uint32 sizeInWords)
{
	RegProtDbEntity0ProtIrq_u  protectedIrq;
	RegProtDbEntity0ProtClearIrq_u protectedIrqClear; 
	TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		
// polling the HW status register 
	protectedIrqClear.val = 0;//reset all bits 
	protectedIrq.val = 0; //reset all bits (although its a read only parameter)
	do
	{
		RegAccess_Read(REG_PROT_DB_IRQ_ADDRESS, &protectedIrq.val); // sample interrupt bit 
	}	while(protectedIrq.val == 0x0); 
		
	protectedIrqClear.bitFields.entity0ProtClearIrqStb = 0x1; //Clear IRQ for next time 
	RegAccess_Write(REG_PROT_DB_CLEAR_IRQ_ADDRESS, protectedIrqClear.val);

	memcpy32((void*)PROT_DB_SHADOW_MEM_BASE_ADDR_VAL,sourceAddr,sizeInWords); 	//copy data to shadow memory 

	ProtectedDbLock_triggerHwCopy(destShramAddr, sizeInWords);

	OSAL_ENABLE_INTERRUPTS(interrupt_save); //free resource for others 
	return;

}

void ProtectedDbLock_ReadModifyWriteReq(uint32* destinationShramAddress, uint32 sizeInwords, 
													uint32* scratchPadAddress,ProtectedDbLock_localModificationFunc modificationFunc, void* pModificationFuncParams)
{
	RegProtDbEntity0ProtIrq_u  protectedIrq;
	RegProtDbEntity0ProtClearIrq_u protectedIrqClear; 
	TX_INTERRUPT_SAVE_AREA;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	// polling the HW status register 
	protectedIrqClear.val = 0;//reset all bits 
	protectedIrq.val = 0; //reset all bits (although its a read only parameter)
	do
	{
		RegAccess_Read(REG_PROT_DB_IRQ_ADDRESS, &protectedIrq.val); // sample interrupt bit 
	}	while(protectedIrq.val == 0x0); 

	protectedIrqClear.bitFields.entity0ProtClearIrqStb = 0x1; //Clear IRQ for next time 
	RegAccess_Write(REG_PROT_DB_CLEAR_IRQ_ADDRESS, protectedIrqClear.val);

	memcpy32((void*)scratchPadAddress,destinationShramAddress,sizeInwords); 	//copy data to scratchpad memory 
	modificationFunc(scratchPadAddress, pModificationFuncParams); 
	memcpy32((void*)PROT_DB_SHADOW_MEM_BASE_ADDR_VAL,scratchPadAddress,sizeInwords); 	//copy data to shadow memory 

	ProtectedDbLock_triggerHwCopy(destinationShramAddress, sizeInwords);

	OSAL_ENABLE_INTERRUPTS(interrupt_save); //free resource for others 


}




#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void ProtectedDbLock_Initialize(void)
{
	uint32 scratchpadAddress;
	uint32* destinationShramAddress = (uint32*)&(StaDbHwEntries[0].common.word4); //dummy write to this address
	RegProtDbEntity0ProtShadowMem8BOffset_u regProtDbEntity0ProtShadowMem8BOffset;
	TX_INTERRUPT_SAVE_AREA;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	memset32((void*)PROT_DB_SHADOW_MEM_BASE_ADDR,0,SHADOW_MEMORY_SIZE_WORDS);	//set zeros in shadow ram

	regProtDbEntity0ProtShadowMem8BOffset.val = 0;
	regProtDbEntity0ProtShadowMem8BOffset.bitFields.entity0ProtShadowMem8BOffset = SHADOW_OFFSET_8B;

	/*Define shdow memory offset for UM and LM*/
	RegAccess_Write(REG_PROT_DB_PROT_SHADOW_MEM_8B_OFFSET_ADDRESS, regProtDbEntity0ProtShadowMem8BOffset.val);


	/*Read without modify and write 1 word to DB through protected DB, in order to trigger the protected DB HW to set the status bit on 1, this will be done once in the LM initialization*/		
	memcpy32(&scratchpadAddress,destinationShramAddress,1);	//copy 1 word  to scratch pad 
	memcpy32((void*)PROT_DB_SHADOW_MEM_BASE_ADDR_VAL,&scratchpadAddress,1);	//copy data to shadow memory 
	ProtectedDbLock_triggerHwCopy(destinationShramAddress, 1);

	OSAL_ENABLE_INTERRUPTS(interrupt_save); //free resource for others 

	return;
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif






