/***********************************************************************************
 File:			secManager_StaManager.c
 Module:		Sec manager STA Manager
 Purpose: 		handle STA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "Utils_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "StaDatabase_Api.h"
#include "TxSelector_Api.h"
#include "Locker_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "shramTxDesc.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "secManager_StaManager.h"
#include "secManager_VapManager.h"
#include "loggerAPI.h"
#include "StaDatabase_Api.h"
#include "RxPp_Api.h"
#include "TsManager_API.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_SECURITY
#define LOG_LOCAL_FID 2


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
typedef struct SecManagerGlobalDb_s
{
	uint8 flushOnSetKey;		/*Whether to flush TID 7 queue on Set Key to remove pending EAPOLS*/
} SecManagerGlobalDb_t;
#endif

typedef void (*SecManager_StaManagerFunc)(SecManagerStaDb_t *staDb);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void SecManager_StaManagerOpenSta(SecManagerStaDb_t *StaDb);
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
static void SecManager_StaManagerLockSta(SecManagerStaDb_t *StaDb);
#endif
static void SecManager_StaManagerLockRekey(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerGphpRekey(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerSetFilter(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerSetKey(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerReKey(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerKeyIndex(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerCloseSta(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerFatal(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerSendConfirm(StaId sid);
static void SecManager_StaManagerChangeState(secManagerStaManager_state state, SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerFinishRekey(SecManagerStaDb_t *StaDb);
static void SecManager_StaManagerFlushQueue(SecManagerStaDb_t *StaDb, uint8 tid);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*STA Open Event Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerStaOpen[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 SecManager_StaManagerOpenSta, 				  /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	 SecManager_StaManagerFatal,				  /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif	 
	 SecManager_StaManagerFatal,				  /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
 	 SecManager_StaManagerFatal,				  /* SEC_MANAGER_STA_MANAGER_OPEN */
	 SecManager_StaManagerFatal,				  /* SEC_MANAGER_STA_MANAGER_REKEY */
};

/*STA Set Filter Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerStaSetFilter[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SecManager_StaManagerFatal,					 /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif	
	SecManager_StaManagerSetFilter, 			 /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_OPEN */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_REKEY */
};

/*STA Set Key Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerStaSetKey[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SecManager_StaManagerFatal,					 /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif
	SecManager_StaManagerSetKey, 			 	 /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
	SecManager_StaManagerReKey, 				 /* SEC_MANAGER_STA_MANAGER_OPEN */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_REKEY */
};

/*STA Set Index Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerStaSetIndex[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SecManager_StaManagerFatal,					 /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif	
	SecManager_StaManagerKeyIndex, 			 	 /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_OPEN */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_REKEY */
};

/*STA Close Event Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerStaClose[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SecManager_StaManagerFatal,					 /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif	
	SecManager_StaManagerCloseSta, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
	SecManager_StaManagerCloseSta, 				 /* SEC_MANAGER_STA_MANAGER_OPEN */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_REKEY */
};

/*STA Lock Event Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerStaLock[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SecManager_StaManagerFatal,					 /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	SecManager_StaManagerLockSta, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif	
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_OPEN */
	SecManager_StaManagerLockRekey, 			 /* SEC_MANAGER_STA_MANAGER_REKEY */
};

/*GPHP Lock Event Handlers*/
static const SecManager_StaManagerFunc SecManager_StaManagerGphpLock[SEC_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SecManager_StaManagerFatal,					 /* SEC_MANAGER_STA_MANAGER_CLOSED */
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_LOCK */
#endif	
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_WAIT_FILTER */
	SecManager_StaManagerFatal, 				 /* SEC_MANAGER_STA_MANAGER_OPEN */
	SecManager_StaManagerGphpRekey, 			 /* SEC_MANAGER_STA_MANAGER_REKEY */
};

#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
static SecManagerGlobalDb_t SecManager_GlobalDb;
#endif

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

SecManagerStaDb_t SecManager_StaManagerDb[HW_NUM_OF_STATIONS];
extern PerClientStatistics_t PerClientStatistics;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

SecManager_StaManagerSendConfirm 


Description:
------------
Sends COnfirmation message to STA manager

Input: 
-----


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
 static void SecManager_StaManagerSendConfirm(StaId sid)
 {
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_SEC_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = sid;
	/*Send confirmation message*/
	ILOG2_DD("Sec Manager, Send Confirmation, Client %d, SID %d", confirmMessage->clientId, sid);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

SecManager_StaManagerChangeState 


Description:
------------
Change STA state


Input: 
-----	
secManagerStaManager_state - event
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerChangeState(secManagerStaManager_state state, SecManagerStaDb_t *StaDb)
{
	ILOG0_DDD("SEC Manager STA Manager, Station %d changed from %d to %d", StaDb->sid, StaDb->state, state);
	StaDb->state = state;
}

/**********************************************************************************

SecManager_StaManagerFlushQueue 


Description:
------------
Utility function that flushes a queue - used during rekey


Input: 
-----	
SecManagerStaDb_t - STA DB entry
uint8 tid
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerFlushQueue(SecManagerStaDb_t *StaDb, uint8 tid)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint8 mfp = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_MFP);
	TxPd_t *currentPacketDescriptor = NULL;
	TxPd_t *lastPacketDescriptorToDiscard = NULL;
	K_MSG* tsManagerMsg = NULL; 
	TsManagerDataPacketDiscardedMessage_t  *tsManagerMessageParameters = NULL;
	uint32	count = 0;
	
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = StaDb->sid;
	hwQueueManagerRequestParams.secondaryAddr = tid;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
	if ((TxPd_t *)NULL_PD != currentPacketDescriptor)
	{
		/*if MFP first packet could be encrypted management*/
		if ((mfp) &&
			(currentPacketDescriptor->pdType == PD_TYPE_MANAGEMENT_ENC) &&
			 (currentPacketDescriptor->retransmissionIndication))
		{
			lastPacketDescriptorToDiscard = currentPacketDescriptor;
		}
		else 
		{
			while (((TxPd_t *)NULL_PD != currentPacketDescriptor) &&
				   (currentPacketDescriptor->pdType == PD_TYPE_DATA) &&
				  	(currentPacketDescriptor->retransmissionIndication))
			{
				/* PD will not be transmitted: set status as NACK */
				currentPacketDescriptor->status = PD_STATUS_NACK;
				
				if(currentPacketDescriptor->aggregationIndication)
				{
					count += currentPacketDescriptor->pdCounter;
					/* A-MSDU - discard the whole A-MSDU. There shouldnt be A-MSDU of 1 MSDU */
					currentPacketDescriptor = (TxPd_t *)CONVERT_OFFSET_TO_PD(currentPacketDescriptor->aMsduTailPointer);
				}
				else
				{
					count += 1;
				}
				lastPacketDescriptorToDiscard = currentPacketDescriptor;
				currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
			}
		}
	}
	if(lastPacketDescriptorToDiscard != NULL)
	{
		hwQueueManagerRequestParams.pTailDesc = lastPacketDescriptorToDiscard;
		HwQManager_PopListFromHead(&hwQueueManagerRequestParams);
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;

		if (count > 0)
		{
#ifdef ENET_INC_ARCH_WAVE600
			GeneralStatistics.dropReasonReKey[ConfigurationManager_GetBandForVap(StaDb->vapId)] += count;
			PerClientStatistics.dropCntReasonReKey[StaDb->sid] += count;
#else
			GeneralStatistics.dropReasonReKey+=count;
#endif //ENET_INC_ARCH_WAVE600
		}
		
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
		/* If PD with retry bit was discarded the low sequence number should be update */
		StaDB_UpdateLowSequenceNumberFromCurrent(StaDb->sid, tid);
		/* Inform the TS manager that packet was discarded - since the whole queue was handled there will  be 
		   only one message per queue */
		tsManagerMsg = OSAL_GET_MESSAGE(sizeof(TsManagerDataPacketDiscardedMessage_t));
		tsManagerMessageParameters = ((TsManagerDataPacketDiscardedMessage_t *)tsManagerMsg->abData);
		tsManagerMessageParameters->stationId = StaDb->sid;
		tsManagerMessageParameters->tid = tid;
		OSAL_SEND_MESSAGE(TS_MANAGER_PACKET_DISCARDED, TASK_TS_MANAGER, tsManagerMsg, StaDb->vapId);
	}
}

/**********************************************************************************

SecManager_StaManagerFinishRekey 


Description:
------------
Utility function that finishes the rekey process


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerFinishRekey(SecManagerStaDb_t *StaDb)
{
	K_MSG *pMsg = NULL;
	UMI_SET_KEY *psSetKey = NULL;
	bool rekeyDisableFlag = FALSE; //rekey is activated
	uint8 mfp = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_MFP);
	uint8 isQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	/*No PMF - we need to flush RX, set the key and resume TX & RX*/
	pMsg = (K_MSG *)StaDb->pMsg;
	psSetKey = (UMI_SET_KEY *)pK_MSG_DATA(pMsg);
	/*Halt RX*/
	RxPp_Halt();
	/*Calll RXPP rekey handler - will flush all TIDs*/
	RxPp_Rekey(StaDb->sid);
	/*Set The Key now*/
	StaDB_SetSecurityKey(psSetKey, rekeyDisableFlag);
	if (mfp == TRUE)
	{
		StaDB_SetManagementEncrypted(StaDb->sid);
	}
	/*Set the default key index here ...*/
	StaDB_SetDefaultKeyIndex(StaDb->sid, psSetKey->u16KeyIndex, TRUE);
	/*Unlock TX queues*/
	if (isQos)
	{
		Locker_UnLockAllStaQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid);
	}
	else
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid, (0x1 << IEEE802_1D_BE_1));
	}
	if (mfp == TRUE)
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, StaDb->vapId, (0x1 << 0));
	}
	/*Resume RX*/
	RxPp_Resume();
	/*Change STA State to Open*/
	SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_OPEN, StaDb);
	/*Clear message pointer*/
	StaDb->pMsg = NULL;
	/*Send SET KEY confirmation*/
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_KEY_CFM, TASK_UM_IF_TASK, pMsg, StaDb->vapId);
}

/**********************************************************************************

SecManager_StaManagerOpenSta 


Description:
------------
handles STA Open event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerOpenSta(SecManagerStaDb_t *StaDb)
{
	K_MSG *pMsg = StaDb->pMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
	uint8 isXFilterOpen = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);
	uint8 isWds = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_WDS);
	uint8 vapCipherSuite;
	
	if (isXFilterOpen == TRUE)
	{
		/*It could be that we are working with WEP - need to check VAP encryption type and if WEP
		copy VAP keys to STA keys*/
		/*For now it is done in staDbFillDbEntry that is called by STA Manager before calling registered modules*/
		/*Change STA State to Open*/
		SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_OPEN, StaDb);
		/*It could be that we are working with WEP - need to check VAP encryption type and if WEP
		copy VAP keys to STA keys - unless STA is WDS*/
		vapCipherSuite = secManager_VapManagerGetCipherSuite(pAddSta->u8VapIndex);
		if (((vapCipherSuite == UMI_RSN_CIPHER_SUITE_WEP104) || (vapCipherSuite == UMI_RSN_CIPHER_SUITE_WEP40)) && 
			((isWds == FALSE) || (pAddSta->u8WDS_client_type != PEER_AP)))
		{
			StaDB_SetWepSecurityKey(pAddSta->u16SID, pAddSta->u8VapIndex, vapCipherSuite);
		}
	}
	else
	{
		/*Change STA State to Wait For Filter*/
		SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_WAIT_FILTER, StaDb);
	}
	/*Copy STA flags*/
	StaDb->flags = pAddSta->u8Flags;
	/*Save VAP ID*/
	StaDb->vapId = pAddSta->u8VapIndex;
	/*Clear message pointer*/
	StaDb->pMsg = NULL;
	/*Send open STA Confirmation to STA Manager*/
	SecManager_StaManagerSendConfirm(StaDb->sid);
}

/**********************************************************************************

SecManager_StaManagerLockSta 


Description:
------------
handles STA Lock event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
static void SecManager_StaManagerLockSta(SecManagerStaDb_t *StaDb)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	K_MSG *pMsg = (K_MSG *)StaDb->pMsg;
	UMI_SET_KEY *psSetKey = (UMI_SET_KEY *)pK_MSG_DATA(pMsg);
	bool rekeyDisableFlag = FALSE; //rekey is activated
	uint8 mfp = 0;
	uint8 isQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	/*Flush TID 7*/
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = StaDb->sid;
	if (isQos)
	{
		hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_VO_2;
	}
	else
	{
		hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_BE_1;
	}
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
	{
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
#ifdef ENET_INC_ARCH_WAVE600
		GeneralStatistics.dropReasonSetKey[ConfigurationManager_GetBandForStation(StaDb->sid)] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
		PerClientStatistics.dropCntReasonSetKey[StaDb->sid] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#else
		GeneralStatistics.dropReasonSetKey += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#endif //ENET_INC_ARCH_WAVE600

		/*head and tail already include the PDs*/
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
	}
	/*Now set the key*/
	/*Handle Set Key*/
	StaDB_SetSecurityKey(psSetKey, rekeyDisableFlag);
	/*Get MFP flag from STA DB*/
	mfp = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_MFP);
	if (mfp == TRUE)
	{
		StaDB_SetManagementEncrypted(StaDb->sid);
	}
	/*Clear message pointer*/
	StaDb->pMsg = NULL;
	/*Send SET KEY confirmation*/
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_KEY_CFM, TASK_UM_IF_TASK, pMsg, StaDb->vapId);
	/*Change STA State to Wait for filter*/
	SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_WAIT_FILTER, StaDb);
	/*Unlock TID 7*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID ,StaDb->sid, (0x1 << IEEE802_1D_VO_2));
}
#endif

/**********************************************************************************

SecManager_StaManagerLockRekey 


Description:
------------
Lock STA event handler in Rekey State


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerLockRekey(SecManagerStaDb_t *StaDb)
{
	uint8 mfp = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_MFP);
	uint8 isQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
	uint8 i = 0;
	RequesterLockParams_t lockRequesterParams;

	/*Flush Retransmissions from Queues*/
	if (isQos)
	{
		/*QoS STA: flush retransmissions from all TX queues*/
		for (i = 0; i < NUM_OF_TID; i++)
		{
			SecManager_StaManagerFlushQueue(StaDb, i);
		}
	}
	else
	{
		/*Non-QoS STA: flush retransmissions from single queue*/
		SecManager_StaManagerFlushQueue(StaDb, IEEE802_1D_BE_1);
	}
	/*If PMF we mayhave retransmitted CSA action frame in GPHP*/
	if (mfp)
	{
		/*need to lock GPHP in case there is a retransmitted CSA frame there*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		lockRequesterParams.callerContext = (void *)StaDb;
		lockRequesterParams.returnMsg = ENCRYPTION_LOCK_GPHP;
		lockRequesterParams.returnTask = TASK_ENCRYPTION;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, StaDb->vapId, 0, &lockRequesterParams);
		/*We stay in rekey state*/
	}
	else
	{
		/*No need to lock GPHP - we can finish the rekey process*/
		SecManager_StaManagerFinishRekey(StaDb);
	}
}

/**********************************************************************************

SecManager_StaManagerGphpRekey 


Description:
------------
Lock GPHP event handler in Rekey State


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerGphpRekey(SecManagerStaDb_t *StaDb)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	TxPd_t *currentPacketDescriptor = NULL;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_GPHP;
	hwQueueManagerRequestParams.primaryAddr = StaDb->vapId;
	hwQueueManagerRequestParams.secondaryAddr = 0;
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
	if (((TxPd_t *)NULL_PD != currentPacketDescriptor) &&
		 ((currentPacketDescriptor->pdType == PD_TYPE_CH_SWITCH_ANN)
#if defined (ENET_INC_ARCH_WAVE600)
		 || (currentPacketDescriptor->pdType == PD_TYPE_CH_SWITCH_ANN_ENC)
#endif
		 ) &&
		 (currentPacketDescriptor->txQStaId == StaDb->sid))
	{
		/*CSA announcement for this STA at head of queue*/
		HwQManager_PopPacket(&hwQueueManagerRequestParams);
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
	}
	/*Now we can finish the rekey process*/
	SecManager_StaManagerFinishRekey(StaDb);
}

/**********************************************************************************

SecManager_StaManagerSetKey



Description:
------------
handles Set Key event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerSetKey(SecManagerStaDb_t *StaDb)
{
	K_MSG *pMsg = (K_MSG *)StaDb->pMsg;
	UMI_SET_KEY *psSetKey = (UMI_SET_KEY *)pK_MSG_DATA(pMsg);
	bool rekeyDisableFlag = TRUE; //rekey is not activated - only init of key
	uint8 mfp = 0;
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	RequesterLockParams_t lockRequesterParams;
	uint8 isStaQos;
#endif

#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	if (SecManager_GlobalDb.flushOnSetKey == FALSE)
#endif		
	{
		/*Handle Set Key*/
		StaDB_SetSecurityKey(psSetKey, rekeyDisableFlag);
		/*Get MFP flag from STA DB*/
		mfp = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_MFP);
		if (mfp == TRUE)
		{
			StaDB_SetManagementEncrypted(StaDb->sid);
		}
		/*Clear message pointer*/
		StaDb->pMsg = NULL;
		/*Send SET KEY confirmation*/
		OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_KEY_CFM, TASK_UM_IF_TASK, pMsg, StaDb->vapId);
		/*We remain in same state - wait for filter*/
	}
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	else
	{
		isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request Locker to lock TID 7*/
		lockRequesterParams.callerContext = (void *)StaDb;
		lockRequesterParams.returnMsg = ENCRYPTION_LOCK_STA;
		lockRequesterParams.returnTask = TASK_ENCRYPTION;
		if (isStaQos)
		{
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_VO_2, &lockRequesterParams);
		}
		else
		{
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_BE_1, &lockRequesterParams);
		}
		/*Change state to wait for lock*/
		SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_WAIT_LOCK, StaDb);
	}
#endif	
}

/**********************************************************************************

SecManager_StaManagerReKey


Description:
------------
handles Re Key event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerReKey(SecManagerStaDb_t *StaDb)
{
	RequesterLockParams_t lockRequesterParams;
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	/*First Lock queues*/
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	lockRequesterParams.callerContext = (void *)StaDb;
	/*Request Locker to lock TID 7*/
	lockRequesterParams.returnMsg = ENCRYPTION_LOCK_STA;
	lockRequesterParams.returnTask = TASK_ENCRYPTION;
	if (isStaQos)
	{
		Locker_LockAllStaQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid, &lockRequesterParams);
	}
	else
	{
		Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_BE_1, &lockRequesterParams);
	}
	/*Change state to rekey*/
	SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_REKEY, StaDb);
}

/**********************************************************************************

SecManager_StaManagerKeyIndex




Description:
------------
handles Set Index event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerKeyIndex(SecManagerStaDb_t *StaDb)
{
	K_MSG *pMsg = (K_MSG *)StaDb->pMsg;
	UMI_DEFAULT_KEY_INDEX *pDefaultKey = (UMI_DEFAULT_KEY_INDEX *)pK_MSG_DATA(pMsg);
	
	StaDB_SetDefaultKeyIndex(pDefaultKey->u16SID, pDefaultKey->u16KeyIndex, TRUE);
	StaDb->pMsg = NULL;
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_DEFAULT_KEY_INDEX_CFM, TASK_UM_IF_TASK, pMsg, StaDb->vapId);
}

/**********************************************************************************

SecManager_StaManagerSetFilter


Description:
------------
handles STA Close event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerSetFilter(SecManagerStaDb_t *StaDb)
{
	uint8 isWds = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WDS);
		
	if ((isWds == TRUE) && (StaDB_GetWdsClientType(StaDb->sid) == FOUR_ADDRESSES_STATION))
	{
		/*If this is a 4 address STA we set the WDS bit because the EAPOLs have to be transmitted in 3 address mode*/
		StaDB_SetNegotiatedWds(StaDb->sid, TRUE);
	}
	/*Change STA State to Open*/
	SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_OPEN, StaDb);
	/*Clear message pointer*/
	StaDb->pMsg = NULL;
	/*Send Close STA Confirmation to STA Manager*/
	SecManager_StaManagerSendConfirm(StaDb->sid);
}

/**********************************************************************************

SecManager_StaManagerClose 


Description:
------------
handles STA Close event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerCloseSta(SecManagerStaDb_t *StaDb)
{
	/*Clear security key*/
	StaDB_ClearSecurityKey(StaDb->sid, StaDb->vapId);
	/*Change STA State to Closed*/
	SecManager_StaManagerChangeState(SEC_MANAGER_STA_MANAGER_CLOSED, StaDb);
	/*Clear message pointer*/
	StaDb->pMsg = NULL;
	/*Send Close STA Confirmation to STA Manager*/
	SecManager_StaManagerSendConfirm(StaDb->sid);
}

/**********************************************************************************

SecManager_StaManagerFatal 


Description:
------------
handles unexpected event


Input: 
-----	
SecManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_StaManagerFatal(SecManagerStaDb_t *StaDb)
{
	UNUSED_PARAM(StaDb);	
	DEBUG_FATAL("SEC Manager STA Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

secManager_StaManagerAddSta 


Description:
------------
STA ADD message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerAddSta(K_MSG* secManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(secManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
 	StaId sid = pAddSta->u16SID;
 	SecManagerStaDb_t *staDb;
		
	/*From ADD STA message extract SID*/
	staDb = &SecManager_StaManagerDb[sid];
	/*Store Message*/
	DEBUG_ASSERT(staDb->pMsg == NULL);
	staDb->pMsg = pMsg;
	/*Call STA open handler*/
	SecManager_StaManagerStaOpen[staDb->state](staDb);
}

/**********************************************************************************

secManager_StaManagerLockSta 


Description:
------------
STA Lock message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerLockSta(K_MSG* secManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(secManagerMessage);
	SecManagerStaDb_t *StaDb = (SecManagerStaDb_t *)lockReg->callerContext;

	/*Extract context from Locker message = Sta DB*/
	SecManager_StaManagerStaLock[StaDb->state](StaDb);
}

/**********************************************************************************

secManager_StaManagerLockGphp 


Description:
------------
GPGP Lock message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerLockGphp(K_MSG* secManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(secManagerMessage);
	SecManagerStaDb_t *StaDb = (SecManagerStaDb_t *)lockReg->callerContext;

	/*Extract context from Locker message = Sta DB*/
	SecManager_StaManagerGphpLock[StaDb->state](StaDb);
}

/**********************************************************************************

secManager_StaManagerSetKey



Description:
------------
STA Set Keyhandler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerSetKey(K_MSG* secManagerMessage)
{
	UMI_SET_KEY *pSetKey = (UMI_SET_KEY *)pK_MSG_DATA(secManagerMessage);
 	StaId sid = pSetKey->u16Sid;
 	SecManagerStaDb_t *staDb;
	
	staDb = &SecManager_StaManagerDb[sid];
	/*Store Message*/
	DEBUG_ASSERT(staDb->pMsg == NULL);
	/*Store the K MSG as we need to send it back*/
	staDb->pMsg = secManagerMessage;
	/*Call STA Set Key handler*/
	SecManager_StaManagerStaSetKey[staDb->state](staDb);
}

/**********************************************************************************

secManager_StaManagerSetIndex




Description:
------------
STA Set Index handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerSetIndex(K_MSG* secManagerMessage)
{
	UMI_DEFAULT_KEY_INDEX *pSetIndex = (UMI_DEFAULT_KEY_INDEX *)pK_MSG_DATA(secManagerMessage);
 	StaId sid = pSetIndex->u16SID;
 	SecManagerStaDb_t *staDb;
		
	staDb = &SecManager_StaManagerDb[sid];
	/*Store Message*/
	DEBUG_ASSERT(staDb->pMsg == NULL);
	/*Store the K MSG as we need to send it back*/
	staDb->pMsg = secManagerMessage;
	/*Call STA Set Index handler*/
	SecManager_StaManagerStaSetIndex[staDb->state](staDb);
}

/**********************************************************************************

secManager_StaManagerSetFilter 


Description:
------------
STA Set Filter handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerSetFilter(K_MSG* secManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(secManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_802_1X_FILTER *pSetFilter = (UMI_802_1X_FILTER *)pK_MSG_DATA(pMsg);
 	StaId sid = pSetFilter->u16SID;
 	SecManagerStaDb_t *staDb;
		
	/*From STA Close message extract SID*/
	staDb = &SecManager_StaManagerDb[sid];
	/*Store Message*/
	DEBUG_ASSERT(staDb->pMsg == NULL);
	staDb->pMsg = pMsg;
	/*Call STA Close handler*/
	SecManager_StaManagerStaSetFilter[staDb->state](staDb);
}

/**********************************************************************************

secManager_StaManagerRemoveSta 


Description:
------------
STA Close message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerRemoveSta(K_MSG* secManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(secManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_REMOVE *pRemoveSta = (UMI_STA_REMOVE *)pK_MSG_DATA(pMsg);
 	StaId sid = pRemoveSta->u16SID;
 	SecManagerStaDb_t *staDb;
		
	/*From STA Close message extract SID*/
	staDb = &SecManager_StaManagerDb[sid];
	/*Store Message*/
	DEBUG_ASSERT(staDb->pMsg == NULL);
	staDb->pMsg = pMsg;
	/*Call STA Close handler*/
	SecManager_StaManagerStaClose[staDb->state](staDb);
}

 
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif



/**********************************************************************************

secManager_StaManagerInit


Description:
------------
Initiailize SEC MAnager STA Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_StaManagerInit(void)
{
 	StaId i;
 
#if defined (SEC_MANAGER_STA_MANAGER_FLUSH_ON_SET_KEY_OPTION)		
	/*Initialize flush on Set key variable*/
	SecManager_GlobalDb.flushOnSetKey = FALSE;
#endif
	/*Initialize DB*/
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		SecManager_StaManagerDb[i].sid = i;
		SecManager_StaManagerDb[i].state = SEC_MANAGER_STA_MANAGER_CLOSED;
		SecManager_StaManagerDb[i].pMsg = NULL;
	}
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif





