/***********************************************************************************
 File:			secManager_VapManager.c
 Module:		Sec manager VAP Manager
 Purpose: 		handle VAP Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "RxManager_Api.h"
#include "Utils_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "VapDatabase_Api.h"
#include "TxSelector_Api.h"
#include "Locker_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "secManager_VapManager.h"
#include "loggerAPI.h"
#include "RxManager_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_SECURITY
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

typedef void (*SecManager_VapManagerFunc)(SecManagerVapDb_t *vapDb);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void SecManager_VapManagerChangeState(secManagerVapManager_state state, SecManagerVapDb_t *StaDb);
static void SecManager_VapManagerSendConfirm(SecManagerVapDb_t *VapDb, vapManagerEvents_e event, uint8 clientId);
static void SecManager_VapManagerSetKeyAndConfirm(SecManagerVapDb_t *VapDb, bool rekeyDisableFlag);
static void SecManager_VapManagerSetIndexAndConfirm(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerAddVapInIdle(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerSetKeyInIdle(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerSetKeyInActive(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerSetIndexInIdle(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerSetIndexInActive(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerLockCfmInWaitLock(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerRemoveInActive(SecManagerVapDb_t *VapDb);
static void SecManager_VapManagerFatal(SecManagerVapDb_t *VapDb);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*VAP Add Event Handlers*/
static const SecManager_VapManagerFunc SecManager_VapManagerVapAdd[SEC_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	 SecManager_VapManagerAddVapInIdle,			  /* SEC_MANAGER_VAP_MANAGER_IDLE */
	 SecManager_VapManagerFatal,				  /* SEC_MANAGER_VAP_MANAGER_WAIT_LOCK */
	 SecManager_VapManagerFatal,				  /* SEC_MANAGER_VAP_MANAGER_ACTIVE */
};

/*VAP Set Key Handlers*/
static const SecManager_VapManagerFunc SecManager_VapManagerVapSetKey[SEC_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	SecManager_VapManagerSetKeyInIdle,			 /* SEC_MANAGER_VAP_MANAGER_IDLE */
	SecManager_VapManagerFatal, 				 /* SEC_MANAGER_VAP_MANAGER_WAIT_LOCK */
	SecManager_VapManagerSetKeyInActive,		 /* SEC_MANAGER_VAP_MANAGER_ACTIVE */
};

/*VAP Set Index Handlers*/
static const SecManager_VapManagerFunc SecManager_VapManagerVapSetIndex[SEC_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	SecManager_VapManagerSetIndexInIdle,		 /* SEC_MANAGER_VAP_MANAGER_IDLE */
	SecManager_VapManagerFatal, 				 /* SEC_MANAGER_VAP_MANAGER_WAIT_LOCK */
	SecManager_VapManagerSetIndexInActive,		 /* SEC_MANAGER_VAP_MANAGER_ACTIVE */
};

/*VAP Lock CFM Handlers*/
static const SecManager_VapManagerFunc SecManager_VapManagerVapLockCfm[SEC_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	SecManager_VapManagerFatal,		 			 /* SEC_MANAGER_VAP_MANAGER_IDLE */
	SecManager_VapManagerLockCfmInWaitLock,		 /* SEC_MANAGER_VAP_MANAGER_WAIT_LOCK */
	SecManager_VapManagerFatal,		 			 /* SEC_MANAGER_VAP_MANAGER_ACTIVE */
};

/*VAP Remove Handlers*/
static const SecManager_VapManagerFunc SecManager_VapManagerVapRemove[SEC_MANAGER_VAP_MANAGER_NUM_STATES] =
{
	SecManager_VapManagerFatal,		 			 /* SEC_MANAGER_VAP_MANAGER_IDLE */
	SecManager_VapManagerFatal, 				 /* SEC_MANAGER_VAP_MANAGER_WAIT_LOCK */
	SecManager_VapManagerRemoveInActive,		 /* SEC_MANAGER_VAP_MANAGER_ACTIVE */
};

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

SecManagerVapDb_t SecManager_VapManagerDb[HW_NUM_OF_VAPS];

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

SecManager_VapManagerChangeState 


Description:
------------
Utiltity function to change VAP state

Input: 
-----
VAP DB entry
New State

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerChangeState(secManagerVapManager_state state, SecManagerVapDb_t *VapDb)
{
	ILOG0_DDD("SEC Manager VAP Manager, Vap %d changed from %d to %d", VapDb->vapId, VapDb->state, state);
	VapDb->state = state;
}

/**********************************************************************************

SecManager_VapManagerSendConfirm 


Description:
------------
Utiltity function to send confirmation to BSS MAnager VAP manager

Input: 
-----
VAP DB entry
New State

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSendConfirm(SecManagerVapDb_t *VapDb, vapManagerEvents_e event, uint8 clientId)
{
	K_MSG*	psMsg = VapDb->pMsg;

	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg, VapDb->vapId, event, clientId);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, VapDb->vapId);
	VapDb->pMsg = NULL;
}

/**********************************************************************************

SecManager_VapManagerSetKeyAndConfirm 


Description:
------------
Utiltity function that sets the Index and sends confirmation

Input: 
-----
VAP DB entry

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSetKeyAndConfirm(SecManagerVapDb_t *VapDb, bool rekeyDisableFlag)
{
	K_MSG *pMsg = VapDb->pMsg;
	UMI_SET_KEY *psSetKey = (UMI_SET_KEY *)pK_MSG_DATA(pMsg);
	uint8 vapId = pMsg->header.vapId;
	uint8* pnStaMode = psSetKey->au8RxSeqNum;
	uint8 keyId = 0;

	if ((psSetKey->u16CipherSuite != UMI_RSN_CIPHER_SUITE_WEP40) && (psSetKey->u16CipherSuite != UMI_RSN_CIPHER_SUITE_WEP104))
	{
		/*Need to notify RX manager*/
		RxManager_SetGroupKey(pMsg->header.vapId, psSetKey);
		keyId = psSetKey->u16KeyIndex;
		// reduce keyId by 1 in order to fit the lastPN array - keyId 1 will be assigned to 0, keyId 2 will be assigned to 1	
		keyId--;
		// set initial PN ( Station mode ) according to Vap index and key Index
		rxManagerSetPnStaMode(vapId, pnStaMode, keyId);
	}
	VapDB_SetSecurityKey(vapId, rekeyDisableFlag, psSetKey);	
	ILOG2_DD("vUM_MAN_SET_KEY_REQ: u16Status:%d psMsg->header.vapId:%d",psSetKey->u16Status,vapId);
	/*Send SET KEY confirmation*/
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_KEY_CFM, TASK_UM_IF_TASK, pMsg, vapId);
	/*Clear message in DB*/
	VapDb->pMsg = NULL;
}

/**********************************************************************************

SecManager_VapManagerSetIndexAndConfirm 


Description:
------------
Utiltity function that sets the Index and sends confirmation

Input: 
-----
VAP DB entry

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSetIndexAndConfirm(SecManagerVapDb_t *VapDb)
{
	K_MSG *pMsg = (K_MSG *)VapDb->pMsg;
	UMI_DEFAULT_KEY_INDEX *pDefaultKey = (UMI_DEFAULT_KEY_INDEX *)pK_MSG_DATA(pMsg);
	
	VapDB_SetDefaultKeyIndex(VapDb->vapId, pDefaultKey->u16KeyIndex);
	VapDb->pMsg = NULL;
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_DEFAULT_KEY_INDEX_CFM, TASK_UM_IF_TASK, pMsg, VapDb->vapId);
}

/**********************************************************************************

SecManager_VapManagerAddVapInIdle 


Description:
------------
ADD VAP in Idle handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerAddVapInIdle(SecManagerVapDb_t *VapDb)
{
	/*If we have WEP keys stored for this VAP - need to set VAP encryption type to WEP*/
	/*Change state to Active*/
	SecManager_VapManagerChangeState(SEC_MANAGER_VAP_MANAGER_ACTIVE, VapDb);
	/*We are done*/
	SecManager_VapManagerSendConfirm(VapDb, VAP_MANAGER_ADD_VAP, BSS_MANAGER_VAP_MANAGER_SEC_MANAGER_CLIENT);
}

/**********************************************************************************

SecManager_VapManagerSetKeyInIdle 


Description:
------------
Set Key in Idle handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSetKeyInIdle(SecManagerVapDb_t *VapDb)
{
	/*Should only happen for WEP - store Keys in Keys DB, mark VAP as WEP*/
	K_MSG *pMsg = VapDb->pMsg;
	UMI_SET_KEY *psSetKey = (UMI_SET_KEY *)pK_MSG_DATA(pMsg);
	bool rekeyDisableFlag = TRUE; //IDLE - therefore this is init key and not rekey

	/* WLANRTSYS-6161: remove assert since there is no prevention from FW side to receive GROUP CCMP key before ADD_VAP.  DEBUG_ASSERT((psSetKey->u16CipherSuite == UMI_RSN_CIPHER_SUITE_WEP40) || (psSetKey->u16CipherSuite == UMI_RSN_CIPHER_SUITE_WEP104)); */
	VapDb->cipherSuite = psSetKey->u16CipherSuite;
	SecManager_VapManagerSetKeyAndConfirm(VapDb, rekeyDisableFlag);
}

/**********************************************************************************

SecManager_VapManagerSetKeyInActive 


Description:
------------
Set Key in Active handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSetKeyInActive(SecManagerVapDb_t *VapDb)
{
	/*Should only happen in CCMP & TKIP - need to lock VAP queue first*/
	K_MSG *pMsg = (K_MSG *)VapDb->pMsg;
	UMI_SET_KEY *psSetKey = (UMI_SET_KEY *)pK_MSG_DATA(pMsg);
	RequesterLockParams_t lockRequesterParams;

	/*Store Cipher Suite*/
	VapDb->cipherSuite = psSetKey->u16CipherSuite;
	/*Request Locker to lock VAP queues*/
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	lockRequesterParams.callerContext = (void *)VapDb;
	lockRequesterParams.returnMsg = ENCRYPTION_LOCK_VAP;
	lockRequesterParams.returnTask = TASK_ENCRYPTION;
	Locker_LockAllStaQueues(HW_TX_Q_TYPE_VAP_TID, VapDb->vapId, &lockRequesterParams);
	/*Change state to wait for lock*/
	SecManager_VapManagerChangeState(SEC_MANAGER_VAP_MANAGER_WAIT_LOCK, VapDb);
}

/**********************************************************************************

SecManager_VapManagerSetIndexInIdle 


Description:
------------
Set Index in Idle handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSetIndexInIdle(SecManagerVapDb_t *VapDb)
{
	/*Should only happen in WEP*/
	/* WLANRTSYS-6161: remove assert since there is no prevention from FW side to receive GROUP CCMP key before ADD_VAP. DEBUG_ASSERT((VapDb->cipherSuite == UMI_RSN_CIPHER_SUITE_WEP40) || (VapDb->cipherSuite == UMI_RSN_CIPHER_SUITE_WEP104)); */
	SecManager_VapManagerSetIndexAndConfirm(VapDb);
}

/**********************************************************************************

SecManager_VapManagerSetIndexInActive 


Description:
------------
Set Index in Active handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerSetIndexInActive(SecManagerVapDb_t *VapDb)
{
	/*Should only happen in WEP - need to loop over all connected STAs and change the default Key Index - except WDS*/
	SecManager_VapManagerSetIndexAndConfirm(VapDb);
}

/**********************************************************************************

SecManager_VapManagerLockCfmInWaitLock 


Description:
------------
Lock CFM In Wait Lock handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerLockCfmInWaitLock(SecManagerVapDb_t *VapDb)
{
	bool rekeyDisableFlag = FALSE; //need to lock queue - therefore this is rekey and not init

	/*Should only happen in CCMP & TKIP - set Key now*/
	SecManager_VapManagerSetKeyAndConfirm(VapDb, rekeyDisableFlag);
	/*Unlock queues*/
	Locker_UnLockAllStaQueues(HW_TX_Q_TYPE_VAP_TID, VapDb->vapId);
	/*Change state to Active again*/
	SecManager_VapManagerChangeState(SEC_MANAGER_VAP_MANAGER_ACTIVE, VapDb);
}

/**********************************************************************************

SecManager_VapManagerRemoveInActive 


Description:
------------
Remove VAP in Active handler

Input: 
-----
VAP DB entry


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerRemoveInActive(SecManagerVapDb_t *VapDb)
{
	/*Clear Security Key*/
	VapDB_ClearSecurityKey(VapDb->vapId);
	/*Clear cipher suite*/
	VapDb->cipherSuite = UMI_RSN_CIPHER_SUITE_NONE;
	/*Change state to Idle*/
	SecManager_VapManagerChangeState(SEC_MANAGER_VAP_MANAGER_IDLE, VapDb);
	/*We are done*/
	SecManager_VapManagerSendConfirm(VapDb, VAP_MANAGER_REMOVE_VAP, BSS_MANAGER_VAP_MANAGER_SEC_MANAGER_CLIENT);	
}

/**********************************************************************************

SecManager_VapManagerFatal 


Description:
------------
handles unexpected event


Input: 
-----
VAP DB entry

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SecManager_VapManagerFatal(SecManagerVapDb_t *VapDb)
{
	UNUSED_PARAM(VapDb);	
	FATAL("SEC Manager STA Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

secManager_VapManagerGetCipherSuite 


Description:
------------
Utility function that returns the VAP cipher suite


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 secManager_VapManagerGetCipherSuite(uint8 vapId)
{
	return SecManager_VapManagerDb[vapId].cipherSuite;
}

/**********************************************************************************

secManager_VapManagerAddVap 


Description:
------------
VAP ADD message message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_VapManagerAddVap(K_MSG* secManagerMessage)
{
	uint8 vapId; 
	UMI_ADD_VAP* addVapStructurePtr = (UMI_ADD_VAP*)EXTRACT_VAP_MANAGER_MSG(secManagerMessage);
	SecManagerVapDb_t *vapDb;

	vapId = addVapStructurePtr->vapId; 
	/*From ADD VAP message extract VAP Index*/
	vapDb = &SecManager_VapManagerDb[vapId];
	/*Store Message*/
	DEBUG_ASSERT(vapDb->pMsg == NULL);
	vapDb->pMsg = secManagerMessage;
	/*Call VAP open handler*/
	SecManager_VapManagerVapAdd[vapDb->state](vapDb);
}

/**********************************************************************************

secManager_VapManagerLockVap 


Description:
------------
VAP Lock message message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_VapManagerLockVap(K_MSG* secManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(secManagerMessage);
	SecManagerVapDb_t *VapDb = (SecManagerVapDb_t *)lockReg->callerContext;

	/*Extract context from Locker message = VapDb DB*/
	SecManager_VapManagerVapLockCfm[VapDb->state](VapDb);
}


/**********************************************************************************

secManager_VapManagerSetKey




Description:
------------
VAP Set Key message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_VapManagerSetKey(K_MSG* secManagerMessage)
{
	/*Ugly - but the VAP ID is in the KMSG header*/
	uint8 vapId = secManagerMessage->header.vapId;
	SecManagerVapDb_t *VapDb;
		
	VapDb = &SecManager_VapManagerDb[vapId ];
	/*Store Message*/
	DEBUG_ASSERT(VapDb->pMsg == NULL);
	/*Store the K MSG as we need to send it back*/
	VapDb->pMsg = secManagerMessage;
	/*Call STA Set Key handler*/
	SecManager_VapManagerVapSetKey[VapDb->state](VapDb);
}

/**********************************************************************************

secManager_VapManagerSetIndex 


Description:
------------
VAP Set Index message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_VapManagerSetIndex(K_MSG* secManagerMessage)
{
	/*Ugly, but the VAP is in the KMSG header*/
	uint8 vapId = secManagerMessage->header.vapId;
	SecManagerVapDb_t *VapDb;
		
	VapDb = &SecManager_VapManagerDb[vapId ];
	/*Store Message*/
	DEBUG_ASSERT(VapDb->pMsg == NULL);
	/*Store the K MSG as we need to send it back*/
	VapDb->pMsg = secManagerMessage;
	/*Call STA Set Key handler*/
	SecManager_VapManagerVapSetIndex[VapDb->state](VapDb);
}

/**********************************************************************************

secManager_VapManagerRemoveVap 


Description:
------------
VAP Remove message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_VapManagerRemoveVap(K_MSG* secManagerMessage)
{
	uint8 vapId; 
	UMI_REMOVE_VAP* removeVapStructurePtr = (UMI_REMOVE_VAP*) EXTRACT_VAP_MANAGER_MSG(secManagerMessage);
	SecManagerVapDb_t *vapDb;

	vapId = removeVapStructurePtr->vapId; 
	/*From remove VAP message extract VAP Index*/
	vapDb = &SecManager_VapManagerDb[vapId];
	/*Store Message*/
	DEBUG_ASSERT(vapDb->pMsg == NULL);
	vapDb->pMsg = secManagerMessage;

	ILOG2_V("secManager_VapManagerRemoveVap");

	/*Call VAP open handler*/
	SecManager_VapManagerVapRemove[vapDb->state](vapDb);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif


/**********************************************************************************

secManager_StaManagerInit


Description:
------------
Initiailize SEC MAnager STA Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void secManager_VapManagerInit(void)
{
	uint8 i;

	/*Initialize DB*/
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		SecManager_VapManagerDb[i].vapId = i;
		SecManager_VapManagerDb[i].state = SEC_MANAGER_VAP_MANAGER_IDLE;
		SecManager_VapManagerDb[i].pMsg = NULL;
		SecManager_VapManagerDb[i].cipherSuite = UMI_RSN_CIPHER_SUITE_NONE;		
	}
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif





