/*************************************************************************/
/***                        Include Files                              ***/
/*************************************************************************/
#include "AlphaFilter.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "lm.h"
#include "MT_Math.h"
#include "Utils_Api.h"
/***************************************************************************/
/* 					Constants                                                  				     */
/***************************************************************************/
#define LOG_LOCAL_GID   GLOBAL_GID_LINK_ADAPTATION
#define LOG_LOCAL_FID 0

#define DEBUG_FIXED_ALPHA 0xcc //fixed alpha = 0.8 in 11 bits fraction =>0.8*2^11

/***************************************************************************/
/***                  Static function prototypes                                 						 ***/
/***************************************************************************/
static void adaptiveAlphaFilterCalc(uint8 filterInput, AdaptiveAlphaFilterParams_t* pFilterParams, uint16 diffTsf, uint16 lnBeta, uint16 dTAverage);
static uint16 alphaCalc (uint16 lnBeta, uint16 lastAlpha);
static uint16  betaCalc (uint16 diffTsf, uint16 lnBeta,uint16 dtAverage);

/***************************************************************************/
/***                  Static data                                       							 ***/
/***************************************************************************/


/****************************************************************************
 **
 ** NAME:          AlphaFilter_updateFilter
 **
 ** PARAMETERS:    
 ** RETURN VALUES:
 **
 ** DESCRIPTION:   
** Function calculates adaptive alpha filter according to the following equation:
** filter result = newAlpha * filterInput + (1 - newAlpha) *  (last filterResult) //32 bits, 15 bits raction
** alpha = (current alpha) / ((current alpha) + beta) //16 bits, 15 bits fraction
** beta = beta^diffTsf = exp(Ln(Beta)*DiffTsf), calculate approximately //16 bits 15 bits fraction
 ***************************************************************************/
void AlphaFilter_updateFilter(AdaptiveAlphaFilterParams_t* pFilterParams, uint8 filterInput, int16 lnBeta, uint8 constAlphaForDtAverage,uint16* dTAverage, bool updateDtAverage) 
{
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	uint16 adaptTsf; 
	uint16 diffTsf;
	uint16 averageDtResult = *dTAverage;
	
	adaptTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);

	if (pFilterParams->lastPacketTsf != TSF_INVALID)
	{
		diffTsf = cyclicMinusOperator (adaptTsf, pFilterParams->lastPacketTsf, TRUE, TSF_RANGE_ALPHA_FILTER); 
		diffTsf <<= DT_AVERAGE_FACTOR;

		/*Calculate constant alpha filter on average DT */
		if (updateDtAverage == TRUE)
		{
			averageDtResult = PHY_SHAPED_FORMULA(diffTsf,*dTAverage, constAlphaForDtAverage);
		}
	}
	else
	{
		/*When Tsf invalid, beta calc should get the maximum diff time so the beta will be zero and last sample will be taken for filter calculation*/
		diffTsf = MAX_UINT16; 
		updateDtAverage = FALSE;
	}

	adaptiveAlphaFilterCalc (filterInput, pFilterParams, diffTsf, lnBeta, averageDtResult);
	pFilterParams->lastPacketTsf = adaptTsf;

	if (updateDtAverage == TRUE)
	{
		*dTAverage = averageDtResult;
	}
//#ifdef LINK_ADAPTATION_LOGS
	//	filterResult = AlphaFilter_GetFilterResult(pFilterParams);
//		ILOG0_DDD("AlphaFilter_updateFilter, currentTsf = %d,lastPacketTsf = %d, adaptTsf = %d",currentTsf,pFilterParams->lastPacketTsf, adaptTsf);
//#endif


}

/****************************************************************************
 **
 ** NAME:          adaptiveAlphaFilterCalc
 **
 ** PARAMETERS:    
 ** RETURN VALUES:
 **
 ** DESCRIPTION:   
 **              
 ***************************************************************************/
static void adaptiveAlphaFilterCalc(uint8 filterInput, AdaptiveAlphaFilterParams_t* pFilterParams, uint16 diffTsf, uint16 lnBeta, uint16 dTAverage)
{
	uint16 lastAlpha = pFilterParams->alpha;
	uint32 lastSn = pFilterParams->filterResult;
	uint16 beta = 0;

	beta = betaCalc(diffTsf, lnBeta, dTAverage);
	ASSERT(!((lastAlpha == 0) && (beta == 0)));
	pFilterParams->alpha = alphaCalc (beta, lastAlpha);
	//pFilterParams->alpha = DEBUG_FIXED_ALPHA;

#ifdef LINK_ADAPTATION_LOGS
//	ILOG0_D("adaptiveAlphaFilterCalc, last filter result = %d", lastSn); 
#endif
	pFilterParams->filterResult = pFilterParams->alpha*filterInput + ((((1 << FILTER_RESULT_NUMBER_OF_FRAC_BITS) - pFilterParams->alpha)*lastSn)>>FILTER_RESULT_NUMBER_OF_FRAC_BITS);

}
/****************************************************************************
 **
 ** NAME:          alphaCalc
 **
 ** PARAMETERS:    
 ** RETURN VALUES:
 **
 ** DESCRIPTION:   
 **              
 ***************************************************************************/
static uint16 alphaCalc (uint16 beta, uint16 lastAlpha)
{
	uint32 lastAlphaWithFact;
	uint32 alpha;

	lastAlphaWithFact = lastAlpha << (FILTER_RESULT_NUMBER_OF_FRAC_BITS);

	alpha = (lastAlphaWithFact/(lastAlpha + beta));

	alpha = (alpha==0) ? 1 : alpha;

	return (uint16)alpha;
}
/***********************************************************************************************************
 **
 ** NAME:          betaCalc
 **
 ** PARAMETERS: diffTsf, Lnbeta  represent with 11 bit fraction
 ** RETURN VALUES:beta 15 bit fraction representation
 **
 ** DESCRIPTION:   
 ** 
 ** Calculation based on SAS (3.23.2.4.7) definition for CCA adaptive alpha filter calculation  
 ** Function calculate approximate beta^difftime using the follwing equation:
 ** beta^diffTime = Exp(Ln(beta)* DiffTime)
 ** exp(x) = (2^k)*(1 + x/Ln2 - k)
 ** k = floor (x/ln2)
**
 ** Function gets Ln(Beta) as parameter in 11 bit fraction representation and use constant LN2 also in 11 bit fraction representation
 ************************************************************************************************************/
static uint16  betaCalc (uint16 diffTsf, uint16 lnBeta,uint16 dTAverage)
{
	uint32 lnBetaMultDiffTsf = 0;
	int32 k;
	uint8 multiplyAccomulatefractionFactor = 0;
	uint32 kFloor = 0;
	uint32 betaRes = 0;
	int32 kfloorNoFactor = 0;
	uint32 diffTsfWithFact;
	uint32 diffTsfDivAvTsfWithFact;
	
	dTAverage = MAX(dTAverage,1); // dTAverage could not be zero!
	
	diffTsfWithFact = (diffTsf << DIFF_TSF_NUMBER_OF_FRC_BITS);
	diffTsfDivAvTsfWithFact = (diffTsfWithFact/dTAverage);

	/*Calculate DiffTime*LnBeta*/
	lnBetaMultDiffTsf = (diffTsfDivAvTsfWithFact*lnBeta) >> BETA_CALC_INTERMEDIATE_MULTIPLY_FACTOR_SHIFT;		//accomulate log2(factor) (shift)=  BETA_CALC_NUMBER_OF_BITS*2 - BETA_CALC_INTERMEDIATE_MULTIPLY_FACTOR_SHIFT
	multiplyAccomulatefractionFactor =  (BETA_CALC_NUMBER_OF_FRAC_BITS + DIFF_TSF_NUMBER_OF_FRC_BITS) - BETA_CALC_INTERMEDIATE_MULTIPLY_FACTOR_SHIFT	;	

	/*Calculate k =  Ln(beta) * DiffTsf *Ln2*/
	k = ((lnBetaMultDiffTsf*LN2_INV_11BIT_FRACTION) >> BETA_CALC_INTERMEDIATE_MULTIPLY_FACTOR_SHIFT);	//accomulate log2(factor) (shift)  = BETA_CALC_NUMBER_OF_BITS*3 - 2*BETA_CALC_INTERMEDIATE_MULTIPLY_FACTOR_SHIFT
	multiplyAccomulatefractionFactor +=  BETA_CALC_NUMBER_OF_FRAC_BITS - BETA_CALC_INTERMEDIATE_MULTIPLY_FACTOR_SHIFT;	

	/*In order to simplify the calculation and use bits shift instead of deviation, Ln beta take as positive number, now we should take into account the negative sign*/
	k*=-1;

	/*Calculate floor(k) =  k / 2^multiplyAccomulatefractionFactor, take only the integer part of the number*/
	kfloorNoFactor = (k >> multiplyAccomulatefractionFactor);
	kfloorNoFactor = -kfloorNoFactor > 31 ? 31 :  -kfloorNoFactor;

	/*Calculate kFloor * 2^multiplyAccomulatefractionFactor factor, so it will be normlized with other  arguments used in betaRes calculation*/
	kFloor = ((k >> multiplyAccomulatefractionFactor) << multiplyAccomulatefractionFactor); //get only the integer part of the number and facorize back to multiplyAccomulatefractionFactor

	/*Calculate beta^diffTime result = ((1 + k - floor(k)) * 2^kfloorNoFactor) / multiplyAccomulatefractionFactor*/
	betaRes = ((1 << multiplyAccomulatefractionFactor) + k - kFloor) >> (kfloorNoFactor);

	DEBUG_ASSERT(multiplyAccomulatefractionFactor >= FILTER_RESULT_NUMBER_OF_FRAC_BITS);

	/*Normelized beta^diffTime result to be FILTER_RESULT_NUMBER_OF_FRAC_BITS number of bits*/
	betaRes >>= (multiplyAccomulatefractionFactor - FILTER_RESULT_NUMBER_OF_FRAC_BITS);

//#ifdef LINK_ADAPTATION_LOGS
//	("betaCalc,diffTsf = %d, diffTsfDivAvTsfWithFact = %d, lnBeta = %d,lnBetaMultDiffTsf = %d, k = %d, kFloor = %d,betaRes = %d, kfloorNoFactor = %d",diffTsf, diffTsfDivAvTsfWithFact, lnBeta, lnBetaMultDiffTsf,k, kFloor,betaRes, (-kfloorNoFactor) );
//#endif
	return (uint16)betaRes;
}
/****************************************************************************/
/*
 **
 ** NAME:          AlphaFilter_Init
 **
 ** PARAMETERS:    
 ** RETURN VALUES:
 **
 ** DESCRIPTION:   
 **              
 ***************************************************************************/
void AlphaFilter_Init(AdaptiveAlphaFilterParams_t* pFilterParams, uint16* DeltaTAverage)
{
	
	pFilterParams->alpha = (1 << FILTER_RESULT_NUMBER_OF_FRAC_BITS);
	pFilterParams->filterResult = 0;
	pFilterParams->lastPacketTsf = TSF_INVALID;
	*DeltaTAverage = (AVERAGE_DT_INIT_VAL<<DT_AVERAGE_FACTOR);
}
/****************************************************************************/
/*
 **
 ** NAME:          AlphaFilter_isFilterResultValid
 **
 ** PARAMETERS:    
 ** RETURN VALUES:
 **
 ** DESCRIPTION:   
 **              
 ***************************************************************************/

bool AlphaFilter_isFilterResultValid (AdaptiveAlphaFilterParams_t* filterParams, uint32 maxValidTsfDiff)
{
	uint32 currentTsf = GET_TSF_TIMER_LOW();
	uint32 cyclicMinusOperatorValue;
	bool retValue;

	currentTsf = CALC_TSF_FOR_ALPHA_FILTER(currentTsf);
	cyclicMinusOperatorValue = cyclicMinusOperator(currentTsf, filterParams->lastPacketTsf, TRUE, TSF_RANGE_ALPHA_FILTER);

	if (filterParams->lastPacketTsf != TSF_INVALID)
	{
		retValue = (cyclicMinusOperatorValue <= maxValidTsfDiff);
	}
	else
	{
		retValue = FALSE;
	}
	return (retValue);
}
/****************************************************************************/
/*
 **
 ** NAME:          AlphaFilter_GetFilterResult
 **
 ** PARAMETERS:    
 ** RETURN VALUES:
 **
 ** DESCRIPTION:   
 **              
 ***************************************************************************/

uint8 AlphaFilter_GetFilterResult(AdaptiveAlphaFilterParams_t* pFilterParams)
{
	return (NORMELIZE_ALPHA_FILTER_RESULT(pFilterParams->filterResult));
}
