/***********************************************************************************
 File:			CalendarWheel.c
 Module:		Calendar wheel 
 Purpose: 		To create a module which supply scheduling events ability
 Description:   This file is the implementation of the Calendar wheel module 
				which is responsible of organizing events to be scheduled. 
				each module that use calendar wheel should create its own istance
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "CalendarWheel_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"

#if defined(ENET_INC_UMAC)
#include "string.h"
#endif /* #ifdef ENET_INC_UMAC */

#include "stringLibApi.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_CALENDAR_WHEEL 
#define LOG_LOCAL_FID 0


/*
DIVIDE BY 12 in 2 cycles.
WARNING!!! - Valid only if x is 32 bits and:
1.	For Input%12=0: x < 24564.
2.	For input%12 !=0: x<2048
*/
#define DIVIDE_BY_12(x) (((x)*1366)>>14) 

/* This is accurate when the are no more than 2047 uints */
//#define DIVIDE_BY_20(x) (((x)*820)>>14) 


#define DIVIDE_BY_13(x) (((x)*53687)>>19) 






/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static uint16 CalendarWheel_DivideBy20(uint32 distanceFromBase);
static uint16 CalendarWheel_DivideBy16(uint32 distanceFromBase);
static uint16 CalendarWheel_DivideBy12(uint32 distanceFromBase);
static uint16 CalendarWheel_DivideBy32(uint32 distanceFromBase);
#ifdef ENET_INC_ARCH_WAVE600
static uint16 CalendarWheel_DivideBy24(uint32 distanceFromBase);
static uint16 CalendarWheel_DivideBy36(uint32 distanceFromBase);
#endif
static void CalendarWheelSetDivFunc(CalendarWheelParameters_t *calendarWheel);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

CalendarWheel_DivideBy20 


Description:
------------
Divide by 20

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 20
	

Returns:
--------
	void - 
	
**********************************************************************************/
static uint16 CalendarWheel_DivideBy20(uint32 distanceFromBase)
{
	return (DIVIDE_BY_20(distanceFromBase));
}

/**********************************************************************************

CalendarWheel_DivideBy16 


Description:
------------
Divide by 16

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 16
	

Returns:
--------
	void - 
	
**********************************************************************************/
static uint16 CalendarWheel_DivideBy16(uint32 distanceFromBase)
{
	return (distanceFromBase >> 4);
}

/**********************************************************************************

CalendarWheel_DivideBy12 


Description:
------------
Divide by 16

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 12
	

Returns:
--------
	void - 
	
**********************************************************************************/
static uint16 CalendarWheel_DivideBy12(uint32 distanceFromBase)
{
	return (DIVIDE_BY_12(distanceFromBase));
}


/**********************************************************************************

CalendarWheel_DivideBy13


Description:
------------
Divide by 13

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 13
	

Returns:
--------
	void - 
	
**********************************************************************************/
static uint16 CalendarWheel_DivideBy13(uint32 distanceFromBase)
{
	return (DIVIDE_BY_13(distanceFromBase));
}

/**********************************************************************************

CalendarWheel_DivideBy24


Description:
------------
Divide by 24

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 24
	

Returns:
--------
	void - 
	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600

static uint16 CalendarWheel_DivideBy24(uint32 distanceFromBase)
{
	return (distanceFromBase/24);
}

#endif
/**********************************************************************************

CalendarWheel_DivideBy32


Description:
------------
Divide by 32

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 32
	

Returns:
--------
	void - 
	
**********************************************************************************/
static uint16 CalendarWheel_DivideBy32(uint32 distanceFromBase)
{
	return (distanceFromBase >> 5);
}

/**********************************************************************************

CalendarWheel_DivideBy36


Description:
------------
Divide by 36

Input:
-----
distance from base

		
Output:
-------
distance in entry size which is 36
	

Returns:
--------
	void - 
	
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600

static uint16 CalendarWheel_DivideBy36(uint32 distanceFromBase)
{
	return (distanceFromBase/36);
}

#endif
/**********************************************************************************

CalendarWheelSetDivFunc 


Description:
------------
Set division function

Input:
-----
calendarWheel - a pointer to the calendar wheel

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void CalendarWheelSetDivFunc(CalendarWheelParameters_t *calendarWheel)
{
	if (calendarWheel->entrySize == 16)
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy16;
	}
	else if (calendarWheel->entrySize == 12)
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy12;
	}
	else if (calendarWheel->entrySize == 13)
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy13;
	}
	else if (calendarWheel->entrySize == 20) // for group manager station entry
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy20;
	}
#ifdef ENET_INC_ARCH_WAVE600

	else if (calendarWheel->entrySize == 24) // for group manager station entry (600)
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy24;
	}
#endif
	else if (calendarWheel->entrySize == 32) // for group manager group entry
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy32;
	}
	
#ifdef ENET_INC_ARCH_WAVE600
	else if (calendarWheel->entrySize == 36) // for group manager entry in 600
	{
		calendarWheel->divFunc = CalendarWheel_DivideBy36;
	}
#endif
	else
	{
		/*If new value is used need to supply div func*/
		ASSERT(FALSE);
	}
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

CalendarWheel_AddEntry 


Description:
------------
add an entry to the calendar wheel

Input:
-----
calendarWheel - a pointer to the calendar wheel
newEntry - the pointer of the entry to add
timeToSchedule - the time this event should happen (start counting from now)

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CalendarWheel_AddEntry(CalendarWheelParameters_t *calendarWheel, CalendarWheelEntry_t *newEntry, uint16 timeToSchedule)
{	
	uint32 slotToScheduleTo = 0;
	uint32 calendarWheelSize = 0;
	CalendarWheelEntry_t *nextEntry = NULL;
	uint16 currentEntryIndex = 0;

	calendarWheelSize = calendarWheel->size;
	slotToScheduleTo = calendarWheel->currentSlot + timeToSchedule;
	if(slotToScheduleTo >= calendarWheelSize)
	{
		/* Wrap around */
		slotToScheduleTo -= calendarWheelSize;
		ASSERT(slotToScheduleTo <= calendarWheelSize);
	}
	/* div function divides the received offset by size of the entry in the calendar wheel. the size of the
	entry of each calendar wheel entity is known*/
	currentEntryIndex =  calendarWheel->divFunc((uint32)newEntry - calendarWheel->baseAddress);
	
	DEBUG_ASSERT(currentEntryIndex != calendarWheel->slots[slotToScheduleTo]);
	/* add new event to the head of the list in the slot, 
	slot field contains the index of the first entry of the linked list
	this index will be set as next of new first entry currently added*/
	newEntry->nextIndex = calendarWheel->slots[slotToScheduleTo];

	/* Put the slot address as the previous in order to simplify remove */
	newEntry->previousIndex =  CALENDAR_WHEEL_CONVERT_SLOT_INDEX_INTO_CALENDAR_INDEX(slotToScheduleTo);


	/* Update the previous if exists  - the slot is not empty*/
	if(CALENDAR_WHEEL_NULL_ENTRY_INDEX != newEntry->nextIndex)
	{
		nextEntry = (CalendarWheelEntry_t *)(calendarWheel->baseAddress + (newEntry->nextIndex * calendarWheel->entrySize));
		/* set the previous field of previous first entry in slot to be my index*/
		nextEntry->previousIndex = currentEntryIndex;
	}

	/* Update the head of the list - set my index in the slot*/
	calendarWheel->slots[slotToScheduleTo] = currentEntryIndex;
	
}


/**********************************************************************************

CalendarWheel_RemoveEntry 


Description:
------------
remove an entry from the calendar wheel

Input:
-----
entry - the pointer of the entry to remove
calendarWheel - a pointer to the calendar wheel


		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CalendarWheel_RemoveEntry(CalendarWheelParameters_t *calendarWheel, CalendarWheelEntry_t *entry)
{	
	uint16 previousIndex = 0;
	uint16 nextIndex = 0;
	CalendarWheelEntry_t *nextEntry = NULL;
	CalendarWheelEntry_t *previousEntry = NULL;

	previousIndex = entry->previousIndex;
	nextIndex = entry->nextIndex;
	
	if(previousIndex & CALENDAR_WHEEL_CALENDAR_INDEX_MASK)
	{
		/* The previous entry is slot index */
		previousIndex = CALENDAR_WHEEL_CONVERT_CALENDAR_INDEX_INTO_SLOT_INDEX(previousIndex);
		calendarWheel->slots[previousIndex] = entry->nextIndex;	
	}
	else
	{
		previousEntry = (CalendarWheelEntry_t *)(calendarWheel->baseAddress + (previousIndex * calendarWheel->entrySize));
		previousEntry->nextIndex = entry->nextIndex;
		DEBUG_ASSERT(entry->previousIndex != previousEntry->nextIndex);
	}

	if(CALENDAR_WHEEL_NULL_ENTRY_INDEX != nextIndex)
	{
		nextEntry = (CalendarWheelEntry_t *)(calendarWheel->baseAddress + (nextIndex * calendarWheel->entrySize));
		nextEntry->previousIndex = entry->previousIndex;	
		
	}
}


/**********************************************************************************

CalendarWheel_RescheduleEntry 


Description:
------------
reschedule entry in the calendar wheel

Input:
-----
calendarWheel - a pointer to the calendar wheel
entry - the pointer of the entry to add
timeToSchedule - the time this event should happen (start counting from now)


		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void CalendarWheel_RescheduleEntry(CalendarWheelParameters_t *calendarWheel, CalendarWheelEntry_t *entry, uint16 timeToSchedule)
{	
	CalendarWheel_RemoveEntry(calendarWheel, entry);
	CalendarWheel_AddEntry(calendarWheel, entry, timeToSchedule); 
}


/**********************************************************************************

CalendarWheel_TimerEvent 


Description:
------------
Return the index of the first entry in the list (can be null entry)

Input:
-----
calendarWheel - a pointer to the calendar wheel

		
Output:
-------


Returns:
--------
index of the first entry in the list	
	
**********************************************************************************/
uint16 CalendarWheel_TimerEvent(CalendarWheelParameters_t *calendarWheel)
{	
	uint16 currentIndex = 0;
	uint32 currentSlot = 0;

    currentSlot = calendarWheel->currentSlot;
	currentIndex = calendarWheel->slots[currentSlot];

	calendarWheel->slots[currentSlot] = CALENDAR_WHEEL_NULL_ENTRY_INDEX;

	currentSlot ++;

	if(currentSlot == calendarWheel->size)
	{
		currentSlot = 0;
	}
	calendarWheel->currentSlot = currentSlot;
	
	return currentIndex;
}


/**********************************************************************************

CalendarWheel_Initialize 


Description:
------------
initialize the specific calendar wheel internal parameters

Input:
-----
calendarWheel - a pointer to the calendar wheel
size - the size of the calendar wheel (in slots)
slotsAddress - the address of the slots array
baseAddress - the base address from which all the entries start
entrySize - the entry size to use in order to find the index of the entry

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization_start" 
#endif
void CalendarWheel_Initialize(CalendarWheelParameters_t *calendarWheel, uint16 size, uint16 *slotsAddress, uint32 baseAddress, uint8 entrySize)
{	
	memset(calendarWheel, 0, sizeof(CalendarWheelParameters_t));

    /* current was set to 0 in memset */
	calendarWheel->size = size;
	calendarWheel->slots = slotsAddress;
	calendarWheel->baseAddress = baseAddress;
	calendarWheel->entrySize = entrySize; 

	/*Set div func according to entry size*/
	CalendarWheelSetDivFunc(calendarWheel);
	
	/* Initialize the slots - size is the number of slots and each slot is an index - 2 bytes*/
	memset(slotsAddress, CALENDAR_WHEEL_NULL_ENTRY_INDEX, CALENDAR_WHEEL_SLOT_SIZE * size);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


