/*******************************************************************************
*    
*   Source File:
*       MT_Math.c
*	
*	AUTHOR: Amir Hitron
*
*   Description:Driver to the Rfic 
*       
*   Copyright: 
*       (C) Metalink Ltd.
*       All rights are strictly reserved. Reproduction or divulgence in any 
*       form whatsoever is not permitted without written authority from the 
*       copyright owner. Issued by Metalink Transmission Devices Ltd in 
*       Israel - 11/94.
*   Revision History:
*
*******************************************************************************/
/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"

#include "MT_Math.h"
#include "ErrorHandler_Api.h"

#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_UTILS
#define LOG_LOCAL_FID 4


uint8 LogLUT5[32] = {0, 1, 3, 4, 5, 7, 8, 9, 10, 11,13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 22, 23, 24, 25, 26, 27, 27, 28, 29, 30, 31, 31};

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/

//************************************
// Method:    Abs
// Purpose:   Get absolute value of number
// Parameter: int32 x
// Returns:   int32
//************************************
int32 Abs(int32 x){return ((int32)((x<0)?(-x):x));}

//************************************
// Method:    Abs64
// Purpose:   Get absolute value of number
// Parameter: int64 x
// Returns:   int64
//************************************
int64 Abs64(int64 x){return ((int64)((x<0)?(-x):x));}






/********************************************************************************
MATH_10Log10

Description:
------------
    Calculates 10*Log10 of input, 4 bits after binary point 

Input:
-----
    x - positive integer	

Returns:
-------
   10*log10(Input)	
********************************************************************************/
int16 MATH_10Log10(uint32 x)
{
	int8 i = 0, d = 0;
	int16 g;
	int32 y = x;
	
	if (x == 0)
	{
		return (0);
	}

	while (x)
	{
		i++;
		x >>= 1;
	}
	
	if (i >= 5)
	{
		d = (y >> (i - 5)) & 0xF;
	}
	else
	{
		d = (y << (5 - i)) & 0xF;
	}
	
	g = (i - 1) << 4 | d;
	
	return (g * 3);
}
/********************************************************************************
MATH_10Log10Res

Description:
------------
	Calculates 10*Log10 of input, n resolution bits after binary point.
	Using interpolation by lookup table rather than linear interpolation.

Input:
-----
	x - positive integer	
	n - number of resolution bits.

Returns:
-------
   10*log10(Input)	
********************************************************************************/
int16 MATH_10Log10Res(uint32 x, uint8 n)
{
	int8 i = 0;
	int8 D, N, d;
	int16 g;
	int32 y = x;
	uint8 *LUTptr;

	// Check input params
	if (x == 0)
	{
		return (0);
	}
	DEBUG_ASSERT(n == LOG_LUT_OFFSET);
	
	LUTptr = LogLUT5;
	
	// calculate log2(x)
	
	//Find integer part
	while (x)
	{
		i++;
		x >>= 1;
	}
	// Find decimal part
	D = (1 << n) - 1; // for 5bit resolution, 31[dec]
	N = n + 1;		  //number of resolution bits + msb bit
	if (i >= N)
	{
		d = (y >> (i - N)) & D;
	}
	else
	{
		d = (y << (N - i)) & D;
	}
	// Interpolation for the decimal part
	d = LUTptr[d];
	// Form the log2 from integer & decimal parts
	g = (i - 1) << n | d;
	//convert from binary to decimal base: log2(x) --> 10log10(x)
	return (g * 3);
}

/*************************************************************************************
Name: void MATH_LongDivision(int32 inNumeratorInt,int32 inDenominatorInt,int32 inFractionBits,int32 *outResult_p)

Input:
	1. inNumeratorInt	: the numerator  (int32)
	2. inDenominatorInt	: the denominator  (int32)
	3. inFractionBits	: the number of bits used after the decimal point for the result (int32)
	
Output: 
	1. outResult_p		: result from the division (type - int32)


Description:	This function returns an integer (int32) fraction-
				the division between 2 integer (int32) numbers. 
				The calculation uses extension to 64 bits of the numerator and division 
				by the denominator
 
*************************************************************************************/
void MATH_LongDivision_DenomUint(int32 inNumeratorInt,uint32 inDenominatorInt,int32 inFractionBits,int32 *outResult_p)
{
	int8 Sign_numerator;
	int64 numerator_64b,result;

	if (inDenominatorInt != 0)
	{
		Sign_numerator=(inNumeratorInt<0)?-1:1;
		inNumeratorInt*=Sign_numerator;		//Make sure inNumeratorInt is positive.
		numerator_64b = inNumeratorInt;
		numerator_64b<<=inFractionBits; //add for the round
		numerator_64b+=(inDenominatorInt>>1);
		result = numerator_64b/inDenominatorInt;
		result*=Sign_numerator;
		*outResult_p = (int32)result; 
	}
	else
		*outResult_p = 0;

	return;
}


void MATH_LongDivision(int32 inNumeratorInt,int32 inDenominatorInt,int32 inFractionBits,int32 *outResult_p)
{
	int8 Sign_numerator;
	int64 numerator_64b,result;

	if (inDenominatorInt != 0)
	{
		Sign_numerator=(inNumeratorInt<0)?-1:1;
		inNumeratorInt*=Sign_numerator;		//Make sure inNumeratorInt is positive.
		numerator_64b = inNumeratorInt;
		numerator_64b<<=inFractionBits; //add for the round
		numerator_64b+=(inDenominatorInt>>1);
		result = numerator_64b/inDenominatorInt;
		result*=Sign_numerator;
		*outResult_p = (int32)result; 
	}
	else
		*outResult_p = 0;

	return;
}
int16 MATH_PowerRes(int32 x, int16 a, uint8 n)
{
	uint8 i;
	int16 y = 1;

	ASSERT(a >= 0);
	
	if (a != 0)
	{
		y = x;
		for (i = 0; i < a - 1; i++)
		{
			y *= x;
			y >>= n;
		}
	}
	return (y);
}

/********************************************************************************
MATH_Sqrt

Description:
------------
	Approximate square root of an input, using "Newton's method".

Input:
-----
	x - positive integer	
Returns:
-------
      y - Approximated value of sqrt(x)	
********************************************************************************/
uint32 MATH_Sqrt(uint32 x)
{
	int32 y = x, last = 0;

	while (Abs(y - last) > 1)
	{
		last = y;
		y = (y + x/y) >> 1;
	}
	return (y);
}
int16 MATH_ArctanRes(int16 x, uint8 n)
{
 	int16 y;
 
	 y = ((x << n) - ((1<<n)/3) * MATH_PowerRes(x,3,n) + ((1<<n)/5) * MATH_PowerRes(x,5,n));
 
 	return (y >> n);
}
int16 MATH_Sin(int16 x) 
{
	return (x);
}

int16 MATH_Cos(int16 x, uint8 n) 
{
	return((1 << n) - (MATH_PowerRes(x,2,n) >> 1));
}


int64 MATH_PowerRes_64(int64 x, int16 a, uint8 n)
{
	uint8 i;
	int64 y = 1;

	ASSERT(a >= 0);
	
	if (a != 0)
	{
		y = x;
		for (i = 0; i < a - 1; i++)
		{
			y *= x;
			y >>= n;
		}
	}
	return ((int64)y);
}

int32 MATH_ArctanRes_32(int32 x, uint8 n)
{
 	int64 y;
	 y = ((((int64)x) << n) - ((1<<n)/3) * MATH_PowerRes_64(x,3,n) + ((1<<n)/5) * MATH_PowerRes_64(x,5,n));
 	return (y >> n);
}

int32 MATH_Cos_64(int32 x, uint8 n) 
{
	int32 y = (1 << n) - (MATH_PowerRes_64(x,2,n) >> 1);	
	ILOG0_DD("cos:(1 << n):%d, PowerRes(x,2,n) >> 1:%d ",(1 << n),(MATH_PowerRes_64(x,2,n) >> 1));
	ILOG0_DD("x:%d, y:%d",x,y);
	
	return y;
}

uint64 MATH_Sqrt_64(uint64 x)
{
	int64 y = x, last = 0;

	while (Abs(y - last) > 1)
	{
		last = y;
		y = (y + x/y) >> 1;
	}
	return (y);
}




/****************************************************************************
 **
 ** NAME:        cyclicMinusOperator
 **
 ** PARAMETERS:  inputA  - first  operand
 **              inputB - second  operand		 
 **				 abs	   - If TRUE return difference in absolute value,
 **							 otherwise, return difference as is.
 **
 ** RETURN VALUES: difference of subtraction 
 **
 ** DESCRIPTION: subtracts two given sequence numbers
 **              
 **
 **   
 ***************************************************************************/
int32 cyclicMinusOperator(uint32 inputA, uint32 inputB, bool abs, uint32 range)
{																			 
	bool     notWrapArround;
	int32    delta  = (int32)(inputA - inputB);
	int8  	 factor = (abs ? 1 : -1);

	/* check if inFirstSn is newer (bigger) or lower (smaller) from inSecondSn [check if distance is more or less than range] */
	notWrapArround = ((((unsigned)delta) & (range-1)) > (range >> 1) ) ? FALSE : TRUE ;
	if (notWrapArround)
		return ( ((unsigned)delta) & (range-1));
	else 
		return ( factor * (((unsigned)(-delta)) & (range-1)) );  
}


