/************************************************************************************
*    
*    File:		  Utils_Misc.c
*    Class/Module: 
*    Description:  A detailed description of the Module, its  purpose attributes and
*		 	  whatever information the user & maintainer might find valuable.
*
*    COPYRIGHT: 
*        (C) Lantiq Israel Ltd.
*        All rights are strictly reserved. Reproduction or divulgence in any   
* 	  form whatsoever is not permitted without written authority from the 
*        copyright owner. Issued by Lantiq Israel Ltd
*
**************************************************************************************/

#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "stringLibApi.h"
#include "database.h"
#ifdef CPU_ARC
#include "arc_intrinsics.h"
#endif // CPU_ARC

#define LOG_LOCAL_GID   GLOBAL_GID_UTILS
#define LOG_LOCAL_FID 5

#ifdef CPU_MIPS
uint32 __CLZ32(uint32);
#endif // CPU_MIPS

/***************************************************************************
**
** NAME         Utils_CountLeadingZeros
**
** PARAMETERS   uint32 num
**
** RETURNS      number of leading zeros (how many '0' before the first '1' from MSB to LSB)
**
** DESCRIPTION  Count Leading Zeros using mips' clz routine.
**
****************************************************************************/
uint32 Utils_CountLeadingZeros(uint32 num)
{
#if defined (CPU_MIPS)
	return __CLZ32(num);
#endif
#if defined (CPU_ARC)
	if (num != 0)
	{
		// The function FLS (Find Last Set) in ARC returns the position of the last (highest) '1' from MSB to LSB.
		// fls(0x1) = 0 ; fls (0x2) = 1 ; fls (0x3) = 1; fls (0x80000001) = 31; 		
		return (31 - _fls(num));
	}
	else
	{
		return 32;
	}
#endif
}

/**********************************************************************************

orOperator64Bit 

Description: This function calc or between two 64 bitmaps
  
Input: 
-----
	
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
uint32 orOperator64Bit(uint32* inputA, uint32* inputB, uint32* result)
{
//	DBASSERT((sizeof(inputA) == sizeof(inputB)) && (sizeof(inputB)== sizeof(result)) && ((sizeof result)== (sizeInWords<<2)));
	uint32  compressedResult = 0;
	uint32 tempResult[2] = {0};

	tempResult[0] = inputA[0] | inputB[0];
	tempResult[1] = inputA[1] | inputB[1];
	result[0] = tempResult[0];
	result[1] = tempResult[1];

	compressedResult = (result[0] | result[1]);	
	
	return compressedResult;
}

/**********************************************************************************

notOperator64Bit 

Description: This function calc or between two 64 bitmaps
  
Input: 
-----
	
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
uint32 notOperator64Bit(uint32* inputA, uint32* result)
{
	uint32 compressedResult = 0;
	uint32 tempResult[2] = {0};

	tempResult[0] = ~(inputA[0]);
	tempResult[1] = ~(inputA[1]);
	result[0] = tempResult[0];
	result[1] = tempResult[1];

	compressedResult = (result[0] | result[1]);	
	
	return compressedResult;
}

/***************************************************************************
**
** NAME         Utils_FindFirstSet
**
** PARAMETERS   uint32 num
**
** RETURNS      Index of first set(1st lit msb)
**
** DESCRIPTION  Count Leading Zeros using mips' clz routine.
**
****************************************************************************/
int32 Utils_FindFirstSet(uint32 num)
{
	// This function actualy implement what is called "Find Last Set". We call it Find First Set by mistake.
	//we want to support only the case where num is != 0
	ASSERT(num != 0);

#if defined (CPU_MIPS)	
	return (31 - __CLZ32(num));
#endif // CPU_MIPS
#if defined (CPU_ARC)
	// The function FLS (Find Last Set) in ARC returns the position of the last (highest) '1' from MSB to LSB.
	// fls(0x1) = 0 ; fls (0x2) = 1 ; fls (0x3) = 1; fls (0x80000001) = 31;
	return _fls(num);
#endif // CPU_ARC
}

/***************************************************************************
**
** NAME         Utils_CountTrailingZeros
**
** PARAMETERS   uint32 num
**
** RETURNS      number of trailing zeros (magnitude of 1st lit lsb)
**
** DESCRIPTION  Count Trailing Zeros using mips' clz routine.
**				The function does not support 0 as input!!
**
****************************************************************************/
uint32 Utils_CountTrailingZeros(uint32 num)
{
#if defined (CPU_MIPS)
	uint32 tempNum;

	//this logic does not support 0 as input
	ASSERT(num != 0);

	//in the following logic, num is automatically casted to int32 for the mathematical expression
	tempNum = num & (-num); //leave only lsb lit

	return (31 - Utils_CountLeadingZeros(tempNum));
#endif // CPU_MIPS
#if defined (CPU_ARC)
	ASSERT(num != 0);

	// The function FFS (Find First Set) in ARC returns the position of the first (lowest) '1' from LSB to MSB.
	// ffs(0x1) = 0 ; ffs (0x2) = 1 ; ffs (0x3) = 0; ffs(0x80000001) = 0; 
	return _ffs(num);
#endif // CPU_ARC
}


/***************************************************************************
**
** NAME         Utils_FindFirstSetAndClear
**
** PARAMETERS   uint32 *bitmap
**
** RETURNS      The index of the first bit set
**
** DESCRIPTION  this function is using mips' clz routine.
**				The function does not support 0 as input!!
**              The function finds the first bit set (MSB)and
**              clears it from the bitmap
**
****************************************************************************/
uint32 Utils_FindFirstSetAndClear(uint32 *bitmap)
{
	uint8 firstBitSet;
	uint32 tempNum;
	uint32 mask;

    tempNum = *bitmap;
	
	/* this logic does not support 0 as input */
	ASSERT(tempNum != 0);
#if defined (CPU_MIPS)	
	firstBitSet = (31 - __CLZ32(tempNum));
#endif // CPU_MIPS
#if defined (CPU_ARC)
	// The function FLS (Find Last Set) in ARC returns the position of the last (highest) '1' from MSB to LSB.
	// fls(0x1) = 0 ; fls (0x2) = 1 ; fls (0x3) = 1; fls (0x80000001) = 31;	
	firstBitSet = _fls(tempNum);
#endif // CPU_ARC

    /* Clear the first trailing '1' */
	mask = 0x1 << firstBitSet;
	*bitmap = tempNum & ~mask;
	
	return firstBitSet;
}

/***************************************************************************
**
** NAME         Utils_FindLastSetAndClear
**
** PARAMETERS   uint32 *bitmap - pointer to array of bitmap
**				uint8  numOfBitsInBitmap - num of bits in bitmap array
**
** RETURNS      The index of the first bit set
**
** DESCRIPTION  this function is using mips' clz routine.
**
**              The function finds the last bit set (LSB) and
**              clears it from the bitmap
**
****************************************************************************/
uint32 Utils_FindLastSetAndClear(uint32 *bitmap, uint16 numOfBitsInBitmap)
{
	uint32 lastBitSet = numOfBitsInBitmap;
	uint32 tempBitmap;
#if defined (CPU_MIPS)
	uint32 lastBitSetBitmap;
#endif //defined (CPU_MIPS)
	uint32 mask;
	uint8  i;
	uint8  numOfWords = CONVERT_BIT_INDEX_TO_WORDS(numOfBitsInBitmap) ;// find the number of words in the bitmap
	
	for(i = 0; i < numOfWords; i++)
	{
		//ILOG0_DD("Utils_FindLastSetAndClear: i %d, val %d", i, bitmap[i]);
		if(bitmap[i] != 0)
		{
		    tempBitmap = bitmap[i];
#if defined (CPU_MIPS)
			/* Save only the last '1' in the bitmap */
			lastBitSetBitmap = tempBitmap & (-tempBitmap);
			lastBitSet = (31 - __CLZ32(lastBitSetBitmap)); 
#endif //defined (CPU_MIPS)
#if defined (CPU_ARC)
			// The function FFS (Find First Set) in ARC returns the position of the first (lowest) '1' from LSB to MSB.
			// ffs(0x1) = 0 ; ffs (0x2) = 1 ; ffs (0x3) = 0; ffs(0x80000001) = 0; 
			lastBitSet = _ffs(tempBitmap);
#endif //defined (CPU_ARC)
		    /* Clear the last trailing '1' */
			mask = 0x1 << lastBitSet;
			bitmap[i] = tempBitmap & ~mask;
			lastBitSet += (i * 32); //add 32 bits times the words that were 0 in the bitmap
			break;
		}
	}

	ASSERT(lastBitSet < numOfBitsInBitmap);

	return lastBitSet;
}

/***************************************************************************
**
** NAME         Utils_FindLastSet
**
** PARAMETERS   uint32 *bitmap - pointer to array of bitmap
**				uint8  numOfBitsInBitmap - num of bits in bitmap array
**
** RETURNS      The index of the first bit set
**
** DESCRIPTION  this function is using mips' clz routine.
**
**              The function finds the last bit set (LSB) 
**
****************************************************************************/
uint32 Utils_FindLastSet(uint32 *bitmap, uint16 numOfBitsInBitmap)
{
	uint32 lastBitSet = numOfBitsInBitmap;
	uint32 tempBitmap;
#if defined (CPU_MIPS)
	uint32 lastBitSetBitmap;
#endif //defined (CPU_MIPS)
	uint8  i;
	uint8  numOfWords = CONVERT_BIT_INDEX_TO_WORDS(numOfBitsInBitmap) ;// find the number of words in the bitmap
	
	for(i = 0; i < numOfWords; i++)
	{
		//ILOG0_DD("Utils_FindLastSetAndClear: i %d, val %d", i, bitmap[i]);
		if(bitmap[i] != 0)
		{
		    tempBitmap = bitmap[i];
#if defined (CPU_MIPS)
			/* Save only the last '1' in the bitmap */
			lastBitSetBitmap = tempBitmap & (-tempBitmap);
			lastBitSet = (31 - __CLZ32(lastBitSetBitmap)); 
#endif //defined (CPU_MIPS)
#if defined (CPU_ARC)
			// The function FFS (Find First Set) in ARC returns the position of the first (lowest) '1' from LSB to MSB.
			// ffs(0x1) = 0 ; ffs (0x2) = 1 ; ffs (0x3) = 0; ffs(0x80000001) = 0; 
			lastBitSet = _ffs(tempBitmap);
#endif //defined (CPU_ARC)
			lastBitSet += (i * 32); //add 32 bits times the words that were 0 in the bitmap
			break;
		}
	}

	ASSERT(lastBitSet < numOfBitsInBitmap);

	return lastBitSet;
}


/********************************************************************************
Utils_CountTrailZeroBitmap


Description:
------------
find the first LSB '1' index in bitmap

Input:
-----
pointer to the bitmap , start bit offset , size of the bitmap

Output:
-------
Returns:
--------
first LSB '1' index.
if the bitmap is empty, the return value will be the last index in bit map + 1 (i.e 128 if bitmap is 0-127)
if offset is greater than BitmapSize, there is DEBUG ASSERT, so need to check it in caller function.

<Descriptions of the function return value>
********************************************************************************/
uint16 Utils_CountTrailZeroBitmap(uint32* bitmap,uint8 offset , uint8 bitmapSize)
{
	uint8  i = 0;
	uint16 index = 0;
	uint16 wordStartIndex = 0;
	uint8  numOfWords = 0;
	
	wordStartIndex = CONVERT_BIT_INDEX_TO_WORDS(offset);// find the relevant word - divide by 32 (bits per word)
	ASSERT(wordStartIndex < CONVERT_BYTES_TO_WORDS(bitmapSize)); // check that word is inside bitmap
	numOfWords = CONVERT_BYTES_TO_WORDS(bitmapSize) - wordStartIndex;// find the number of words between the offset and the bitmap end
	bitmap += wordStartIndex; // set the pointer to the relevant word
	index = CONVERT_WORDS_TO_BIT_INDEX(wordStartIndex);// the bit index is the start of word bit index
	offset = offset - index;
	
	
	if ((*bitmap>>offset) != 0)// if there is at least 1 bit ='1' in the first checked word
	{
		index = index + offset + Utils_CountTrailingZeros(*bitmap>>offset);
	}
	else
	{
		bitmap++;
		numOfWords--;
		index += 32;
		
		for (i=0 ; i < numOfWords ;i++)
		{
			if (*bitmap != 0)// if there is at least 1 bit ='1'
			{
				index = index + Utils_CountTrailingZeros(*bitmap);// the result is num of bits from the word start + num of bits offset from the bitmap start to the word start
				break;
			}
			else
			{
				index += 32; //increase the offset from the bitmap start by size of word (32 bits)
			}
			bitmap++;//jump to the next word
		}
		

	}
	
	return index;


}



/********************************************************************************
Utils_ZeroBitInBitmap

Description:
------------
zero bit in bitmap

Input:
-----
pointer to the bitmap , bit index

Output:
-------
Returns:
--------

<Descriptions of the function return value>
********************************************************************************/
void Utils_ZeroBitInBitmap(uint32* bitmap,uint16 index)
{
	uint16 word;
	uint16 bitLocationInWord;
	uint32 mask = 0;
	
	/*Find word */
	word = CONVERT_BIT_INDEX_TO_WORDS(index);// offset in word units from the bitmap start
	/* Find location in word */
	bitLocationInWord = index & 0x1F;

	mask = ~(0x1 << bitLocationInWord);
	bitmap[word] &= mask;
}


/********************************************************************************
Utils_SetBitInBitmap


Description:
------------
set bit in bitmap

Input:
-----
pointer to the bitmap , bit index

Output:
-------
Returns:
--------

<Descriptions of the function return value>
********************************************************************************/
void Utils_SetBitInBitmap(uint32* bitmap,uint16 index)
{
	uint16 word;
	uint16 bitLocationInWord;
	uint32 mask = 0;

	/*Find word */
	word = CONVERT_BIT_INDEX_TO_WORDS(index);// offset in word units from the bitmap start
	/* Find location in word */
	bitLocationInWord = index & 0x1F;
	
	mask = 0x1 << bitLocationInWord;
	bitmap[word] |= mask;
}


uint8 Utils_GetBitFromBitmap(uint32* bitmap,uint16 index)
{
	uint16 wordOffset;
	uint16 bitOffset;
	uint8 bitValue;
	
	wordOffset = CONVERT_BIT_INDEX_TO_WORDS(index);// offset in word units from the bitmap start
	bitOffset = CONVERT_WORDS_TO_BIT_INDEX(wordOffset);// offset between the bitmap start and the word start in bits unit
	bitmap += wordOffset;
	
	
	bitValue = (*bitmap >> (index - bitOffset)) & 0x1;

	return bitValue;
}

uint16 Utils_RoundRobinBitmap(uint32* bitmap,uint8 startBit , uint8 bitmapSizeInBytes)
{
	uint16 index;
	uint8 bitmapSize;
		
	index = Utils_CountTrailZeroBitmap(bitmap, startBit,  bitmapSizeInBytes);
	bitmapSize = CONVERT_BYTES_TO_BIT_INDEX(bitmapSizeInBytes);
	if (index >= bitmapSize)
	{
		index = Utils_CountTrailZeroBitmap(bitmap, 0, bitmapSizeInBytes);
	}
	
	return index;
}


/********************************************************************************

Utils_Shift32bitArrayRight


Description:
------------
logic bit shift right of entire array

Input:
-----
s - pointer to source array
d - pointer to destination array (can be same as source)
l - length of array
shift - number of bits to shift

Output:
-------
none

********************************************************************************/
//void Utils_Shift32bitArrayRight(uint32  *s,uint32  *d, uint32 l, uint32 shift)
//{
//	uint32 i, w, b;
//
//	if (shift >= (l * 32))
//	{
//		memset(d, 0, (sizeof(uint32) * l));
//	}
//	else
//	{
//		w = shift / 32;
//		b = shift % 32;
//
//		for (i = 0; i < l; i++)
//		{
//			if (i < (l - w - 1))
//			{
//				d[i] = (s[i + w] >> b) | ((s[i + w + 1] & ((1 << b) - 1)) << (32 - b));
//			}
//			else if (i == (l - w - 1))
//			{
//				d[i] = (s[i + w] >> b);
//			}
//			else
//			{
//				d[i] = 0;
//			}
//		}
//	}
//}

/********************************************************************************

Utils_Shift32bitArrayLeft


Description:
------------
logic bit shift left of entire array

Input:
-----
s - pointer to source array
d - pointer to destination array (can be same as source)
l - length of array
shift - number of bits to shift

Output:
-------
none

********************************************************************************/
void Utils_Shift32bitArrayLeft(uint32  *s,uint32  *d, uint32 l, uint32 shift)
{
	uint32 	w, b, i;
	
	if (shift >= (l * 32))
	{
		memset(d, 0, (sizeof(uint32) * l));
	}
	else
	{
		w = shift / 32;
		b = shift % 32;

		for (i = l; i > 0; i--)
		{
			if ((i-1) >= (w + 1))
			{
				d[i-1] = (s[i -1 - w] << b) | ((s[i -1 - w - 1] & ((1 << b) - 1)) >> (32 - b));
			}
			else if (i == w)
			{
				d[i -1] = (s[i -1 - w] << b);
			}
			else
			{
				d[i -1] = 0;
			}
		}
	}
}



/********************************************************************************
Utils_IsBitmapNotEmpty


Description:
------------
find whether the bitmap is empty, or at least one bit is set.

Input:
-----
pointer to the bitmap , size of the bitmap in bits

Output:
-------
Returns:
--------
if the bitmap is empty, return TRUE, else return FALSE

********************************************************************************/
bool Utils_IsBitmapNotEmpty(uint32* Bitmap, uint16 numOfBitsInBitmap)
{
	uint8  i;
	uint8  numOfWords = CONVERT_BIT_INDEX_TO_WORDS(numOfBitsInBitmap) ;// find the number of words in the bitmap
	
	for(i = 0; i < numOfWords; i++)
	{
		if(Bitmap[i]!=0)
		{
			return TRUE;
		}
	}

	return FALSE;
}




/********************************************************************************
Utils_determineSourceTypeAndDmaDirection

Author: Sion Cohen
Data:   23/12/2015

Description:
------------
	
Input:
-----
		addr - > The given address to check which component data is allocated to
Output:
-------
	fill  dmaDirection_e *direction:
		DMA_DIRECTION_HOST_TO_SHRAM,
		DMA_DIRECTION_SHRAM_TO_HOST,
		DMA_DIRECTION_SHRAM_TO_SHRAM,
		DMA_DIRECTION_HOST_IF_QOS_TO_HOST,
		DMA_DIRECTION_HOST_IF_TO_HOST,
		DMA_DIRECTION_RX_COORDINATOR_TO_HOST,
		DMA_DIRECTION_RXCOUNTERS_TO_HOST,
		DMA_DIRECTION_BAACOUNTERS_TO_HOST,
		DMA_DIRECTION_INVALID,


	fill statisticSourceType_e *sourceType:
		SOURCE_IRAM,	
		SOURCE_SHRAM,
		SOURCE_HOSTIF_QOS,		
		SOURCE_HOSTIF,	
		SOURCE_RX_COORDINATOR_COUNTERS,
		SOURCE_RXCOUNTERS,
		SOURCE_BAACOUNTERS,
		SOURCE_BSRCOUNTERS,


Returns:
--------
	dmaDirection_e:
		DMA_DIRECTION_HOST_TO_SHRAM,
		DMA_DIRECTION_SHRAM_TO_HOST,
		DMA_DIRECTION_SHRAM_TO_SHRAM,
		DMA_DIRECTION_HOST_IF_QOS_TO_HOST,
		DMA_DIRECTION_HOST_IF_TO_HOST,
		DMA_DIRECTION_RX_COORDINATOR_TO_HOST,
		DMA_DIRECTION_RXCOUNTERS_TO_HOST,
		DMA_DIRECTION_BAACOUNTERS_TO_HOST,
		DMA_DIRECTION_BSRCOUNTERS_TO_HOST,
		DMA_DIRECTION_INVALID,
**********************************************************************************/
dmaDirection_e Utils_determineSourceTypeAndDmaDirection (uint32 addr, dmaDirection_e *direction, SourceType_e *sourceType)
{

	if (IS_ADDRESS_IN_IRAM(addr))
	{
		*direction 	= DMA_DIRECTION_SHRAM_TO_HOST;
		*sourceType	= SOURCE_IRAM;
	}
	else if (IS_ADDRESS_IN_SHRAM(addr))
	{
		*direction 	= DMA_DIRECTION_SHRAM_TO_HOST;
		*sourceType	= SOURCE_SHRAM;
	}
	else if (IS_ADDRESS_IN_HOST_IF(addr))
	{
		*direction	 = DMA_DIRECTION_HOST_IF_TO_HOST;
		*sourceType	 = SOURCE_HOST_IF_COUNTERS;
	}	
	else if (IS_ADDRESS_IN_RXCOUNTERS(addr))
	{
		*direction	 = DMA_DIRECTION_RXCOUNTERS_TO_HOST;
		*sourceType	 = SOURCE_RX_COUNTERS;
	}
	else if (IS_ADDRESS_IN_BAACOUNTERS(addr))
	{
		*direction	 = DMA_DIRECTION_BAACOUNTERS_TO_HOST;
		*sourceType	 = SOURCE_BAACOUNTERS;
	}
	else if (IS_ADDRESS_IN_BSRCOUNTERS(addr))
	{
		*direction	 = DMA_DIRECTION_BSRCOUNTERS_TO_HOST;
		*sourceType	 = SOURCE_BSRCOUNTERS;
	}
#ifdef ENET_INC_ARCH_WAVE600D2
	else if (IS_ADDRESS_IN_HOST_IF_QOS(addr))
	{
		*direction	 = DMA_DIRECTION_HOST_IF_QOS_TO_HOST;
		*sourceType  = SOURCE_HOST_IF_QOS_COUNTERS;
	}
	else if (IS_ADDRESS_IN_RX_COOR_COUNTERS(addr))
	{
		*direction	 = DMA_DIRECTION_RX_COORDINATOR_TO_HOST;
		*sourceType  = SOURCE_RX_COORDINATOR_COUNTERS;
	}
#endif
	else
	{
		FATAL("Utils_determineSourceTypeAndDmaDirection");
	}
	
	return *direction;
}



/**********************************************************************************
Utils_GetNumAntsFromMask	

Description:
------------
	Calculate number of antennas from a given antenna mask
Input:
-----
	uint8 mask - antennas bitmap
Returns:
--------
	uint8 numAnts - number of '1's in the bitmap
**********************************************************************************/
uint8 Utils_GetNumAntsFromMask(uint8 mask)
{
	uint8 numAnts = 0;
	
	while (mask != 0)
	{
		// Clear the least significant '1' bit.
		mask = ((mask - 1)&(mask));
		numAnts++;
	}

	return numAnts;
}


/**********************************************************************************

Utils_GetPrimaryChannel

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 Utils_GetPrimaryChannel(uint8 low_chan_num, uint8 low_chan_num2, uint8 chan_width, uint8 primary_chan_idx)
{
	uint8 primaryChannel = 0;
	UNUSED_PARAM(low_chan_num2);
	
	switch (chan_width)
	{
	case BANDWIDTH_TWENTY:
	case BANDWIDTH_FOURTY:
	case BANDWIDTH_EIGHTY:
	case BANDWIDTH_ONE_HUNDRED_SIXTY:
		primaryChannel = low_chan_num + (0x4 * primary_chan_idx);
		break;
	case BANDWIDTH_EIGHTY_EIGHTY:
		DEBUG_FATAL("Utils_GetPrimaryChannel: 80+80 not supported");
		break;
	default:
		DEBUG_FATAL("Utils_GetPrimaryChannel: Illegal rate");
		break;
	}

	return primaryChannel;
}


/***************************************************************************
**
** NAME         Utils_CountNumOfOnes
**
** PARAMETERS   uint32 num
**
** RETURNS      number of 1 in num
**
** DESCRIPTION  Count the number of ones in the input parameter.
**
****************************************************************************/
uint8 Utils_CountNumOfOnes(uint32 num)
{
	uint8 numOfOnes = 0;

	while(num != 0)
	{
		numOfOnes ++;
		num = num & (num-1);
	}
	return numOfOnes;

}


