// *************************************************
// ABB_API.c
// *************************************************
// author: lusin, pridnig
//
// history
// march/april 2018: initial version
// 12.4.2018: added debugging enabling/disabling
//  9.5.2018: added abb_fcsirf
// 22.5.2018: adjusted abb_rx_calibration
// 28.5.2018: fully enabled rx_rctune
//  8.8.2018: added rx_calibration_addon*
// 30.8.2018: added recal_ssi
//  6.9.2018: added rx_calibration_bubble_correction
// 26.9.2018: rx_ssi_calibration_rcmeas_update
// 14.12.2018: force_rxon
// 30.1.2019: rxoff2 (ARC only, not implemented as mex function)
//  6.8.2019: changed int8_t abb_rx_rctune
// *************************************************

#define API_DEBUG

#include "mex.h"
#include <math.h>
#include <strings.h>

#include "interface_abb.h"
uint8_t ENABLE_DEBUGS = 1;
#include "fcsi_access_functions.c"
#include "lib_abb_api.c"
#include "lib_abb_sequences.c"

// Declaration
void check_parameter(int parameter1, int parameter2);

// The wrapper "MAIN" function
void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[])
{
    char        *api_cmd;
    double      *lhparam0;
    double      result;
    double      *pdouble_param1;
              
    uint8_t    mcal[3][32];
    
    int16_t        int16_param1;
    uint16_t        uint16_param1;
    uint16_t        uint16_param2;
    uint8_t         uint8_param1;
    uint8_t         uint8_param2;
    uint8_t         uint8_param3;
//  int16_t         int16_param1;
    int16_t         int16_param2;
//  int32_t         int32_param1;
    uint8_t    fuse[0x13];
    
   

    int8_t         int8_param1;
    int8_t         int8_param2;
     
    uint16_t   rccount[3];
    uint16_t   read_val[5];

     
//    double          *pdouble_param1;

    uint16_t    i;
    uint16_t    temp;
//    uint16_t    int16_data[16];
    
    if (nrhs == 0)
    {
        mexErrMsgTxt("API-> no command detected");
    }

    api_cmd  = mxArrayToString(prhs[0]);
    
    plhs[0]  = mxCreateDoubleMatrix(1,1, mxREAL);
    lhparam0 = mxGetPr(plhs[0]);
    result   = 815;
    
    mexPrintf("API-> ");
    
// ---------------------------------------------------
// Check API version in Repo
// ---------------------------------------------------
    
    if (0 == strcmp(api_cmd,"version"))                                //Compare input string
    {   mexPrintf("version 21, 20190806\n");
        result = (double)21;
    }
    
    else if (0 == strcmp(api_cmd,"abb_api_version"))        
    {   mexPrintf("abb_api_version\n");
        uint8_param1 = abb_api_version();
        mexPrintf("abb_api_version : return=%d\n", uint8_param1);
        result = (double)uint8_param1;
    }   
    
    else if (0 == strcmp(api_cmd,"enable_debugs"))
    {   mexPrintf("debug statements enabled\n");
        ENABLE_DEBUGS=1;
    }
    
    else if (0 == strcmp(api_cmd,"disable_debugs"))
    {   mexPrintf("debug statements disabled\n");
        ENABLE_DEBUGS=0;
    }  
    
    // LL-API -------------------------------------------------------------

    else if (0 == strcmp(api_cmd,"abb_api_fcsiwrite_softisolation_central"))        
    {   mexPrintf("abb_api_fcsiwrite_softisolation_central\n");
        abb_api_fcsiwrite_softisolation_central();
    }

    else if (0 == strcmp(api_cmd,"abb_api_fcsiread_softisolation_central"))        
    {   mexPrintf("abb_api_fcsiread_softisolation_central\n");
        int16_param1 = abb_api_fcsiread_softisolation_central();
        debug_printf("abb_api_fcsiread_softisolation_central : return=%d\n", int16_param1);
        result = (double)int16_param1;
    }
    
    else if (0 == strcmp(api_cmd,"abb_api_fcsiwrite_softisolation_antenna"))        
    {   mexPrintf("abb_api_fcsiwrite_softisolation_antenna\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_api_fcsiwrite_softisolation_antenna(uint16_param1);
    }
    
       else if (0 == strcmp(api_cmd,"abb_api_fcsiread_softisolation_antenna"))        
    {   mexPrintf("abb_api_fcsiread_softisolation_antenna\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        int16_param2 = abb_api_fcsiread_softisolation_antenna(uint16_param1);
        debug_printf("abb_api_fcsiread_softisolation_antenna : return=%d\n", int16_param2);
        result = (double)int16_param2;
    }
    
    else if (0 == strcmp(api_cmd,"abb_central_fusing"))        
    {   mexPrintf("abb_central_fusing\n");
        check_parameter(nrhs,1);
        
        temp = mxGetN(prhs[1]);     // number of rows
        debug_printf(" array with %d elements detected\n", temp);
        // from 0x018000, 0x114 words
        if (temp < 0x13)  mexErrMsgTxt("array size must be at least 19 paramters");
        pdouble_param1 = mxGetPr(prhs[1]);                                  //uint8_t *fuse fuse_table

        //Matlab can handle only double as default -> cast 
        for(i=0; i<0x13; i++)  {
	       fuse[i] = (uint8_t) pdouble_param1[i];
        }
        
        abb_central_fusing(&fuse[0]);
    }
    else if (0 == strcmp(api_cmd,"abb_central_init_pll"))        
    {   mexPrintf("abb_central_init_pll\n");
        abb_central_init_pll();
    }
    else if (0 == strcmp(api_cmd,"abb_central_init_clkd"))        
    {   mexPrintf("abb_central_init_clkd\n");
        abb_central_init_clkd();
    }
    else if (0 == strcmp(api_cmd,"abb_rcmeas"))        
    {   mexPrintf("abb_rcmeas\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t dsmsel
        debug_printf("dsmsel=%d\n", uint8_param1);   
        int16_param1 = abb_rcmeas(&rccount[0], uint8_param1);
        debug_printf("abb_rcmeas : ccal=%d, val1=%d, val2=%d, return=%d\n", rccount[0], rccount[1], rccount[2], int16_param1);
        result = (double)int16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_adctune"))        
    {   mexPrintf("abb_adctune\n");
        check_parameter(nrhs,1);
        
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //int16_t count_calc
        
        int8_param1 = abb_adctune(uint16_param1, &int8_param2);
        debug_printf("abb_adctune : rctune_calc=%d, return=%d\n", int8_param2, int8_param1);
        if (int8_param1 == 0)
        {   result = (double)int8_param2; }                                //int8_t rctune_calc
        else
        {   result = (double)int8_param1; }                                // error code
    }
    else if (0 == strcmp(api_cmd,"abb_rx_rctune"))        
    {   mexPrintf("abb_rx_rctune\n");
        check_parameter(nrhs,1);
        
        temp = mxGetN(prhs[1]);     // number of rows
        debug_printf(" array with %d elements detected\n", temp);
        // from 0x018000, 0x114 words
        if (temp < 0x13)  mexErrMsgTxt("array size must be at least 19 paramters");
        pdouble_param1 = mxGetPr(prhs[1]);                                  //uint8_t *fuse fuse_table

        //Matlab can handle only double as default -> cast 
        for(i=0; i<0x13; i++)  {
	       fuse[i] = (uint8_t) pdouble_param1[i];
        }
        
        int8_param1 = abb_rx_rctune(&fuse[0]);
        debug_printf("abb_rx_rctune : return=%d\n", int8_param1);
        result = (double) int8_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_init"))        
    {   mexPrintf("abb_antenna_init\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_antenna_init(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_fusing"))        
    {   mexPrintf("abb_antenna_fusing\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask

        temp = mxGetN(prhs[2]);     // number of rows
        debug_printf(" array with %d elements detected\n", temp);
        // from 0x018000, 0x13 words
        if (temp < 0x13)  mexErrMsgTxt("array size must be at least 19 paramters");
        pdouble_param1 = mxGetPr(prhs[2]);                                  //uint8_t *fuse fuse_table

        //Matlab can handle only double as default -> cast 
        for(i=0; i<0x13; i++)  {
	       fuse[i] = (uint8_t) pdouble_param1[i];
        }
        
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_antenna_fusing(uint16_param1, &fuse[0]);
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_rx_calibration"))        
    {   mexPrintf("abb_antenna_rx_calibration\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        //uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint16_t rctune
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        int8_param1 = abb_antenna_rx_calibration(uint16_param1);
        debug_printf("abb_antenna_rx_calibration : return=%d\n", int8_param1);
        result = (double)int8_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_force_rxon"))        
    {   mexPrintf("abb_antenna_force_rxon\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        //uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint16_t rctune
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_antenna_force_rxon(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_force_rxoff"))        
    {   mexPrintf("abb_antenna_force_rxoff\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        //uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint16_t rctune
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_antenna_force_rxoff(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_rx_calibration_addon1"))        
    {   mexPrintf("abb_antenna_rx_calibration_addon1\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint8_t start
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        debug_printf("start=%d\n", uint8_param1);   
        int8_param1 = abb_antenna_rx_calibration_addon1(uint16_param1,uint8_param1);
        debug_printf("abb_antenna_rx_calibration_addon1 : return=%d\n", int8_param1);
        result = (double)int8_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_rx_calibration_addon2"))        
    {   mexPrintf("abb_antenna_rx_calibration_addon2\n");
        check_parameter(nrhs,3);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint8_t iqsel
        int8_param1 =   (int8_t)  mxGetScalar(prhs[3]);                      //int8_t decision
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        debug_printf("iqsel=%d\n", uint8_param1);   
        debug_printf("decision=%d\n", int8_param1);   
        int8_param2 = abb_antenna_rx_calibration_addon2(uint16_param1,uint8_param1,int8_param1);
        debug_printf("abb_antenna_rx_calibration_addon2 : return=%d\n", int8_param2);
        result = (double)int8_param2;
    } 
    else if (0 == strcmp(api_cmd,"abb_antenna_rx_calibration_bubble_correction"))        
    {   mexPrintf("abb_antenna_rx_calibration_bubble_correction\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_antenna_rx_calibration_bubble_correction(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_rx_qtmanen"))        
    {   mexPrintf("abb_antenna_rx_qtmanen\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint8_t  qtcalen
        debug_printf("_ant_mask=%d\n", uint16_param1);  
        debug_printf("qtcalen=%d\n", uint8_param1);  
        abb_antenna_rx_qtmanen(uint16_param1,uint8_param1);
    }
    
    else if (0 == strcmp(api_cmd,"abb_antenna_rx_adccalman"))        
    {   mexPrintf("abb_antenna_rx_adccalman\n");
        check_parameter(nrhs,5);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);  

        pdouble_param1 = mxGetPr(prhs[2]);                                   //uint16_t  mCal[1]
        temp = mxGetN(prhs[2]);                                              //number of elements
        debug_printf(" array with %d elements detected\n", temp);
        if (temp < 32)  mexErrMsgTxt("array size must be 32 paramters");
        for(i=0; i<32; i++)  {
                debug_printf("arg = %d %d\n ", (uint8_t) pdouble_param1[i],i);
                mcal[0][i]= (uint8_t) pdouble_param1[i];
       }
        
        pdouble_param1 = mxGetPr(prhs[3]);                                   //uint16_t  mCal[2]
        temp = mxGetN(prhs[3]);                                              //number of elements
        debug_printf(" array with %d elements detected\n", temp);
        if (temp < 32)  mexErrMsgTxt("array size must be 32 paramters");
        for(i=0; i<32; i++)  {
                debug_printf("arg = %d %d\n ", (uint8_t) pdouble_param1[i],i);
                mcal[1][i]= (uint8_t) pdouble_param1[i];
       }

        pdouble_param1 = mxGetPr(prhs[4]);                                   //uint16_t  mCal[3]
        temp = mxGetN(prhs[4]);                                              //number of elements
        debug_printf(" array with %d elements detected\n", temp);
        if (temp < 32)  mexErrMsgTxt("array size must be 32 paramters");
        for(i=0; i<32; i++)  {
                debug_printf("arg = %d %d\n ", (uint8_t) pdouble_param1[i],i);
                mcal[2][i]= (uint8_t) pdouble_param1[i];
       }
        
        uint8_param1 =  (uint8_t)  mxGetScalar(prhs[5]);                     //uint8_t   nComp

        debug_printf("ncomp=%d\n", uint8_param1);  
                
        //test
        /*
        int row=0;
        int col=0;
        for(row=0; row<3; row++)  {
            for (col=0; col<32; col++) {
               mcal[row][col] = (uint8_t)3; //(uint8_t) pdouble_param1[i];
               debug_printf("row= %d col= %d con =%d\n",row,col,mcal[row][col]);
            }
        }*/
        
        abb_antenna_rx_adccalman(uint16_param1,&mcal[0,0],uint8_param1);
        
    }
    
    else if (0 == strcmp(api_cmd,"abb_antenna_config"))        
    {   mexPrintf("abb_antenna_config\n");
        check_parameter(nrhs,4);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        uint8_param1 =  (uint8_t)  mxGetScalar(prhs[2]);                     //uint8_t abb_mode
        int8_param1 =   (int8_t)   mxGetScalar(prhs[3]);                     //int8_t rctune
        int8_param2 =   (int8_t)   mxGetScalar(prhs[4]);                     //int8_t rctune
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("abb_mode=%d, ", uint8_param1);   
        debug_printf("rctune1=%d, ", int8_param1);   
        debug_printf("rctune2=%d\n", int8_param2); 
        if ((uint8_param1!=80) & (uint8_param1!=160) & (uint8_param1!=20)& (uint8_param1!=0))
        { mexErrMsgTxt("Wrong abb_mode parameter detected."); }
        abb_antenna_config(uint16_param1, uint8_param1, int8_param1, int8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_antenna_pd"))        
    {   mexPrintf("abb_antenna_pd\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                     //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_antenna_pd(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_rcmeas_start"))        
    {   mexPrintf("abb_api_rcmeas_start\n");
        check_parameter(nrhs,2);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t polarity
        uint8_param2 = (uint8_t) mxGetScalar(prhs[2]);                     //uint8_t dsmsel
        debug_printf("polarity=%d, ", int8_param1);   
        debug_printf("dsmsel=%d\n", uint8_param2);   
        abb_api_rcmeas_start(uint8_param1, uint8_param2);
    }

// Section: LL API Functions    
    else if (0 == strcmp(api_cmd,"abb_api_central_trim_ldo0v95fsys"))        
    {   mexPrintf("abb_api_central_trim_ldo0v95fsys\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t trim
        debug_printf("trim=%d\n", uint8_param1);   
        abb_api_central_trim_ldo0v95fsys(uint8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_init_central_pll_bypass"))        
    {   mexPrintf("abb_api_init_central_pll_bypass\n");
        abb_api_init_central_pll_bypass();
    }
    else if (0 == strcmp(api_cmd,"abb_api_init_central_pll_enable"))        
    {   mexPrintf("abb_api_init_central_pll_enable\n");
        abb_api_init_central_pll_enable();
    }
    else if (0 == strcmp(api_cmd,"abb_api_init_central_pll_enable_fcsicontrol"))        
    {   mexPrintf("abb_api_init_central_pll_enable_fcsicontrol\n");
        abb_api_init_central_pll_enable_fcsicontrol();
    }
    else if (0 == strcmp(api_cmd,"abb_api_central_trim_ldo0v95ref"))        
    {   mexPrintf("abb_api_central_trim_ldo0v95ref\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t trim
        debug_printf("trim=%d\n", uint8_param1);   
        abb_api_central_trim_ldo0v95ref(uint8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_central_trim_ldo0v95post"))        
    {   mexPrintf("abb_api_central_trim_ldo0v95post\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t trim
        debug_printf("trim=%d\n", uint8_param1);   
        abb_api_central_trim_ldo0v95post(uint8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_central_trim_vreg1v8hv"))        
    {   mexPrintf("abb_api_central_trim_vreg1v8hv\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t trim
        debug_printf("trim=%d\n", uint8_param1);   
        abb_api_central_trim_vreg1v8hv(uint8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_central_trim_bias"))        
    {   mexPrintf("abb_api_central_trim_vreg1v8hv\n");
        check_parameter(nrhs,3);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t vref
        uint8_param2 = (uint8_t) mxGetScalar(prhs[2]);                     //uint8_t iptat
        uint8_param3 = (uint8_t) mxGetScalar(prhs[3]);                     //uint8_t ibias
        debug_printf("vref=%d, ", uint8_param1);   
        debug_printf("iptat=%d, ", uint8_param2);   
        debug_printf("ibias=%d\n", uint8_param3);   
        abb_api_central_trim_bias(uint8_param1, uint8_param2, uint8_param3);
    }
    else if (0 == strcmp(api_cmd,"abb_api_init_central_pup_clkd"))        
    {   mexPrintf("abb_api_init_central_pup_clkd\n");
        abb_api_init_central_pup_clkd();
    }
    else if (0 == strcmp(api_cmd,"abb_api_init_central_pd_clkd"))        
    {   mexPrintf("abb_api_init_central_pd_clkd\n");
        abb_api_init_central_pd_clkd();
    }
    else if (0 == strcmp(api_cmd,"abb_api_central_trim_ldo0v95clkd"))        
    {   mexPrintf("abb_api_central_trim_ldo0v95clkd\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t trim
        debug_printf("trim=%d\n", uint8_param1);   
        abb_api_central_trim_ldo0v95clkd(uint8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_central_version"))        
    {   mexPrintf("abb_api_read_central_version\n");
        uint16_param1 = abb_api_read_central_version();
        debug_printf("abb_api_read_central_version : return=%d\n", uint16_param1);
        result = (double)uint16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_lcpll_lock"))        
    {   mexPrintf("abb_api_read_lcpll_lock\n");
        uint16_param1 = abb_api_read_lcpll_lock();
        debug_printf("abb_api_read_lcpll_lock : return=%d\n", uint16_param1);
        result = (double)uint16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_lcpll_prelock"))        
    {   mexPrintf("abb_api_read_lcpll_prelock\n");
        uint16_param1 = abb_api_read_lcpll_prelock();
        debug_printf("abb_api_read_lcpll_prelock : return=%d\n", uint16_param1);
        result = (double)uint16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_rcmeas_default"))        
    {   mexPrintf("abb_api_rcmeas_default\n");
        abb_api_rcmeas_default();
    }
    else if (0 == strcmp(api_cmd,"abb_api_rcmeas_running"))        
    {   mexPrintf("abb_api_rcmeas_running\n");
        uint16_param1 = abb_api_rcmeas_running();
        debug_printf("abb_api_rcmeas_running : return=%d\n", uint16_param1);
        result = (double)uint16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_rcmeas_finished"))        
    {   mexPrintf("abb_api_rcmeas_finished\n");
        uint16_param1 = abb_api_rcmeas_finished();
        debug_printf("abb_api_rcmeas_finished : return=%d\n", uint16_param1);
        result = (double)uint16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_rcmeas"))        
    {   mexPrintf("abb_api_read_rcmeas\n");
        uint16_param1 = abb_api_read_rcmeas();
        debug_printf("abb_api_read_rcmeas : return=%d\n ", uint16_param1);   
        result = (double)uint16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_init_ant"))        
    {   mexPrintf("abb_api_init_ant\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   
        abb_api_init_ant(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ant_trim_daclb"))        
    {   mexPrintf("abb_api_ant_trim_daclb\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        uint8_param2 =  (uint8_t)  mxGetScalar(prhs[2]);                   //uint8_t trim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("trim=%d\n", uint8_param2);   
        abb_api_ant_trim_daclb(uint16_param1, uint8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ant_trim_ldo0v95a"))        
    {   mexPrintf("abb_api_ant_trim_ldo0v95a\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        uint8_param2 =  (uint8_t)  mxGetScalar(prhs[2]);                   //uint8_t trim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("trim=%d\n", uint8_param2);   
        abb_api_ant_trim_ldo0v95a(uint16_param1, uint8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ant_trim_ldo0v95d"))        
    {   mexPrintf("abb_api_ant_trim_ldo0v95d\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        uint8_param2 =  (uint8_t)  mxGetScalar(prhs[2]);                   //uint8_t trim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("trim=%d\n", uint8_param2);   
        abb_api_ant_trim_ldo0v95d(uint16_param1, uint8_param2);
    }
    
    else if (0 == strcmp(api_cmd,"abb_api_read_antenna_version"))        
    {   mexPrintf("abb_api_read_antenna_version\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        uint16_param2 = abb_api_read_antenna_version(uint16_param1);
        debug_printf("abb_api_read_antenna_version : return=%d\n", uint16_param2);
        result = (double)uint16_param2;
    }
    else if (0 == strcmp(api_cmd,"abb_api_pd_ant"))        
    {   mexPrintf("abb_api_pd_ant\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_read_antenna_version(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_tx"))        
    {   mexPrintf("abb_api_start_tx\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_start_tx(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_tx_lp"))        
    {   mexPrintf("abb_api_start_tx_lp\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_start_tx_lp(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_standby_tx"))        
    {   mexPrintf("abb_api_standby_tx\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_standby_tx(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_trim_rx_rctune"))        
    {   mexPrintf("abb_api_trim_rx_rctune\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //int8_t rctune
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rctune=%d\n", int8_param1);   

        abb_api_trim_rx_rctune(uint16_param1,int8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_set_rx_rctune_qtrim"))        
    {   mexPrintf("abb_api_start_rx_11ax\n");
        check_parameter(nrhs,3);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //int8_t rctune
        int8_param2 =   (int8_t)   mxGetScalar(prhs[3]);                   //int8_t qtrim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rctune=%d, ", uint16_param1);   
        debug_printf("qtrim=%d\n", int8_param2);   

        abb_api_start_rx_11ax(uint16_param1, int8_param1, int8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_rx_11ax"))        
    {   mexPrintf("abb_api_start_rx_11ax\n");
        check_parameter(nrhs,3);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //int8_t rctune
        int8_param2 =   (int8_t)   mxGetScalar(prhs[3]);                   //int8_t qtrim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rctune=%d, ", uint16_param1);   
        debug_printf("qtrim=%d\n", int8_param2);   

        abb_api_start_rx_11ax(uint16_param1, int8_param1, int8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_rx_ssb"))        
    {   mexPrintf("abb_api_start_rx_ssb\n");
        check_parameter(nrhs,3);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //int8_t rctune
        int8_param2 =   (int8_t)   mxGetScalar(prhs[3]);                   //int8_t qtrim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rctune=%d, ", uint16_param1);   
        debug_printf("qtrim=%d\n", int8_param2);   

        abb_api_start_rx_ssb(uint16_param1, int8_param1, int8_param2);
    }
     else if (0 == strcmp(api_cmd,"abb_api_start_rx_lp"))        
    {   mexPrintf("abb_api_start_rx_lp\n");
        check_parameter(nrhs,3);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //int8_t rctune
        int8_param2 =   (int8_t)   mxGetScalar(prhs[3]);                   //int8_t qtrim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rctune=%d, ", uint16_param1);   
        debug_printf("qtrim=%d\n", int8_param2);   

        abb_api_start_rx_lp(uint16_param1, int8_param1, int8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_rx_qt"))        
    {   mexPrintf("abb_api_read_rx_qt\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        uint16_param2 = abb_api_read_rx_qt(uint16_param1);
        debug_printf("abb_api_read_rx_qt : return=%d\n", uint16_param2);
        result = (double)uint16_param2;
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_rx_qtcaldone"))        
    {   mexPrintf("abb_api_read_rx_qtcaldone\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        uint16_param2 = abb_api_read_rx_qtcaldone(uint16_param1);
        debug_printf("abb_api_read_rx_qtcaldone : return=%d\n", uint16_param2);
        result = (double)uint16_param2;
    }
     else if (0 == strcmp(api_cmd,"abb_api_start_rx_loopback"))        
    {   mexPrintf("abb_api_start_rx_loopback\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d \n", uint16_param1);   
        
        abb_api_start_rx_loopback(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_rx_lfsr"))        
    {   mexPrintf("abb_api_start_rx_lfsr\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_start_rx_lfsr(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_rx_ts_lfsr"))        
    {   mexPrintf("abb_api_start_rx_ts_lfsr\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_start_rx_ts_lfsr(uint16_param1);
    }
     else if (0 == strcmp(api_cmd,"abb_api_standby_rx"))        
    {   mexPrintf("abb_api_standby_rx\n");
        check_parameter(nrhs,3);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //int8_t rctune
        int8_param2 =   (int8_t)   mxGetScalar(prhs[3]);                   //int8_t qtrim
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rctune=%d, ", uint16_param1);   
        debug_printf("qtrim=%d\n", int8_param2);   

        abb_api_standby_rx(uint16_param1, int8_param1, int8_param2);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_ssi"))        
    {   mexPrintf("abb_api_start_ssi\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_start_ssi(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_recal_ssi"))        
    {   mexPrintf("abb_api_recal_ssi\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_recal_ssi(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_start_ssi_lfsr"))        
    {   mexPrintf("abb_api_start_ssi_lfsr\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_start_ssi_lfsr(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_standby_ssi"))        
    {   mexPrintf("abb_api_standby_ssi\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_standby_ssi(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_read_ssi_cal_comp"))        
    {   mexPrintf("abb_api_read_ssi_cal_comp\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        int16_param1 = abb_api_read_ssi_cal_comp(uint16_param1);
        debug_printf("abb_api_read_ssi_cal_comp : return=%d\n", int16_param1);
        result = (double)int16_param1;
    }
    else if (0 == strcmp(api_cmd,"abb_api_ssi_enable_exton"))        
    {   mexPrintf("abb_api_ssi_enable_exton\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_ssi_enable_exton(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ssi_set_extssion"))        
    {   mexPrintf("abb_api_ssi_set_extssion\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //uint8_t ssion
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("ssion=%d\n", int8_param1);   

        abb_api_ssi_set_extssion(uint16_param1, int8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ssi_set_exttxon"))        
    {   mexPrintf("abb_api_ssi_set_exttxon\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //uint8_t txon
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("txon=%d\n", int8_param1);   

        abb_api_ssi_set_exttxon(uint16_param1, int8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ssi_set_extrxon"))        
    {   mexPrintf("abb_api_ssi_set_extrxon\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        int8_param1 =   (int8_t)   mxGetScalar(prhs[2]);                   //uint8_t rxon
        debug_printf("_ant_mask=%d, ", uint16_param1);   
        debug_printf("rxon=%d\n", int8_param1);   

        abb_api_ssi_set_extrxon(uint16_param1, int8_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_ssi_disable_exton"))        
    {   mexPrintf("abb_api_ssi_disable_exton\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_ssi_disable_exton(uint16_param1);
    }
// DFT CENTRAL BIAS   
    else if (0 == strcmp(api_cmd,"abb_api_anamux_centralbias_RFIC1V8_atITRIM"))        
    {   mexPrintf("abb_api_anamux_centralbias_RFIC1V8_atITRIM\n");
        abb_api_anamux_centralbias_RFIC1V8_atITRIM();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_centralbias_ICONST_atITRIM"))        
    {   mexPrintf("abb_api_anamux_centralbias_ICONST_atITRIM\n");
        abb_api_anamux_centralbias_ICONST_atITRIM();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_centralbias_IPTAT_atITRIM"))        
    {   mexPrintf("abb_api_anamux_centralbias_IPTAT_atITRIM\n");
        abb_api_anamux_centralbias_IPTAT_atITRIM();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_centralbias_VBGP_atITRIM"))        
    {   mexPrintf("abb_api_anamux_centralbias_VBGP_atITRIM\n");
        abb_api_anamux_centralbias_VBGP_atITRIM();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_centralbias_disable"))        
    {   mexPrintf("abb_api_anamux_centralbias_disable\n");
        abb_api_anamux_centralbias_disable();
    }
// DFT CENTRAL CLKD
    else if (0 == strcmp(api_cmd,"abb_api_anamux_clkd_VREFLDO_atVREF"))        
    {   mexPrintf("abb_api_anamux_clkd_VREFLDO_atVREF\n");
        abb_api_anamux_clkd_VREFLDO_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_clkd_LDO0V95CLKD_atVREF"))        
    {   mexPrintf("abb_api_anamux_clkd_LDO0V95CLKD_atVREF\n");
        abb_api_anamux_clkd_LDO0V95CLKD_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_clkd_disable"))        
    {   mexPrintf("abb_api_anamux_clkd_disable\n");
        abb_api_anamux_clkd_disable();
    }
// DFT CENTRAL LCPLL
    else if (0 == strcmp(api_cmd,"abb_api_anamux_lcpll_VREG1V8HV_atVREF"))        
    {   mexPrintf("abb_api_anamux_lcpll_VREG1V8HV_atVREF\n");
        abb_api_anamux_lcpll_VREG1V8HV_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_lcpll_LDO0V95PLLREF_atVREF"))        
    {   mexPrintf("abb_api_anamux_lcpll_LDO0V95PLLREF_atVREF\n");
        abb_api_anamux_lcpll_LDO0V95PLLREF_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_lcpll_LDO0V95PLLPOST_atVREF"))        
    {   mexPrintf("abb_api_anamux_lcpll_LDO0V95PLLPOST_atVREF\n");
        abb_api_anamux_lcpll_LDO0V95PLLPOST_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_lcpll_ATESTOUT_atVREF"))        
    {   mexPrintf("abb_api_anamux_lcpll_ATESTOUT_atVREF\n");
        abb_api_anamux_lcpll_ATESTOUT_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_lcpll_disable"))        
    {   mexPrintf("abb_api_anamux_lcpll_disable\n");
        abb_api_anamux_lcpll_disable();
    }
// DFT CENTRAL CLKIN    
    else if (0 == strcmp(api_cmd,"abb_api_anamux_clkin_VREFLDO_atVREF"))        
    {   mexPrintf("abb_api_anamux_clkin_VREFLDO_atVREF\n");
        abb_api_anamux_clkin_VREFLDO_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_clkin_LDO0V95FSYS_atVREF"))        
    {   mexPrintf("abb_api_anamux_clkin_LDO0V95FSYS_atVREF\n");
        abb_api_anamux_clkin_LDO0V95FSYS_atVREF();
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_clkin_disable"))        
    {   mexPrintf("abb_api_anamux_clkin_disable\n");
        abb_api_anamux_clkin_disable();
    }
// DFT ANTENNA BIAS    
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antbias_VREF0V95_atVREF"))        
    {   mexPrintf("abb_api_anamux_antbias_VREF0V95_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antbias_VREF0V95_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antbias_LDO0V95A_atVREF"))        
    {   mexPrintf("abb_api_anamux_antbias_LDO0V95A_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antbias_LDO0V95A_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antbias_LDO0V95D_atVREF"))        
    {   mexPrintf("abb_api_anamux_antbias_LDO0V95D_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antbias_LDO0V95D_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antbias_VSSAANT_atVREF"))        
    {   mexPrintf("abb_api_anamux_antbias_VSSAANT_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antbias_VSSAANT_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antbias_disable"))        
    {   mexPrintf("abb_api_anamux_antbias_disable\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antbias_disable(uint16_param1);
    }
// DFT ANTENNA SSI
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antssi_vrefn_atVREF"))        
    {   mexPrintf("abb_api_anamux_antssi_vrefn_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antssi_vrefn_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antssi_vrefp_atVREF"))        
    {   mexPrintf("abb_api_anamux_antssi_vrefp_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antssi_vrefp_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antssi_vcm_r_atVREF"))        
    {   mexPrintf("abb_api_anamux_antssi_vcm_r_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antssi_vcm_r_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antssi_vcm_atVREF"))        
    {   mexPrintf("abb_api_anamux_antssi_vcm_atVREF\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antssi_vcm_atVREF(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_api_anamux_antssi_disable"))        
    {   mexPrintf("abb_api_anamux_antssi_disable\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_api_anamux_antssi_disable(uint16_param1);
    }

// SEQUENCES
    
    else if (0 == strcmp(api_cmd,"abb_sequence_read_central_fcsi"))        
    {   mexPrintf("abb_sequence_read_central_fcsi\n");
        abb_sequence_read_central_fcsi();
    }
    else if (0 == strcmp(api_cmd,"abb_sequence_read_antenna_fcsi"))        
    {   mexPrintf("abb_sequence_read_antenna_fcsi\n");
        check_parameter(nrhs,1);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _ant_mask
        debug_printf("_ant_mask=%d\n", uint16_param1);   

        abb_sequence_read_antenna_fcsi(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_sequence_central_pll_active"))        
    {   mexPrintf("abb_sequence_central_pll_active\n");
        check_parameter(nrhs,1);
        uint8_param1 = (uint8_t) mxGetScalar(prhs[1]);                     //uint8_t prelock
        debug_printf("prelock=%d\n", uint8_param1);   

        abb_sequence_central_pll_active(uint16_param1);
    }
    else if (0 == strcmp(api_cmd,"abb_sequence_central_pll_bypass"))        
    {   mexPrintf("abb_sequence_central_pll_bypass\n");
        abb_sequence_central_pll_bypass();
    }
    else if (0 == strcmp(api_cmd,"abb_sequence_dft"))        
    {   mexPrintf("abb_sequence_dft\n");
        abb_sequence_dft();
    }
    else if (0 == strcmp(api_cmd,"abb_sequence_clock_reprogramming"))        
    {   mexPrintf("abb_sequence_clock_reprogramming\n");
        check_parameter(nrhs,2);
        uint16_param1 = (uint16_t) mxGetScalar(prhs[1]);                   //uint16_t _op_freq
        uint16_param2 = (uint16_t) mxGetScalar(prhs[2]);                   //uint16_t _div
        debug_printf("_op_freq=%d, ", uint16_param1);   
        debug_printf("_div=%d\n", uint16_param2);   

        abb_sequence_clock_reprogramming(uint16_param1, uint16_param2);
    }
    
    else if (0 == strcmp(api_cmd,"abb_fcsirf"))        
    {   mexPrintf("abb_fcsirf\n");
        check_parameter(nrhs,3);
        uint8_param1 =   (int8_t)   mxGetScalar(prhs[1]);                   //uint8_t sr
        uint8_param2 =   (int8_t)   mxGetScalar(prhs[2]);                   //uint8_t smt
        uint8_param3 =   (int8_t)   mxGetScalar(prhs[3]);                   //uint8_t drivestrength
        debug_printf("sr=%d, ", uint8_param1);   
        debug_printf("smt=%d, ", uint8_param2);   
        debug_printf("drivestrength=%d \n ", uint8_param3);   

        abb_fcsirf(uint8_param1, uint8_param2, uint8_param3);
    }

    else                                                        //Function not found

    {
        mexPrintf("%s ->", api_cmd);
        mexErrMsgTxt("API command not found.");
    }
    
    *lhparam0 = result;      //pass up one parameter
     
    mxFree(api_cmd);
    return;
}


// Local function definition, needed by mapper

void check_parameter(int parameter1, int parameter2)
{
    if (parameter1 < parameter2+1)      //1- api_sting, 2-parameter1, 3-...
            mexErrMsgTxt("Wrong amount of parameters detected.");
}



