// ----------------------------------------------------------------------------
// Project    : WAVE600
// ----------------------------------------------------------------------------
// File       : lib_abb_api.c
// Author     : Pridnig Peter
// Created    : 2017-03-24
// Standard   : System Verilog
// ----------------------------------------------------------------------------
// Copyright (c) 2018 Intel GmbH
// ---------------------------------------------------------------------------
// Revisions  :
// Date        Version  Author    Description
// 2017-05-02  D1p1     Pridnig
// 2017-05-16  D1p2     Pridnig
// 2017-10-13  D2p0     Pridnig
// 2017-10-25  D2p01    Pridnig
// 2018-02-16  LA1p1    Pridnig
// 2018-03-14  LA1p11   Pridnig    Bringup version
// 2018-04-12  LA1p2    Pridnig    Bringup version fixes
// 2018-05-08  LA1p21   Pridnig    added abb_fcsirf
// 2018-05-28  LA1p3    Pridnig    adjusted abb_antenna_rx_calibration and abb_rx_rctune
// 2018-06-06  LA1p31   Pridnig    adjusted abb_antenna_rx_calibration with loopback mode
// 2018-06-26  LA1p32   Pridnig    updated 11ax rx OPA programming
// 2018-07-25  LA1p33   Pridnig    bugfix start_tx_lp and update rx_calibration: OPA prog & CALDACMODE=2

// 2018-07-25  SPLIT B0 ------- B0 MONO ------- B0 MONO ------- B0 MONO ------- B0 MONO -------

// 2018-10-09  LA1p341  Pridnig    B0 updates 
//                       	   adjustd abb_api_start_ssi to local cal
//                                 added abb_api_recal_ssi
// 2019-01-14  LA1p351  Pridnig    rx calibration updates
//				   ssiadc wide calibration range
// 	      			   rxoff_11ax via FCSI
// 2019-01-31  LA1p361  Pridnig    rx_off2 ARC


// ----------------------------------------------------------------------------
// History    :
// 2017-05-02 copied from tc_ABB_A11
//            removed *clko*
//   	      changed prefix to abb_	
// 2017-05-03 added abb_api_read_lcpll_prelock()
// 2017-05-12 added abb_api_start_rcmeas / abb_api_read_rcmeas / abb_api_rcmeas_running
// 2017-05-15  _wait->ABB_wait_us
//            abb_api_init_ant: 0x44, 0x0ABF
// 2017-05-16 cleaned up debug_printf statements;
//            moved fcsi_read to tests.c
// 2017-05-17 combined tx_config and tx_start including WAIT
// 2017-05-19 updated IDACD0 = 0x47 = 0x000 defaultvalue
// 2017-05-22 added abb_api_start_rx_lp
// 2017-05-26 updated op_freq_sel_sc_d2a DIRECT_WRITES bus access
//            renamed abb11ax->abb
// 2017-06-02 changed dac to 2s complement input
// 2017-07-03 updated DFT functions
// 2017-07-14 updated RX and RX_LP settings; 
//	      added defaults for ant_init
//	      fixed DFT anamux mask
//	      add rx_w320 dummy function
// 2017-07-25 added ant ldo fcsi reset release into init_ant
//            renamed w320 mode to ssb mode
//            updated ant ldo default config 0x3C
// 2017-07-26 dac dten=1
//            adc renamed start_rx->start_rx_11ax; 
//            updated 11ax/lp/ssb settings
// 2017-08-04 changed dten back to 0
//            changed ant ldo target to 1.03
//            changed central ldo target to 0.98
// 2017-08-11 added ssi_exton features
// 2017-09-25 added abb_api_read_rx_qt
//            updated start_rx_? functions
// 2017-10-11 updated ant reg 0x4C: vcmprog=2 default
//            added abb_api_ant_trim_daclb
// 2017-10-13 updated 0x46 IDACA0 to default 06A0 (itrim=01)
//            updated abb_api_pd_rx
// 2017-10-18 added abb_adctune(uint16_t count_calc, uint16_t* ctune)
// 2017-10-23 added abb_api_rctune_rx
//            renamed abb_api_disable_rcmeas to abb_api_rcmeas_disable
// 2017-11-16 abb_api_init_central_pll* added PLL LDO fcsi setting to 0.95V
// 2018-01-15 added abb_central_fusing
// 2018-01-17 added abb_central_init_pll and abb_central_init_clkd
// 2018-01-19 added abb_rx_rctune
// 2018-01-22 added abb_rcmeas; renamed abb_api_start_rcmeas to abb_api_rcmeas_start
// 2018-01-23 updated abb_antenna_init; added abb_antenna_fusing
// 2018-01-26 added abb_antenna_rx_calibration and abb_api_read_rx_qtcaldone
// 2018-01-30 updated abb_antenna_config
//            updated abb_api_start_rx_11ax
// 2018-01-31 used int8_t instead of uint8_t wherever -1 error code occurs
//            added abb_api_set_rx_rctune_qtrim
// 2018-02-01 updated start_rx_*
// 2018-02-12 updated *pd* and *dft*
// 2018-02-14 renamed abb_api_anamux_antbias_VREF0V95_atVREF
//	      renamed abb_api_anamux_antbias_VSSAANT_atVREF
//            updated abb_api_start_rx_loopback
// 2018-02-16 LA1p1 release: abb_rcmeas not called; delivers nominal values
//                           sdadc qtrim = 0
// 2018-02-19 removed rctune/qtrim from abb_api_start_rx_loopback
// 2018-02-20 updated abb_api_start_ssi (IBREFSEL=CONST & ENCRS=ON)
// 2018-03-08 added abb_api_fcsiwrite_softisolation_central
//            and abb_api_fcsiwrite_softisolation_antenna
// 2018-03-09 added abb_api_fcsiread_softisolation_central
//            and abb_api_fcsiread_softisolation_antenna
//            bringup version: abb_rx_rctune returns always RCTUNE=12
// 2018-03-14 added abb_api_version
// 2018-04-10 adjusted abb_api_read_rcmeas
//            renamed abb_api_pd_tx/rx/ssi -> abb_api_standby_tx/rx/ssi
//            renamed ABB_OFF -> ABB_STANDBY
// 2018-40-12 updated abb_central_fusing: at trimindicator=0: fuse default
// 2018-05-08 added abb_fcsirf
// 2018-05-18 update abb_antenna_rx_calibration
// 2018-05-28 updated abb_adctune: RC=func[CNT-38]
//            fully enabled abb_rx_rctune including abb_rcmeas call
// 2018-06-06 update abb_antenna_rx_calibration: tx loopback mode
// 2018-06-26 updated "abb_api_start_rx_11ax" "abb_api_standby_rx"
//	      to new rx OPA programming "opa1=1 opa2=2 opa3=3"
// 2018-07-25 start_tx_lp     BINARYIN=0
//            rx_calibration: OPA programming "opa1=1 opa2=2 opa3=3"
//              	      CALDACMODE=2
//	      update start_rx_*: CALDACMODE=2 

// 2018-07-25 B0 REDESIGN -------- B0 REDESIGN -------- B0 REDESIGN -------- B0 REDESIGN --------

// 2018-07-25 B0 updates: IADCA1 frclfsrresn/def=0/th_trim05/def=1
//                        IADCA2 caldacmode/def=10
//                        IADCA4 calclkpolarity/def=1
//                        IDACA0 frctxclkon/def=0
//			  start_rx_lfsr: added commented out statement to force A0 lfrs reset behavior
// 2018-08-13 IADCA1<12> clkinv 0->1 (CLKINV_NOTINV->CLKINV_INV)
// 2018-08-20 uint8_t->uint16_t abb_api_version()
// 2018-09-24 updated abb_api_start_ssi (offset calibration fix: local calibration)
//            added abb_api_recal_ssi
// 2018-11-09 IADCA1<0>  rename UNIQT to OVMASK (mask reset in overload conditions)
//	      added abb_api_read_rx_qtcalactive
// 2018-11-19 added abb_api_rx_set_mask_reset_overload(on/off) (mask reset in overload conditions)
// 2018-11-23 update abb_api_start_ssi (local calibration and wide calibration range)
//            changed rx_calibration to lp mode
// 2018-01-10 changed rx_calibration back to 11ax mode
// 2018-01-14 added abb_antenna_rxoff_11ax
// 2019-01-31 added abb_antenna_rxoff2_11ax
// 2019-03-28 updated function() with function(void)
//	      removed unreachable break statemens
// 2019-08-05 change "int8_t abb_rx_rctune"


#include <stdint.h>
#include <math.h>
#include <stdio.h>

//target.h not needed; to be commented out
//#include "target.h"
//#include "lib_test.h"

/* global definitions */
#include "abb_tb_defines.h"   	// link to sv testbench.. benchports
#include "abb_mode_defines.h"

/* target interface definitions */
#include "interface_abb.h"

// API definitions
#include "lib_abb_api.h"

//Spectool2 c-outputs
#include "central_fcsi_fieldpkg.h" // register for CENTRAL
#include "central_fcsi_regpkg.h"   // register for CENTRAL

#include "ant_fcsi_fieldpkg.h"     // register for ANT
#include "ant_fcsi_regpkg.h"       // register for ANT



uint16_t abb_api_version(void)
{

 // 2019-01-31  LA1p361  Pridnig    B0
 
 return 1361;
 
}


// ---------------------------------------------------------
// Section: API Functions for DRIVER USAGE
// ---------------------------------------------------------

void abb_central_fusing(uint8_t *fuse)
{

 //WAV600 Fusetable:
 //https://sharepoint.gar.ith.intel.com/sites/WAVE600C/Shared%20Documents/Chip%20Development/Micro-architecture%20and%20Design/Top/WAV600%20Efuse%20Mapping.xlsx

    //function is called after FCSI reset!

    // reg0, bit 7:4
    int ibiastrim = (fuse[0] >> 4) & 0x0F;

    // reg0, bit 3:0
    int vreftrim = fuse[0] & 0x0F;

    // reg1, bit 7
    int trim_indicator = fuse[1] >> 7 & 0x01;
    

    // reg1, bit 6:4
    int ldo_central_pll1v8 = (fuse[1] >> 4) & 0x07;
    
    // reg1, bit 3:0
    int ldo_central_pllref = fuse[1] & 0x0F;
    
    // reg2, bit 7:4
    int ldo_central_pllpost = (fuse[2] >> 4) & 0x0F;  

    // reg2, bit 3:0
    int iptattrim = fuse[2] & 0x0F;
    
    // reg3, bit 7:4
    int ldo_central_clkd = (fuse[3] >> 4) & 0x0F;      

    // reg3, bit 3:0
    int ldo_central_clkin = fuse[3] & 0x0F;
    
    /*
    //debug output
    debug_printf("fuse[0]        = %d;\n",fuse[0]);
    debug_printf("fuse[1]        = %d;\n",fuse[1]);
    debug_printf("fuse[2]        = %d;\n",fuse[2]);
    debug_printf("fuse[3]        = %d;\n",fuse[3]);

    debug_printf("trim_indicator = %d;\n",trim_indicator);   

    debug_printf("ibiastrim      = %d;\n",ibiastrim);    
    debug_printf("vreftrim       = %d;\n",vreftrim);

    debug_printf("trim_indicator      = %d;\n",trim_indicator);
    debug_printf("ldo_central_pll1v8  = %d;\n",ldo_central_pll1v8);
    debug_printf("ldo_central_pllref  = %d;\n",ldo_central_pllref);

    debug_printf("ldo_central_pllpost = %d;\n",ldo_central_pllpost);
    debug_printf("iptattrim           = %d;\n",iptattrim);

    debug_printf("ldo_central_clkd    = %d;\n",ldo_central_clkd);
    debug_printf("ldo_central_clkin   = %d;\n",ldo_central_clkin);
    */	

 if (trim_indicator == 1)
    {
     	abb_api_central_trim_bias(vreftrim,iptattrim,ibiastrim);

	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_IVREG0V950, 	
		( CENTRAL_FCSI_IVREG0V950__INIT  & CENTRAL_FCSI_IVREG0V950_OUTTRIM__ISMSK   ) | 
		   	(ldo_central_clkin & CENTRAL_FCSI_IVREG0V950_OUTTRIM__MSK) );

	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_IVREG1V8HV0,  	
		( CENTRAL_FCSI_IVREG1V8HV0__INIT & CENTRAL_FCSI_IVREG1V8HV0_OUTTRIM__ISMSK  ) | 
			(ldo_central_pll1v8 & CENTRAL_FCSI_IVREG1V8HV0_OUTTRIM__MSK) );

	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ILDOPLLREF0,   
		( CENTRAL_FCSI_ILDOPLLREF0__INIT  & CENTRAL_FCSI_ILDOPLLREF0_OUTTRIM__ISMSK  ) | 
			(ldo_central_pllref       & CENTRAL_FCSI_ILDOPLLREF0_OUTTRIM__MSK  ) );
			
	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ILDOPLLPOST0,   
		( CENTRAL_FCSI_ILDOPLLPOST0__INIT  & CENTRAL_FCSI_ILDOPLLPOST0_OUTTRIM__ISMSK  ) | 
			(ldo_central_pllpost       & CENTRAL_FCSI_ILDOPLLPOST0_OUTTRIM__MSK  ) );		
	
	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ILDOCLKD0,     
		( CENTRAL_FCSI_ILDOCLKD0__INIT & CENTRAL_FCSI_ILDOCLKD0_OUTTRIM__ISMSK ) | 
			(ldo_central_clkd       & CENTRAL_FCSI_ILDOCLKD0_OUTTRIM__MSK  ) );
	
    }
    else
    {
    	// function is called after FCSI reset
        // trim biasing and central ldos to default where required
	
	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ICENTBIAS0,
		CENTRAL_FCSI_ICENTBIAS0__INIT);
	
	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_IVREG0V950, 	
		CENTRAL_FCSI_IVREG0V950__INIT );

	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_IVREG1V8HV0,  	
		CENTRAL_FCSI_IVREG1V8HV0__INIT  );	

	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ILDOPLLREF0, 	
		( CENTRAL_FCSI_ILDOPLLREF0__INIT    & CENTRAL_FCSI_ILDOPLLREF0_OUTTRIM__ISMSK  ) | 
			CENTRAL_FCSI_ILDOPLLREF0_OUTTRIM_0V951__SVAL );
	
	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ILDOPLLPOST0, 	
		( CENTRAL_FCSI_ILDOPLLPOST0__INIT   & CENTRAL_FCSI_ILDOPLLPOST0_OUTTRIM__ISMSK ) | 
			CENTRAL_FCSI_ILDOPLLPOST0_OUTTRIM_0V951__SVAL );
			
	ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ILDOCLKD0,     
		CENTRAL_FCSI_ILDOCLKD0__INIT  );				

    }
}


void abb_central_init_pll(void)
{ 
 // program reset values
 ABB_DIR_CTRL_WRITE(lcpll_pup_d2a,        0x0);
 ABB_DIR_CTRL_WRITE(lcpll_resn_d2a,       0x0);  
 ABB_DIR_CTRL_WRITE(pd_clk_sc_d2a,        0x1);
 ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x0);
 ABB_DIR_CTRL_WRITE(lcpll_div_d2a,        0x0);
 ABB_DIR_CTRL_WRITES(spare_config_i_d2a,8,0x00);
 
 // lcpll powerup
 ABB_DIR_CTRL_WRITE(lcpll_pup_d2a,        0x1);   
 ABB_DIR_CTRL_WRITE(lcpll_resn_d2a,       0x1); 

}


void abb_central_init_clkd(void)
{

 //affects: 640M and 160M clk_sc_a2d and stb_sc_a2d and
 //         abb internal clock distribution 
 //CTRLSYNCDL=100=default
 
 ABB_DIR_CTRL_WRITE(pd_clk_sc_d2a,0x0);
 
}


// ---------------------------------------------------------
// Section: API Functions for FW USAGE
// ---------------------------------------------------------


void abb_fcsirf(uint8_t sr, uint8_t smt, uint8_t drvstrength)
{

 ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ICLKIN0, 	
		( CENTRAL_FCSI_ICLKIN0__INIT            & 
		  CENTRAL_FCSI_ICLKIN0_FCSIRFE__ISMSK   & 
		  CENTRAL_FCSI_ICLKIN0_FCSIRFSMT__ISMSK &
		  CENTRAL_FCSI_ICLKIN0_FCSIRFSR__ISMSK  ) | 
		   	(sr          & CENTRAL_FCSI_ICLKIN0_FCSIRFE__MSK   ) << CENTRAL_FCSI_ICLKIN0_FCSIRFSR__POS  |
			(smt         & CENTRAL_FCSI_ICLKIN0_FCSIRFSMT__MSK ) << CENTRAL_FCSI_ICLKIN0_FCSIRFSMT__POS |
			(drvstrength & CENTRAL_FCSI_ICLKIN0_FCSIRFE__MSK   ) << CENTRAL_FCSI_ICLKIN0_FCSIRFE__POS
		);
}


int16_t abb_rcmeas(int16_t *rccount, uint8_t dsmsel)
{
 // FCSI default state:
 // RESET =1
 // START =0
 // COMP_ENABLE=0
 // POL=0

  uint16_t measdone = 0;
  uint16_t countvalue1 = 0;
  uint16_t countvalue2 = 0;
  uint16_t count_calc = 0;
  uint16_t i = 0;

 abb_api_rcmeas_default();
 
 abb_api_rcmeas_start(ABB_POL_LO,dsmsel);
     
     
 //at 128MHz ca. 212ns per each FCSI access
 //rcmeas duration ca. 724ns
 //limit waiting time to ca. 2.1us
   
 while ((!measdone) && (i++ < 10)) 
    {
        measdone=abb_api_rcmeas_finished();
	
    }
    
 //debug_printf("abb_rcmeas/measdone: %d\n", measdone);
    
 if (measdone == 1)
    {
        countvalue1=abb_api_read_rcmeas();
    }
    else
    {
       return -1;
    }

 // read measdone and 2nd countvalue
 measdone = 0;
 i = 0;
   
 abb_api_rcmeas_default();

 abb_api_rcmeas_start(ABB_POL_HI,dsmsel);
 
 while ((!measdone) && (i++ < 15))
    {
        measdone=abb_api_rcmeas_finished();
    }
    
 if (measdone == 1)
    {
        countvalue2=abb_api_read_rcmeas();
    }
    else
    {
       return -1;
    }

 count_calc = (countvalue1 + countvalue2)/2;
	
 abb_api_rcmeas_default();
	
 rccount[0] = count_calc;
 rccount[1] = countvalue1;
 rccount[2] = countvalue2;
	
 return (int16_t) count_calc;  

}


int8_t abb_adctune(int16_t count_calc, int8_t* rctune)
{
  int8_t rctune_calc = 0;
  
  if (count_calc != 0)
  {
     // CORNER  FF  TT  SS
     // RCMEAS 304 468 662

     // rctune = round(A2 x count_calc^2 + A1 x count_calc + A0)
     //          A2 = 0.0000080772
     //          A1 = 0.091601
     //          A0 = -27.10037  
     // 29.5.2018
          
     // \Wave600_MS\01_A11\04_Design\00_SDADC80\RCtune_formula\Wave600ABB11ax_RCtune.pptx
     // Update 25.5.2018
     // Selected Formula: RC=func[CNT-38]
     
          if(count_calc-38 < 310) { rctune_calc =  0; }
     else if(count_calc-38 < 322) { rctune_calc =  1; }
     else if(count_calc-38 < 333) { rctune_calc =  2; }    
     else if(count_calc-38 < 345) { rctune_calc =  3; }    
     else if(count_calc-38 < 357) { rctune_calc =  4; }    
     else if(count_calc-38 < 368) { rctune_calc =  5; }    
     else if(count_calc-38 < 380) { rctune_calc =  6; }
     else if(count_calc-38 < 392) { rctune_calc =  7; }    
     else if(count_calc-38 < 403) { rctune_calc =  8; }    
     else if(count_calc-38 < 415) { rctune_calc =  9; }    
     else if(count_calc-38 < 427) { rctune_calc = 10; }        
     else if(count_calc-38 < 439) { rctune_calc = 11; }
     else if(count_calc-38 < 451) { rctune_calc = 12; }    
     else if(count_calc-38 < 463) { rctune_calc = 13; }    
     else if(count_calc-38 < 474) { rctune_calc = 14; }    
     else if(count_calc-38 < 486) { rctune_calc = 15; }        
     else if(count_calc-38 < 498) { rctune_calc = 16; }        
     else if(count_calc-38 < 510) { rctune_calc = 17; }
     else if(count_calc-38 < 522) { rctune_calc = 18; }    
     else if(count_calc-38 < 534) { rctune_calc = 19; }    
     else if(count_calc-38 < 546) { rctune_calc = 20; }    
     else if(count_calc-38 < 559) { rctune_calc = 21; }     
     else if(count_calc-38 < 571) { rctune_calc = 22; }        
     else if(count_calc-38 < 583) { rctune_calc = 23; }    
     else if(count_calc-38 < 595) { rctune_calc = 24; }    
     else if(count_calc-38 < 607) { rctune_calc = 25; }        
     else if(count_calc-38 < 619) { rctune_calc = 26; }        
     else if(count_calc-38 < 632) { rctune_calc = 27; }
     else if(count_calc-38 < 644) { rctune_calc = 28; }    
     else if(count_calc-38 < 656) { rctune_calc = 29; }    
     else if(count_calc-38 < 669) { rctune_calc = 30; }    
     else                      { rctune_calc = 31; }
     
     *rctune = rctune_calc;
      
     return 0;
  }
  else
  {
     return -1;
  }
}


int8_t abb_rx_rctune(uint8_t *fuse)
{
 // reg1, bit 7
 int trim_indicator = fuse[1] >> 7 & 0x01;
    
 // reg12, bit 7:4 LSBs = rccount[3:0]
 // reg13, bit 5:0 MSBs = rccount[9:4]
 int rccount = ((fuse[13] & 0x3F) << 4 ) + ( (fuse[12] >> 4) & 0xF );
  
 //initial version:
 //rccount=445;  
 
 //debug output
 //debug_printf("fuse[1]         = %d;\n",fuse[1]);
 //debug_printf("fuse[12]        = %d;\n",fuse[12]);
 //debug_printf("fuse[13]        = %d;\n",fuse[13]);
    
 //debug_printf("trim_indicator = %d;\n",trim_indicator);
      
 //debug_printf("rccount fused   = %d;\n",rccount);    
 
  
 int8_t rctune=0;
 int8_t ret_val_adctune=0;


 if (trim_indicator == 0)
 { 
 	int16_t rccountp[3];

	//rccount = 445 ; // initial version: do not call rcmeas 471=>14 445=>12
	rccount=abb_rcmeas(rccountp,ABB_DSM_OFF); //ABB_DSM_ON: provoke error!
	
	//debug_printf("abb_rx_rctune/rccount:  %d\n",rccount);
	
	if (rccount == -1)
	{
		return -1;
	}
 }

 //debug_printf("rccount used   = %d;\n",rccount);    
 
 ret_val_adctune = abb_adctune(rccount, &rctune);
	
 if (ret_val_adctune == -1)
	 {
	    return -1;
	 }
	 else
	{
	    return (int8_t) rctune;	 
	}

}


void abb_antenna_init(uint16_t _ant_mask)
{
 // pup antenna ldo
 // pup antenna biasing
 // trim biasing (not implemented)
 // switch on clk inside antennas
 // trim clk biasing (not implemented)
  
 //FCSI access ANT / enable antenna LDOA
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ILDOA0, 
		( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) | ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
 
 //FCSI access ANT / enable antenna LDOD
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ILDOD0, 
		( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) | ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );
 
 //FCSI access ANT / enable antenna local bias; daclbtrim=00
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ILOCBIAS0, 
		ANT_FCSI_ILOCBIAS0_ENLB_ON__SVAL);
  
 //FCSI access ANT / activate clock distribution and release LDO domain FCSI reset
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ICLKDIST0, 
		ANT_FCSI_ICLKDIST0_FCSIANTFD_LO__SVAL   	|
		ANT_FCSI_ICLKDIST0_FCSIANTRSTQ_HI__SVAL		|
		ANT_FCSI_ICLKDIST0_ITRIMLBCLKD_10__SVAL	 	|
		ANT_FCSI_ICLKDIST0_ENLBCLKD_ON__SVAL 		|
		ANT_FCSI_ICLKDIST0_ENRXCLK_ON__SVAL 		|
		ANT_FCSI_ICLKDIST0_ENRXSTB_ON__SVAL 		|
		ANT_FCSI_ICLKDIST0_ENSSICLK_ON__SVAL 		|
		ANT_FCSI_ICLKDIST0_SELDIVSSI_CLK640M__SVAL 	|
		ANT_FCSI_ICLKDIST0_ENTXCLK_ON__SVAL 		|
		ANT_FCSI_ICLKDIST0_ENTXSTB_ON__SVAL );
  
}


void abb_antenna_fusing(uint16_t _ant_mask, uint8_t *fuse)
{

 //WAV600 Fusetable:
 //https://sharepoint.gar.ith.intel.com/sites/WAVE600C/Shared%20Documents/Chip%20Development/Micro-architecture%20and%20Design/Top/WAV600%20Efuse%20Mapping.xlsx

    // reg1, bit 7
    int trim_indicator = fuse[1] >> 7 & 0x01;
    
    // reg8, bit 7:4
    int ldoa_ant3 = (fuse[8] >> 4) & 0xF;
    
    // reg8, bit 3:0
    int ldoa_ant2 = fuse[8] & 0xF;
    
     // reg9, bit 7:4
    int ldoa_ant1 = (fuse[9] >> 4) & 0xF;
    
    // reg9, bit 3:0
    int ldoa_ant0 = fuse[9] & 0xF;  
    
    // reg10, bit 7:4
    int ldod_ant3 = (fuse[10] >> 4) & 0xF;
    
    // reg10, bit 3:0
    int ldod_ant2 = fuse[10] & 0xF;
    
    // reg11, bit 7:4
    int ldod_ant1 = (fuse[11] >> 4) & 0xF;
    
    // reg11, bit 3:0
    int ldod_ant0 = fuse[11] & 0xF;      
    
    /*
    //debug output
    debug_printf("fuse[1]        = %d;\n",fuse[1]);
    debug_printf("fuse[8]        = %d;\n",fuse[8]);
    debug_printf("fuse[9]        = %d;\n",fuse[9]);
    debug_printf("fuse[10]       = %d;\n",fuse[10]);
    debug_printf("fuse[11]       = %d;\n",fuse[11]);

    debug_printf("trim_indicator = %d;\n",trim_indicator);   

    debug_printf("ldoa_ant3    = %d;\n",ldoa_ant3);
    debug_printf("ldoa_ant2    = %d;\n",ldoa_ant2);
    debug_printf("ldoa_ant1    = %d;\n",ldoa_ant1);    
    debug_printf("ldoa_ant0    = %d;\n",ldoa_ant0);    
    
    debug_printf("ldod_ant3    = %d;\n",ldod_ant3);
    debug_printf("ldod_ant2    = %d;\n",ldod_ant2);
    debug_printf("ldod_ant1    = %d;\n",ldod_ant1);    
    debug_printf("ldod_ant0    = %d;\n",ldod_ant0);    
    */

 if (trim_indicator == 1)
    {
     	//set fused values and enable ldo
	
	if (_ant_mask & ABB_ANTENNA0)
	{
	ABB_FCSI_WRITE(ABB_ANTENNA0, 
		ANT_FCSI_ILDOA0, 
			( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_OUTTRIM__ISMSK & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) | 
			( ldoa_ant0 & ANT_FCSI_ILDOA0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			
	ABB_FCSI_WRITE(ABB_ANTENNA0, 
		ANT_FCSI_ILDOD0, 
			( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_OUTTRIM__ISMSK & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) | 
			( ldod_ant0 & ANT_FCSI_ILDOD0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );
	}		

	if (_ant_mask & ABB_ANTENNA1)
	{
	ABB_FCSI_WRITE(ABB_ANTENNA1, 
		ANT_FCSI_ILDOA0, 
			( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_OUTTRIM__ISMSK & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) | 
			( ldoa_ant1 & ANT_FCSI_ILDOA0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			
	ABB_FCSI_WRITE(ABB_ANTENNA1, 
		ANT_FCSI_ILDOD0, 
			( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_OUTTRIM__ISMSK & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) | 
			( ldod_ant1 & ANT_FCSI_ILDOD0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );
	}	

	if (_ant_mask & ABB_ANTENNA2)
	{
	ABB_FCSI_WRITE(ABB_ANTENNA2, 
		ANT_FCSI_ILDOA0, 
			( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_OUTTRIM__ISMSK & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) | 
			( ldoa_ant2 & ANT_FCSI_ILDOA0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			
	ABB_FCSI_WRITE(ABB_ANTENNA2, 
		ANT_FCSI_ILDOD0, 
			( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_OUTTRIM__ISMSK & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) | 
			( ldod_ant2 & ANT_FCSI_ILDOD0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );
	}		

	if (_ant_mask & ABB_ANTENNA3)
	{
	ABB_FCSI_WRITE(ABB_ANTENNA3, 
		ANT_FCSI_ILDOA0, 
			( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_OUTTRIM__ISMSK & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) | 
			( ldoa_ant3 & ANT_FCSI_ILDOA0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			
	ABB_FCSI_WRITE(ABB_ANTENNA3, 
		ANT_FCSI_ILDOD0, 
			( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_OUTTRIM__ISMSK & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) | 
			( ldod_ant3 & ANT_FCSI_ILDOD0_OUTTRIM__MSK) |
			ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );
	}	

	
    }
    else
    {
    
    	//set default values and enable ldo
    
  	if (_ant_mask & ABB_ANTENNA0)
	{  
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOA0, 
			 ( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			 
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOD0, 
			 ( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );		 
			 
	}
	
  	if (_ant_mask & ABB_ANTENNA1)
	{  
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOA0, 
			 ( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			 
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOD0, 
			 ( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );		 
			 
	}

  	if (_ant_mask & ABB_ANTENNA2)
	{  
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOA0, 
			 ( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			 
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOD0, 
			 ( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );		 
			 
	}
	
  	if (_ant_mask & ABB_ANTENNA3)
	{  
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOA0, 
			 ( ANT_FCSI_ILDOA0__INIT & ANT_FCSI_ILDOA0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOA0_ENLDO_ENABLE__SVAL );
			 
	ABB_FCSI_WRITE(_ant_mask, 
		ANT_FCSI_ILDOD0, 
			 ( ANT_FCSI_ILDOD0__INIT & ANT_FCSI_ILDOD0_ENLDO__ISMSK ) |
			 ANT_FCSI_ILDOD0_ENLDO_ENABLE__SVAL );		 
			 
	}
	
    }
}


int8_t abb_antenna_rx_calibration(uint16_t _ant_mask)
{
 // productive version: QTRIM mode = 2

 //enable ADC and force reset
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL |
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 	| // ->11ax
		ANT_FCSI_IADCD0_RXONMANVAL_3__SVAL 	| // en=1; rst=0
		ANT_FCSI_IADCD0_RXONMANSEL_MANUAL__SVAL | // rxonmansel=MANUAL
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 	|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL		|
		ANT_FCSI_IADCD0_ENFRCRST_ENABLE__SVAL );  // en_forcereset=1
 
 //set loopback mode
  ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IADCA0, 
		ANT_FCSI_IADCA0_R1_2__SVAL               | // R1=10=LOOPBACK
		ANT_FCSI_IADCA0_ELDRES_9__SVAL		 | // ->11ax
		ANT_FCSI_IADCA0_W40FILTEN_OFF__SVAL	 | // ->11ax
		ANT_FCSI_IADCA0_W320_OFF__SVAL		 |
		ANT_FCSI_IADCA0_OPA3_3__SVAL		 | // ->11ax
		ANT_FCSI_IADCA0_OPA2_2__SVAL		 | // ->11ax
		ANT_FCSI_IADCA0_OPA1_1__SVAL		); // 1/2/3 changed for LA1p33
  
 //set ELDCAP=0  
  ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA1,
		ANT_FCSI_IADCA1_ELDCAP__PUT(ANT_FCSI_IADCA1_ELDCAP_0__VAL , ANT_FCSI_IADCA1__INIT) 
	); // update 23.11.2018  
  
 //enable cal start 
 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IADCA2, 
		ANT_FCSI_IADCA2_CALDACMODE_2__SVAL | // changed for LA1p33
		ANT_FCSI_IADCA2_SELRSTCTL_1__SVAL  |
		ANT_FCSI_IADCA2_QTRIM1EN_0__SVAL   |
		ANT_FCSI_IADCA2_QTRIM2EN_1__SVAL   | //qtrim2=1 (diode trimming)
		ANT_FCSI_IADCA2_R1VAL__INIT  << ANT_FCSI_IADCA2_R1VAL__POS	|
		ANT_FCSI_IADCA2_RCTUNE__INIT << ANT_FCSI_IADCA2_RCTUNE__POS	);

 //at 128MHz ca. 212ns per each FCSI access
 //qt duration ca. 200us
 //limit waiting time to max. 1ms
 
 int caldone=0;
 int i=0;
   
 while ((!caldone) && (i++ < 4702)) 
    {
        caldone=abb_api_read_rx_qtcaldone(_ant_mask);
	//debug_printf("trying qtcaldone=%d\n",caldone);
    }
  
 //int caldone=1; //=0: provoke error for simulation purpose

 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IADCA0, 
		ANT_FCSI_IADCA0__INIT );
  
 //release ADC reset and rx on manual selection
   ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0, ANT_FCSI_IADCD0__INIT );
		
 if (caldone)
 {
  return 2;
  // 0 = no calibration active
  // 1 = qtrim1
  // 2 = qtrim2 => DEFAULT
 }
 else
 {
  return -1;
  // ERROR
 }
 
}


void abb_antenna_config(uint16_t _ant_mask, uint8_t abb_mode, int8_t rctune, int8_t qtrim)
{
 // configuration of SDADC/DAC/SSI   
 
 if (abb_mode == ABB_11AX) // 11ax mode
  { 
  abb_api_start_tx(_ant_mask);
  abb_api_start_ssi(_ant_mask);
  abb_api_start_rx_11ax(_ant_mask, rctune, qtrim);
  }
 else
 if (abb_mode == ABB_LOPOWER) // lp mode
  { 
  abb_api_start_tx_lp(_ant_mask);
  abb_api_start_ssi(_ant_mask); 
  abb_api_start_rx_lp(_ant_mask, rctune, qtrim);
  }
 else
 if (abb_mode == ABB_SSB) // sdadc ssb mode
  {
  abb_api_start_tx_lp(_ant_mask);
  abb_api_start_ssi(_ant_mask);
  abb_api_start_rx_ssb(_ant_mask, rctune, qtrim);
  }  
 else
 if (abb_mode == ABB_STANDBY ) // switch off adc/dac/ssi
  {
  abb_api_standby_tx(_ant_mask);
  abb_api_standby_ssi(_ant_mask);
  abb_api_standby_rx(_ant_mask, rctune, qtrim);
  } 
  
}


void abb_antenna_rxoff_11ax(uint16_t _ant_mask)
{

 // Register content after init
 //      INIT
 // D0 = 0000
 // A4 = 8000
 // A1 = D9F3


 // Register content before RXOFF
 //      11ax   (lp    ssb )
 // D0 = 0A00   (0A08  0A00)
 // A4 = 8rrr   (8rrr  8rrr) (r=read register signal)
 // A1 = D9F3   (D9BF  D9F3)



 // IADCD0 -> 8A08 (=INIT+phcen+rxen + enfrcrst=1 & w40clk=1)

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_640M__SVAL 		| //=1
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| //=1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| //=1
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_ENABLE__SVAL ); 	  //=1
			
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCD0,0x8A08);



 //IADCA4 -> 0rrr/8rrr/0rrr/8rrr/0rrr/8rrr/0rrr/8rrr (=INIT + clkpolarity)

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_0__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x0000);	
	
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_1__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x8000);	

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_0__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x0000);	
	
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_1__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x8000);	
		
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_0__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x0000);	
	
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_1__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x8000);	

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_0__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x0000);	
	
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4_CALCLKPOLARITY__PUT(ANT_FCSI_IADCA4_CALCLKPOLARITY_1__VAL ,ANT_FCSI_IADCA4__INIT ) 
	);
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA4,0x8000);	

 
 
 // IADCA1 = C9F3/D9F3/C9F3/D9F3 (=INIT+clkinv)
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_CLKINV__PUT(ANT_FCSI_IADCA1_CLKINV_NOTINV__VAL, ANT_FCSI_IADCA1__INIT)
		 ); 
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA1,0xC9F3);	
	
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_CLKINV__PUT(ANT_FCSI_IADCA1_CLKINV_INV__VAL, ANT_FCSI_IADCA1__INIT)
		 ); 
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA1,0xD9F3);		
	
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_CLKINV__PUT(ANT_FCSI_IADCA1_CLKINV_NOTINV__VAL, ANT_FCSI_IADCA1__INIT)
		 ); 
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA1,0xC9F3);	
	
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_CLKINV__PUT(ANT_FCSI_IADCA1_CLKINV_INV__VAL, ANT_FCSI_IADCA1__INIT)
		 ); 
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCA1,0xD9F3);		
	
	
 //IADCD0 -> 0A08 enfrcreset=0 (=INIT+phcen+rxen - enfrcreset)

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_640M__SVAL 		| //=1
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| //=1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| //=1
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 	  //=0!

 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCD0,0x0A08);
 
 
 //IADCD0 -> 0A00 w40clk=0 (=INIT+phcen+rxen-w40clk)

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		| //=0!
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| //=1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| //=1
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 	  //=0!

			
 //ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCD0,0x0A00);


 
 // Register content after RXOFF
 //      11ax   (lp    ssb )
 // D0 = 0A00   (0A08  0A00)
 // A4 = 8rrr   (8rrr  8rrr) (r=read register signal)
 // A1 = D9F3   (D9BF  D9F3)

 			
}


void abb_antenna_rxoff2_11ax(uint16_t _ant_mask)
{
			
 ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCD0,0x8A00);
 ABB_FCSI_WRITE(_ant_mask,ANT_FCSI_IADCD0,0x0A00);	
 			
}


void abb_antenna_pd(uint16_t _ant_mask)
{

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ILDOA0, 
 		ANT_FCSI_ILDOA0__INIT );

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ILDOD0, 
		ANT_FCSI_ILDOD0__INIT );
 

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ILOCBIAS0, 
		ANT_FCSI_ILOCBIAS0__INIT );

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ICLKDIST0, 
		ANT_FCSI_ICLKDIST0__INIT );
 
}



// ---------------------------------------------------------
// Section: LL API Functions
// ---------------------------------------------------------


void abb_api_fcsiwrite_softisolation_central()
{

 // dummy "non destructive" central write accesses
 // PLL up and running
 // central clk distribution up and running
 
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x40, 0x0081); //0x0080 (def)
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x41, 0x0002); //0x0000 (def)
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x42, 0x001f); //0x0001 (def)
	//                          0x43 rcmeas read only
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x44, 0x2615); //0x2415 (def)
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x45, 0x0009); //0x0008 (def)
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x46, 0x0204); //0x0004 (def)
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x47, 0x0009); //0x0008 (def)
	//                          0x48 not used
	//			    ...
	//			    0x4F not used
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x50, 0x7002); //0x7000 (def)
	//                          0x51 PLLW1 do not touch => 0x2000
	//                          0x52 PLLW2 do not touch => 0x0085
	//                          0x53 PLLW3 do not touch => 0x0000
	//                          0x54 PLLW4 do not touch => 0x0C38
	ABB_FCSI_WRITE(ABB_CENTRAL, 0x55, 0x0002); //0x0000 (def)
	//                          0x56 PLLW6 do not touch => 0x1C36
	//                          0x57 PLLW7 do not touch => 0x04C0
	//                          0x58 PLLR0 read only	
	//                          0x59 PLLR1 read only	
	//                          0x5A PLLR2 read only	
	//                          0x5B PLLR3 read only
	//                          0x5C LDOPLLLREF do not touch => 0x0005 (unfused!)
	//                          0x5D LDOPLLPOST do not touch => 0x0005 (unfused!)
	//                          0x5E LDOREV1V8H do not touch => 0x0004 (unfused!)
	//                          0x5F cversion read only      => 0x0002	

}

int16_t abb_api_fcsiread_softisolation_central()
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 0x40, data); if (data[ABB_CENTRAL_INDEX] != 0x0081) { return - 0x40 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x41, data); if (data[ABB_CENTRAL_INDEX] != 0x0002) { return - 0x41 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x42, data); if (data[ABB_CENTRAL_INDEX] != 0x001f) { return - 0x42 ; } 
 //                         0x43 rcmeas read only
 ABB_FCSI_READ(ABB_CENTRAL, 0x44, data); if (data[ABB_CENTRAL_INDEX] != 0x2615) { return - 0x44 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x45, data); if (data[ABB_CENTRAL_INDEX] != 0x0009) { return - 0x45 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x46, data); if (data[ABB_CENTRAL_INDEX] != 0x0204) { return - 0x46 ; } 
 ABB_FCSI_READ(ABB_CENTRAL, 0x47, data); if (data[ABB_CENTRAL_INDEX] != 0x0009) { return - 0x47 ; } 
 //                         0x48 not used
 //			    ...
 //			    0x4F not used 
 ABB_FCSI_READ(ABB_CENTRAL, 0x50, data); if (data[ABB_CENTRAL_INDEX] != 0x7002) { return - 0x50 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x51, data); if (data[ABB_CENTRAL_INDEX] != 0x2000) { return - 0x51 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x52, data); if (data[ABB_CENTRAL_INDEX] != 0x0085) { return - 0x52 ; } 
 ABB_FCSI_READ(ABB_CENTRAL, 0x53, data); if (data[ABB_CENTRAL_INDEX] != 0x0000) { return - 0x53 ; } 
 ABB_FCSI_READ(ABB_CENTRAL, 0x54, data); if (data[ABB_CENTRAL_INDEX] != 0x0C38) { return - 0x54 ; }  
 ABB_FCSI_READ(ABB_CENTRAL, 0x55, data); if (data[ABB_CENTRAL_INDEX] != 0x0002) { return - 0x55 ; }   
 ABB_FCSI_READ(ABB_CENTRAL, 0x56, data); if (data[ABB_CENTRAL_INDEX] != 0x1C36) { return - 0x56 ; }
 ABB_FCSI_READ(ABB_CENTRAL, 0x57, data); if (data[ABB_CENTRAL_INDEX] != 0x04C0) { return - 0x57 ; }
 //                         0x58 PLLR0 read only	
 //                         0x59 PLLR1 read only	
 //                         0x5A PLLR2 read only	
 //                         0x5B PLLR3 read onl 
 ABB_FCSI_READ(ABB_CENTRAL, 0x5C, data); if (data[ABB_CENTRAL_INDEX] != 0x0005) { return - 0x5C ; } 
 ABB_FCSI_READ(ABB_CENTRAL, 0x5D, data); if (data[ABB_CENTRAL_INDEX] != 0x0005) { return - 0x5D ; } 
 ABB_FCSI_READ(ABB_CENTRAL, 0x5E, data); if (data[ABB_CENTRAL_INDEX] != 0x0004) { return - 0x5E ; }  
 ABB_FCSI_READ(ABB_CENTRAL, 0x5F, data); if (data[ABB_CENTRAL_INDEX] != 0x0002) { return - 0x5F ; }  
	 
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL 

 return 0 ;
}



void abb_api_fcsiwrite_softisolation_antenna(uint16_t _ant_mask)
{

 // dummy "non destructive" antenna write accesses
 // antenna initialized with "abb_antenna_init" 
 // i.e. antenna LDOs up and running
 // TX/RX/SSI FCSI resets released
 // expected (=>) values "untrimmed"
 
 //                        0x40 LDOA0 do not touch    => 0x013C (unfused!)
 //                        0x41 LDOD0 do not touch    => 0x013C (unfused!)
 //                        0x42 LOCBIAS0 do not touch => 0x0001
 ABB_FCSI_WRITE(_ant_mask, 0x43, 0x0002); //0x0000 (def) 
 //                        0x44 CLKDIST0 do not touch => 0x1abf
 //                        0x45 aversion read only    => 0x2123
 ABB_FCSI_WRITE(_ant_mask, 0x46, 0x06E0); //0x06A0 (def)  
 ABB_FCSI_WRITE(_ant_mask, 0x47, 0x0020); //0x0000 (def)  
 ABB_FCSI_WRITE(_ant_mask, 0x48, 0xE4B2); //0xE4B1 (def)  
 ABB_FCSI_WRITE(_ant_mask, 0x49, 0xE1F2); //0xE1F3 (def)  
 ABB_FCSI_WRITE(_ant_mask, 0x4A, 0x23AE); //0x238E (def)  
 ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0040); //0x0000 (def)   
 ABB_FCSI_WRITE(_ant_mask, 0x4C, 0x5001); //0x5000 (def) 
 //                        0x4D QT read only   
 //                        0x4E not used
 //			   0x4F not used 
 ABB_FCSI_WRITE(_ant_mask, 0x50, 0x0000); //0x0200 (def)    
 ABB_FCSI_WRITE(_ant_mask, 0x51, 0x27F1); //0x67F1 (def)
 ABB_FCSI_WRITE(_ant_mask, 0x52, 0x0017); //0x0013 (def) 
 ABB_FCSI_WRITE(_ant_mask, 0x53, 0x0020); //0x0000 (def)  

}


uint16_t abb_api_fcsiread_softisolation_antenna(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 uint16_t antindex=0;
  
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL


switch(_ant_mask) { 
  	case ABB_ANTENNA0: antindex= ABB_ANT0_INDEX; break;	
 	case ABB_ANTENNA1: antindex= ABB_ANT1_INDEX; break;
 	case ABB_ANTENNA2: antindex= ABB_ANT2_INDEX; break;
 	case ABB_ANTENNA3: antindex= ABB_ANT3_INDEX; break;
	default: return -1; // return value only if single antenna is specified
	}
	
 ABB_FCSI_READ(_ant_mask, 0x40, data); if (data[antindex] != 0x013C) { return - 0x40 ; }
 ABB_FCSI_READ(_ant_mask, 0x41, data); if (data[antindex] != 0x013C) { return - 0x41 ; }
 ABB_FCSI_READ(_ant_mask, 0x42, data); if (data[antindex] != 0x0001) { return - 0x42 ; } 
 ABB_FCSI_READ(_ant_mask, 0x43, data); if (data[antindex] != 0x0002) { return - 0x43 ; } 
 ABB_FCSI_READ(_ant_mask, 0x44, data); if (data[antindex] != 0x1abf) { return - 0x44 ; }
 ABB_FCSI_READ(_ant_mask, 0x45, data); if (data[antindex] != 0x2123) { return - 0x45 ; }
 ABB_FCSI_READ(_ant_mask, 0x46, data); if (data[antindex] != 0x06E0) { return - 0x46 ; } 
 ABB_FCSI_READ(_ant_mask, 0x47, data); if (data[antindex] != 0x0020) { return - 0x47 ; } 
 ABB_FCSI_READ(_ant_mask, 0x48, data); if (data[antindex] != 0xE4B2) { return - 0x48 ; }
 ABB_FCSI_READ(_ant_mask, 0x49, data); if (data[antindex] != 0xE1F2) { return - 0x49 ; }
 ABB_FCSI_READ(_ant_mask, 0x4A, data); if (data[antindex] != 0x23AE) { return - 0x4A ; } 
 ABB_FCSI_READ(_ant_mask, 0x4B, data); if (data[antindex] != 0x0040) { return - 0x4B ; } 
 ABB_FCSI_READ(_ant_mask, 0x4C, data); if (data[antindex] != 0x5001) { return - 0x4C ; }  
 //                         0x4D QT read only   
 //                         0x4E not used
 //			    0x4F not used 
 ABB_FCSI_READ(_ant_mask, 0x50, data); if (data[antindex] != 0x0000) { return - 0x50 ; }
 ABB_FCSI_READ(_ant_mask, 0x51, data); if (data[antindex] != 0x27F1) { return - 0x51 ; }
 ABB_FCSI_READ(_ant_mask, 0x52, data); if (data[antindex] != 0x0017) { return - 0x52 ; } 
 ABB_FCSI_READ(_ant_mask, 0x53, data); if (data[antindex] != 0x0020) { return - 0x53 ; } 
	
 return 0;
	
}


void abb_api_central_trim_ldo0v95fsys(uint8_t trim)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_IVREG0V950, 
 	CENTRAL_FCSI_IVREG0V950_OUTTRIM__ISMSK, 
		trim & CENTRAL_FCSI_IVREG0V950_OUTTRIM__MSK);
  
}


void abb_api_init_central_pll_bypass()
{
 // TESTSOC legacy
 // no central fcsi access neded

 // reset values
 ABB_DIR_CTRL_WRITE(lcpll_pup_d2a,0x0);
 ABB_DIR_CTRL_WRITE(lcpll_resn_d2a,0x0);  
 ABB_DIR_CTRL_WRITE(pd_clk_sc_d2a,0x1);
 ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x0);
 ABB_DIR_CTRL_WRITE(lcpll_div_d2a,0x0);

 // lcpll bypass powerup
 ABB_DIR_CTRL_WRITES(spare_config_i_d2a,8,0x10); //MUX: PLL bypass active
 ABB_DIR_CTRL_WRITE(lcpll_resn_d2a,0x1); 
 
}


void abb_api_init_central_pll_enable()
{
 abb_central_init_pll();
}


void abb_api_init_central_pll_enable_fcsicontrol()
{ 
 
 ABB_FCSI_WRITE(ABB_CENTRAL, 0x56, 0x0C39); // pll: cal_enable=1; resn=0; en=0; en_fcsicontrol=1
 ABB_DIR_CTRL_WRITE(lcpll_resn_d2a,0x0);           // resn for clockdist

 ABB_DIR_CTRL_WRITE(pd_clk_sc_d2a,0x1);
 ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x0);
 ABB_DIR_CTRL_WRITE(lcpll_div_d2a,0x0);
 
 ABB_DIR_CTRL_WRITES(spare_config_i_d2a,8,0x00);     //MUX: PLL active; testsoc only!
 //ABB_FCSI_WRITE(ABB_CENTRAL, 0x56, 0x1C3B); // pll:  ckdsmen=1  en=1 was fixed for abb11ax!

 ABB_FCSI_WRITE(ABB_CENTRAL, 0x5C, 0x0005); //LDOREF=0.95V
 ABB_FCSI_WRITE(ABB_CENTRAL, 0x5D, 0x0005); //LDOPOST=0.95V


 ABB_DIR_CTRL_WRITE(lcpll_resn_d2a,0x1);    // disable resn for clockdist
 ABB_FCSI_WRITE(ABB_CENTRAL, 0x56, 0x1C37); // pll: cal_enable=0; resn=1;
 //_wait(1,T_USEC);
  
 // PLL lock time ~1.8ms 

}


void abb_api_central_trim_ldo0v95ref(uint8_t trim)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_ILDOPLLREF0, 
 	CENTRAL_FCSI_ILDOPLLREF0_OUTTRIM__ISMSK, 
		trim & CENTRAL_FCSI_ILDOPLLREF0_OUTTRIM__MSK);
}


void abb_api_central_trim_ldo0v95post(uint8_t trim)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_ILDOPLLPOST0, 
 	CENTRAL_FCSI_ILDOPLLPOST0_OUTTRIM__ISMSK, 
		trim & CENTRAL_FCSI_ILDOPLLPOST0_OUTTRIM__MSK);
}


void abb_api_central_trim_vreg1v8hv(uint8_t trim)
{ 
 ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_IVREG1V8HV0, 
 	trim & CENTRAL_FCSI_IVREG1V8HV0_OUTTRIM__MSK);
}


void abb_api_central_trim_bias(uint8_t vref, uint8_t iptat, uint8_t ibias)
{
 ABB_FCSI_WRITE(ABB_CENTRAL, CENTRAL_FCSI_ICENTBIAS0, 
 	(vref  & CENTRAL_FCSI_ICENTBIAS0_VREFTRIM__MSK) << CENTRAL_FCSI_ICENTBIAS0_VREFTRIM__POS | 
 	(iptat & CENTRAL_FCSI_ICENTBIAS0_PTATTRIM__MSK) << CENTRAL_FCSI_ICENTBIAS0_PTATTRIM__POS |
	(ibias & CENTRAL_FCSI_ICENTBIAS0_BIASTRIM__MSK) << CENTRAL_FCSI_ICENTBIAS0_BIASTRIM__POS );
}


void abb_api_init_central_pup_clkd()
{
 abb_central_init_clkd();
}


void abb_api_init_central_pd_clkd()
{
 ABB_DIR_CTRL_WRITE(pd_clk_sc_d2a,0x1);
}


void abb_api_central_trim_ldo0v95clkd(uint8_t trim)
{ 
  ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_ILDOCLKD0, 
  	CENTRAL_FCSI_ILDOCLKD0_OUTTRIM__ISMSK, 
		trim & CENTRAL_FCSI_ILDOCLKD0_OUTTRIM__MSK);
}


uint16_t abb_api_read_central_version()
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICVERSION0,
		 data);
	 
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL 

 return data[4] & CENTRAL_FCSI_ICVERSION0_CTRLVER__MSK ;
}


uint16_t abb_api_read_lcpll_lock()
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 
 	CENTRAL_FCSI_IPLLR3,
		data);
		
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

 return CENTRAL_FCSI_IPLLR3_LOCK__GET(data[4] & CENTRAL_FCSI_IPLLR3_LOCK__SMSK ) ;

 //return data[4] & 1;

}


uint16_t abb_api_read_lcpll_prelock(void)
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 
 	CENTRAL_FCSI_IPLLR3,
		data); 
 
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

 return CENTRAL_FCSI_IPLLR3_PRELOCK__GET(data[4] & CENTRAL_FCSI_IPLLR3_PRELOCK__SMSK );
 
 //data[4] & 16;

}

void abb_api_rcmeas_start(uint8_t polarity, uint8_t dsmsel)
{
 // start from default configuration

 //debug_printf("abb_api_rcmeas_start/dsmsel: %d\n",dsmsel);
 
 //enable clkrcmeas 640M  
 ABB_FCSI_WRITE(ABB_CENTRAL, 
  	CENTRAL_FCSI_ICLKDIST0, 
		(CENTRAL_FCSI_ICLKDIST0__INIT & CENTRAL_FCSI_ICLKDIST0_ENCLKRCMEAS__ISMSK) | //CTRLSYNCDL=100=default 
		CENTRAL_FCSI_ICLKDIST0_ENCLKRCMEAS_1__SVAL ); 
 

 // RESET=0 (=> counter=0)
 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS0,
		(dsmsel   & CENTRAL_FCSI_IRCMEAS0_DSMSEL__MSK) << CENTRAL_FCSI_IRCMEAS0_DSMSEL__POS |
		(polarity & CENTRAL_FCSI_IRCMEAS0_POL__MSK   ) <<  CENTRAL_FCSI_IRCMEAS0_POL__POS   |
		CENTRAL_FCSI_IRCMEAS0_CMPEN_OFF__SVAL	    |
		CENTRAL_FCSI_IRCMEAS0_START_0__SVAL         |
		CENTRAL_FCSI_IRCMEAS0_RESET_DISABLE__SVAL    );

 // COMP_ENABLE=1
  ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS0,
		(dsmsel   & CENTRAL_FCSI_IRCMEAS0_DSMSEL__MSK) << CENTRAL_FCSI_IRCMEAS0_DSMSEL__POS |
		(polarity & CENTRAL_FCSI_IRCMEAS0_POL__MSK   ) <<  CENTRAL_FCSI_IRCMEAS0_POL__POS   |
		CENTRAL_FCSI_IRCMEAS0_CMPEN_ON__SVAL	    |
		CENTRAL_FCSI_IRCMEAS0_START_0__SVAL         |
		CENTRAL_FCSI_IRCMEAS0_RESET_DISABLE__SVAL    );

// START=1
// wait >200n (= 1 fcsi access) 
  ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS0,
		(dsmsel   & CENTRAL_FCSI_IRCMEAS0_DSMSEL__MSK) << CENTRAL_FCSI_IRCMEAS0_DSMSEL__POS |
		(polarity & CENTRAL_FCSI_IRCMEAS0_POL__MSK   ) <<  CENTRAL_FCSI_IRCMEAS0_POL__POS   |
		CENTRAL_FCSI_IRCMEAS0_CMPEN_ON__SVAL	    |
		CENTRAL_FCSI_IRCMEAS0_START_1__SVAL         |
		CENTRAL_FCSI_IRCMEAS0_RESET_DISABLE__SVAL    );

// during operation: running=1; finished=0

}

void abb_api_rcmeas_default(void)
{

 //disable clkrcmeas 640M
 ABB_FCSI_WRITE(ABB_CENTRAL, 
  	CENTRAL_FCSI_ICLKDIST0, 
		CENTRAL_FCSI_ICLKDIST0__INIT ); //CTRLSYNCDL=100=default 
 
 // establish default configuration:
 // RESET=1
 // START=0
 // COMP_ENABLE=0
 // POL=0
 // DSMSEL=0

 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS0, 
		CENTRAL_FCSI_IRCMEAS0_RESET__INIT);

}


uint16_t abb_api_rcmeas_running(void)
{
 
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS1, 
		data);
	
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

 return CENTRAL_FCSI_IRCMEAS1_RUNNING__GET(data[ABB_CENTRAL_INDEX]) ;
 
 //return (data[4] & 0x4000) >> 14;
}


uint16_t abb_api_rcmeas_finished(void)
{
 
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS1, 
		data);
		
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

 return CENTRAL_FCSI_IRCMEAS1_FINISHED__GET(data[ABB_CENTRAL_INDEX]) ;

 //return (data[4] & 0x8000) >> 15;

}


uint16_t abb_api_read_rcmeas(void)
{
 // read and return countervalue;
 // expected  -40   +70   125 deg
 // TT        455 / 471 / 480
 // SS             ~671 
 // FF             ~304
 // firmware subtracts average counter delay!
 
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(ABB_CENTRAL, 
 	CENTRAL_FCSI_IRCMEAS1, 
		data);
		
 return CENTRAL_FCSI_IRCMEAS1_RCCOUNT__GET(data[ABB_CENTRAL_INDEX]) ;


 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

 //return CENTRAL_FCSI_IRCMEAS1_FINISHED__GET(data[ABB_CENTRAL_INDEX]) ;

 //return data[4] & 0x03FF; 
}


void abb_api_init_ant(uint16_t _ant_mask)
{
  // compatibility with previously released API libs
  abb_antenna_init(_ant_mask);
}

void abb_api_ant_trim_daclb(uint16_t _ant_mask, uint8_t trim)
{ 
  ABB_FCSI_RMW(_ant_mask, 0x42, 0xFF3F, trim << 4);
}


void abb_api_ant_trim_ldo0v95a(uint16_t _ant_mask, uint8_t trim)
{ 
  ABB_FCSI_RMW(_ant_mask, 
  	ANT_FCSI_ILDOA0, 
		ANT_FCSI_ILDOA0_OUTTRIM__ISMSK, 
			trim & ANT_FCSI_ILDOA0_OUTTRIM__MSK );
}


void abb_api_ant_trim_ldo0v95d(uint16_t _ant_mask, uint8_t trim)
{ 
  ABB_FCSI_RMW(_ant_mask, 
  	ANT_FCSI_ILDOD0, 
		ANT_FCSI_ILDOD0_OUTTRIM__ISMSK, 
			trim & ANT_FCSI_ILDOD0_OUTTRIM__MSK );  
  
}


uint16_t abb_api_read_antenna_version(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(_ant_mask, 
 	ANT_FCSI_IAVERSION0,
		data );
 
 //0x45, data);
 
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

switch(_ant_mask) { 
  	case ABB_ANTENNA0: return data[0] & ANT_FCSI_IAVERSION0__R_MSK;
 	case ABB_ANTENNA1: return data[1] & ANT_FCSI_IAVERSION0__R_MSK;
 	case ABB_ANTENNA2: return data[2] & ANT_FCSI_IAVERSION0__R_MSK;
 	case ABB_ANTENNA3: return data[3] & ANT_FCSI_IAVERSION0__R_MSK;
	default: return -1; // return value only if single antenna is specified
	}
}


void abb_api_pd_ant(uint16_t _ant_mask)
{
 // for compatibility with previously released API versions
 abb_antenna_pd(_ant_mask);
}


void abb_api_start_tx(uint16_t _ant_mask)
{

 // clock gating enabled by default: FRCTXCLKON_DISABLE
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IDACA0,
		ANT_FCSI_IDACA0_PUPBUF_OFF__SVAL 	|
		ANT_FCSI_IDACA0_PUPREF_ON__SVAL		| // pup_ref=1
		ANT_FCSI_IDACA0_GAINABS_DEFAULT__SVAL	|
		ANT_FCSI_IDACA0_LPMODE_DISABLE__SVAL	| 
		ANT_FCSI_IDACA0_ITRIM_50U__SVAL		|
		ANT_FCSI_IDACA0_CNFLCLBIAS_PTAT__SVAL	|
		ANT_FCSI_IDACA0_ENLB_ENABLE__SVAL	| // en_lb=1
		ANT_FCSI_IDACA0_INVCLK_NOTINV__SVAL 	|
		ANT_FCSI_IDACA0_FRCTXCLKON_DISABLE__SVAL);
 
 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,
		ANT_FCSI_IDACD0_DEMMODE_0__SVAL		|
		ANT_FCSI_IDACD0_DTEN_DISABLE__SVAL	| // dten=0
		ANT_FCSI_IDACD0_RSTQ_0__SVAL		|
		ANT_FCSI_IDACD0_BINARYIN_DISABLE__SVAL	| // binary=0
		ANT_FCSI_IDACD0_STOPSYNC_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_PHCEN_0__SVAL		|
		ANT_FCSI_IDACD0_XCHGDATA_0__SVAL	|
		ANT_FCSI_IDACD0_TXEN_DISABLE__SVAL	);

 ABB_wait_us(20); // required for DAC to settle!

 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,
		ANT_FCSI_IDACD0_DEMMODE_0__SVAL		|
		ANT_FCSI_IDACD0_DTEN_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_RSTQ_0__SVAL		|
		ANT_FCSI_IDACD0_BINARYIN_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_STOPSYNC_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_PHCEN_1__SVAL		| // phc_en 0->1
		ANT_FCSI_IDACD0_XCHGDATA_0__SVAL	|
		ANT_FCSI_IDACD0_TXEN_DISABLE__SVAL	);

 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,
		ANT_FCSI_IDACD0_DEMMODE_0__SVAL		|
		ANT_FCSI_IDACD0_DTEN_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_RSTQ_1__SVAL		| // reset_q 0->1
		ANT_FCSI_IDACD0_BINARYIN_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_STOPSYNC_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_PHCEN_1__SVAL		|
		ANT_FCSI_IDACD0_XCHGDATA_0__SVAL	|
		ANT_FCSI_IDACD0_TXEN_ENABLE__SVAL	);// tx_en 0->1
		
 //Toplevel TX_ON 0->1 (->pup_buf=1)
 
}

void abb_api_start_tx_lp(uint16_t _ant_mask)
{
 
 // clock gating enabled by default: FRCTXCLKON_DISABLE
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IDACA0,
		ANT_FCSI_IDACA0_PUPBUF_OFF__SVAL 	|
		ANT_FCSI_IDACA0_PUPREF_ON__SVAL		| // pup_ref=1
		ANT_FCSI_IDACA0_GAINABS_DEFAULT__SVAL	| 
		ANT_FCSI_IDACA0_LPMODE_ENABLE__SVAL	| // lpmode=1 
		ANT_FCSI_IDACA0_ITRIM_50U__SVAL		|
		ANT_FCSI_IDACA0_CNFLCLBIAS_PTAT__SVAL	|
		ANT_FCSI_IDACA0_ENLB_ENABLE__SVAL	| // en_lb=1
		ANT_FCSI_IDACA0_INVCLK_NOTINV__SVAL 	|
		ANT_FCSI_IDACA0_FRCTXCLKON_DISABLE__SVAL);

 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,
		ANT_FCSI_IDACD0_DEMMODE_0__SVAL		|
		ANT_FCSI_IDACD0_DTEN_DISABLE__SVAL	| // dten=0
		ANT_FCSI_IDACD0_RSTQ_0__SVAL		|
		ANT_FCSI_IDACD0_BINARYIN_DISABLE__SVAL	| // binary=0
		ANT_FCSI_IDACD0_STOPSYNC_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_PHCEN_0__SVAL		|
		ANT_FCSI_IDACD0_XCHGDATA_0__SVAL	|
		ANT_FCSI_IDACD0_TXEN_DISABLE__SVAL	);


// ABB_FCSI_WRITE(_ant_mask, 0x46, 0x0EE2); //en_lb=1; pup_ref=1; lpmode=1
// ABB_FCSI_WRITE(_ant_mask, 0x47, 0x0020); //binary=1; dten=0

 ABB_wait_us(20); // required for DAC to settle!

 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,
		ANT_FCSI_IDACD0_DEMMODE_0__SVAL		|
		ANT_FCSI_IDACD0_DTEN_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_RSTQ_0__SVAL		|
		ANT_FCSI_IDACD0_BINARYIN_DISABLE__SVAL	| // binary=0
		ANT_FCSI_IDACD0_STOPSYNC_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_PHCEN_1__SVAL		| // phc_en 0->1
		ANT_FCSI_IDACD0_XCHGDATA_0__SVAL	|
		ANT_FCSI_IDACD0_TXEN_DISABLE__SVAL	);

 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,
		ANT_FCSI_IDACD0_DEMMODE_0__SVAL		|
		ANT_FCSI_IDACD0_DTEN_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_RSTQ_1__SVAL		| // reset_q 0->1
		ANT_FCSI_IDACD0_BINARYIN_DISABLE__SVAL	| // binary=0
		ANT_FCSI_IDACD0_STOPSYNC_DISABLE__SVAL	|
		ANT_FCSI_IDACD0_PHCEN_1__SVAL		|
		ANT_FCSI_IDACD0_XCHGDATA_0__SVAL	|
		ANT_FCSI_IDACD0_TXEN_ENABLE__SVAL	);// tx_en 0->1

// ABB_FCSI_WRITE(_ant_mask, 0x47, 0x2020); //phc_en 0->1; binary=1
// ABB_FCSI_WRITE(_ant_mask, 0x47, 0xA028); //tx_en 0->1; reset_q 0->1; binary=1; dten=0

 //Toplevel TX_ON 0->1 (->pup_buf=1)
 
}


void abb_api_standby_tx(uint16_t _ant_mask)
{
 //Toplevel TX_ON 1->0 (->pup_buf=0)
 
 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACA0,
		ANT_FCSI_IDACA0__INIT );
	
 ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IDACD0,	
		ANT_FCSI_IDACD0__INIT );
 
 
 //ABB_FCSI_WRITE(_ant_mask, 0x46, 0x06A0);
 //ABB_FCSI_WRITE(_ant_mask, 0x47, 0x0000);
 
}


void abb_api_trim_rx_rctune(uint16_t _ant_mask, int8_t rctune)
{

 ABB_FCSI_RMW(_ant_mask, ANT_FCSI_IADCA2, 
 	ANT_FCSI_IADCA2_RCTUNE__ISMSK & ANT_FCSI_IADCA2_R1VAL__ISMSK,
		(rctune & ANT_FCSI_IADCA2_RCTUNE__MSK) |
		((2*rctune) &  ANT_FCSI_IADCA2_R1VAL__MSK) << ANT_FCSI_IADCA2_R1VAL__POS);

}

void abb_api_rx_set_mask_reset_overload(uint16_t _ant_mask, int8_t enable_mask_reset)
{

 ABB_FCSI_RMW(_ant_mask, ANT_FCSI_IADCA1, 
 	ANT_FCSI_IADCA1_OVMASK__ISMSK,
		(enable_mask_reset &  ANT_FCSI_IADCA1_OVMASK__MSK) << ANT_FCSI_IADCA1_OVMASK__POS);

}



void abb_api_set_rx_rctune_qtrim(uint16_t _ant_mask, int8_t rctune, int8_t qtrim)
{

 int myANT_FCSI_IADCA2_R1VAL  = ANT_FCSI_IADCA2_R1VAL_29__SVAL;
 int myANT_FCSI_IADCA2_RCTUNE = ANT_FCSI_IADCA2_RCTUNE_505OHM263__SVAL;

 if (rctune != -1)
 {
  myANT_FCSI_IADCA2_R1VAL  = ((2*rctune) &  ANT_FCSI_IADCA2_R1VAL__MSK) << ANT_FCSI_IADCA2_R1VAL__POS;
  myANT_FCSI_IADCA2_RCTUNE = (rctune & ANT_FCSI_IADCA2_RCTUNE__MSK);
 }

 int myANT_FCSI_IADCA2_QTRIM1EN = ANT_FCSI_IADCA2_QTRIM1EN_0__SVAL;
 int myANT_FCSI_IADCA2_QTRIM2EN = ANT_FCSI_IADCA2_QTRIM2EN_0__SVAL;

 if (qtrim == 1)
 {
  myANT_FCSI_IADCA2_QTRIM1EN = ANT_FCSI_IADCA2_QTRIM1EN_1__SVAL;
 }
 else if (qtrim == 2)
 {
  myANT_FCSI_IADCA2_QTRIM2EN = ANT_FCSI_IADCA2_QTRIM2EN_1__SVAL;
 }

  ABB_FCSI_WRITE(_ant_mask, 
  	ANT_FCSI_IADCA2, 
		ANT_FCSI_IADCA2_CALDACMODE_2__SVAL | // changed for LAp133
		ANT_FCSI_IADCA2_SELRSTCTL_1__SVAL  |
		myANT_FCSI_IADCA2_QTRIM1EN	   |
		myANT_FCSI_IADCA2_QTRIM2EN	   |
		myANT_FCSI_IADCA2_R1VAL		   |
		myANT_FCSI_IADCA2_RCTUNE	   );

}


void abb_api_start_rx_11ax(uint16_t _ant_mask, int8_t rctune, int8_t qtrim)
{

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA0, 
		ANT_FCSI_IADCA0_OPA1_1__SVAL 		| // opa123=1,2,3 ! changed for LA1p32
		ANT_FCSI_IADCA0_OPA2_2__SVAL 		|
		ANT_FCSI_IADCA0_OPA3_3__SVAL 		|
		ANT_FCSI_IADCA0_W320_OFF__SVAL 		|
		ANT_FCSI_IADCA0_W40FILTEN_OFF__SVAL 	|
		ANT_FCSI_IADCA0_ELDRES_9__SVAL 		| // eldres=9; 
		ANT_FCSI_IADCA0_R1_3__SVAL ); 
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_OVMASK_ENABLE__SVAL 	|
		ANT_FCSI_IADCA1_ELDCAP_9__SVAL 		| // eldcap=9
		ANT_FCSI_IADCA1_LFSRSEL_3__SVAL 	| // lfsr=3; 
		ANT_FCSI_IADCA1_ITRIM_3__SVAL 		|
		ANT_FCSI_IADCA1_PTATEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCA1_PTAT2EN_0__SVAL 	|
		ANT_FCSI_IADCA1_TH_TRIM05_ENABLE__SVAL  | // B0 update
		ANT_FCSI_IADCA1_CLKINV_INV__SVAL 	| // B0 update
		ANT_FCSI_IADCA1_FRCLFSRRESN_DISABLE__SVAL| // B0 update, was "ctldac1en"
		ANT_FCSI_IADCA1_CTLDAC2EN_DISABLE__SVAL	|
		ANT_FCSI_IADCA1_CTLDAC3EN_DISABLE__SVAL ); 

 abb_api_set_rx_rctune_qtrim(_ant_mask, rctune,qtrim);
		
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL |
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 	|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 	|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL		|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA3,
		ANT_FCSI_IADCA3_VCMPROG__PUT(ANT_FCSI_IADCA3_VCMPROG_2__VAL , ANT_FCSI_IADCA3__INIT) );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4__INIT );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| // phc_en 0->1		
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL 			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| // rx_en 0->1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 
 
}


void abb_api_start_rx_ssb(uint16_t _ant_mask, int8_t rctune, int8_t qtrim)
{
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA0, 
		ANT_FCSI_IADCA0_OPA1_7__SVAL 		| // opa123=7,7,3; 
		ANT_FCSI_IADCA0_OPA2_7__SVAL 		|
		ANT_FCSI_IADCA0_OPA3_3__SVAL 		|
		ANT_FCSI_IADCA0_W320_ON__SVAL 		| // w320=1
		ANT_FCSI_IADCA0_W40FILTEN_OFF__SVAL 	|
		ANT_FCSI_IADCA0_ELDRES_9__SVAL 		| // eldres=9; 
		ANT_FCSI_IADCA0_R1_3__SVAL ); 
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_OVMASK_ENABLE__SVAL 	|
		ANT_FCSI_IADCA1_ELDCAP_9__SVAL 		| // eldcap=9
		ANT_FCSI_IADCA1_LFSRSEL_3__SVAL 	| // lfsr=3; 
		ANT_FCSI_IADCA1_ITRIM_3__SVAL 		|
		ANT_FCSI_IADCA1_PTATEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCA1_PTAT2EN_0__SVAL 	|
		ANT_FCSI_IADCA1_TH_TRIM05_ENABLE__SVAL  | // B0 update, was 0
		ANT_FCSI_IADCA1_CLKINV_INV__SVAL 	| // B0 update
		ANT_FCSI_IADCA1_FRCLFSRRESN_DISABLE__SVAL| // B0 update, was ctldac1en
		ANT_FCSI_IADCA1_CTLDAC2EN_DISABLE__SVAL	|
		ANT_FCSI_IADCA1_CTLDAC3EN_DISABLE__SVAL ); 

 abb_api_set_rx_rctune_qtrim(_ant_mask, rctune,qtrim);
		
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL |
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 	|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 	|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL		|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA3,
		ANT_FCSI_IADCA3_VCMPROG__PUT(ANT_FCSI_IADCA3_VCMPROG_2__VAL , ANT_FCSI_IADCA3__INIT) );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4__INIT );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| // phc_en 0->1		
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL 			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| // rx_en 0->1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 
 
}


void abb_api_start_rx_lp(uint16_t _ant_mask, int8_t rctune, int8_t qtrim)
{
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA0, 
		ANT_FCSI_IADCA0_OPA1_0__SVAL 		| // opa123=0,1,1; 
		ANT_FCSI_IADCA0_OPA2_1__SVAL 		|
		ANT_FCSI_IADCA0_OPA3_1__SVAL 		|
		ANT_FCSI_IADCA0_W320_OFF__SVAL 		| // w320=0
		ANT_FCSI_IADCA0_W40FILTEN_ON__SVAL 	| // w40filen=ON
		ANT_FCSI_IADCA0_ELDRES_15__SVAL		| // eldres=15; 
		ANT_FCSI_IADCA0_R1_3__SVAL ); 
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_OVMASK_ENABLE__SVAL 	|
		ANT_FCSI_IADCA1_ELDCAP_15__SVAL		| // eldcap=15
		ANT_FCSI_IADCA1_LFSRSEL_1__SVAL 	| // lfsr=1; 
		ANT_FCSI_IADCA1_ITRIM_3__SVAL 		|
		ANT_FCSI_IADCA1_PTATEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCA1_PTAT2EN_0__SVAL 	|
		ANT_FCSI_IADCA1_TH_TRIM05_ENABLE__SVAL  | // B0 update, was 0
		ANT_FCSI_IADCA1_CLKINV_INV__SVAL 	| // B0 update
		ANT_FCSI_IADCA1_FRCLFSRRESN_DISABLE__SVAL| // B0 update, was ctldac1en
		ANT_FCSI_IADCA1_CTLDAC2EN_DISABLE__SVAL	|
		ANT_FCSI_IADCA1_CTLDAC3EN_DISABLE__SVAL ); 

 abb_api_set_rx_rctune_qtrim(_ant_mask, rctune,qtrim);
		
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL |
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_W40CLKEN_640M__SVAL 	| // w40clken=640m
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 	|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 	|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL		|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA3,
		ANT_FCSI_IADCA3_VCMPROG__PUT(ANT_FCSI_IADCA3_VCMPROG_2__VAL , ANT_FCSI_IADCA3__INIT) );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4__INIT );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_640M__SVAL 		| // w40clken=640m
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 		|		
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL 			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_640M__SVAL 		| // w40clken=640m
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| // phc_en 0->1		
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL 			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_640M__SVAL 		| // w40clken=640m
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| // rx_en 0->1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 
 
}


uint16_t abb_api_read_rx_qt(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(_ant_mask, 
 	ANT_FCSI_IADCA4, 
		data);
		
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL
 
 switch(_ant_mask) { 
  	case ABB_ANTENNA0: return data[0] & 0x3FFF;
 	case ABB_ANTENNA1: return data[1] & 0x3FFF;
 	case ABB_ANTENNA2: return data[2] & 0x3FFF;
 	case ABB_ANTENNA3: return data[3] & 0x3FFF;
	//case ABB_CENTRAL:  return data[0] & 0x3FFF; break;
	default:           return -1; // return value only if single antenna is specified
	}
}


uint16_t abb_api_read_rx_qtcalactive(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(_ant_mask, 
 	ANT_FCSI_IADCA4, 
		data);
		
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL
 
 switch(_ant_mask) { 
  	case ABB_ANTENNA0: return (data[0] & ANT_FCSI_IADCA4_QTCALACTIVE__SMSK) >> ANT_FCSI_IADCA4_QTCALACTIVE__POS;
 	case ABB_ANTENNA1: return (data[1] & ANT_FCSI_IADCA4_QTCALACTIVE__SMSK) >> ANT_FCSI_IADCA4_QTCALACTIVE__POS;
 	case ABB_ANTENNA2: return (data[2] & ANT_FCSI_IADCA4_QTCALACTIVE__SMSK) >> ANT_FCSI_IADCA4_QTCALACTIVE__POS;
 	case ABB_ANTENNA3: return (data[3] & ANT_FCSI_IADCA4_QTCALACTIVE__SMSK) >> ANT_FCSI_IADCA4_QTCALACTIVE__POS;
	//case ABB_CENTRAL:  return (data[0] & ANT_FCSI_IADCA4_QTCALACTIVE__SMSK) >> ANT_FCSI_IADCA4_QTCALACTIVE__POS; break;
	default:           return -1; // return value only if single antenna is specified
	}
}


uint16_t abb_api_read_rx_qtcaldone(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(_ant_mask, 
 	ANT_FCSI_IADCA4, 
		data);
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

 
 int caldone=0;
 
 if ((_ant_mask & ABB_ANTENNA0) | 
     (_ant_mask & ABB_ANTENNA1) |
     (_ant_mask & ABB_ANTENNA2) |
     (_ant_mask & ABB_ANTENNA3))
 {
  caldone=1;
 }
 
 //debug_printf("qtcaldone data_o0: %d\n",data[0]);
 //debug_printf("qtcaldone data_o1: %d\n",data[1]);
 //debug_printf("qtcaldone data_o2: %d\n",data[2]);
 //debug_printf("qtcaldone data_o3: %d\n",data[3]);
 
 
 if (_ant_mask & ABB_ANTENNA0)
  { 
    caldone = caldone & ((data[0] & ANT_FCSI_IADCA4_QTCALDONE__SMSK) >> ANT_FCSI_IADCA4_QTCALDONE__POS);
  }
 
 if (_ant_mask & ABB_ANTENNA1)
  { 
    caldone = caldone & ((data[1] & ANT_FCSI_IADCA4_QTCALDONE__SMSK) >> ANT_FCSI_IADCA4_QTCALDONE__POS);
  }
   
 if (_ant_mask & ABB_ANTENNA2)
  { 
    caldone = caldone & ((data[2] & ANT_FCSI_IADCA4_QTCALDONE__SMSK) >> ANT_FCSI_IADCA4_QTCALDONE__POS);
  }
 if (_ant_mask & ABB_ANTENNA3)
  { 
    caldone = caldone & ((data[3] & ANT_FCSI_IADCA4_QTCALDONE__SMSK) >> ANT_FCSI_IADCA4_QTCALDONE__POS);
  }
   
 if (caldone)
 {
  return 1;
 }
 else
 {
  return 0;
 } 
   
}


void abb_api_start_rx_loopback(uint16_t _ant_mask)
{
 //ABB_FCSI_WRITE(_ant_mask, 0x48, 0xA0D1); // R1=10=> loopback; eldres=8; opa123=1,2,3 
 //ABB_FCSI_WRITE(_ant_mask, 0x49, 0xE1D3); // lfsr=2; eldcap=9
 //ABB_FCSI_WRITE(_ant_mask, 0x4A, 0x238E); // caldacmode = 0;  rctune=DEFAULT
 //ABB_FCSI_WRITE(_ant_mask, 0x4C, 0x5000); // vcmprog = 2; qtdebug = DISABLED

 //ABB_FCSI_WRITE(_ant_mask, 0x4D, 0x0000); // qtunbreak = 0
 
 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0800); // phc_en 0->1
 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0A00); // rx_en 0->1
 
 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_IADCA0,
		ANT_FCSI_IADCA0_R1__ISMSK,
			ANT_FCSI_IADCA0_R1_2__SVAL ); // R1=10=LOOPBACK

} 


void abb_api_start_rx_lfsr(uint16_t _ant_mask)
{

 // force A0 behavior => release lfsr reset and trigger by FCSI
 //ABB_FCSI_RMW(_ant_mask,
 //	ANT_FCSI_IADCA1,
 //		ANT_FCSI_IADCA1_FRCLFSRRESN__SMSK,
 //			ANT_FCSI_IADCA1_FRCLFSRRESN_ENABLE__SVAL ); // frclfrresn=1
			

 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0002); // en_adc_lfsr=1; 
 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0802); // phc_en 0->1
 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0A02); // rx_en 0->1
 
   ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_ENABLE__SVAL  	| // enadclfsr=1
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ENADCLFSR_ENABLE__SVAL 		| // enadclfsr=1
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| // phc_en 0->1		
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL 			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL	|
		ANT_FCSI_IADCD0_ENADCLFSR_ENABLE__SVAL 		| // enadclfsr=1
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| // rx_en 0->1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 
}


void abb_api_start_rx_ts_lfsr(uint16_t _ant_mask)
{

 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0001); //en_tsrx_lfsr=1; 
 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0801); // phc_en 0->1
 //ABB_FCSI_WRITE(_ant_mask, 0x4B, 0x0A01); // rx_en 0->1
  
    ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_ENABLE__SVAL 	| // entsrxlfsr=1
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_ENABLE__SVAL 	| // entsrxlfsr=1
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL		| 
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		| // phc_en 0->1		
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL 			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_ENABLE__SVAL		| // enatsrxlfsr=1
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL		|
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 		|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 		|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 		|
		ANT_FCSI_IADCD0_RXEN_ENABLE__SVAL 		| // rx_en 0->1
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 		|
		ANT_FCSI_IADCD0_PHCEN_ENABLE__SVAL 		|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL			|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL ); 
 
}


void abb_api_standby_rx(uint16_t _ant_mask, int8_t rctune, int8_t qtrim)
{
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA0, 
		ANT_FCSI_IADCA0_OPA1_1__SVAL 		| // opa123=1,2,3 ! changed for LA1p32
		ANT_FCSI_IADCA0_OPA2_2__SVAL 		|
		ANT_FCSI_IADCA0_OPA3_3__SVAL 		|
		ANT_FCSI_IADCA0_W320_OFF__SVAL 		|
		ANT_FCSI_IADCA0_W40FILTEN_OFF__SVAL 	|
		ANT_FCSI_IADCA0_ELDRES_9__SVAL 		| // eldres=9; 
		ANT_FCSI_IADCA0_R1_3__SVAL ); 
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_IADCA1, 
		ANT_FCSI_IADCA1_OVMASK_ENABLE__SVAL 	|
		ANT_FCSI_IADCA1_ELDCAP_9__SVAL 		| // eldcap=9
		ANT_FCSI_IADCA1_LFSRSEL_3__SVAL 	| // lfsr=3; 
		ANT_FCSI_IADCA1_ITRIM_3__SVAL 		|
		ANT_FCSI_IADCA1_PTATEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCA1_PTAT2EN_0__SVAL 	|
		ANT_FCSI_IADCA1_TH_TRIM05_ENABLE__SVAL  | // B0 update, was 0
		ANT_FCSI_IADCA1_CLKINV_INV__SVAL 	| // B0 update
		ANT_FCSI_IADCA1_FRCLFSRRESN_DISABLE__SVAL| // B0 update, was ctldac1en
		ANT_FCSI_IADCA1_CTLDAC2EN_DISABLE__SVAL	|
		ANT_FCSI_IADCA1_CTLDAC3EN_DISABLE__SVAL ); 

 abb_api_set_rx_rctune_qtrim(_ant_mask, rctune,qtrim);
		
 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCD0,
		ANT_FCSI_IADCD0_ENLSRXLFSR_DISABLE__SVAL|
		ANT_FCSI_IADCD0_ENADCLFSR_DISABLE__SVAL |
		ANT_FCSI_IADCD0_RXCLKINV_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_W40CLKEN_2G56__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANVAL_0__SVAL 	|
		ANT_FCSI_IADCD0_RXONMANSEL_AUTO__SVAL 	|
		ANT_FCSI_IADCD0_RXONDEL_50NS__SVAL 	|
		ANT_FCSI_IADCD0_RXEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_STOPSYNC_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_PHCEN_DISABLE__SVAL 	|
		ANT_FCSI_IADCD0_ELDDEL_0__SVAL		|
		ANT_FCSI_IADCD0_ENFRCRST_DISABLE__SVAL );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA3,
		ANT_FCSI_IADCA3_VCMPROG__PUT(ANT_FCSI_IADCA3_VCMPROG_2__VAL , ANT_FCSI_IADCA3__INIT) );

 ABB_FCSI_WRITE(_ant_mask,
 	ANT_FCSI_IADCA4,
		ANT_FCSI_IADCA4__INIT );
 
}


void abb_api_start_ssi(uint16_t _ant_mask)
{
 //start from reset values
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA0, 
	ANT_FCSI_ISSIA0_RXRESSEL_RES10K__SVAL		|
	ANT_FCSI_ISSIA0_INFILTON_DISCONNECTED__SVAL	);
 //0x0200);
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ENCAL_OFF__SVAL		| // offset cal fix
		ANT_FCSI_ISSIA1_IBREFSEL_ICONST__SVAL	| // update request by Sergio 20.2.2018
		ANT_FCSI_ISSIA1_IBCOMPSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBRBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBINBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_REFSELI_P0V75N0V25__SVAL|
		ANT_FCSI_ISSIA1_ANAMUXSEL_3__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_OFF__SVAL	);
 //0x67F1);
 
 ABB_FCSI_WRITE(_ant_mask, 
	 ANT_FCSI_ISSID0,
		ANT_FCSI_ISSID0_SARCLKFREQ_640M__SVAL	|
		ANT_FCSI_ISSID0_ENCRS_ON__SVAL		| // update request by Sergio 20.2.2018
		ANT_FCSI_ISSID0_GLOBCALON_OFF__SVAL	| // local calibration strategy 24.9.2018
		ANT_FCSI_ISSID0_CMPCALFRZ_OFF__SVAL	|
		ANT_FCSI_ISSID0_CMPCALEXD_ON__SVAL	| // update request by Sergio 23.11.2018
		ANT_FCSI_ISSID0_CALEXTEN_OFF__SVAL	|
		ANT_FCSI_ISSID0_CALEXTSET_0__SVAL	);	 
 //0x0013);

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_DFTOUTSEL_DATAO__SVAL	| // outsel=00=10bit adc output
		ANT_FCSI_ISSID1_RESN_DISABLED__SVAL	| // resn 0->1
		ANT_FCSI_ISSID1_ENEXTON_ENABLED__SVAL	| // enexton 0->1
		ANT_FCSI_ISSID1_SSION_ENABLED__SVAL	| // ssion 0->1
		ANT_FCSI_ISSID1_TXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_RXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_CAL_COMP_0__SVAL	);
 // 0x0004); // resn=1; outsel=00=10bit adc output
  
 ABB_wait_us(10); // required for ssi_on settling
  
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ENCAL_ON__SVAL		| // encal 0->1
		ANT_FCSI_ISSIA1_IBREFSEL_ICONST__SVAL	|
		ANT_FCSI_ISSIA1_IBCOMPSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBRBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBINBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_REFSELI_P0V75N0V25__SVAL|
		ANT_FCSI_ISSIA1_ANAMUXSEL_3__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_OFF__SVAL	); 
  
  
 ABB_wait_us(4); // offset calibration
 
  ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_DFTOUTSEL_DATAO__SVAL	| // outsel=00=10bit adc output
		ANT_FCSI_ISSID1_RESN_DISABLED__SVAL	| // resn 1
		ANT_FCSI_ISSID1_ENEXTON_DISABLED__SVAL	| // enexton 1->0
		ANT_FCSI_ISSID1_SSION_DISABLED__SVAL	| // ssion 1->0
		ANT_FCSI_ISSID1_TXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_RXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_CAL_COMP_0__SVAL	);  
  
}


void abb_api_recal_ssi(uint16_t _ant_mask)
{
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ENCAL_OFF__SVAL		| // encal 1->0
		ANT_FCSI_ISSIA1_IBREFSEL_ICONST__SVAL	|
		ANT_FCSI_ISSIA1_IBCOMPSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBRBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBINBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_REFSELI_P0V75N0V25__SVAL|
		ANT_FCSI_ISSIA1_ANAMUXSEL_3__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_OFF__SVAL	);
 

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_DFTOUTSEL_DATAO__SVAL	|
		ANT_FCSI_ISSID1_RESN_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_ENEXTON_ENABLED__SVAL	| // enexton 0->1
		ANT_FCSI_ISSID1_SSION_ENABLED__SVAL	| // ssion 1->0
		ANT_FCSI_ISSID1_TXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_RXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_CAL_COMP_0__SVAL	);

  
 ABB_wait_us(1); // for encal to react
 
  ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ENCAL_ON__SVAL		| // encal 0->1
		ANT_FCSI_ISSIA1_IBREFSEL_ICONST__SVAL	|
		ANT_FCSI_ISSIA1_IBCOMPSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBRBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_IBINBUFSEL_50UA__SVAL	|
		ANT_FCSI_ISSIA1_REFSELI_P0V75N0V25__SVAL|
		ANT_FCSI_ISSIA1_ANAMUXSEL_3__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_OFF__SVAL	);
  
 ABB_wait_us(4); // offset calibration
 
  ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_DFTOUTSEL_DATAO__SVAL	|
		ANT_FCSI_ISSID1_RESN_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_ENEXTON_DISABLED__SVAL	| // enexton 1->0
		ANT_FCSI_ISSID1_SSION_DISABLED__SVAL	| // ssion 1->0
		ANT_FCSI_ISSID1_TXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_RXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_CAL_COMP_0__SVAL	);  
  
}



void abb_api_start_ssi_lfsr(uint16_t _ant_mask)
{
 //reset values

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA0,
		ANT_FCSI_ISSIA0__INIT);
 //0x50, 0x0200);
  
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1__INIT);
 //0x51, 0x67F1);
  
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID0,
		ANT_FCSI_ISSID0__INIT);
 //0x52, 0x0013); 

 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_DFTOUTSEL_LFSR__SVAL	| // outsel=11=3x5bit lfsr
		ANT_FCSI_ISSID1_RESN_DISABLED__SVAL	| // resn=1
		ANT_FCSI_ISSID1_ENEXTON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_SSION_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_TXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_RXON_DISABLED__SVAL	|
		ANT_FCSI_ISSID1_CAL_COMP_0__SVAL	);
 //0x53, 0x0007); // resn=1; outsel=11=3x5bit lfsr

}


void abb_api_standby_ssi(uint16_t _ant_mask)
{
 //reset values
 
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA0,
		ANT_FCSI_ISSIA0__INIT);
  //0x50, 0x0200);
  
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1__INIT);
  //0x51, 0x67F1);
  
 ABB_FCSI_WRITE(_ant_mask, 
 	ANT_FCSI_ISSID0,
		ANT_FCSI_ISSID0__INIT);
  //0x52, 0x0013); 
  
 ABB_FCSI_WRITE(_ant_mask, 
	 ANT_FCSI_ISSID1,
	 	ANT_FCSI_ISSID1__INIT);
  //0x53, 0x0000); // resn=0;
 
}


int16_t abb_api_read_ssi_cal_comp(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(_ant_mask, 
 	ANT_FCSI_ISSID1,
		data );
 //0x53
 
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL

switch(_ant_mask) { 
  	case ABB_ANTENNA0: return ANT_FCSI_ISSID1_CAL_COMP__GET(data[0]& ANT_FCSI_ISSID1_CAL_COMP__SMSK);
	case ABB_ANTENNA1: return ANT_FCSI_ISSID1_CAL_COMP__GET(data[1]& ANT_FCSI_ISSID1_CAL_COMP__SMSK);
	case ABB_ANTENNA2: return ANT_FCSI_ISSID1_CAL_COMP__GET(data[2]& ANT_FCSI_ISSID1_CAL_COMP__SMSK);
	case ABB_ANTENNA3: return ANT_FCSI_ISSID1_CAL_COMP__GET(data[3]& ANT_FCSI_ISSID1_CAL_COMP__SMSK);
	default: return -1; // return value only if single antenna is specified
	}
}


void abb_api_ssi_enable_exton(uint16_t _ant_mask)
{

 ABB_FCSI_RMW(_ant_mask,
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_ENEXTON__ISMSK,
			ANT_FCSI_ISSID1_ENEXTON_ENABLED__SVAL ); // enexton=1

// ABB_FCSI_RMW(_ant_mask, 0x53, 0xFFEF, 0x0010); // enexton=1

/* //ABB_FCSI_WRITE(_ant_mask, 0x53, 0x0010);

 uint16_t data[NR_OF_ANTENNAS];
 
 ABB_FCSI_READ(_ant_mask, 0x53, data);
 debug_printf("SSI API TEST1: %d xxx %d\n",data[0], (data[0] & 0xFFEF) | 0x0010);
 debug_printf("SSI API TEST2: %d xxx %d\n",data[0] >> 8 );
 ABB_FCSI_WRITE(_ant_mask, 0x53, (data[0] & 0xFFEF) | 0x0010);*/
 
}


void abb_api_ssi_set_extssion(uint16_t _ant_mask, uint8_t ssion)
{

 ABB_FCSI_RMW(_ant_mask,
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_SSION__ISMSK,
			(ssion & ANT_FCSI_ISSID1_SSION__MSK) << ANT_FCSI_ISSID1_SSION__POS );

 //if (ssion==SIG_ON) {
 // ABB_FCSI_RMW(_ant_mask, 0x53, 0xFFDF, 0x0020);
 //} else
 //{
 // ABB_FCSI_RMW(_ant_mask, 0x53, 0xFFDF, 0x0000);
 // }
}


void abb_api_ssi_set_exttxon(uint16_t _ant_mask, uint8_t txon)
{

 ABB_FCSI_RMW(_ant_mask,
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_TXON__ISMSK,
			(txon & ANT_FCSI_ISSID1_TXON__MSK) << ANT_FCSI_ISSID1_TXON__POS );

 //if (txon==SIG_ON) {
 // ABB_FCSI_RMW(_ant_mask, 0x53, 0xFFBF, 0x0040);
 //} else
 //{
 // ABB_FCSI_RMW(_ant_mask, 0x53, 0xFFBF, 0x0000);
 // }
}


void abb_api_ssi_set_extrxon(uint16_t _ant_mask, uint8_t rxon)
{

 ABB_FCSI_RMW(_ant_mask,
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_RXON__ISMSK,
			(rxon & ANT_FCSI_ISSID1_RXON__MSK) << ANT_FCSI_ISSID1_RXON__POS );

 // if (rxon==SIG_ON) {
 // ABB_FCSI_RMW(_ant_mask, 0x53, 0xFF7F, 0x0080);
 //} else
 //{
 // ABB_FCSI_RMW(_ant_mask, 0x53, 0xFF7F, 0x0000);
 // }
}


void abb_api_ssi_disable_exton(uint16_t _ant_mask)
{
 ABB_FCSI_RMW(_ant_mask,
 	ANT_FCSI_ISSID1,
		ANT_FCSI_ISSID1_ENEXTON__ISMSK,
			ANT_FCSI_ISSID1_ENEXTON_DISABLED__SVAL ); // enexton=0		

 //ABB_FCSI_RMW(_ant_mask, 0x53, 0xFFEF, 0x0000); // enexton=0
}


// DFT CENTRAL BIAS

void  abb_api_anamux_centralbias_RFIC1V8_atITRIM(void)
{
 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON
 
 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICANAMUX0,
		CENTRAL_FCSI_ICANAMUX0_ENAMUX_ON__SVAL |
		CENTRAL_FCSI_ICANAMUX0_SELAMUX_0__SVAL	);
  
 //0x41, 0x0001); 	//en_anamux=1 sel_anamux=00
 
 
}


void  abb_api_anamux_centralbias_ICONST_atITRIM(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICANAMUX0,
		CENTRAL_FCSI_ICANAMUX0_ENAMUX_ON__SVAL |
		CENTRAL_FCSI_ICANAMUX0_SELAMUX_1__SVAL	);
 
 //0x41, 0x0003); 	//en_anamux=1 sel_anamux=01 
}


void  abb_api_anamux_centralbias_IPTAT_atITRIM(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICANAMUX0,
		CENTRAL_FCSI_ICANAMUX0_ENAMUX_ON__SVAL |
		CENTRAL_FCSI_ICANAMUX0_SELAMUX_2__SVAL	); 
 
 //0x41, 0x0005); 	//en_anamux=1 sel_anamux=10  
}


void  abb_api_anamux_centralbias_VBGP_atITRIM(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICANAMUX0,
		CENTRAL_FCSI_ICANAMUX0_ENAMUX_ON__SVAL |
		CENTRAL_FCSI_ICANAMUX0_SELAMUX_3__SVAL	);  
		
 //0x41, 0x0007); 	//en_anamux=1 sel_anamux=11   
}


void  abb_api_anamux_centralbias_disable(void)
{ 
 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_OFF__SVAL ); //central anamux OFF
 
 ABB_FCSI_WRITE(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICANAMUX0,
		CENTRAL_FCSI_ICANAMUX0_ENAMUX_OFF__SVAL |
		CENTRAL_FCSI_ICANAMUX0_SELAMUX_0__SVAL	); 
 
 //0x41, 0x0000); 	//en_anamux=0 
}


// DFT CENTRAL CLKD

void  abb_api_anamux_clkd_VREFLDO_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(ABB_CENTRAL, 
	 CENTRAL_FCSI_ICLKDIST0,
	 	CENTRAL_FCSI_ICLKDIST0_ENAMUX__ISMSK	&
	 	CENTRAL_FCSI_ICLKDIST0_SELAMUX__ISMSK	,
		CENTRAL_FCSI_ICLKDIST0_ENAMUX_ON__SVAL	|
		CENTRAL_FCSI_ICLKDIST0_SELAMUX_0__SVAL	);
		
 //0x46, 0xFCFF, 0x0100); //en_anamux=1 sel_anamux=0   
}


void  abb_api_anamux_clkd_LDO0V95CLKD_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON

 ABB_FCSI_RMW(ABB_CENTRAL, 
	 CENTRAL_FCSI_ICLKDIST0,
	 	CENTRAL_FCSI_ICLKDIST0_ENAMUX__ISMSK	&
	 	CENTRAL_FCSI_ICLKDIST0_SELAMUX__ISMSK	,
		CENTRAL_FCSI_ICLKDIST0_ENAMUX_ON__SVAL	|
		CENTRAL_FCSI_ICLKDIST0_SELAMUX_1__SVAL	); 	
 
 //0x46, 0xFCFF, 0x0300); //en_anamux=1 sel_anamux=1   
}


void  abb_api_anamux_clkd_disable(void)
{ 

 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_OFF__SVAL ); //central anamux OFF

 ABB_FCSI_RMW(ABB_CENTRAL, 
	 CENTRAL_FCSI_ICLKDIST0,
	 	CENTRAL_FCSI_ICLKDIST0_ENAMUX__ISMSK	&
	 	CENTRAL_FCSI_ICLKDIST0_SELAMUX__ISMSK	,
		CENTRAL_FCSI_ICLKDIST0_ENAMUX_OFF__SVAL	|
		CENTRAL_FCSI_ICLKDIST0_SELAMUX_0__SVAL	);
		 	 
 //0x46, 0xFEFF, 0x0000); //en_anamux=0
}


// DFT CENTRAL LCPLL

void  abb_api_anamux_lcpll_VREG1V8HV_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_WRITE(ABB_CENTRAL, 
	CENTRAL_FCSI_IPLLW5,
		CENTRAL_FCSI_IPLLW5_ENAMUX_ON__SVAL 	|
		CENTRAL_FCSI_IPLLW5_SELAMUX_0__SVAL	);
 
 //0x55, 0x0001); 	//en_anamux=1 sel_anamux=0 
}

void  abb_api_anamux_lcpll_LDO0V95PLLREF_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON

 ABB_FCSI_WRITE(ABB_CENTRAL, 
	CENTRAL_FCSI_IPLLW5,
		CENTRAL_FCSI_IPLLW5_ENAMUX_ON__SVAL 	|
		CENTRAL_FCSI_IPLLW5_SELAMUX_1__SVAL	); 
 
 //0x55, 0x0003); 	//en_anamux=1 sel_anamux=01  
}

void  abb_api_anamux_lcpll_LDO0V95PLLPOST_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON

 ABB_FCSI_WRITE(ABB_CENTRAL, 
	CENTRAL_FCSI_IPLLW5,
		CENTRAL_FCSI_IPLLW5_ENAMUX_ON__SVAL 	|
		CENTRAL_FCSI_IPLLW5_SELAMUX_2__SVAL	);  
 
 //0x55, 0x0005); 	//en_anamux=1 sel_anamux=10 
}

void  abb_api_anamux_lcpll_ATESTOUT_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON

 ABB_FCSI_WRITE(ABB_CENTRAL, 
	CENTRAL_FCSI_IPLLW5,
		CENTRAL_FCSI_IPLLW5_ENAMUX_ON__SVAL 	|
		CENTRAL_FCSI_IPLLW5_SELAMUX_3__SVAL	);  
 
 //0x55, 0x0007); 	//en_anamux=1 sel_anamux=11
}

void  abb_api_anamux_lcpll_disable(void)
{  
 
  ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_OFF__SVAL ); //central anamux OFF

 ABB_FCSI_WRITE(ABB_CENTRAL, 
	CENTRAL_FCSI_IPLLW5,
		CENTRAL_FCSI_IPLLW5_ENAMUX_OFF__SVAL 	|
		CENTRAL_FCSI_IPLLW5_SELAMUX_0__SVAL	);   
 
 //0x55, 0x0000); 	//en_anamux=0 
}


// DFT CENTRAL CLKIN

void  abb_api_anamux_clkin_VREFLDO_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0, 
		CENTRAL_FCSI_ICLKIN0_ENAMUX__ISMSK	&
		CENTRAL_FCSI_ICLKIN0_SELAMUX__ISMSK	,
			CENTRAL_FCSI_ICLKIN0_ENAMUX_ON__SVAL	|
			CENTRAL_FCSI_ICLKIN0_SELAMUX_0__SVAL	);
	
 //0x44, 0xFCFF, 0x0100); //en_anamux=1 sel_anamux=0
}

void  abb_api_anamux_clkin_LDO0V95FSYS_atVREF(void)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON

 ABB_FCSI_RMW(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUX__ISMSK	&
		CENTRAL_FCSI_ICLKIN0_SELAMUX__ISMSK	,
			CENTRAL_FCSI_ICLKIN0_ENAMUX_ON__SVAL	|
			CENTRAL_FCSI_ICLKIN0_SELAMUX_1__SVAL	); 
 
 //0x44, 0xFCFF, 0x0300); //en_anamux=1 sel_anamux=1
}

void  abb_api_anamux_clkin_disable(void)
{ 

 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_OFF__SVAL ); //central anamux OFF

 ABB_FCSI_RMW(ABB_CENTRAL, 
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUX__ISMSK	&
		CENTRAL_FCSI_ICLKIN0_SELAMUX__ISMSK	,
			CENTRAL_FCSI_ICLKIN0_ENAMUX_OFF__SVAL	|
			CENTRAL_FCSI_ICLKIN0_SELAMUX_0__SVAL	); 
 
 //0x44, 0xFCFF, 0x0000); //en_anamux=0
}


// DFT ANTENNA BIAS

void  abb_api_anamux_antbias_VREF0V95_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ILANAMUX0, 
		ANT_FCSI_ILANAMUX0_ENAMUX__ISMSK	&
		ANT_FCSI_ILANAMUX0_SELAMUX__ISMSK	,
		ANT_FCSI_ILANAMUX0_ENAMUX_ON__SVAL	|
		ANT_FCSI_ILANAMUX0_SELAMUX_0__SVAL);
	
 //0x43, 0xFFF8, 0x0001); //en_anamux=1 sel_anamux=00
}

void  abb_api_anamux_antbias_LDO0V95A_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ILANAMUX0, 
		ANT_FCSI_ILANAMUX0_ENAMUX__ISMSK	&
		ANT_FCSI_ILANAMUX0_SELAMUX__ISMSK	,
		ANT_FCSI_ILANAMUX0_ENAMUX_ON__SVAL	|
		ANT_FCSI_ILANAMUX0_SELAMUX_1__SVAL); 
 
 
 //0x43, 0xFFF8, 0x0003); //en_anamux=1 sel_anamux=01 
}

void  abb_api_anamux_antbias_LDO0V95D_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ILANAMUX0, 
		ANT_FCSI_ILANAMUX0_ENAMUX__ISMSK	&
		ANT_FCSI_ILANAMUX0_SELAMUX__ISMSK	,
		ANT_FCSI_ILANAMUX0_ENAMUX_ON__SVAL	|
		ANT_FCSI_ILANAMUX0_SELAMUX_2__SVAL); 
 
 //0x43, 0xFFF8, 0x0005); //en_anamux=1 sel_anamux=10
}

void  abb_api_anamux_antbias_VSSAANT_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ILANAMUX0, 
		ANT_FCSI_ILANAMUX0_ENAMUX__ISMSK	&
		ANT_FCSI_ILANAMUX0_SELAMUX__ISMSK	,
		ANT_FCSI_ILANAMUX0_ENAMUX_ON__SVAL	|
		ANT_FCSI_ILANAMUX0_SELAMUX_3__SVAL);  
 
 
 //0x43, 0xFFF8, 0x0007); //en_anamux=1 sel_anamux=11
}

void  abb_api_anamux_antbias_disable(uint8_t _ant_mask)
{ 

 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_OFF__SVAL ); //central anamux OFF

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ILANAMUX0, 
		ANT_FCSI_ILANAMUX0_ENAMUX__ISMSK	&
		ANT_FCSI_ILANAMUX0_SELAMUX__ISMSK	,
		ANT_FCSI_ILANAMUX0_ENAMUX_OFF__SVAL	|
		ANT_FCSI_ILANAMUX0_SELAMUX_0__SVAL);  
  
 //0x43, 0xFFF8, 0x0000); //en_anamux=0
}


// DFT ANTENNA SSI

void abb_api_anamux_antssi_vrefn_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ANAMUXSEL__ISMSK	&
		ANT_FCSI_ISSIA1_ANAMUXEN__ISMSK	,
		ANT_FCSI_ISSIA1_ANAMUXSEL_0__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_ON__SVAL );
 
 //0x51, 0x1FFF, 0x8000); //en_anamux=1 sel_anamux=00
}

void abb_api_anamux_antssi_vrefp_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ANAMUXSEL__ISMSK	&
		ANT_FCSI_ISSIA1_ANAMUXEN__ISMSK	,
		ANT_FCSI_ISSIA1_ANAMUXSEL_1__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_ON__SVAL ); 
 
 //0x51, 0x1FFF, 0xA000); //en_anamux=1 sel_anamux=01
}

void abb_api_anamux_antssi_vcm_r_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ANAMUXSEL__ISMSK	&
		ANT_FCSI_ISSIA1_ANAMUXEN__ISMSK	,
		ANT_FCSI_ISSIA1_ANAMUXSEL_2__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_ON__SVAL ); 
 
 //0x51, 0x1FFF, 0xC000); //en_anamux=1 sel_anamux=10
}

void abb_api_anamux_antssi_vcm_atVREF(uint8_t _ant_mask)
{ 
 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_ON__SVAL ); //central anamux ON 

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ANAMUXSEL__ISMSK	&
		ANT_FCSI_ISSIA1_ANAMUXEN__ISMSK	,
		ANT_FCSI_ISSIA1_ANAMUXSEL_3__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_ON__SVAL ); 
 
 //0x51, 0x1FFF, 0xE000); //en_anamux=1 sel_anamux=11
}

void abb_api_anamux_antssi_disable(uint8_t _ant_mask)
{ 

 ABB_FCSI_RMW(ABB_CENTRAL,
 	CENTRAL_FCSI_ICLKIN0,
		CENTRAL_FCSI_ICLKIN0_ENAMUXPAD__ISMSK,
			CENTRAL_FCSI_ICLKIN0_ENAMUXPAD_OFF__SVAL ); //central anamux OFF

 ABB_FCSI_RMW(_ant_mask, 
 	ANT_FCSI_ISSIA1,
		ANT_FCSI_ISSIA1_ANAMUXSEL__ISMSK	&
		ANT_FCSI_ISSIA1_ANAMUXEN__ISMSK	,
		ANT_FCSI_ISSIA1_ANAMUXSEL_3__SVAL	|
		ANT_FCSI_ISSIA1_ANAMUXEN_OFF__SVAL );  
 
 //0x51, 0x1FFF, 0x0000); //en_anamux=0
}
