// ----------------------------------------------------------------------------
// Project    : WAVE600
// ----------------------------------------------------------------------------
// File       : abb_api.c
// Author     : Pridnig Peter
// Created    : 2017-03-24
// Standard   : Sytem Verilog
// ----------------------------------------------------------------------------
// Copyright (c) 2016 Intel GmbH
// ----------------------------------------------------------------------------
// Revisions  :
// Date        Version  Author    Description
// 2017-05-02  1.0      Pridnig   D1p1
// 2017-05-16  1.0      Pridnig   D1p2
// 2017-09-25  1.2      Pridnig   D2p0
// ----------------------------------------------------------------------------
// History    :
// 2017-05-02 copied from tc_ABB_A11
// 2017-05-03 changed "abb_sequence_central_pll_active" to pre-lock
// 2017-05-12 added abb_sequence_run_rcmeas
// 2017-0515  _wait->ABB_wait_us
// 2017-05-26 updated op_freq_sel_sc_d2a DIRECT_WRITES bus access
//            renamed abb11ax->abb
// 2018-01-22 moved abb_sequence_run_rcmeas to lib_abb_api.c


#include <stdint.h>
#include <math.h>
#include <stdio.h>
#include "target.h"
//#include "lib_test.h"

/* global definitions */
#include "abb_tb_defines.h"   	// link to sv testbench.. benchports
#include "abb_mode_defines.h"

/* target interface  definitions */
#include "interface_abb.h"

// API definitions
#include "lib_abb_api.h"
#include "lib_abb_sequences.h"


//Spectool2 c-outputs
//#include "central_fcsi_fieldpkg.h" // register for CENTRAL
//#include "ant_fcsi_fieldpkg.h"     // register for ANT


 // ---------------------------------------------------------
 // Section: TB only API functions (Sequences)
 // ---------------------------------------------------------

 void abb_sequence_read_central_fcsi()
{
 uint16_t data[NR_OF_ANTENNAS];
 int i;
 
 debug_printf("Central FCSI Readout Started:\n");

 for (i=64;i<=95;i++)
 {
 ABB_FCSI_READ(ABB_CENTRAL, i, data);
 
 debug_printf("Central FCSI Readout Addr: 0x%04x Data:0x%04x\n",i,data[4]);
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL
 }
 
  debug_printf("Central FCSI Readout Finished.\n");
  
}

 void abb_sequence_read_antenna_fcsi(uint16_t _ant_mask)
{
 uint16_t data[NR_OF_ANTENNAS];
 int i;
 
 debug_printf("Antenna FCSI Readout Started:\n");
  
 for (i=64;i<=83;i++)
 {
 ABB_FCSI_READ(_ant_mask, i, data);

 switch(_ant_mask) {
  	case ABB_ANTENNA0: debug_printf("Antenna FCSI Readout Ant#0 Addr: 0x%04x Data:0x%04x\n",i,data[0]); break;
  	case ABB_ANTENNA1: debug_printf("Antenna FCSI Readout Ant#1 Addr: 0x%04x Data:0x%04x\n",i,data[1]); break;
 	case ABB_ANTENNA2: debug_printf("Antenna FCSI Readout Ant#2 Addr: 0x%04x Data:0x%04x\n",i,data[2]); break;
  	case ABB_ANTENNA3: debug_printf("Antenna FCSI Readout Ant#3 Addr: 0x%04x Data:0x%04x\n",i,data[3]); break;
 	}
 // data[   0    1    2    3       4]
 //      ANT0 ANT1 ANT2 ANT3 CENTRAL
 }
 
 debug_printf("Antenna FCSI Readout Finished.\n");
 
}

// ---------------------------------------------------------
// Section: Sequence PLL Active
// --------------------------------------------------------- 

void abb_sequence_central_pll_active(uint8_t prelock)
{ 
 int i;
 int lcpll_lock;
 int lcpll_prelock;
 int lock;

 ABB_DIR_CTRL_WRITE(ENABLE_FSYS,0x1); 	// FSYS =  40MHz;
 ABB_DIR_CTRL_WRITE(BB_CLKSRC,0x0);
 ABB_DIR_CTRL_WRITE(ENABLE_BB_CLKIN,0x0);   // BB_CLKIN = 40MHz
 
 abb_api_init_central_pll_enable();
 //abb_api_init_central_pll_enable_fcsicontrol();

 i=0;
 lcpll_lock=0;
 lcpll_prelock=0;
 lock=0;

 if (prelock==1)
  debug_printf("start LCPLL pre-locking phase\n");
 else
  debug_printf("start LCPLL locking phase\n");

 do {
  ABB_wait_us(10);
  i+=1;
  debug_printf("LCPLL locking since %d us\n",i*10);
  lcpll_lock=abb_api_read_lcpll_lock();
  lcpll_prelock=abb_api_read_lcpll_prelock();
  
  lock = ( prelock==1 ? lcpll_prelock : lcpll_lock );
  
  debug_printf("LCPLL lock signal: %d\n",lcpll_lock);
  debug_printf("LCPLL prelock signal: %d\n",lcpll_prelock);
  
 } while ( (lock==0) && (i < 200) );
 
 if (i<200)
	 debug_printf("LCPLL locked after %d us!\n",i*10);
	else
	{
	 debug_printf("FAIL: LCPLL not locked after %d us!\n",i*10);
	 }

 if (!prelock) {
  ABB_DIR_CTRL_WRITE(ASSERT_lcpll_lock_a2d,0x1);
  ABB_wait_us(0.1);
  ABB_DIR_CTRL_WRITE(ASSERT_lcpll_lock_a2d,0x0);
 }
 
 abb_api_init_central_pup_clkd();
 ABB_wait_us(1);

 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_640m,0x1);
 ABB_wait_us(0.1);
 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_640m,0x0);

}

// ---------------------------------------------------------
// Section: Sequence PLL Bypass
// --------------------------------------------------------- 

void abb_sequence_central_pll_bypass()
{
 // TESTSOC legacy
 
 ABB_DIR_CTRL_WRITE(ENABLE_CLK_IN,0x1); //CLK_IN    = 2560MHz;
 
 abb_api_init_central_pll_bypass();

 abb_api_init_central_pup_clkd();
 ABB_wait_us(1);
 
 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_640m,0x1);
 ABB_wait_us(0.1);
 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_640m,0x0);
 
}


// ---------------------------------------------------------
// Section: Sequence DfT
// --------------------------------------------------------- 

void abb_sequence_dft()
{

  abb_api_anamux_centralbias_RFIC1V8_atITRIM();ABB_wait_us(0.5);
  abb_api_anamux_centralbias_ICONST_atITRIM();ABB_wait_us(0.5);
  abb_api_anamux_centralbias_IPTAT_atITRIM();ABB_wait_us(0.5);
  abb_api_anamux_centralbias_VBGP_atITRIM();ABB_wait_us(0.5);
  abb_api_anamux_centralbias_disable();ABB_wait_us(0.5);

  abb_api_anamux_clkd_VREFLDO_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_clkd_LDO0V95CLKD_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_clkd_disable();ABB_wait_us(0.5);

  abb_api_anamux_lcpll_VREG1V8HV_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_lcpll_LDO0V95PLLREF_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_lcpll_LDO0V95PLLPOST_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_lcpll_ATESTOUT_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_lcpll_disable();ABB_wait_us(0.5);

  abb_api_anamux_clkin_VREFLDO_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_clkin_LDO0V95FSYS_atVREF();ABB_wait_us(0.5);
  abb_api_anamux_clkin_disable();ABB_wait_us(0.5);

  abb_api_anamux_antbias_VREF0V95_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antbias_LDO0V95A_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antbias_LDO0V95D_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antbias_VSSAANT_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antbias_disable(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);

  abb_api_anamux_antssi_vrefn_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antssi_vrefp_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antssi_vcm_r_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antssi_vcm_atVREF(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
  abb_api_anamux_antssi_disable(ABB_ANTENNA0|ABB_ANTENNA1|ABB_ANTENNA2|ABB_ANTENNA3);ABB_wait_us(0.5);
 
 
}

// ---------------------------------------------------------
// Section: Sequence Clock Reprogramming
// --------------------------------------------------------- 

void abb_sequence_clock_reprogramming(int _op_freq, int _div)
{
// _div has precedence!
 
 abb_api_init_central_pd_clkd();
 ABB_wait_us(1);
 
  switch(_op_freq) {
 	case 0: ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x0); break; //640M
 	case 1: ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x1); break; //320M
	case 2: ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x2); break; //160M
	case 3: ABB_DIR_CTRL_WRITES(op_freq_sel_sc_d2a,2,0x3); break; //80M
	}	

  switch(_div) {
 	case 0: ABB_DIR_CTRL_WRITE(lcpll_div_d2a,0x0); break; //7g68 (=>640MHz)
 	case 1: ABB_DIR_CTRL_WRITE(lcpll_div_d2a,0x1); break; //7g68 x 3/4 (=>480MHz)
	}
  
 abb_api_init_central_pup_clkd();
 ABB_wait_us(1);

 if (_div==0)
 {
   switch(_op_freq) {
 	case 0:  ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_640m,0x1);
		 ABB_wait_us(0.1);
		 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_640m,0x0);
		 break; //640M
 	case 1:   ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_320m,0x1);
		 ABB_wait_us(0.1);
		 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_320m,0x0);
		 break; //320M
	case 2:  ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_160m,0x1);
		 ABB_wait_us(0.1);
		 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_160m,0x0);
		 break; //160M
	case 3:  ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_80m,0x1);
		 ABB_wait_us(0.1);
		 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_80m,0x0); 
		 break; //80M
	}
 } else
 {	
  switch(_div) {
 	case 0:  break; //7g68 (=>640MHz)
 	case 1:  ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_480m,0x1);
		 ABB_wait_us(0.1);
		 ABB_DIR_CTRL_WRITE(ASSERT_clk_sc_a2d_480m,0x0);break; //7g68 x 3/4 (=>480MHz)
	}
 }
	
}
