/*! \mainpage DPI Library Overview
 *
 *  \section intro_sec Introduction
 *
 *  The lib_dpi library contains a set of C to Systemverilog Driver Interface functions
 *
 *  \section usage_sec Usage
 *
 *  this library is the interface from the Systemverilog Driver to the C API Layer\n
 *
 *  Test Structure Overview (Linux  ): /home/lqatproject1/ip_modules/03_Documentation/1_Developement_Documentation/02_Specification/SWIP_sv_driver/ams_testcase-structure.vsd
 *

 * <img src="/home/lqatproject1/ip_modules/03_Documentation/1_Developement_Documentation/02_Specification/SWIP_sv_driver/ams_testcase-structure.jpg" alt="Screenshot"> 
 * <img src="I:\ip_modules\03_Documentation\1_Developement_Documentation\02_Specification\SWIP_sv_driver\ams_testcase-structure.jpg" alt="Screenshot">

 *  Example Tree:
 *  \sa FROM_CL <pre> All functions callable from Testcase </pre>
 * \arg User functions for text printing\n
 * - \ref debug_printf\n
 * - \ref raw_printf\n
 * \arg User functions for data io\n
 * - \ref _in\n 
 * - \ref _out\n
 * \arg User functions for time control\n
 * - \ref _nop\n
 * - \ref _wait\n
 * \arg User functions for port IO\n
 * - \ref _port_write\n
 * - \ref _port_read\n 
 * - \ref _port_set\n
 * - \ref _port_get\n
 * - \ref _ports_set\n
 * - \ref _ports_get\n
 * \arg User functions for configuration\n
 * - \ref _config_set\n
 * - \ref _config_get\n
 *  \sa FROM_SV <pre> All functions called from Systemverilog </pre>
 *  \sa COMP_CL <pre> Compatibility functions to older Busdriver versions </pre>
 */


/*! \file lib_dpi.c
 *  \brief DPI Core C Function interface  
 *  \date Juli 3, 2014
 *  \author: nusserh
*/

// include libary header files here
#include <stdio.h>
#include <dlfcn.h>
#include <string.h>
// include generated header files here
#include "dpi_func.h"

// include user header files here
#include "lib_dpi.h"
#include <vpi_user.h>


// user defines
// ************

#define LINE_LEN 1024
#define DPI_PRINT_DETAILS 1
#define DPI_PRINT_RAW     0

// user defined macros
// *******************

// structure definitions:
// **********************
// note: this struct is local in order to keep "lib_dpi.h" Sv-Free!
typedef struct 
{
  svScope Scope;  // scope pointer
  uint32_t clk_p; // clk period of that scope
}stScopeCtxt_t;

// external function prototypes
// ****************************

extern int testcase_main(void);

// global/static variables
// ***********************

static char*    ptmp;
static uint32_t ScopeIdx = 0;  // scope index counter
static char     tstline[LINE_LEN]; // temporyry line buffer
static char*    pTokens[10];      // pointer to tokens in scopename
static uint32_t ScopeTokens=0; // amount of tokens in scopename
static void*    so_file_handle[DPI_MAX_SCOPES]={NULL};
static uint32_t so_index=0;
static int32_t  so_file_id[DPI_MAX_SCOPES]={0};
static uint32_t so_file_uartcnt=0;
static uint32_t dpi_do_io_print =1;

static stScopeCtxt_t top_scope[DPI_MAX_SCOPES]; // struct for all scope informations
uint8_t scope_init_done=0;

// static functions
// ****************

static int32_t search_scope (svScope scope)
{
  uint32_t i;
  for(i=0;i<DPI_MAX_SCOPES;i++)
    {
      //debug_printf("Idx : %d %x scope : 0x%x\n",i,scope,top_scope[i].Scope);
      if(top_scope[i].Scope == scope)
	{
	  return i;
	}
    }
  return -1;
}

// conversion of systemverilog logic data type to uint32_t
static uint32_t logictouint32(svLogicVecVal* svData)
{
  uint32_t Data;

#ifdef V_IF_USE_SELECT_CONV

#else
  //  Data = svData->aval;
  Data = svData->a;
  //debug_printf("[C] logic2uint32 aval: 0x%x\n",svData->aval);
  //debug_printf("[C] logic2uint32 bval: 0x%x\n",svData->bval);

  //if(svData->bval != 0x0)
  if(svData->b != 0x0)
    {
      debug_printf("logic2uint32: Warning, X or Z detected !\n");
    }
#endif
  return Data;
}

// conversion of uint32_t  to system veriog logic data type
static uint32_t uint32tologic(svLogicVecVal* svData, uint32_t Value)
{

#ifdef V_IF_USE_SELECT_CONV
  uint32_t i;
  uint8_t bit;
  svLogic svbit;

  for(i=0;i<32;i++)
    {
      bit = (Value>>i)&0x1;
      svbit=bit;
      debug_printf("uint32tologic: converted to logic bitpos %d Data:%d\n",i,svbit);
      svPutSelectLogic(svData,i,svbit);
      //printf("svData.aval :0x%08x\n",svData->aval);
      //printf("svData.bval :0x%08x\n",svData->bval);
    }
#else
  //svData->aval = Value;
  //svData->bval = 0x0;

  svData->a = Value;
  svData->b = 0x0;

#endif

  return 0;
}

static int set_clk_p(void)
{
  svScope p_scope=NULL;
  uint32_t i;
  uint32_t time1,time2;
  _nop();
  time1= _get_time(BASE_UNIT);
  //debug_printf("time1: %d\n",time1);
  _nop();
  time2= _get_time(BASE_UNIT);
  //debug_printf("time2: %d\n",time2);

// connect clk period to dedicated scope
  p_scope = svGetScope();

  for(i=0;i<ScopeIdx;i++)
    {
      //debug_printf("p_scope: %x top_scope[%d]: %x\n",p_scope,i,top_scope[i].Scope);
      if(p_scope == top_scope[i].Scope)
	{
	  top_scope[i].clk_p = time2-time1;
          debug_printf("Scope Index: %d time added: %d ns\n",i,top_scope[i].clk_p);
	  return 0;
	}
    }
  
  debug_printf("Warning: clk_p for scope %x not inserted\n",p_scope);
  return -1;
}

// helper function for hierarchy search
static unsigned long c_dot_search(const char* str, char* pTok[])
{

	uint16_t i;

	// clean up scratch buffer
	i=0;
	while(i<LINE_LEN)
	{
		tstline[i]=' ';
		i++;
	}

	i=0;

	// copy whole string to scratch buffer
	do
	{
		tstline[i]=str[i];
		i++;
	}while(str[i] != '\0');

	tstline[i]='\0';

	ptmp = strtok (tstline,".:");

	// search for the "." or ":" token
	i=0;
	while (ptmp != NULL)
	{
		pTok[i]=ptmp;
		i++;
		ptmp= strtok (NULL, ".:");
	}

	return i;
}


static int _debug_printf (uint32_t Mode, const char *msg, ...)
{
  va_list ap;
  long long time_val;
  //uint64_t time_val;
  svScope p_scope;
  char ScopeName[128];    // scopename buffer
  const char * pName = &ScopeName[0];// pointer to scopename buffer

#ifdef USE_DIFFERENT_TOP_SCOPE
  svSetScope(top_scope[0].Scope);
#else
  p_scope = svGetScope();
  svSetScope(p_scope);
#endif

  //int i;

  //debug_printf("Simulation with C Started \n"); 
  
  // get the scope name to derive the path for the testcase.so-file
  pName = svGetNameFromScope(p_scope);

  //debug_printf("Scope ptr at Idx %d is : %x\n",ScopeIdx,p_scope); 
  //debug_printf("Scope name is: %s \n\n ",svGetNameFromScope(p_scope ));

  c_dot_search(pName,pTokens);

  va_start( ap, msg );

#ifdef V_IF_USE_RTL_PRINT // print to simulator terminal
  char     str[1024];
  vsprintf(str, msg, ap);
  HAL_puts(str);
#else // print to terminal

  //get_time_v(&time_val);
  //time_val = _get_time_base();
  time_val = _get_time(T_NSEC);
  if(Mode == DPI_PRINT_DETAILS)
    {
      vpi_printf("Time: %010llu ns %-20s [C] ",time_val,pTokens[1]);
    }
  vpi_vprintf( (PLI_BYTE8 *)msg, ap );
  //vpi_vprintf( msg, ap ); 
#endif

  va_end( ap );
  return 0;
}



// LIBRARY Functions:
// void* get_so_handle       // called from "server" in case we have f.e 2 uarts connected to 2 terminals
// int c_main(void)          // called from verilog module via testbench
// int c_vief_if_scope(void) // called from every dpi verilog module
// ******************


/*! \addtogroup FROM_SV
 * \brief C-Function which are called from Systemverilog, or calls Systemverilog functions
 *  \{
 */





/*! \fn int c_main(int * Value) 
 *  \brief main function called out of systemverilog. it opens the precompiled shared object and runs main testcase function inside.
 *        
 *
 *	\param[in] pointer to return value, since return value of DPI called c function has to be always 0 
 *      \return always 0
 */


int c_main(int * Value) 
{
  typedef int (*tc_function_t)(void);
 
  char  testname[128];
  int i;
  char* so_name=&testname[0];
  char* strptr=NULL;
  int   tc_exit_value;
  tc_function_t testcase_function;

  svScope p_scope;        // scope pointer from SV
  char ScopeName[128];    // scopename buffer

  const char * dlsym_error;
  const char * pName = &ScopeName[0];// pointer to scopename buffer

  debug_printf("Simulation with C Started \n"); 
  
  // derive path and open shared object
  // **********************************

  p_scope = svGetScope();
  pName = svGetNameFromScope(p_scope);

  debug_printf("Scope ptr at Idx %d is : %x\n",ScopeIdx,p_scope); 
  debug_printf("Scope name is: %s \n",svGetNameFromScope(p_scope ));

  ScopeTokens = c_dot_search(pName,pTokens);

  debug_printf("%d tokens found\n",ScopeTokens);
  debug_printf("%d inputValue\n",*Value);

  strcpy(so_name,"./");


  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] Scope name is: rf_4x4_tb.fcsi2_n_driver[0].fcsi2_n_driver_i 
  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] 3 tokens found
  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] 1 inputValue
  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] Opening Shared Library ./TESTCASE_libs/testcase_fcsi2_n_driver_0_.so

  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] Scope name is: rf_4x4_tb.spi_driver_i 
  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] 2 tokens found
  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] 1 inputValue
  //Time: 0000000000 ns fcsi2_n_driver[0]    [C] Opening Shared Library ./TESTCASE_libs/testcase_spi_driver_i.so


  if(*Value == 1) //we support new driver link dir structure
    {
      strcat(so_name,"TESTCASE_libs/");
      strcat(so_name,"testcase_");

      if(ScopeTokens == 3) //  if inputval =1 and a tc_libs sub directory for .so
        {
          strcat(so_name,pTokens[ScopeTokens-2]);

          // replace [ = _  &&  ] == _ 
          for(i=0;i<128;i++)
            {
              if(so_name[i] == '[' || so_name[i] == ']')
                {
                  so_name[i] = '_';
                } 
            }
      
        }

      else
        {
          strcat(so_name,pTokens[ScopeTokens-1]);
        }

      strcat(so_name,".so");
    }
  else
    {
      strcat(so_name,pTokens[ScopeTokens-1]);
      strcat(so_name,"/testcase.so");
    }


  debug_printf("Opening Shared Library %s\n",so_name);
  so_file_handle[so_index]=dlopen(so_name,RTLD_LAZY);


  // check if we have a "uart"
  // *************************

  strptr = strstr(so_name,"uart");
  if(strptr != NULL)
    {
      debug_printf("UART Device %d detected at so index: % d\n",so_file_uartcnt,so_index);
      so_file_id[so_index]=so_file_uartcnt;
      so_file_uartcnt++;
    }
  else 
    {
      so_file_id[so_index]=-1;
    }


  // check open retval and load "main"
  // *********************************

  if (!so_file_handle[so_index]) 
    {
      debug_printf("Warning: Cannot load Library %s handle: %x \n",so_name, so_file_handle[so_index]);
      debug_printf("dlerror: %s\n", dlerror() );

      *Value = -2;
      return 0;
    } 
  else 
    {
      debug_printf("Library opened %s handle: %x \n",so_name, so_file_handle[so_index]);
    
      debug_printf("Loading Sysmbol 'testcase_main'\n");
      testcase_function = (tc_function_t) dlsym(so_file_handle[so_index], "testcase_main");
      dlsym_error = dlerror();
      
      if (dlsym_error) 
	{
	  debug_printf("Cannot load symbol 'testcase_main' error:%d\n",dlsym_error);
	  dlclose(so_file_handle[so_index]);
	  *Value =-3;
	  return 0;
	} 
    }
  so_index++;
  
  _nop();      // wait for next rising edge
  set_clk_p(); // get/store clock periode for wait usage ! 

  // main function call
  // ******************

  tc_exit_value = testcase_function(); // main testcase entry
  //dlclose(handle);


  // return to SV
  // ************
  _nop();
  *Value = tc_exit_value;

  return 0;
}


/*! \fn int c_view_if_scope(void)
 * \brief this function is called from the systemverilog interface and registers each scope in a context
 * \return always 0 
 */

int c_view_if_scope(void)
{
  uint32_t i;
  svScope p_scope;
  //uint64_t time1;
  //uint64_t time2;

  p_scope = svGetScope();

  // initialize all the scopes at very first call
  if(scope_init_done == 0)
    {
      debug_printf("scope_view: running scope init\n");
      for(i=0;i<DPI_MAX_SCOPES; i++)
	{
	  top_scope[i].clk_p = 0;
	  top_scope[i].Scope = NULL;
	}
      scope_init_done =1;
    }


  if(ScopeIdx <  DPI_MAX_SCOPES)
    { 
      top_scope[ScopeIdx].Scope = p_scope;
      debug_printf("scope view: Scope ptr at Idx %d is : %x\n",ScopeIdx,p_scope); 
      debug_printf("scope view: Scope name is: %s \n",svGetNameFromScope(p_scope ));
      ScopeIdx++;

    }
  else 
    {
      debug_printf("Error: maximum amount of scopes reached\n");
    }
  return 0;


/*! \}
*/ 






/*! \fn void* get_so_handle(uint32_t Idx)
 *  \brief Function for delivering a shared object pointer to a given index, primarily used for communicating
 *         to dedicated uart shared object
 *
 *	\param[in] Idx of shared object to be adressed
 *      \return pointer to shared object
 */

void* get_so_handle(uint32_t Idx)
{
  uint32_t i;
  if(Idx<so_index)
    {
      // we are searching for the uart index
      for(i=0;i<so_index;i++)
        {
          if(so_file_id[i]==Idx)
            {
              return so_file_handle[i];
            }
        }
          //return so_file_handle[Idx];
    }
  return NULL;
}





/*! \fn uint32_t dpi_cnt_scopes(void) 
 *  \brief this functions shows the amount of registered drivers
 *      \return amount of registered drivers
 */


uint32_t dpi_cnt_scopes(void)
{
  return ScopeIdx;
}

/*! \fn int32_t dpi_show_scope(uint32_t Index, stScope_t* ScopeInfo)
 * \brief this functions gives a string pointer and a clock period information from all the drivers which are registered, 
 * and returns the id of the "current"
 *	\param[in] Idx of shared object from which infos should be stored to scopeinfo pointer
 *	\param[in] pointer to scopeinfo where infos are stored to
 *      \return index of registered driver ( as double check ) 
*/

int32_t dpi_show_scope(uint32_t Index, stScope_t* ScopeInfo)
{
  svScope p_scope;
  p_scope = svGetScope();

  ScopeInfo->Name  = (char *)svGetNameFromScope(top_scope[Index].Scope);
  ScopeInfo->clk_p = top_scope[Index].clk_p;

  return search_scope(p_scope);
}






}


/*! \fn uint64_t _get_time_base(void)
 *  \brief this function returns the current simulation time from systemverilog
 *  \return 64 bit simulation time in "base units"
 */

uint64_t _get_time_base(void)
{
  long long v_p_time;
  get_time_v(&v_p_time);
  //debug_printf("_get_time: received time: %13llu\n",v_p_time);
  return v_p_time;
}


/*! \addtogroup FROM_CL
 * \brief C-Function which are called from higher layer C
 *  \{

// global functions
// ****************

// /*! \fn int _set_clk_p(void);
 *  \brief This function triggers Clock period remeasurement
 *  \return 0:success -1:period not inserted in context
 */
int _set_clk_p(void)
{
  return set_clk_p();
}



/*! \fn int debug_printf (const char *msg, ...)
 * \brief this function does a printout on the simulation terminal with detailed module and time information
 * \param[in]: message which should be printed, variable argument length 
 * \return always 0 
 */

int debug_printf (const char *msg, ...)
{

  char myString[1024];
  va_list    ap;

  va_start(ap,msg);
  vsprintf(myString,msg,ap);
  _debug_printf(DPI_PRINT_DETAILS, "%s",myString);
  va_end(ap);

  return 0;
}


/*! \fn int raw_printf (const char *msg, ...)
 * \brief this function does a printout on the simulation terminal \b without any additional information
 * \param[in]: message which should be printed, variable argument length 
 * \return always 0 
 */

int raw_printf (const char *msg, ...)
{
  char myString[1024];
  va_list    ap;

  va_start(ap,msg);
  vsprintf(myString,msg,ap);
  _debug_printf(DPI_PRINT_RAW, "%s",myString);
  va_end(ap);

  return 0;
}




// IO and Interface ROUTINES, entry point from C to verilog!
// all _v function are tasks inside verilog and exported to C 
// **********************************************************



/*! \fn void _wait(uint32_t cnt, uint32_t unit)
 * \brief this function waits for a given amount of units, aligned to clock edges
 * \param[in]: Amount of units to wait
 * \param[in]: units (T_MSEC,T_USEC,T_NSEC)
 */
  
void _ABB_wait(uint32_t cnt, uint32_t unit)
{
  
  //uint32_t wait_val;

  svScope p_scope;

#ifdef USE_DIFFERENT_TOP_SCOPE
  svSetScope(top_scope[0].Scope);
#else
  p_scope = svGetScope();
  svSetScope(p_scope);
#endif

  if(unit< T_NSEC || unit > T_SEC)
    {
      debug_printf("Error, invalid wait unit :%d\n",unit);
    }
  else 
    {
#ifdef V_IF_NO_UNIT_WAIT
      uint64_t clk_p;
      uint8_t wait_cnt;
      uint64_t wait_val;
      uint32_t i;
      uint32_t clk_idx;
      
      clk_idx = search_scope(p_scope);
      if(clk_idx == -1)
	{
	  debug_printf("Error, cant wait for that scope: %x !\n",p_scope);
	  return;
	}
      //else 
      //{
      //  debug_printf("Scope found on index %d\n",clk_idx);
      //}
      
      //debug_printf("calling wait with units %d cnt %d\n",unit, cnt);

      //clk_p = top_scope[0].clk_p *2; // unit is BASE_UNIT (fs is default) // TODO *2 is wrong here ?
      clk_p = top_scope[clk_idx].clk_p;    // unit is BASE_UNIT (ps is default) // TODO *2 is wrong here ?
      //debug_printf("wait_cnt = %d\n",clk_p);      

      wait_cnt = unit - BASE_UNIT -1; // we assume that base unit is psec, but smallest waits with NSEC only !
      //debug_printf("wait_cnt = %d\n",wait_cnt);

      clk_p = clk_p /1000;
      wait_val=1;
      for(i=0;i<wait_cnt;i++)
	{
	  wait_val = wait_val *1000;
	}

      wait_val = wait_val *cnt;
      //debug_printf("we will wait for %d fs\n",wait_val);

      //debug_printf("wait_val = %d\n",wait_val);
      wait_val = wait_val / clk_p;
      
      //debug_printf("clk_p = %d\n",clk_p);
      //debug_printf("wait_val new = %d\n",wait_val);
      for(i=0;i<wait_val;i++)
	{
	  _nop();
	}
#else
      wait_val = cnt;
      debug_printf("calling wait with units %d cnt %d\n",unit, wait_val);
      wait_v(unit,wait_val);
#endif
      
    }
}

/*! \fn void dpi_io_config(uint32_t Idx, uint32_t Value)
 * \brief this function configures printouts on dpi lib layer
 * \param[in]: Idx (DPI_CFG_PRINT_IDX)
 * \param[in]: Value (0,1)
 */

void dpi_io_config(uint32_t Idx, uint32_t Value)
{
  if(Idx == DPI_CFG_PRINT_IDX)
    {
      dpi_do_io_print=Value;
    }
}


/*! \fn void dpi_io_v_config(uint32_t Value)
 * \brief this function enables/disables printouts on SV layer
 * \param[in]: Value (0,1)
 */

void dpi_io_v_config(uint32_t Value)
{
  io_print_cfg_v(Value); 
  
}




// C - DPI -API function


/*! \fn void _port_set(uint32_t PortId, uint32_t level)
 * \brief this function sets a connected benchport to a given level and waits one clock cycle 
 * \param[in]: PortId 
 * \param[in]: level: level can be 0,1,LOGIC_X,LOGIC_Z
 */

void _port_set(uint32_t PortId, uint32_t level)
{
  _bench_write(PortId,level);
  _nop();
}

/*! \fn uint32_t _port_get(uint32_t PortId)
 * \brief this function reads the level of a given port and waits one clock cycle 
 * \param[in]: PortId
 * \return: level can be 0,1,LOGIC_X,LOGIC_Z
 */


uint32_t _port_get(uint32_t PortId)
{
  uint32_t value;
  value = _bench_read(PortId);
  _nop();
  return value;
}

/*! \fn void _port_write(uint32_t PortId, uint32_t level)
 * \brief this function sets a connected benchport to a given level \b without waiting 
 * \param[in]: PortId 
 * \param[in]: level: level can be 0,1,LOGIC_X,LOGIC_Z
 */

void _port_write(uint32_t PortId, uint32_t level)
{
  _bench_write(PortId,level);
}


/*! \fn uint32_t _port_read(uint32_t PortId)
 * \brief this function reads the level of a given port \b without waiting
 * \param[in]: PortId
 * \return: level can be 0,1,LOGIC_X,LOGIC_Z
 */

uint32_t _port_read(uint32_t PortId)
{
  uint32_t value;
  value = _bench_read(PortId);
  return value;
}



/*! \fn uint32_t _port_wait(uint32_t Addr)
 * \brief this function waits for value "1" on a dedicated port 
 * \param[in]: PortId
 */

void _port_wait(uint32_t PortId)
{
  while (_port_read(PortId) == 0) 
    {
      _nop();
    }
}

/*! \fn uint32_t _in(uint32_t Addr)
 * \brief this function does the Read from an given Address 
 * \param[in]: 32bit Address
 * \return: 32bit Data
 */

uint32_t _in(uint32_t Addr)
{
  svLogicVecVal svAddr;
  svLogicVecVal svData;
  uint32_t Data;
  int returnval;
  svScope p_scope;

#ifdef USE_DIFFERENT_TOP_SCOPE
  svSetScope(top_scope[0].Scope);
#else
  p_scope = svGetScope();
  svSetScope(p_scope);
#endif

  // convert the Address to logic format
  uint32tologic(&svAddr,Addr);

  // call verilog function for "read"
  io_read_v     (&svAddr,   &svData , &returnval);

  // convert the read result back to uint32_t
  Data = logictouint32(&svData);
  if(dpi_do_io_print == 1)
    {
      debug_printf("in : Addr: 0x%08x , Data: 0x%08x\n",Addr,Data);
    }

  if(returnval<0)
    {
      debug_printf("Error: Module is not accessable\n");
    }

  return Data;
}

/*! \fn void _out(uint32_t Addr, uint32_t Data)
 * \brief this function does the Write to an given Address 
 * \param[in]: 32bit Address
 * \param[in]: 32bit Data
 */

void _out (uint32_t Addr, uint32_t Data)
{
  svLogicVecVal svAddr;
  svLogicVecVal svData;
  int returnval;

  svScope p_scope;

#ifdef USE_DIFFERENT_TOP_SCOPE
  svSetScope(top_scope[0].Scope);
#else
  p_scope = svGetScope();
  svSetScope(p_scope);
#endif

  if(dpi_do_io_print == 1)
    {
      debug_printf("out: Addr: 0x%08x , Data: 0x%08x\n",Addr,Data);
    }

  // convert to logic format
  uint32tologic(&svAddr,Addr);
  uint32tologic(&svData,Data);

  // call verilog function for write !
  io_write_v(&svAddr,&svData,&returnval);

  if(returnval<0)
    {
      debug_printf("Error: Module is not accessable\n");
    }
}


/*! \fn uint32_t _int_get(void)
 * \brief Compatibility function \sa _get_int
 */


uint32_t _int_get(void)
{
  svScope p_scope;
  svLogicVecVal svData;
  uint32_t Data;

#ifdef USE_DIFFERENT_TOP_SCOPE
  svSetScope(top_scope[0].Scope);
#else
  p_scope = svGetScope();
  svSetScope(p_scope);
#endif


  io_int_v(&svData);

  Data = logictouint32(&svData);

  if(dpi_do_io_print == 1)
    {
      debug_printf("int get: Data: 0x%08x\n",Data);
    }

  return Data;
}


/*! \fn uint32_t _get_int(void)
 * \brief Abandoned: this function reads the level of all interupts (max 32) \b without waiting
 * \return: 2 value logic vector
 */


uint32_t _get_int(void)
{
  return _int_get();
}



/*! \fn uint32_t _get_time(uint32_t unit)
 * \brief this function calculates the simulation time, according to the slected units
 * \param[in]: time units \sa T_NSEC
 * \return: 32 bit simulation time value
 */


uint32_t _get_time(uint32_t unit)
{
  uint64_t time;
  uint32_t unitcnt,i;
  time = _get_time_base(); // base unit is ps ?
  
  unitcnt = unit - BASE_UNIT;

  for(i=0;i<unitcnt;i++)
    {
      time = time/1000;
    }
  return time;
}


/*! \fn int _nop(void)
 * \brief this function waits a single clock cycle
 * \return: always 0
 */

int _nop(void)
{
  svScope p_scope;
#ifdef USE_DIFFERENT_TOP_SCOPE
  svSetScope(top_scope[0].Scope);
#else
  p_scope = svGetScope();
  svSetScope(p_scope);
#endif

  nop_v();
  return 0;
}


/*! \fn int32_t _config_set(int index, int value)
 * \brief this function calls sv config set function with input params
 * \param[in]:index: any configuration index
 * \param[in]: any configuration value
 * \return: always 0
 */
int32_t _config_set(int index, int value)
{
  io_config_set_v(index,value);
  return 0;
}

/*! \fn int32_t _config_get(int index)
 * \brief this function calls sv config get function with input params
 * \param[in]:index: any configuration index
 * \return: configuration value
 */
int32_t _config_get(int index)
{
  int32_t Retval;
  io_config_get_v(index,&Retval);
  return Retval;
}


/*! \fn int32_t _bench_write(int port_id,int level)
 * \brief Compatibility function \sa _port_write
 */

int32_t _bench_write(int port_id,int level)
{
  //debug_printf("Sorry, benchports -write currently not supported\n");
  io_port_set_v(port_id,level);

  return 0;
  
}

/*! \fn int32_t _bench_read(int port_id)
 * \brief Compatibility function \sa _port_read
 */

int32_t _bench_read(int port_id)
{
  int32_t Value;
  //debug_printf("Sorry, benchports -write currently not supported\n");
  io_port_get_v(port_id, &Value);
  return Value;
  //return 0;
}

/*! \fn void _ports_set(uint32_t PortNum, uint32_t Width, uint32_t Value)
 * \brief this function sets a group of benchports and waits for one clock cycle
 * \param[in]: PortNum: lower id of portgroup which should be set
 * \param[in]: Width: width/amount of ports which should be set
 * \param[in]: Value: 2 value logic vector which should be set to the portgroup
 */

void _ports_set(uint32_t PortNum, uint32_t Width, uint32_t Value)
{
  int i;
  for(i=0;i<Width;i++)
    {
      _bench_write(PortNum+i,(Value>>i)&1);
    }
  _nop();
}


/*! \fn uint32_t _ports_get(uint32_t PortNum, uint32_t Width)
 * \brief this function reads a group of benchports and waits for one clock cycle
 * \param[in]: PortNum: lower id of portgroup which should be read
 * \param[in]: Width: width/amount of ports which should be read
 * \return: Value: 2 value logic vector which was read from the portgroup
 */

uint32_t _ports_get(uint32_t PortNum, uint32_t Width)
{
  uint32_t Value=0x0;
  uint32_t PortIn;
  int i;

  for(i=0;i<Width;i++)
    {
      PortIn = _bench_read(PortNum+i);
      if(PortIn >= LOGIC_X)
	{
	  debug_printf("Warning, _ports_get: Port[%d]:  X or Z detected\n",PortNum+i);
	  PortIn = 1;
	}
      Value |= PortIn<<i;
    }
  _nop();
  return Value;
}



/*! \}
*/ 


/*! \addtogroup COMP_CL
 * \brief C-Functions which are compatible to prototypes of older busdriver versions
 *  \{
 */

// *******************************
//    COMPATIBILITY SECTION
// *******************************

// compatibility functions to GPON-Busdriver

/*! \fn void _io_write(uint32_t Addr, uint32_t Data)
 * \brief Compatibility function 
 * \sa _out
 */

void _io_write(uint32_t Addr, uint32_t Data)
{
  _out(Addr,Data);
}

/*! \fn uint32_t _io_read(uint32_t Addr)
 * \brief Compatibility function 
 * \sa _in
 */

uint32_t _io_read(uint32_t Addr)
{
  return _in(Addr);
}


// compatibility functions for DFEV-Busdriver 

/*! \fn void c_io_bus_write(uint32_t Addr, uint32_t Data)
 * \brief Compatibility function 
 * \sa _out
 */

void c_io_bus_write(uint32_t Addr, uint32_t Data)
{
  _out(Addr,Data);
}


/*! \fn uint32_t c_io_bus_read(uint32_t Addr)
 * \brief Compatibility function 
 * \sa _in
 */

uint32_t c_io_bus_read(uint32_t Addr)
{
  return _in(Addr);
}
// end of compatibility section!

/*! \}
*/


int _burst(stBurst_t * AccessTable)
{
  io_master_module__AccessTable_t Table;

  Table._a0 = AccessTable->len;

  // TODO: go over loop til len ?

  //pTable = (io_master_module__AccessTable_t *)&AccessTable
  debug_printf("burst len is: %d\n", Table._a0); 

  io_burst_v(&Table);
  return 0;
}
