/*
 * lib_wav600_api.c
 *
 *  Created on: Sep 3, 2017
 *      Author: delishan
 */
#include "lib_wav600_api.h"
#ifdef VLSI_PLATFORM
	#include "rfsys_platform.c"
#endif

#define UNUSED_PARAM(pMsg) do { (void)(pMsg); } while (0)

// ######################################################################################################
// # COS LUT																    																	         #
// ######################################################################################################
int16_t cosLUT[COS_LUT_LENGTH] =
{
	0x2000, 
	0x2000, 
	0x1FFF, 
	0x1FFF, 
	0x1FFE, 
	0x1FFC, 
	0x1FFA, 
	0x1FF8, 
	0x1FF6, 
	0x1FF4, 
	0x1FF1, 
	0x1FED, 
	0x1FEA, 
	0x1FE6, 
	0x1FE2, 
	0x1FDD, 
	0x1FD9, 
	0x1FD3, 
	0x1FCE, 
	0x1FC8, 
	0x1FC2, 
	0x1FBC, 
	0x1FB5, 
	0x1FAF, 
	0x1FA7, 
	0x1FA0, 
	0x1F98, 
	0x1F90, 
	0x1F87, 
	0x1F7F, 
	0x1F76, 
	0x1F6C, 
	0x1F63, 
	0x1F59, 
	0x1F4E, 
	0x1F44, 
	0x1F39, 
	0x1F2E, 
	0x1F22, 
	0x1F17, 
	0x1F0A, 
	0x1EFE, 
	0x1EF1, 
	0x1EE5, 
	0x1ED7, 
	0x1ECA, 
	0x1EBC, 
	0x1EAE, 
	0x1E9F, 
	0x1E91, 
	0x1E81, 
	0x1E72, 
	0x1E63, 
	0x1E53, 
	0x1E42, 
	0x1E32, 
	0x1E21, 
	0x1E10, 
	0x1DFF, 
	0x1DED, 
	0x1DDB, 
	0x1DC9, 
	0x1DB6, 
	0x1DA4, 
	0x1D90, 
	0x1D7D, 
	0x1D69, 
	0x1D55, 
	0x1D41, 
	0x1D2D, 
	0x1D18, 
	0x1D03, 
	0x1CED, 
	0x1CD8, 
	0x1CC2, 
	0x1CAC, 
	0x1C95, 
	0x1C7F, 
	0x1C68, 
	0x1C50, 
	0x1C39, 
	0x1C21, 
	0x1C09, 
	0x1BF0, 
	0x1BD8, 
	0x1BBF, 
	0x1BA6, 
	0x1B8C, 
	0x1B73, 
	0x1B59, 
	0x1B3E, 
	0x1B24, 
	0x1B09, 
	0x1AEE, 
	0x1AD3, 
	0x1AB7, 
	0x1A9B, 
	0x1A7F, 
	0x1A63, 
	0x1A46, 
	0x1A2A, 
	0x1A0D, 
	0x19EF, 
	0x19D2, 
	0x19B4, 
	0x1996, 
	0x1977, 
	0x1959, 
	0x193A, 
	0x191B, 
	0x18FC, 
	0x18DC, 
	0x18BD, 
	0x189C, 
	0x187C, 
	0x185C, 
	0x183B, 
	0x181A, 
	0x17F9, 
	0x17D8, 
	0x17B6, 
	0x1794, 
	0x1772, 
	0x1750, 
	0x172D, 
	0x170A, 
	0x16E7, 
	0x16C4, 
	0x16A1, 
	0x167D, 
	0x1659, 
	0x1635, 
	0x1611, 
	0x15EC, 
	0x15C7, 
	0x15A3, 
	0x157D, 
	0x1558, 
	0x1533, 
	0x150D, 
	0x14E7, 
	0x14C1, 
	0x149A, 
	0x1474, 
	0x144D, 
	0x1426, 
	0x13FF, 
	0x13D8, 
	0x13B0, 
	0x1388, 
	0x1360, 
	0x1338, 
	0x1310, 
	0x12E8, 
	0x12BF, 
	0x1296, 
	0x126D, 
	0x1244, 
	0x121A, 
	0x11F1, 
	0x11C7, 
	0x119D, 
	0x1173, 
	0x1149, 
	0x111F, 
	0x10F4, 
	0x10C9, 
	0x109F, 
	0x1074, 
	0x1048, 
	0x101D, 
	0x0FF1, 
	0x0FC6, 
	0x0F9A, 
	0x0F6E, 
	0x0F42, 
	0x0F16, 
	0x0EE9, 
	0x0EBD, 
	0x0E90, 
	0x0E63, 
	0x0E36, 
	0x0E09, 
	0x0DDC, 
	0x0DAF, 
	0x0D81, 
	0x0D53, 
	0x0D26, 
	0x0CF8, 
	0x0CCA, 
	0x0C9C, 
	0x0C6D, 
	0x0C3F, 
	0x0C10, 
	0x0BE2, 
	0x0BB3, 
	0x0B84, 
	0x0B55, 
	0x0B26, 
	0x0AF7, 
	0x0AC8, 
	0x0A98, 
	0x0A69, 
	0x0A39, 
	0x0A0A, 
	0x09DA, 
	0x09AA, 
	0x097A, 
	0x094A, 
	0x091A, 
	0x08EA, 
	0x08B9, 
	0x0889, 
	0x0858, 
	0x0828, 
	0x07F7, 
	0x07C6, 
	0x0796, 
	0x0765, 
	0x0734, 
	0x0703, 
	0x06D2, 
	0x06A1, 
	0x066F, 
	0x063E, 
	0x060D, 
	0x05DB, 
	0x05AA, 
	0x0579, 
	0x0547, 
	0x0515, 
	0x04E4, 
	0x04B2, 
	0x0480, 
	0x044E, 
	0x041D, 
	0x03EB, 
	0x03B9, 
	0x0387, 
	0x0355, 
	0x0323, 
	0x02F1, 
	0x02BF, 
	0x028D, 
	0x025B, 
	0x0229, 
	0x01F6, 
	0x01C4, 
	0x0192, 
	0x0160, 
	0x012E, 
	0x00FB, 
	0x00C9, 
	0x0097, 
	0x0065, 
	0x0032
};

int16_t get_cos_value(int16_t tone)
{
	int16_t N = FFT_LENGTH;
	int16_t k; 
	int16_t Sign = 1;
	int16_t Result = 0;

	while (tone < 0)
		tone += N;
	while (tone >= N)
		tone -= N;
	
	if (tone <= (N/4)) 		     // First Quadrant 
	{
		k = tone;
	}
	else if (tone <= (N/2))  	 // Second Quadrant 
	{
		k = N/2 - tone;
		Sign = -1;
	}
	else if (tone <= (3*(N/4))) // Third Quadrant 
	{
		k = tone - (N/2);
		Sign = -1;
	}
	else
	{
		k = N - tone; 			 // Fourth Quadrant 
	}

	if (k < COS_LUT_LENGTH)
	{
		Result = cosLUT[k];
	}
	else
	{
		Result = 0;
	}

	return Result * Sign;
}

int16_t	get_sin_value(int16_t tone)
{
	return  get_cos_value(tone - COS_LUT_LENGTH);
}

int32_t	matlab_test(int16_t input)
{
	return 1;
}

// ######################################################################################################
// # Segment (band) configurations																    	#
// ######################################################################################################

void bbic_get_rx_ant_mask (uint8_t* AntMsk_rtrn, uint8_t* NumOfAnt_rtrn)
{
	RegPhyRxTdPhyRxtdReg07B_u Reg_Host_Antenna_Enable;// REG_PHY_RX_TD_PHY_RXTD_REG07B 0x1EC
	uint8_t i;
	// read the antenna mask
	Reg_Host_Antenna_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG07B);
	*AntMsk_rtrn = Reg_Host_Antenna_Enable.bitFields.hostAntennaEn;
	// calculate number of active antennas
	*NumOfAnt_rtrn = 0;
	for (i=0; i<4; i++)
		if (*AntMsk_rtrn & 1<<i) (*NumOfAnt_rtrn)++;
}

// # AFE Standby

void bbic_rxon (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable; // REG_PHY_RX_TD_IF_AFE_DYN_CTRL
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn ON RX bits according to the antenna mask
	if (AntMsk & 1<<0) Reg_AFE_Enable.bitFields.rx0On = 1;
	if (AntMsk & 1<<1) Reg_AFE_Enable.bitFields.rx1On = 1;
	if (AntMsk & 1<<2) Reg_AFE_Enable.bitFields.rx2On = 1;
	if (AntMsk & 1<<3) Reg_AFE_Enable.bitFields.rx3On = 1;
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_rxoff (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable;
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn OFF RX bits according to the antenna mask
	if (AntMsk & 1<<0) Reg_AFE_Enable.bitFields.rx0On = 0;
	if (AntMsk & 1<<1) Reg_AFE_Enable.bitFields.rx1On = 0;
	if (AntMsk & 1<<2) Reg_AFE_Enable.bitFields.rx2On = 0;
	if (AntMsk & 1<<3) Reg_AFE_Enable.bitFields.rx3On = 0;
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_txon (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable;
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn ON TX bits according to the antenna mask
	if (AntMsk & 1<<0) Reg_AFE_Enable.bitFields.tx0On = 1;
	if (AntMsk & 1<<1) Reg_AFE_Enable.bitFields.tx1On = 1;
	if (AntMsk & 1<<2) Reg_AFE_Enable.bitFields.tx2On = 1;
	if (AntMsk & 1<<3) Reg_AFE_Enable.bitFields.tx3On = 1;
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_txoff (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable;
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn OFF TX bits according to the antenna mask
	if (AntMsk & 1<<0) Reg_AFE_Enable.bitFields.tx0On = 0;
	if (AntMsk & 1<<1) Reg_AFE_Enable.bitFields.tx1On = 0;
	if (AntMsk & 1<<2) Reg_AFE_Enable.bitFields.tx2On = 0;
	if (AntMsk & 1<<3) Reg_AFE_Enable.bitFields.tx3On = 0;
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_auxadc_on (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable; // REG_PHY_RX_TD_IF_AFE_DYN_CTRL
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn ON RX bits according to the antenna mask
	if (AntMsk & 1<<0) Reg_AFE_Enable.bitFields.aux0AdcOn = 1;
	if (AntMsk & 1<<1) Reg_AFE_Enable.bitFields.aux1AdcOn = 1;
	if (AntMsk & 1<<2) Reg_AFE_Enable.bitFields.aux2AdcOn = 1;
	if (AntMsk & 1<<3) Reg_AFE_Enable.bitFields.aux3AdcOn = 1;
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_auxadc_off (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable;
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn OFF RX bits according to the antenna mask
	if (AntMsk & 1<<0) Reg_AFE_Enable.bitFields.aux0AdcOn = 0;
	if (AntMsk & 1<<1) Reg_AFE_Enable.bitFields.aux1AdcOn = 0;
	if (AntMsk & 1<<2) Reg_AFE_Enable.bitFields.aux2AdcOn = 0;
	if (AntMsk & 1<<3) Reg_AFE_Enable.bitFields.aux3AdcOn = 0;
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_rxon_all (uint8_t AntMsk)
{
	RegPhyRxTdIfAfeDynCtrl_u Reg_AFE_Enable; // REG_PHY_RX_TD_IF_AFE_DYN_CTRL
	// read the current AFE state
	Reg_AFE_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_AFE_DYN_CTRL);
	// turn ON RX bits according to the antenna mask
	if (AntMsk & 1<<0) {
		Reg_AFE_Enable.bitFields.rx0On = 1;
		Reg_AFE_Enable.bitFields.aux0AdcOn = 1;
	}
	if (AntMsk & 1<<1) {
		Reg_AFE_Enable.bitFields.rx1On = 1;
		Reg_AFE_Enable.bitFields.aux1AdcOn = 1;
	}
	if (AntMsk & 1<<2) {
		Reg_AFE_Enable.bitFields.rx2On = 1;
		Reg_AFE_Enable.bitFields.aux2AdcOn = 1;
	}
	if (AntMsk & 1<<3) {
		Reg_AFE_Enable.bitFields.rx3On = 1;
		Reg_AFE_Enable.bitFields.aux3AdcOn = 1;
	}
	// write the state to AFE
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_AFE_DYN_CTRL, Reg_AFE_Enable.val);
}

void bbic_tx_reset(void)
{
	RegPhyTxTxSwReset_u Reg_Tx_Reset_Save; 	// REG_PHY_TX_TX_SW_RESET 0x0
	RegPhyTxTxBeReg06_u Reg_Tx_BE_Reset_Save;	// REG_PHY_TX_TX_BE_REG_06 0x18
	Reg_Tx_Reset_Save.val = BBIC_Reg_Read(REG_PHY_TX_TX_SW_RESET);
	Reg_Tx_BE_Reset_Save.val = BBIC_Reg_Read(REG_PHY_TX_TX_BE_REG_06);
	BBIC_Reg_Write(REG_PHY_TX_TX_SW_RESET, 0x00000000);
	BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_06, 0x00000000);
	BBIC_Reg_Write(REG_PHY_TX_TX_SW_RESET, Reg_Tx_Reset_Save.val);
	BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_06, Reg_Tx_BE_Reset_Save.val);
}
void bbic_set_tx_reset(void)
{
	//RegPhyTxTxSwReset_u Reg_Tx_Reset_Save; 	// REG_PHY_TX_TX_SW_RESET 0x0
	//RegPhyTxTxBeReg06_u Reg_Tx_BE_Reset_Save;	// REG_PHY_TX_TX_BE_REG_06 0x18
	BBIC_Reg_Write(REG_PHY_TX_TX_SW_RESET, 0x00000000);
	BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_06, 0x00000000);
}
void bbic_release_tx_reset(void)
{
	//RegPhyTxTxSwReset_u Reg_Tx_Reset_Save; 	// REG_PHY_TX_TX_SW_RESET 0x0
	//RegPhyTxTxBeReg06_u Reg_Tx_BE_Reset_Save;	// REG_PHY_TX_TX_BE_REG_06 0x18
	BBIC_Reg_Write(REG_PHY_TX_TX_SW_RESET, 0xFFFFFFFF);
	BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_06, 0xFFFFFFFF);
}
void bbic_set_endless_tx_reset(void)
{
	//RegPhyTxTxSwReset_u Reg_Tx_Reset_Save; 	// REG_PHY_TX_TX_SW_RESET 0x0
	//RegPhyTxTxBeReg06_u Reg_Tx_BE_Reset_Save;	// REG_PHY_TX_TX_BE_REG_06 0x18
	BBIC_Reg_Write(REG_PHY_TX_TX_SW_RESET, 0x00000000);
	//HW said to remove BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_06, 0x00000000);
}

// ######################################################################################################
// # RXIQ Compensation & Calibration blocks																#							#
// ######################################################################################################

//-------------------------------------------------------------------------------------------------------
// IQRAM API
//-------------------------------------------------------------------------------------------------------
void bbic_iqram_read_trigger(uint8_t AntMsk)
{
	// Sends a pulse to trigger the IQRAM read operation
	RegPhyRxtdAnt0PhyRxtdAntReg28_u Reg_RAM_Read_Trigger; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG28
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_RAM_Read_Trigger.val = 0;
			Reg_RAM_Read_Trigger.bitFields.iqRamReadTrigger = 1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG28 + i*(ANT_OFFSET), Reg_RAM_Read_Trigger.val);
		}
	}
}

void bbic_bypass_ram_xtalk(uint8_t AntMsk, BOOL BypassRamMode)
{
	RegPhyRxtdAnt0PhyRxtdAntReg65_u Reg_XTalk_Ram_Bypass; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 0x194;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_XTalk_Ram_Bypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET));
			Reg_XTalk_Ram_Bypass.bitFields.iqRamXtalkBypass = BypassRamMode;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET), Reg_XTalk_Ram_Bypass.val);
		}
	}
}

void bbic_bypass_ram_fdl(uint8_t AntMsk, BOOL BypassRamMode)
{
	RegPhyRxtdAnt0PhyRxtdAntReg24_u Reg_Fdl_Ram_Bypass; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG24 0x90;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Fdl_Ram_Bypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG24 + i*(ANT_OFFSET));
			Reg_Fdl_Ram_Bypass.bitFields.iqFdlBypassRam = BypassRamMode;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG24 + i*(ANT_OFFSET), Reg_Fdl_Ram_Bypass.val);
		}
	}
}
/*
void bbic_block_write(uint8_t AntMsk, uint32_t BaseOffset, uint32_t Array[][ANT_NUM], uint16_t ArraySize, uint8_t WordLength)
{
	// fill contiguous area in the memory starting from BaseOffset, at 4 bytes offsets, limiting the row width to WordLength bits
	int AntIdx, RowIdx;
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
		if (AntMsk & 1<<AntIdx)
			for (RowIdx=0; RowIdx<ArraySize; RowIdx++)
				BBIC_Reg_Write(BaseOffset + AntIdx*(ANT_OFFSET) + 4*RowIdx, Array[RowIdx][AntIdx] & ((1<<WordLength) - 1));
}

void bbic_block_read(uint8_t AntMsk, uint32_t BaseOffset, uint32_t Array[][ANT_NUM], uint16_t ArraySize)
{
	// read from contiguous area in the memory starting from BaseOffset, at 4 bytes offsets
	int AntIdx, RowIdx;
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
		if (AntMsk & 1<<AntIdx)
			for (RowIdx=0; RowIdx<ArraySize; RowIdx++)
				Array[RowIdx][AntIdx] = BBIC_Reg_Read(BaseOffset + AntIdx*(ANT_OFFSET) + 4*RowIdx);
}
*/
void bbic_write_ram_fdl(uint8_t AntMsk, uint8_t BwIdx, int16_t CoeffArray[PGC_GAIN_NUM][ANT_NUM])
{
	// fill the RXIQ RAM FDL section according to the BwIdx (0,1,2,3 for 20/40/80/160MHz respectively)
	uint32_t FdlBaseOffset = RAM_FDL_XTALK_BASE_ADDRESS + 4*PGC_GAIN_NUM*BwIdx;
	uint32_t RamRow;
	uint8_t Bits = 12;
	int AntIdx, RowIdx;
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
	{
		if (AntMsk & 1<<AntIdx) 
		{
			for (RowIdx=0; RowIdx<PGC_GAIN_NUM; RowIdx++)
			{
				RamRow = fxp_limit_sign((int32_t)CoeffArray[RowIdx][AntIdx], Bits);
				BBIC_Reg_Write(FdlBaseOffset + AntIdx*(ANT_OFFSET) + 4*RowIdx, RamRow & ((1<<Bits) - 1));
			}
		}
	}
}

void bbic_read_ram_fdl(uint8_t AntMsk, uint8_t BwIdx, int16_t CoeffArray_rtrn[PGC_GAIN_NUM][ANT_NUM])
{
	// read RXIQ RAM FDL section according to the BwIdx (0,1,2,3 for 20/40/80/160MHz respectively)
	uint32_t FdlBaseOffset = RAM_FDL_XTALK_BASE_ADDRESS + 4*PGC_GAIN_NUM*BwIdx;
	union
	{
		uint32_t val;
		struct
		{
			int16_t FdlCoeff:12;
			uint32_t NA:20;
		} ramFields;
	} RamRow;

	int AntIdx, RowIdx;
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
	{
		if (AntMsk & 1<<AntIdx) 
		{		
			for (RowIdx=0; RowIdx<PGC_GAIN_NUM; RowIdx++)
			{
				RamRow.val = BBIC_Reg_Read(FdlBaseOffset + AntIdx*(ANT_OFFSET) + 4*RowIdx);
				CoeffArray_rtrn[RowIdx][AntIdx] = RamRow.ramFields.FdlCoeff;
			}
		}
	}
}

void bbic_write_ram_xtalk(uint8_t AntMsk, uint8_t BwIdx, int8_t CoeffArrayKa[PGC_GAIN_NUM][ANT_NUM], int8_t CoeffArrayKb[PGC_GAIN_NUM][ANT_NUM])
{
	// fill the RXIQ RAM XTalk section according to the BwIdx (0,1,2,3 for 20/40/80/160MHz respectively)
	uint32_t XtalkBaseOffset = RAM_FDL_XTALK_BASE_ADDRESS + 4*PGC_GAIN_NUM*BW_NUM + 4*PGC_GAIN_NUM*(3-BwIdx);
	uint32_t RamRow;
	int16_t Ka, Kb;
	uint8_t Bits = 6;
	int AntIdx, RowIdx;
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			for (RowIdx=0; RowIdx<PGC_GAIN_NUM; RowIdx++)
			{
				Ka = fxp_limit_sign(CoeffArrayKa[RowIdx][AntIdx], Bits) & ((1<<Bits) - 1);
				Kb = fxp_limit_sign(CoeffArrayKb[RowIdx][AntIdx], Bits) & ((1<<Bits) - 1);
				RamRow = (1 << Bits) * Ka + Kb;
				BBIC_Reg_Write(XtalkBaseOffset + AntIdx*(ANT_OFFSET) + 4*RowIdx, RamRow & ((1<<2*Bits) - 1));
			}
		}
	}
}

void bbic_read_ram_xtalk(uint8_t AntMsk, uint8_t BwIdx, int8_t CoeffArrayKa_rtrn[PGC_GAIN_NUM][ANT_NUM], int8_t CoeffArrayKb_rtrn[PGC_GAIN_NUM][ANT_NUM])
{
	// read the RXIQ RAM XTalk section according to the BwIdx (0,1,2,3 for 20/40/80/160MHz respectively)
	uint32_t XtalkBaseOffset = RAM_FDL_XTALK_BASE_ADDRESS + 4*PGC_GAIN_NUM*BW_NUM + 4*PGC_GAIN_NUM*(3-BwIdx);
	union
	{
		uint32_t val;
		struct
		{
			int32_t Kb:6;
			int32_t Ka:6;
			uint32_t NA:20;
		} ramFields;
	} RamRow;

	int AntIdx, RowIdx;
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			for (RowIdx=0; RowIdx<PGC_GAIN_NUM; RowIdx++)
			{
				RamRow.val = BBIC_Reg_Read(XtalkBaseOffset + AntIdx*(ANT_OFFSET) + 4*RowIdx);
				CoeffArrayKa_rtrn[RowIdx][AntIdx] = RamRow.ramFields.Ka;
				CoeffArrayKb_rtrn[RowIdx][AntIdx] = RamRow.ramFields.Kb;
			}
		}
	}
}

//-------------------------------------------------------------------------------------------------------
// IIR API
//-------------------------------------------------------------------------------------------------------
void bbic_enable_iir(uint8_t AntMsk, BOOL IsEnabled)
{
	// Enables IIR DC removal filter
	RegPhyRxtdAnt0PhyRxtdAntReg01_u Reg_IIR_Bypass; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG01 0x4;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_IIR_Bypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG01 + i*(ANT_OFFSET));
			Reg_IIR_Bypass.bitFields.iqMismatchBypassIir = 1 - IsEnabled;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG01 + i*(ANT_OFFSET), Reg_IIR_Bypass.val);
		}
	}
}

void bbic_set_iir_shift(uint8_t AntMsk, uint8_t IirShift[ANT_NUM])
{
	// IirShift - bits to shift in feedback of DC removal IIR
	RegPhyRxtdAnt0PhyRxtdAntReg02_u Reg_IIR_Mu; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 0x8;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_IIR_Mu.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 + i*(ANT_OFFSET));
			Reg_IIR_Mu.bitFields.iqMismatchIirMu = IirShift[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 + i*(ANT_OFFSET), Reg_IIR_Mu.val);
		}
	}
}

void bbic_get_iir_shift(uint8_t AntMsk, uint8_t IirShift_rtrn[ANT_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg02_u Reg_IIR_Mu; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 0x8;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_IIR_Mu.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 + i*(ANT_OFFSET));
			IirShift_rtrn[i] = Reg_IIR_Mu.bitFields.iqMismatchIirMu;
		}
	}
}

//-------------------------------------------------------------------------------------------------------
// XTalk API
//-------------------------------------------------------------------------------------------------------
void bbic_enable_xtalk(uint8_t AntMsk, BOOL IsEnabled)
{
	// Enables XTalk block
	RegPhyRxtdAnt0PhyRxtdAntReg65_u Reg_XTalk_Bypass; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 0x194;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_XTalk_Bypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET));
			Reg_XTalk_Bypass.bitFields.iqXtalkBypass = 1 - IsEnabled;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET), Reg_XTalk_Bypass.val);
		}
	}
}

void bbic_set_xtalk_coeffs(uint8_t AntMsk, uint8_t Ka[ANT_NUM], uint8_t Kb[ANT_NUM])
{
	// Ka, Kb - XTalk compensation coefficients
	RegPhyRxtdAnt0PhyRxtdAntReg65_u Reg_XTalk_Coeff; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 0x194;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_XTalk_Coeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET));
			Reg_XTalk_Coeff.bitFields.iqXtalkKa = Ka[i];
			Reg_XTalk_Coeff.bitFields.iqXtalkKb = Kb[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET), Reg_XTalk_Coeff.val);
		}
	}
}

void bbic_get_xtalk_coeffs_pm(uint8_t AntMsk, uint8_t Ka_rtrn[ANT_NUM], uint8_t Kb_rtrn[ANT_NUM])
{
	// Read XTalk ProgModel coefficients
	RegPhyRxtdAnt0PhyRxtdAntReg65_u Reg_XTalk_Coeff; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 0x194;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_XTalk_Coeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET));
			Ka_rtrn[i] = Reg_XTalk_Coeff.bitFields.iqXtalkKa;
			Kb_rtrn[i] = Reg_XTalk_Coeff.bitFields.iqXtalkKb;

		}
	}
}

void bbic_get_xtalk_coeffs(uint8_t AntMsk, uint8_t Ka_rtrn[ANT_NUM], uint8_t  Kb_rtrn[ANT_NUM])
{
	// Read XTalk currently used coefficients
	RegPhyRxtdAnt0PhyRxtdAntReg72_u Reg_XTalk_Coeff; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 0x194;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_XTalk_Coeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG65 + i*(ANT_OFFSET));
			Ka_rtrn[i] = Reg_XTalk_Coeff.bitFields.stsIqXtalkKa;
			Kb_rtrn[i] = Reg_XTalk_Coeff.bitFields.stsIqXtalkKb;
		}
	}
}

//-------------------------------------------------------------------------------------------------------
// Decorrelator (W1W2) API
//-------------------------------------------------------------------------------------------------------
void bbic_enable_decorr(uint8_t AntMsk, BOOL IsEnabled)
{
	// Enables De-correlator block
	RegPhyRxtdAnt0PhyRxtdAntReg25_u Reg_Decorr_Bypass; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 0x94;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Decorr_Bypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 + i*(ANT_OFFSET));
			Reg_Decorr_Bypass.bitFields.iqW12Bypass = 1 - IsEnabled;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 + i*(ANT_OFFSET), Reg_Decorr_Bypass.val);
		}
	}
}

void bbic_set_decorr_coeffs(uint8_t AntMsk, uint16_t W1[ANT_NUM], int16_t W2[ANT_NUM], uint8_t BW)
{
	// W1, W2 - Decorrelator compensation coefficients
	// BW - [0,1,2,3] for 20/40/80/160MHz
	RegPhyRxtdAnt0PhyRxtdAntReg66_u reg_iq_w1w2;
	//RegPhyRxtdAnt0PhyRxtdAntReg66_u reg_iq_w1w2_bw20; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG66 0x198;
	//RegPhyRxtdAnt0PhyRxtdAntReg67_u reg_iq_w1w2_bw40; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG67 0x19C;
	//RegPhyRxtdAnt0PhyRxtdAntReg68_u reg_iq_w1w2_bw80; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG68 0x1A0;
	//RegPhyRxtdAnt0PhyRxtdAntReg69_u reg_iq_w1w2_bw160; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG69 0x1A4;
	uint32_t BaseOffset = REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG66;
	uint8_t BwOffset = REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG67 - REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG66;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			reg_iq_w1w2.val = BBIC_Reg_Read(BaseOffset + BW*BwOffset + i*(ANT_OFFSET));
			reg_iq_w1w2.bitFields.iqW1Bw20 = W1[i];
			reg_iq_w1w2.bitFields.iqW2Bw20 = W2[i];
			BBIC_Reg_Write(BaseOffset + BW*BwOffset + i*(ANT_OFFSET), reg_iq_w1w2.val);
		}
	}
}

void bbic_get_decorr_coeffs(uint8_t AntMsk, uint16_t W1_rtrn[ANT_NUM], int16_t W2_rtrn[ANT_NUM], uint8_t BW)
{
	RegPhyRxtdAnt0PhyRxtdAntReg66_u reg_iq_w1w2;
	uint32_t BaseOffset = REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG66;
	uint8_t BwOffset = REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG67 - REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG66;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			reg_iq_w1w2.val = BBIC_Reg_Read(BaseOffset + BW*BwOffset + i*(ANT_OFFSET));
			W1_rtrn[i] = reg_iq_w1w2.bitFields.iqW1Bw20;
			W2_rtrn[i] = reg_iq_w1w2.bitFields.iqW2Bw20;
			SIGN_EXTEND(W2_rtrn[i], 11);
		}
	}
}

//-------------------------------------------------------------------------------------------------------
// RXIQ FDL API
//-------------------------------------------------------------------------------------------------------
void bbic_enable_fdl(uint8_t AntMsk, BOOL IsEnabled)
{
	// Enables/disables FDL RXIQ compensation in the FDL block (FDL block bypass state is not touched here)
	RegPhyRxtdAnt0PhyRxtdAntReg24_u Reg_Fdl_Bypass; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG24 0x90;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Fdl_Bypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG24 + i*(ANT_OFFSET));
			Reg_Fdl_Bypass.bitFields.iqFdlBypass = 1 - IsEnabled;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG24 + i*(ANT_OFFSET), Reg_Fdl_Bypass.val);
		}
	}
}

void bbic_set_fdl_coeffs(uint8_t AntMsk, int16_t D[ANT_NUM])
{
	// Set D fractional delay to the FDL IQ
	RegPhyRxtdAnt0PhyRxtdAntReg50_u Reg_Fdl_Coeff; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 0x140;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Fdl_Coeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i*(ANT_OFFSET));
			Reg_Fdl_Coeff.bitFields.fdlIqComp = D[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i*(ANT_OFFSET), Reg_Fdl_Coeff.val);
		}
	}
}

void bbic_get_fdl_coeffs_pm(uint8_t AntMsk, int16_t D_rtrn[ANT_NUM])
{
	// Read FDL ProgModel coefficients
	RegPhyRxtdAnt0PhyRxtdAntReg50_u Reg_Fdl_Coeff; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 0x140;
	//TODO: check the field for IQ compensation is signed
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Fdl_Coeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i*(ANT_OFFSET));
			D_rtrn[i] = Reg_Fdl_Coeff.bitFields.fdlIqComp;
		}
	}
}

void bbic_get_fdl_coeffs(uint8_t AntMsk, int16_t D_rtrn[ANT_NUM])
{
	// Read FDL IQ currently used coefficient
	//TODO: change to a new parameter when available in SoC online
	RegPhyRxtdAnt0PhyRxtdAntReg50_u Reg_Fdl_Coeff; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 0x140;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Fdl_Coeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i*(ANT_OFFSET));
			D_rtrn[i] = Reg_Fdl_Coeff.bitFields.fdlIqComp;
		}
	}
}

//-------------------------------------------------------------------------------------------------------
// Accumulators API
//-------------------------------------------------------------------------------------------------------
void bbic_set_accum_nos(uint8_t AntMsk,uint32_t NOS[ANT_NUM])
{
	// NOS - number of samples to accumulate
	RegPhyRxtdAnt0PhyRxtdAntReg04_u Reg_Accum_Counter; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG04;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Accum_Counter.val = 0;
			Reg_Accum_Counter.bitFields.iqMismatchEstAccCounter = NOS[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG04 + i*(ANT_OFFSET), Reg_Accum_Counter.val);
		}
	}
}

void bbic_set_accum_shift(uint8_t AntMsk, uint8_t AccumShift[ANT_NUM])
{
	// AccumShift - bits to round after squaring (active in PWR mode only)
	RegPhyRxtdAnt0PhyRxtdAntReg26_u Reg_Accum_Input; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG26;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Accum_Input.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG26 + i*(ANT_OFFSET));
			Reg_Accum_Input.bitFields.iqMismatchEstShift = AccumShift[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG26 + i*(ANT_OFFSET), Reg_Accum_Input.val);
		}
	}
}

void bbic_set_accum_params(uint8_t AntMsk,uint32_t NOS[ANT_NUM], uint8_t DCMode[ANT_NUM], uint8_t RssiMode[ANT_NUM], uint8_t Rate[ANT_NUM], uint8_t AccumShift[ANT_NUM])
{
	// NOS - number of samples to accumulate
	// DCMode - accumulate sample values (1-DC mode) or squared values (0-PWR mode)
	// RssiMode - connect I-accumulator to RSSI signal (1) or to I/Q inputs (0)
	// Rate - connect accumulators before dec chan filter (1) or after dec chan filter (0)
	// AccumShift - bits to round after squaring (active in PWR mode only)
	
	RegPhyRxtdAnt0PhyRxtdAntReg04_u Reg_Accum_Counter; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG04;
	RegPhyRxtdAnt0PhyRxtdAntReg02_u Reg_Accum_DCMode; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02;
	RegPhyRxtdAnt0PhyRxtdAntReg26_u Reg_Accum_Input; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG26;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			// accumulator counter
			Reg_Accum_Counter.val = 0;
			Reg_Accum_Counter.bitFields.iqMismatchEstAccCounter = NOS[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG04 + i*(ANT_OFFSET), Reg_Accum_Counter.val);
			
			// DC mode
			Reg_Accum_DCMode.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 + i*(ANT_OFFSET));
			Reg_Accum_DCMode.bitFields.iqMismatchEstRegularISum = DCMode[i];
			Reg_Accum_DCMode.bitFields.iqMismatchEstRegularQSum = DCMode[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG02 + i*(ANT_OFFSET), Reg_Accum_DCMode.val);
			
			// Rssi mode, Rate and Shift round
			Reg_Accum_Input.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG26 + i*(ANT_OFFSET));
			Reg_Accum_Input.bitFields.iqMismatchEstRate = Rate[i];
			Reg_Accum_Input.bitFields.iqMismatchAccumRssi = RssiMode[i];
			Reg_Accum_Input.bitFields.iqMismatchEstShift = AccumShift[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG26 + i*(ANT_OFFSET), Reg_Accum_Input.val);
		}
	}
}

void bbic_start_accum (uint8_t AntMsk)
{
	// reset and start the accumulators operation
	RegPhyRxtdAnt0PhyRxtdAntReg05_u Reg_Accum_Reset_Start;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{

			Reg_Accum_Reset_Start.val = 0;
			// Reset registers, counter and the valid bit
			Reg_Accum_Reset_Start.bitFields.iqMismatchEstResetRegs = 1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG05 + i*(ANT_OFFSET), Reg_Accum_Reset_Start.val);
			// Release from reset (set reset bit to 0)
			Reg_Accum_Reset_Start.bitFields.iqMismatchEstResetRegs = 0;
			// START pulse starts the accumulation
			Reg_Accum_Reset_Start.bitFields.iqMismatchEstStartWork = 1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG05 + i*(ANT_OFFSET), Reg_Accum_Reset_Start.val);
		}
	}
}

uint8_t bbic_wait_accum (uint8_t AntMsk, uint16_t Timeout)
{
	// wait till accumulation stop condition
	// return value: 0 - success; if failed on the time limit, returns mask of the failed antennas

	RegPhyRxtdAnt0PhyRxtdAntReg05_u Reg_Accum_Valid;
	//uint32_t Reg_Address = REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG05;

	uint32_t startTime = 0;
	uint16_t endTime = 0;
	uint8_t maxTimeExceeded = 0;
	uint8_t isValid = 0;
	uint8_t ValidMask = 0;
	int i;

	startTime = BBIC_Time_Stamp(0,startTime);
	while (!isValid && !maxTimeExceeded) // wait for all antennas valid bit, while max time is not exceeded
	{
		for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				Reg_Accum_Valid.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG05 + i*(ANT_OFFSET));
				ValidMask |= Reg_Accum_Valid.bitFields.iqMismatchEstValidIq << i;
			}
		}
		isValid = (AntMsk == ValidMask);
		endTime = BBIC_Time_Stamp(1,startTime);
		if (endTime > Timeout)
			maxTimeExceeded = 1;
	}
	return ValidMask;
}

void bbic_get_accum (uint8_t AntMsk, CorrResults_t pAccumResults_rtrn[ANT_NUM], CorrOF_t pAccumOF_rtrn[ANT_NUM])
{
	// reads registers values; this function should be called after valid bit becomes 1
	// II - I or I^2 accumulated value
	// QQ - Q or Q^2 accumulated value
	// IQ - I*Q accumulated value
	// OF_II - overflow indicator for I^2, II>=2^32 for pwr mode or (II>=2^31 || II<-2^31) for dc mode
	// OF_QQ - overflow indicator for Q^2
	// OF_IQ - overflow indicator for I*Q, (II>=2^31 || II<-2^31)

	RegPhyRxtdAnt0PhyRxtdAntReg27_u Reg_Accum_OF;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG27
	RegPhyRxtdAnt0PhyRxtdAntReg06_u	Reg_Accum_II;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG06
	RegPhyRxtdAnt0PhyRxtdAntReg07_u	Reg_Accum_QQ;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG07
	RegPhyRxtdAnt0PhyRxtdAntReg08_u	Reg_Accum_IQ;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG08
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			// check overflow
			Reg_Accum_OF.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG27 + i*(ANT_OFFSET));
			pAccumOF_rtrn[i].II = Reg_Accum_OF.bitFields.iqMismatchEstOfIi;
			pAccumOF_rtrn[i].QQ = Reg_Accum_OF.bitFields.iqMismatchEstOfQq;
			pAccumOF_rtrn[i].IQ = Reg_Accum_OF.bitFields.iqMismatchEstOfIq;
			// get II, QQ & IQ registers values
			Reg_Accum_II.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG06 + i*(ANT_OFFSET));
			Reg_Accum_QQ.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG07 + i*(ANT_OFFSET));
			Reg_Accum_IQ.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG08 + i*(ANT_OFFSET));
			pAccumResults_rtrn[i].II.pwrVal = Reg_Accum_II.bitFields.iqMismatchEstRegIi;
			pAccumResults_rtrn[i].QQ.pwrVal = Reg_Accum_QQ.bitFields.iqMismatchEstRegQq;
			pAccumResults_rtrn[i].IQ = Reg_Accum_IQ.bitFields.iqMismatchEstRegIq;
		}
	}
}

uint8_t bbic_read_accum (uint8_t AntMsk, uint16_t Timeout, CorrResults_t pAccumResults_rtrn[ANT_NUM], CorrOF_t pAccumOF_rtrn[ANT_NUM])
{
	// performs the accumulators measurement sequence:
	// (1) - start the accumulation (2) - wait for valid bit (3) read result from accumulator registers
	// Input parameters:
	// MaxTime - the maximum time in [us] to wait for the valid bit
	// AccumResults - the structure array which holds II, QQ and IQ results
	// AccumOF - the structure array which holds II, QQ and IQ overflow indicator values
	
	bbic_start_accum (AntMsk);								// start accumulators
	uint8_t ValidMask = bbic_wait_accum (AntMsk, Timeout);	// wait till valid data is available
	bbic_get_accum (AntMsk, pAccumResults_rtrn, pAccumOF_rtrn); 		// get accumulators values

	return ValidMask;
}

//-------------------------------------------------------------------------------------------------------
// Goertzel API
//-------------------------------------------------------------------------------------------------------
void bbic_set_goert_tone (uint8_t AntMsk, int16_t Tone[ANT_NUM])
{
	// Tone - tone number (in 312.5KHz bin resolution) to set to both Goertzel_0 & Goertzel_1
	RegPhyRxtdAnt0PhyRxtdAntReg0D_u Reg_Goert0_Mu; 		// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0D
	RegPhyRxtdAnt0PhyRxtdAntReg0E_u Reg_Goert1_Mu; 		// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0E
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			// Goertzel multiplier (Tone0)
			Reg_Goert0_Mu.val = 0;
			Reg_Goert0_Mu.bitFields.geortzel0MulI = get_cos_value(Tone[i]);
			Reg_Goert0_Mu.bitFields.geortzel0MulQ = get_sin_value(Tone[i]);
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0D + i*(ANT_OFFSET), Reg_Goert0_Mu.val);

			// Goertzel multiplier (Tone1)
			Reg_Goert1_Mu.val = 0;
			Reg_Goert1_Mu.bitFields.geortzel1MulI = get_cos_value(Tone[i]);
			Reg_Goert1_Mu.bitFields.geortzel1MulQ = get_sin_value(Tone[i]);
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0E + i*(ANT_OFFSET), Reg_Goert1_Mu.val);
		}
	}
}

void bbic_set_goert_length (uint8_t AntMsk,uint32_t Length[ANT_NUM])
{
	// Length - number of samples to accumulate
	RegPhyRxtdAnt0PhyRxtdAntReg0C_u Reg_Goert_Length; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0C
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Goert_Length.val = 0;
			Reg_Goert_Length.bitFields.geortzelLength = Length[i]-1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0C + i*(ANT_OFFSET), Reg_Goert_Length.val);
		}
	}
}

void bbic_set_goert_cycles (uint8_t AntMsk, uint8_t NumOfCycles[ANT_NUM])
{
	// Cycles - number of Goertzel cycles to average
	RegPhyRxtdAnt0PhyRxtdAntReg0F_u Reg_Goert_Cycles; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0F
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Goert_Cycles.val = 0;
			Reg_Goert_Cycles.bitFields.geortzelValidThr = NumOfCycles[i]-1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0F + i*(ANT_OFFSET), Reg_Goert_Cycles.val);
		}
	}
}

void bbic_set_goert_params (uint8_t AntMsk,uint32_t Length[ANT_NUM], uint8_t NumOfCycles[ANT_NUM], int16_t Tone0[ANT_NUM], int16_t Tone1[ANT_NUM])
{
	// Length - number of samples to accumulate
	// Cycles - number of Goertzel cycles to average
	// Tone0 - tone number (in 312.5KHz bin resolution) to set Goertzel_0
	// Tone1 - tone number (in 312.5KHz bin resolution) to set Goertzel_1

	RegPhyRxtdAnt0PhyRxtdAntReg0C_u Reg_Goert_Length; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0C
	RegPhyRxtdAnt0PhyRxtdAntReg0F_u Reg_Goert_Cycles; 	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0F
	RegPhyRxtdAnt0PhyRxtdAntReg0D_u Reg_Goert0_Mu; 		// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0D
	RegPhyRxtdAnt0PhyRxtdAntReg0E_u Reg_Goert1_Mu; 		// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0E
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			// Goertzel length
			Reg_Goert_Length.val = 0;
			Reg_Goert_Length.bitFields.geortzelLength = Length[i]-1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0C + i*(ANT_OFFSET), Reg_Goert_Length.val);

			// Goertzel number of cycles
			Reg_Goert_Cycles.val = 0;
			Reg_Goert_Cycles.bitFields.geortzelValidThr = NumOfCycles[i]-1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0F + i*(ANT_OFFSET), Reg_Goert_Cycles.val);

			// Goertzel multiplier (Tone0)
			Reg_Goert0_Mu.val = 0;
			Reg_Goert0_Mu.bitFields.geortzel0MulI = get_cos_value(Tone0[i]);
			Reg_Goert0_Mu.bitFields.geortzel0MulQ = get_sin_value(Tone0[i]);
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0D + i*(ANT_OFFSET), Reg_Goert0_Mu.val);

			// Goertzel multiplier (Tone1)
			Reg_Goert1_Mu.val = 0;
			Reg_Goert1_Mu.bitFields.geortzel1MulI = get_cos_value(Tone1[i]);
			Reg_Goert1_Mu.bitFields.geortzel1MulQ = get_sin_value(Tone1[i]);
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0E + i*(ANT_OFFSET), Reg_Goert1_Mu.val);
		}
	}
}

void bbic_start_goert (uint8_t AntMsk)
{
	// reset and start the Goertzel operation (issues the pulse)
	RegPhyRxtdAnt0PhyRxtdAntReg0B_u Reg_Goert_Start;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0B

	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Goert_Start.bitFields.geortzelResetAcum = 1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG0B + i*(ANT_OFFSET), Reg_Goert_Start.val);
		}
	}
}

uint8_t bbic_wait_goert (uint8_t AntMsk, uint16_t Timeout)
{
	// wait till Goertzel stop condition
	// return value: 0 - success; if failed on the time limit, returns mask of the failed antennas
	// Notes:
	// 1. The valid bit is located in the MSB bit of the same register as the Goertzel_Imag result
	// 2. There are 2 valid bits, in Goertzel_I and Goertzel_Q. Since both Goertzel blocks are triggered with the same pulse and
	// 	  have the same NOS and number of cycles, both will finish at the same time, so we poll just one Goertzel valid bit
	RegPhyRxtdAnt0PhyRxtdAntReg3E_u Reg_Goert_Valid; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3E 0xF8
	uint32_t startTime = 0;
	uint16_t endTime = 0;
	uint8_t maxTimeExceeded = 0;
	uint8_t isValid = 0;
	uint8_t ValidMask = 0;
	int i;

	startTime = BBIC_Time_Stamp(0,startTime);
	while (!isValid && !maxTimeExceeded) // wait for all antennas valid bit, while max time is not exceeded
	{
		for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				Reg_Goert_Valid.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3E + i*(ANT_OFFSET));
				ValidMask |= Reg_Goert_Valid.bitFields.geortzel0DataValid << i;
			}
		}
		isValid = (AntMsk == ValidMask);
		endTime = BBIC_Time_Stamp(1,startTime);
		if (endTime > Timeout)
			maxTimeExceeded = 1;
	}
	return ValidMask;
}

void bbic_get_goert (uint8_t AntMsk, GoertzelResults_t pGoertResult_rtrn[ANT_NUM])
{
	// reads Goertzel registers values; this function should be called after valid bit becomes 1
	// pGoertResult - Goertzel I & Q complex results
	//RegPhyRxtdAnt0PhyRxtdAntReg3D_u Reg_Goert_I_Real;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3D 0xF4
	//RegPhyRxtdAnt0PhyRxtdAntReg3E_u Reg_Goert_I_Imag; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3E 0xF8
	//RegPhyRxtdAnt0PhyRxtdAntReg3F_u Reg_Goert_Q_Real;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3F 0xFC
	//RegPhyRxtdAnt0PhyRxtdAntReg40_u Reg_Goert_Q_Imag;	// REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG40 0x100
	GoertResult_u ResultVal;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			// get I and Q registers complex values
			ResultVal.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3D + i*(ANT_OFFSET));
			pGoertResult_rtrn[i].Result_I_Real = ResultVal.bitFields.Result;

			ResultVal.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3E + i*(ANT_OFFSET));
			pGoertResult_rtrn[i].Result_I_Imag = ResultVal.bitFields.Result;

			ResultVal.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3F + i*(ANT_OFFSET));
			pGoertResult_rtrn[i].Result_Q_Real = ResultVal.bitFields.Result;

			ResultVal.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG40 + i*(ANT_OFFSET));
			pGoertResult_rtrn[i].Result_Q_Imag = ResultVal.bitFields.Result;
		}
	}
}

uint8_t bbic_read_goert (uint8_t AntMsk, uint16_t Timeout, GoertzelResults_t pGoertResult_rtrn[ANT_NUM])
{
	// performs the Goertzel measurement sequence:
	// (1) - start the operation (2) - wait for the valid bit (3) read result from the results registers
	// Input parameters:
	// Timeout - the maximum time in [us] to wait for the valid bit
	// pGoertResult - the structure array which holds I and Q Goertzel complex results

	bbic_start_goert (AntMsk);								// start Goertzel
	uint8_t ValidMask = bbic_wait_goert (AntMsk, Timeout);	// wait till valid data is available
	bbic_get_goert (AntMsk, pGoertResult_rtrn); 					// get Goertzel result values

	return ValidMask;
}

// ######################################################################################################
// # Tone Generator																						#
// ######################################################################################################
BOOL bbic_generate_tone_gen(int16_t tone, int8_t scale, int16_t digGain[ANT_NUM], uint8_t AntMsk)
{
	// A whole sequence for tone generation
	BOOL timeout;
	
	bbic_set_tx_reset();
	bbic_release_tx_reset();
	bbic_enable_tone_gen(TRUE);
	bbic_clear_gen_tone_all();
	bbic_set_gen_tone(tone);
	bbic_set_tx_scale(AntMsk, scale);
	bbic_enable_tpc_accelerator(AntMsk, FALSE);
	bbic_set_tx_diggain(AntMsk, digGain);
	bbic_tx_fft_in_scale(AntMsk,SCALE_FFT_TONE);
	bbic_start_tone_gen();
	timeout = bbic_wait_tone_gen(100);
	
	return timeout;
}

BOOL bbic_generate_multi_tone_gen(int16_t* tones, int8_t numOfTones, int8_t scale, int16_t digGain[ANT_NUM], uint8_t AntMsk)
{
	BOOL timeout;
	int i;
	bbic_set_tx_reset();
	bbic_release_tx_reset();
	bbic_enable_tone_gen(TRUE);
	bbic_clear_gen_tone_all();
	for ( i = 0; i < numOfTones; i++)
	{
		bbic_set_gen_tone(tones[i]);
	}
	bbic_set_tx_scale(AntMsk, scale);
	bbic_enable_tpc_accelerator(AntMsk, FALSE);
	bbic_set_tx_diggain(AntMsk, digGain);
	bbic_tx_fft_in_scale(AntMsk,SCALE_FFT_MULTI_TONE);
	bbic_start_tone_gen();
	timeout = bbic_wait_tone_gen(100);//is it enogh??
	
	return timeout;
}

void bbic_clear_and_disable_tone_gen(uint8_t AntMsk)
{	
	bbic_enable_tone_gen(FALSE);
	bbic_clear_gen_tone_all();	
	bbic_enable_tpc_accelerator(AntMsk, TRUE);
	bbic_set_tx_reset();
	bbic_release_tx_reset();
}

uint16_t tone2idx(int16_t Tone)
{
	/*
	// inband 2's complement tone index calculation:
	uint16_t ToneIdx;
	if (Tone >= 0 && Tone < 256)
		ToneIdx = Tone; 						// [0:255],	0...80MHz-312.5KHz
	else if (Tone >= -256 && Tone < 0)
		ToneIdx = Tone + 512; 					// [256:511], -80MHz...-312.5KHz
	// out-of-band 2's complement index calculation; TODO: verify after VLSI implementation
	else if (Tone >= 256 && Tone < 512)
		ToneIdx = Tone + 256; 				// [512:767], 80MHz...160MHz-312.5KHz
	else if (Tone >= -512 && Tone < -256)
		ToneIdx = Tone + 1024 + 256; 	// [768:1023], -160MHz...-80MHz-312.5KHz
	*/
	uint16_t FftSize = 1024;
	uint16_t FftSizeInband = 512;
	if (Tone >= -256 && Tone < 256) // inband tone
		return (Tone + FftSizeInband) % FftSizeInband;
	else							// out-of-band tone - TODO: verify after VLSI implementation
		return (Tone + FftSize + FftSizeInband/2) % FftSize;
}
void bbic_set_gen_tone (int16_t Tone)
{
	// Tone - tone number (in 312.5KHz bin resolution), signed (in [-256...255) range for inband tones; full range is [-512...511))
	// The tone mapping in the chip long_preamble RAM is as follows:
	// In-band: 0...511 (0...255: 0MHz...80MHz-312.5KHz, 256...511: -80MHz...-312.5KHz
	// Out-of-band: 512...1023 (512...767: 80MHz...160MHz-312.5KHz, 768...1023: -160MHz...-80MHz-312.5KHz
	uint32_t ToneGenBaseOffset = RAM_TONE_GEN_BASE_ADDRESS;
	uint8_t WordLength = 32;
	uint16_t ToneIdx = tone2idx(Tone);
	uint8_t RowIdx = ToneIdx / WordLength;
	uint8_t BitIdx = ToneIdx % WordLength;
	uint32_t RamRow = BBIC_Reg_Read(ToneGenBaseOffset + 4*RowIdx);
	//RamRow |= (1 << (31-BitIdx)); // set 1 to the tone bin position
	RamRow |= 1 << BitIdx; // set 1 to the tone bin position
	BBIC_Reg_Write(ToneGenBaseOffset + 4*RowIdx, RamRow);
}

void bbic_clear_gen_tone (int16_t Tone)
{
	// clear the single tone from the RAM
	// Tone - tone number (in 312.5KHz bin resolution), signed, in the full FFT range [-512...511)
	uint32_t ToneGenBaseOffset = RAM_TONE_GEN_BASE_ADDRESS;
	uint8_t WordLength = 32;
	uint16_t ToneIdx = tone2idx(Tone);
	uint8_t RowIdx = ToneIdx / WordLength;
	uint8_t BitIdx = ToneIdx % WordLength;
	uint32_t RamRow = BBIC_Reg_Read(ToneGenBaseOffset + 4*RowIdx);
	//RamRow &= ~(1 << (31-BitIdx)); // set 0 to the tone bin position
	RamRow &= ~(1 << BitIdx); // set 0 to the tone bin position
	BBIC_Reg_Write(ToneGenBaseOffset + 4*RowIdx, RamRow);
}

void bbic_set_gen_two_tones (int16_t Tone1, int16_t Tone2)
{
	// Tone1, Tone2 - tones number (in 312.5KHz bin resolution), signed, in the full FFT range [-512...511)
	bbic_set_gen_tone (Tone1);
	bbic_set_gen_tone (Tone2);
}

void bbic_clear_gen_two_tones (int16_t Tone1, int16_t Tone2)
{
	bbic_clear_gen_tone (Tone1);
	bbic_clear_gen_tone (Tone2);
}

void bbic_set_gen_missing_tone (int16_t ToneStart, int16_t ToneStop, int16_t ToneMissing)
{
	// Tone - tone number (in 312.5KHz bin resolution), signed, in [-256...255) range for inband tones
	uint32_t ToneGenBaseOffset = RAM_TONE_GEN_BASE_ADDRESS;
	uint8_t WordLength = 32;
	uint16_t FftSizeInband = 512;
	uint16_t ToneIdxStart = tone2idx(ToneStart);
	uint16_t ToneIdxStop = tone2idx(ToneStop);
	uint16_t RamRowIdxStart;
	uint16_t RamRowIdxStop;
	uint16_t BitIdxStart;
	uint16_t BitIdxStop;
	int RowIdx;
	if (ToneIdxStop > ToneIdxStart) // all the bins between ToneIdxStart and ToneIdxStop will be set to 1
	{
		// fill all the 32-bins RAM rows
		RamRowIdxStart = ToneIdxStart / WordLength;
		BitIdxStart = ToneIdxStart % WordLength;
		RamRowIdxStop = ToneIdxStop / WordLength;
		BitIdxStop = ToneIdxStop % WordLength;
		for (RowIdx=RamRowIdxStart+1; RowIdx<RamRowIdxStop; RowIdx++)
			BBIC_Reg_Write(ToneGenBaseOffset + 4*RowIdx, 0xFFFFFFFF);
		// fill the row with start tone
		//BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStart, (uint32_t) (1 << (32-BitIdxStart))-1);
		BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStart, (uint32_t) (1 << (1+BitIdxStart))-1);
		// fill the row with stop tone
		//BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStop, ~ (uint32_t) ((1 << (32-BitIdxStop))-1));
		BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStop, ~ (uint32_t) ((1 << (1+BitIdxStop))-1));
	}
	else	// two contiguous regions will be set, one for negative tones and another for positive tones
	{
		// 1st region (negative)
		RamRowIdxStart = ToneIdxStart / WordLength;
		BitIdxStart = ToneIdxStart % WordLength;
		RamRowIdxStop = (FftSizeInband-1) / WordLength;
		BitIdxStop = WordLength - 1;
		// fill all the full 32-bins RAM rows
		for (RowIdx=RamRowIdxStart+1; RowIdx<RamRowIdxStop; RowIdx++)
			BBIC_Reg_Write(ToneGenBaseOffset + 4*RowIdx, 0xFFFFFFFF);
		// fill the row with start tone
		//BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStart, (uint32_t) (1 << (32-BitIdxStart))-1);
		BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStart, (uint32_t) (1 << (1+BitIdxStart))-1);
		// 2nd region (positive)
		RamRowIdxStart = 0;
		BitIdxStart = 0;
		RamRowIdxStop = ToneIdxStop / WordLength;
		BitIdxStop = ToneIdxStop % WordLength;
		// fill all the full 32-bins RAM rows
		for (RowIdx=RamRowIdxStart+1; RowIdx<RamRowIdxStop; RowIdx++)
			BBIC_Reg_Write(ToneGenBaseOffset + 4*RowIdx, 0xFFFFFFFF);
		// fill the row with stop tone
		//BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStop, ~ (uint32_t) ((1 << (32-BitIdxStop))-1));
		BBIC_Reg_Write(ToneGenBaseOffset + 4*RamRowIdxStop, ~ (uint32_t) ((1 << (1+BitIdxStop))-1));
	}
	// clear the missing tone
	bbic_clear_gen_tone (ToneMissing);
}

void bbic_clear_gen_tone_all (void)
{
	// Tone - tone number (in 312.5KHz bin resolution), signed (in [-256...255) range for inband tones; full range is [-512...511))
	// The tone mapping in the chip long_preamble RAM is as follows:
	// In-band: 0...511 (0...255: 0MHz...80MHz-312.5KHz, 256...511: -80MHz...-312.5KHz
	// Out-of-band: 512...1023 (512...767: 80MHz...160MHz-312.5KHz, 768...1023: -160MHz...-80MHz-312.5KHz
	// execution time: 26clk@640M x 32 = 1.300us
	uint32_t ToneGenBaseOffset = RAM_TONE_GEN_BASE_ADDRESS;
	uint8_t RamLength = 32;
	int RowIdx;
	for (RowIdx=0; RowIdx<RamLength; RowIdx++)
		BBIC_Reg_Write(ToneGenBaseOffset + 4*RowIdx, 0);
}

void bbic_enable_tone_gen (BOOL isEnabled)
{
	// turn on Tone Generator
	RegPhyTxToneGen_u Reg_Tone_Gen; // REG_PHY_TX_TONE_GEN 0xEF0
	Reg_Tone_Gen.val = BBIC_Reg_Read(REG_PHY_TX_TONE_GEN);
	Reg_Tone_Gen.bitFields.toneGenOn = isEnabled;
	Reg_Tone_Gen.bitFields.toneGenCycles = 0; 	// endless loop
	Reg_Tone_Gen.bitFields.toneGenLstfMode = 0; // CW mode
	BBIC_Reg_Write(REG_PHY_TX_TONE_GEN, Reg_Tone_Gen.val);
}

void bbic_start_tone_gen (void)
{
	// turn on Tone Generator
	RegPhyTxToneGen_u Reg_Tone_Gen; // REG_PHY_TX_TONE_GEN 0xEF0
	Reg_Tone_Gen.val = BBIC_Reg_Read(REG_PHY_TX_TONE_GEN);
	Reg_Tone_Gen.bitFields.toneGenStart = 1;
	BBIC_Reg_Write(REG_PHY_TX_TONE_GEN, Reg_Tone_Gen.val);
}

uint8_t bbic_wait_tone_gen (uint16_t Timeout)
{
	// wait till Tone Generator Active bit goes high (starts playing the samples)
	// return value: 0 - success; 1 - timeout
	RegPhyTxToneGen_u Reg_Tone_Gen; // REG_PHY_TX_TONE_GEN 0xEF0
	uint32_t startTime = 0;
	uint16_t endTime = 0;
	uint8_t maxTimeExceeded = 0;
	uint8_t isActive = 0;
	startTime = BBIC_Time_Stamp(0,startTime);
	while (!isActive && !maxTimeExceeded) // wait for active bit, while max time is not exceeded
	{
		Reg_Tone_Gen.val = BBIC_Reg_Read(REG_PHY_TX_TONE_GEN);
		isActive = Reg_Tone_Gen.bitFields.toneGenActive;
		endTime = BBIC_Time_Stamp(1,startTime);
		if (endTime > Timeout)
			maxTimeExceeded = 1;
	}
	return (isActive) ? 0 : 1;
}

void bbic_set_tx_diggain (uint8_t AntMsk, int16_t DigGain[ANT_NUM])
{
	// fine tuning of the signal level
	// input: DigGain in [dec], 9bit, 0-2 range
	RegPhyTxtdAnt0TxDigitalGain_u Reg_Dig_Gain; // REG_PHY_TXTD_ANT0_TX_DIGITAL_GAIN 0x9C
	RegPhyTxSpareRegs0_u        Reg_TPCACC_pulse;  /*REG_PHY_TX_SPARE_REGS_0 0xCC4 */	
	int i;
		for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				Reg_Dig_Gain.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_DIGITAL_GAIN + i*(ANT_OFFSET));
				Reg_Dig_Gain.bitFields.pmTxDigitalGain = DigGain[i];
				Reg_Dig_Gain.bitFields.pmTxGainEn = 1;
				BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TX_DIGITAL_GAIN + i*(ANT_OFFSET), Reg_Dig_Gain.val);
			}
		}
	 
	// Give pulse to enable APCACC (and take digital gain)
	Reg_TPCACC_pulse.val = BBIC_Reg_Read(REG_PHY_TX_SPARE_REGS_0);
#if defined(ENET_INC_ARCH_WAVE600B) || defined(ENET_INC_ARCH_WAVE600D2)
	Reg_TPCACC_pulse.bitFields.spareGpr00 = 1;
#else
	Reg_TPCACC_pulse.bitFields.spareGpr0 = 1;
#endif
    BBIC_Reg_Write(REG_PHY_TX_SPARE_REGS_0, Reg_TPCACC_pulse.val);
}

void bbic_get_tx_diggain (uint8_t AntMsk, int16_t DigGain_rtrn[ANT_NUM])
{
	// fine tuning of the signal level
	// input: DigGain in [dec], 9bit, 0-2 range
	int i;
		for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				DigGain_rtrn[i] = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_DIGITAL_GAIN + i*(ANT_OFFSET));
			}
		}
}


void bbic_enable_tpc_accelerator(uint8_t AntMsk, BOOL IsEnabled)
{

	RegPhyTxtdAnt0TpcAccelerator_u Reg_Tpc_Acc_Status; // REG_PHY_TXTD_ANT0_TPC_ACCELERATOR 
	int i;
	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Tpc_Acc_Status.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR + i * (ANT_OFFSET));
			Reg_Tpc_Acc_Status.bitFields.tpcaccenable = IsEnabled;
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR + i * (ANT_OFFSET), Reg_Tpc_Acc_Status.val);
		}
	}
}


void bbic_set_tx_pgc2_gain_select(uint8_t AntMsk, int8_t tpcgainidx) //Set dc offset reg per tpc (pgc2)
{

	RegPhyTxtdAnt0TpcAccelerator_u Reg_Tpc_Acc_Status; // REG_PHY_TXTD_ANT0_TPC_ACCELERATOR 
	int i;
	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Tpc_Acc_Status.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR + i * (ANT_OFFSET));
			Reg_Tpc_Acc_Status.bitFields.pmTxPgc2GainSelect = tpcgainidx;
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR + i * (ANT_OFFSET), Reg_Tpc_Acc_Status.val);
		}
	}
}


void bbic_set_tx_scale (uint8_t AntMsk, int8_t TxTdScale)
{
	// coarse tuning of the tone level
	// input: TxTdScale - shift left the signal by TxTdScale bits
	RegPhyTxToneGenScale_u Reg_Gen_Scale; 	// REG_PHY_TX_TONE_GEN_SCALE 0xF84
	if (AntMsk != 0)
	{
		Reg_Gen_Scale.val = BBIC_Reg_Read(REG_PHY_TX_TONE_GEN_SCALE);
		Reg_Gen_Scale.bitFields.fdScaleShift = TxTdScale;
		BBIC_Reg_Write(REG_PHY_TX_TONE_GEN_SCALE , Reg_Gen_Scale.val);
	}	
}

uint32 bbic_get_tx_scale (void)
{
	RegPhyTxToneGenScale_u Reg_Gen_Scale; 
	Reg_Gen_Scale.val = BBIC_Reg_Read(REG_PHY_TX_TONE_GEN_SCALE);
	return Reg_Gen_Scale.bitFields.fdScaleShift;

}


void bbic_set_gen_backoff (uint8_t AntMsk)
{
	// Backoff - tone generator backoff from the full scale in Q5 (1/32dB) resolution
	// ToneGenScale (FD) -> TD
	RegPhyTxToneGenScale_u Reg_Gen_Scale; 	// REG_PHY_TX_TONE_GEN_SCALE 0xF84
	// set tone generator bpsk scaling
	if (AntMsk != 0)
	{
		Reg_Gen_Scale.val = BBIC_Reg_Read(REG_PHY_TX_TONE_GEN_SCALE);
		Reg_Gen_Scale.bitFields.toneGenBpskVal = 2047; // max value in [-1...1)range
		// set shifter value
		int8_t ScaleVal = 2;
		bbic_set_tx_scale (AntMsk, ScaleVal);
		// set digital gain
		int16_t DigGain[ANT_NUM] = {256};
		bbic_set_tx_diggain (AntMsk, DigGain);
	}
}

// ######################################################################################################
// # Streamer 																						#
// ######################################################################################################

void bbic_enable_streamer(uint8_t StreamerMode, BOOL IsEnabled)
{
	// Set streamer_mode (1= cyclic), IsEnabled - streamer on/off
	RegPhyRxFdPhyRxfdRegf5_u Reg_Streamer_Enable; /*REG_PHY_RX_FD_PHY_RXFD_REGF5 0x3D4 */
	Reg_Streamer_Enable.bitFields.streamerEnable = IsEnabled;
	Reg_Streamer_Enable.bitFields.streamerMode	= StreamerMode;
	BBIC_Reg_Write(REG_PHY_RX_FD_PHY_RXFD_REGF5, Reg_Streamer_Enable.val);
}

void bbic_set_streamer_size(int16_t Length, int16_t Offset)
{
	// Set start and stop indexes to play, according to the length provided, starting at StartIndex offset
	RegPhyRxFdPhyRxfdRegf6_u Reg_Streamer_Length; /*REG_PHY_RX_FD_PHY_RXFD_REGF6 0x3D8 */
	Reg_Streamer_Length.bitFields.streamerStartAddress = Offset;
	Reg_Streamer_Length.bitFields.streamerEndAddress = Offset + Length - 1;
	BBIC_Reg_Write(REG_PHY_RX_FD_PHY_RXFD_REGF6, Reg_Streamer_Length.val);
}

void bbic_streamer_txon(uint8_t AntMsk)
{
	// turn on the streamer path per antenna
	RegPhyTxtdAnt0Streamer_u Reg_Streamer_Ant_Enable; /*REG_PHY_TXTD_ANT0_STREAMER 0xB0 */
		int i;
			for (i=0; i<ANT_NUM; i++)
			{
				if (AntMsk & 1<<i)
				{
					Reg_Streamer_Ant_Enable.bitFields.streamerEnable = 1;
					BBIC_Reg_Write(REG_PHY_TXTD_ANT0_STREAMER + i*(ANT_OFFSET), Reg_Streamer_Ant_Enable.val);
				}
			}
}
void bbic_streamer_txoff(uint8_t AntMsk)
{
	// turn off the streamer path per antenna
	RegPhyTxtdAnt0Streamer_u Reg_Streamer_Ant_Enable; /*REG_PHY_TXTD_ANT0_STREAMER 0xB0 */
		int i;
			for (i=0; i<ANT_NUM; i++)
			{
				if (AntMsk & 1<<i)
				{
					Reg_Streamer_Ant_Enable.bitFields.streamerEnable = 0;
					BBIC_Reg_Write(REG_PHY_TXTD_ANT0_STREAMER + i*(ANT_OFFSET), Reg_Streamer_Ant_Enable.val);
				}
			}
}

// ######################################################################################################
// # Fixed Point Arithmetic - to be merged into FW library																						#
// ######################################################################################################

int32_t fxp_limit_sign(int32_t Val, uint8_t Bits)
{
	if (Val < -(1<<(Bits-1)))
		return -(1<<(Bits-1));
	else if (Val >= 1<<(Bits-1))
		return (1<<(Bits-1)) - 1;
	else
		return Val;
}

uint32_t fxp_limit_unsign(uint32_t Val, uint8_t Bits)
{
	if (Val >= (1<<Bits))
		return (1<<Bits) - 1;
	else
		return Val;
}

// ######################################################################################################
// # Auxiliary functions for VLSI tests																	#
// ######################################################################################################

void bbic_detector_rotate (void)
{
	RegPhyRxTdPhyRxtdReg07B_u Reg_Host_Antenna_Enable;// REG_PHY_RX_TD_PHY_RXTD_REG07B 0x1EC
	RegPhyRxTdPhyRxtdReg044_u	Reg_Detection_Mask;		// REG_PHY_RX_TD_PHY_RXTD_REG044 0x110
	RegPhyRxTdPhyRxtdReg04_u Reg_Cyclic_Period; 		// REG_PHY_RX_TD_PHY_RXTD_REG04 0x10
	uint8_t AntMsk;
	uint8_t NumOfAnt;
	uint8_t Period;
	uint32_t Rotate;
	int i;
	uint8_t j;
	// read the antenna mask
	Reg_Host_Antenna_Enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG07B);
	AntMsk = Reg_Host_Antenna_Enable.bitFields.hostAntennaEn;
	// calculate number of active antennas
	NumOfAnt = 0;
	for (i=0; i<4; i++)
		if (AntMsk & 1<<i) NumOfAnt++;
	// set detector rotation according to the antenna mask
	// get period value
	Period = 2*NumOfAnt - 1;
	// get rotate value
	Rotate = 0;
	j = 0;
	for (i=3; i>=0; i--) {
		if (AntMsk & 1<<i) {
			Rotate += AntMsk << (4*(2*j));
			Rotate += (AntMsk & ~(1<<i)) << (4*(2*j+1));
			j++;
		}
	}
	if (AntMsk == 15) // strange outlier in the detector rotate mask lookup table
		Rotate = 0xbfdfef7f;
	// write the rotate and period values to PHY
	Reg_Detection_Mask.val = Rotate;
	Reg_Cyclic_Period.val = 0;
	Reg_Cyclic_Period.bitFields.scCyclicPeriod = Period;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG044, Reg_Detection_Mask.val);
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG04, Reg_Cyclic_Period.val);
}

//**** Lev's Part

void bbic_reset_channel_filter(void)
{
	RegPhyRxTdIfPhyRxtdIf8F_u Reg_rxtdif_reset;
	uint32 Temp_val;
	
	Reg_rxtdif_reset.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_PHY_RXTD_IF8F);
	Temp_val = (Reg_rxtdif_reset.val & 0xFFFFFFFB);
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_PHY_RXTD_IF8F, Temp_val);
	Temp_val = (Reg_rxtdif_reset.val | 0x4);
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_PHY_RXTD_IF8F, Temp_val);
}

void bbic_reset_rxtd(BOOL IsEnabled)
{
	RegPhyRxTdPhyRxtdReg00_u Reg_rxtd_reset;
	uint32 Temp_val;
	
	Reg_rxtd_reset.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG00);
	Temp_val = ((Reg_rxtd_reset.val & 0xFFFC0000) | ((1 - IsEnabled) * ((1<<18) - 1)));
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, Temp_val);
	
}

void bbic_enable_rxtd_fifo(BOOL IsEnabled)
{
	RegB0PhyRxTdPhyRxtdReg2F_RFSys_u Reg_rxtd_enable;

	Reg_rxtd_enable.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG2F);
	Reg_rxtd_enable.bitFields.rxtd_fifo = IsEnabled;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG2F, Reg_rxtd_enable.val);
}

void bbic_enable_detector(BOOL IsEnabled)
{
	RegPhyRxTdIfPhyRxtdIf4C_u Reg_detector_state;  //REG_PHY_RX_TD_IF_PHY_RXTD_IF4C

	Reg_detector_state.val = BBIC_Reg_Read(REG_PHY_RX_TD_IF_PHY_RXTD_IF4C);
	Reg_detector_state.bitFields.detectorBypassEn = 1 - IsEnabled;
	BBIC_Reg_Write(REG_PHY_RX_TD_IF_PHY_RXTD_IF4C, Reg_detector_state.val);

}

void bbic_enable_cal_mode(BOOL IsEnabled)
{
	RegB0PhyRxTdPhyRxtdReg2F_RFSys_u Reg_cal_mode;

	Reg_cal_mode.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG2F);
	Reg_cal_mode.bitFields.iq_cal = IsEnabled;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG2F, Reg_cal_mode.val);
}


void bbic_set_lms_params (uint8_t AntMsk, BOOL CalMode , uint32_t NumOfSteps, uint8_t Mu, uint8_t Alpha )
{
	int i;
	uint8_t CurrentMsk;
	
	RegPhyRxtdAnt0PhyRxtdAntReg35_u Reg_Mu;
	RegPhyRxtdAnt0PhyRxtdAntReg36_u Reg_Alpha;
	RegPhyRxtdAnt0PhyRxtdAntReg38_u Reg_Steps;
	//RegPhyRxtdAnt0PhyRxtdAntReg25_u Reg_CalMode;

	for (i=0; i<ANT_NUM; i++)
	{
		CurrentMsk = AntMsk & 1<<i;
		if (CurrentMsk)
		{
			//Mu,Alpha and Steps has to be set for every LMS cycle and there's no reason to set them separately.
			Reg_Mu.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG35 + (i * ANT_OFFSET));
			Reg_Alpha.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG36 + (i * ANT_OFFSET));
			Reg_Steps.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG38 + (i * ANT_OFFSET));

			Reg_Mu.bitFields.iqLmsMuShift = Mu;
			Reg_Alpha.bitFields.iqLmsAlpha = Alpha;
			Reg_Steps.bitFields.iqLmsStepsToRun = NumOfSteps;

			//set Mu
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG35 + (i * ANT_OFFSET), Reg_Mu.val);
			//set alpha
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG36 + (i * ANT_OFFSET), Reg_Alpha.val);
			//set number of steps
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG38 + (i * ANT_OFFSET), Reg_Steps.val);


		}
	}
	//The following parameters were set before in the loop - To be verified during bring up

	//Set NLMS Cal mode
	bbic_set_lms_cal_mode(AntMsk, CalMode);
	return;

}

void bbic_set_valid_coeffs (uint8_t AntMsk, uint32_t NumOfEnabledCoeffs)
{
	int i;
	RegPhyRxtdAnt0PhyRxtdAntReg37_u Reg_EnabledCoeffs;
	uint16_t EnabledCoeffs;

	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i) 
		{
			EnabledCoeffs = (1<<NumOfEnabledCoeffs) - 1;
			Reg_EnabledCoeffs.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG37 + (i * ANT_OFFSET));
			Reg_EnabledCoeffs.bitFields.iqLmsCoeffEn = EnabledCoeffs;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG37 + (i * ANT_OFFSET), Reg_EnabledCoeffs.val);
		}
	}

	return;
}

void bbic_set_lms_cal_mode (uint8_t AntMsk, BOOL NlmsCalMode)
{
	int i;
	RegPhyRxtdAnt0PhyRxtdAntReg25_u Reg_CalMode;

	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i) 
		{
			Reg_CalMode.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 + (i * ANT_OFFSET));
			Reg_CalMode.bitFields.iqEqCalMode = NlmsCalMode;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 + (i * ANT_OFFSET), Reg_CalMode.val);
		}
	}
	return;
}

void bbic_set_bypass_ram_coeffs (uint8_t AntMsk, BOOL BypassRamMode)
{
	int i;
	RegPhyRxtdAnt0PhyRxtdAntReg25_u Reg_CalMode;

	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i) 
		{
			Reg_CalMode.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 + (i * ANT_OFFSET));
			Reg_CalMode.bitFields.iqEqBypassRam = BypassRamMode;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG25 + (i * ANT_OFFSET), Reg_CalMode.val);
		}
	}
	return;
}
void bbic_start_lms_cycle (uint8_t AntMsk)
{
	int i;
	RegPhyRxtdAnt0PhyRxtdAntReg39_u Reg_CalMode;

	Reg_CalMode.val = 0;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG39 + (i * ANT_OFFSET));
			Reg_CalMode.bitFields.iqLmsStartWork = 1;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG39 + (i * ANT_OFFSET), Reg_CalMode.val);
		}
	}
	return;
}

void bbic_adc_onoff ( BOOL AdcStatus)
{
	RegPhyRxTdPhyRxtdReg18_u Reg_AdcStatus;
	Reg_AdcStatus.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG18);
	Reg_AdcStatus.bitFields.adcActiveReg = AdcStatus;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG18, Reg_AdcStatus.val );
	return;
}

void bbic_enable_fir(uint8_t AntMsk, BOOL IsEnabled)
{
	RegPhyRxtdAnt0PhyRxtdAntReg01_u Reg_FirBypass;

	int i;


	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_FirBypass.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG01 + (i * ANT_OFFSET));
			Reg_FirBypass.bitFields.iqMismatchBypassEq = 1 - IsEnabled;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG01 + (i * ANT_OFFSET), Reg_FirBypass.val);
		}
	}
	return;
}


void bbic_set_iq_delays(uint8_t AntMsk, uint8_t EqLength_q, uint8_t IqFirDelay )
{
	RegPhyRxtdAnt0PhyRxtdAntReg03_u Reg_QDelay;
	RegPhyRxtdAnt0PhyRxtdAntReg69_u Reg_IqFirDelay;

	int i;

	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{

			Reg_QDelay.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG03 + (i * ANT_OFFSET));
			Reg_QDelay.bitFields.iqMismatchEqLengthQ = EqLength_q;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG03 + (i * ANT_OFFSET), Reg_QDelay.val);
			Reg_IqFirDelay.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG69 + (i * ANT_OFFSET));
			Reg_IqFirDelay.bitFields.iqFirDelay = IqFirDelay;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG69 + (i * ANT_OFFSET), Reg_IqFirDelay.val);
		}
	}
	return;

}



void bbic_get_fir_coeffs(uint8_t AntMsk, uint32_t pCoeffs_rtrn[ANT_NUM][FIR_COEFFS_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg29_u Reg_FirCoeff;


	int j, i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			for (j=0; j<FIR_COEFFS_NUM; j++)
			{
				//Reg_FirCoeff.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG10 + i * ANT_OFFSET + j * TAP_OFFSET);
				//pCoeffs[i][j]= Reg_FirCoeff.bitFields.iqEqTap0;
			//}
				Reg_FirCoeff.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG29 + (i * ANT_OFFSET)  + (j * TAP_OFFSET));
				pCoeffs_rtrn[i][j]= Reg_FirCoeff.bitFields.iqEqUsedTap0;

			/*
			 Reg_FirCoeff0.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG10 + i * ANT_OFFSET );
			 pCoeffs[i][0]= Reg_FirCoeff0.bitFields.iqEqTap0;
			 Reg_FirCoeff1.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG11 + i * ANT_OFFSET );
			 pCoeffs[i][1]= Reg_FirCoeff1.bitFields.iqEqTap1;
			 Reg_FirCoeff2.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG12 + i * ANT_OFFSET );
			 pCoeffs[i][2]= Reg_FirCoeff2.bitFields.iqEqTap2;
			 Reg_FirCoeff3.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG13 + i * ANT_OFFSET );
			 pCoeffs[i][3]= Reg_FirCoeff3.bitFields.iqEqTap3;
			 Reg_FirCoeff4.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG14 + i * ANT_OFFSET );
			 pCoeffs[i][4]= Reg_FirCoeff4.bitFields.iqEqTap4;
			 Reg_FirCoeff5.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG15 + i * ANT_OFFSET );
			 pCoeffs[i][5]= Reg_FirCoeff5.bitFields.iqEqTap5;
			 Reg_FirCoeff6.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG16 + i * ANT_OFFSET );
			 pCoeffs[i][6]= Reg_FirCoeff6.bitFields.iqEqTap6;
			 Reg_FirCoeff7.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG17 + i * ANT_OFFSET );
			 pCoeffs[i][7]= Reg_FirCoeff7.bitFields.iqEqTap7;
			 Reg_FirCoeff8.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG18 + i * ANT_OFFSET );
			 pCoeffs[i][8]= Reg_FirCoeff8.bitFields.iqEqTap8;
			 Reg_FirCoeff9.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG19 + i * ANT_OFFSET );
			 pCoeffs[i][9]= Reg_FirCoeff9.bitFields.iqEqTap9;
			 Reg_FirCoeff10.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG1A + i * ANT_OFFSET );
			 pCoeffs[i][10]= Reg_FirCoeff10.bitFields.iqEqTap10;
			 Reg_FirCoeff11.val =BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG1B + i * ANT_OFFSET );
			 pCoeffs[i][11]= Reg_FirCoeff11.bitFields.iqEqTap11;
			*/
			}
		}

	}
	return;

}

void bbic_set_fir_coeffs_to_progmodel(uint8_t AntMsk, uint32_t pCoeffs[ANT_NUM][FIR_COEFFS_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg10_u Reg_FirCoeff;

	int j, i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			for (j=0; j<FIR_COEFFS_NUM; j++)
			{
				Reg_FirCoeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG10 + (i * ANT_OFFSET) + (j * TAP_OFFSET));
				Reg_FirCoeff.bitFields.iqEqTap0 = pCoeffs[i][j];
				BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG10 + (i * ANT_OFFSET) + (j * TAP_OFFSET), Reg_FirCoeff.val);

			}
		}
	}
}

void bbic_get_fir_coeffs_from_progmodel(uint8_t AntMsk, uint32_t pCoeffs_rtrn[ANT_NUM][FIR_COEFFS_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg10_u Reg_FirCoeff;

	int j, i;
		for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				for (j=0; j<FIR_COEFFS_NUM; j++)
				{
					Reg_FirCoeff.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG10 + (i * ANT_OFFSET) + (j * TAP_OFFSET));
					pCoeffs_rtrn[i][j] = Reg_FirCoeff.bitFields.iqEqTap0;

				}
			}
		}
}

void bbic_write_ram_fir_coeffs(uint8_t AntMsk, uint32_t pCoeffs[ANT_NUM][FIR_COEFFS_NUM], uint8_t iBW, uint8_t PgcSetting)
{
	int i, coef;
	uint32_t Two_coeffs_value; //coef 0/2/4/6/8/20 LSB, coef 1/3/5/7/9/11 MSB 

	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			for (coef = 0; coef < FIR_COEFFS_NUM; coef += 2)
			{
				Two_coeffs_value = (pCoeffs[i][coef + 1] << 10) + pCoeffs[i][coef] ; //coef 0/2/4/6/8/20 LSB, coef 1/3/5/7/9/11 MSB 
				//BBIC_Reg_Write(RAM_FIREQ_BASE_ADDRESS + (i * ANT_OFFSET) + (PgcSetting * BW_NUM * 4 * FIR_COEFFS_NUM / 2) + ((3-iBW) * 4 * FIR_COEFFS_NUM / 2) + 4*(coef / 2) , Two_coeffs_value);z
				BBIC_Reg_Write(RAM_FIREQ_BASE_ADDRESS + (i * ANT_OFFSET) + ((3-iBW) * PGC_GAIN_NUM * 4 * FIR_COEFFS_NUM / 2) + (PgcSetting * 4 * FIR_COEFFS_NUM / 2) + 4*(coef / 2) , Two_coeffs_value);

			}
			
		}
	}
}
void bbic_read_ram_fir_coeffs(uint8_t AntMsk, uint32_t pCoeffs_rtrn[ANT_NUM][FIR_COEFFS_NUM], uint8_t iBW, uint8_t PgcSetting)
{
	int i, coef;
	uint32_t Two_coeffs_value; //coef 0/2/4/6/8/20 LSB, coef 1/3/5/7/9/11 MSB 

	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			for (coef = 0; coef < FIR_COEFFS_NUM; coef += 2)
			{
				
				//Two_coeffs_value = BBIC_Reg_Read(RAM_FIREQ_BASE_ADDRESS + i * (ANT_OFFSET)+(PgcSetting * BW_NUM * 4 * FIR_COEFFS_NUM / 2) + ((3-iBW) * 4 * FIR_COEFFS_NUM / 2) + 4*(coef / 2));
				Two_coeffs_value = BBIC_Reg_Read(RAM_FIREQ_BASE_ADDRESS + (i * ANT_OFFSET) + ((3-iBW) * PGC_GAIN_NUM * 4 * FIR_COEFFS_NUM / 2) + (PgcSetting * 4 * FIR_COEFFS_NUM / 2) + 4*(coef / 2));
				
				pCoeffs_rtrn[i][coef] = (Two_coeffs_value & 0x3FF);
				pCoeffs_rtrn[i][coef + 1] = Two_coeffs_value >> 10;
				 
			}
		}
	}
}

void bbic_get_lms_error_indicator(uint8_t AntMsk, uint32_t pErrorInd_rtrn[ANT_NUM], uint8_t pErrorValid_rtrn[ANT_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg3A_u Reg_LmsErrorIndicator;

	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{

			Reg_LmsErrorIndicator.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3A + i * ANT_OFFSET);
			pErrorValid_rtrn[i] = Reg_LmsErrorIndicator.bitFields.iqLmsErrorValid;
			if ( pErrorValid_rtrn[i] ==1)
			{
				pErrorInd_rtrn[i] = Reg_LmsErrorIndicator.bitFields.iqLmsErrorIndicator;
			}
			else
			{
				pErrorInd_rtrn[i] = 0;
			}

		}
		else
		{
			pErrorInd_rtrn[i] = 0;
		}

	}
		return;
}

void bbic_lms_reset(void)
{
	RegPhyRxTdPhyRxtdReg00_u Reg_SwReset;
	//uint32 temp;

	Reg_SwReset.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG00);
	//temp = (Reg_SwReset.bitFields.swResetNReg % 8)/4;
	Reg_SwReset.bitFields.swResetNReg = Reg_SwReset.bitFields.swResetNReg | 4;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, Reg_SwReset.val);
	//Reg_SwReset.bitFields.swResetNReg += 4;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, Reg_SwReset.val);
	return;
}

void bbic_enable_clk_bypass(BOOL isEnabled)
{
	RegPhyRxTdPhyRxtdReg2F_u Reg_Rx_Clk;
	RegPhyTxGclkControl_u Reg_Tx_Clk;

	Reg_Rx_Clk.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG2F );
	Reg_Tx_Clk.val = BBIC_Reg_Read(REG_PHY_TX_GCLK_CONTROL);
	if (isEnabled == TRUE)
	{
		Reg_Rx_Clk.bitFields.gclkBypassEn = Reg_Rx_Clk.bitFields.gclkBypassEn | 0x20000;
		Reg_Tx_Clk.bitFields.gclkEnBypass = Reg_Tx_Clk.bitFields.gclkEnBypass | 0x80;
	}
	else
	{
		Reg_Rx_Clk.bitFields.gclkBypassEn = Reg_Rx_Clk.bitFields.gclkBypassEn & ~(0x20000);
		Reg_Tx_Clk.bitFields.gclkEnBypass = Reg_Tx_Clk.bitFields.gclkEnBypass & ~(0x80);
	}
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG2F, Reg_Rx_Clk.val);
	BBIC_Reg_Write(REG_PHY_TX_GCLK_CONTROL, Reg_Tx_Clk.val);
}

void bbic_mac_emu_set_tcr0_common(void)
{
	RegPhyTxMacEmuEmuPhyTxPhyTcr00_u Reg_Tcr0_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcr01_u Reg_Tcr0_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcr02_u Reg_Tcr0_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcr03_u Reg_Tcr0_3;

	Reg_Tcr0_0.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_0);
	Reg_Tcr0_1.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_1);
	Reg_Tcr0_2.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_2);
	Reg_Tcr0_3.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_3);

}

void bbic_mac_emu_set_inter_packet_gap(uint8_t Gap)
{
	RegPhyTxMacEmuTxMacEmuReg0E_u Reg_Mac_Emu_Ipg_Gap;

	Reg_Mac_Emu_Ipg_Gap.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG0E);
	Reg_Mac_Emu_Ipg_Gap.bitFields.emuIpgDly = (Gap-2)*640;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG0E, Reg_Mac_Emu_Ipg_Gap.val);

	return;
}
void bbic_mac_emu_activate( )
{
	RegPhyTxMacEmuTxMacEmuReg00_u Reg_Activate_Mac_Emu;
	RegPhyTxMacEmuMacEmuMode_u    Reg_Mac_Emu_Mode;
    RegPhyTxSpareRegs0_u        Reg_TPCACC_pulse;  /*REG_PHY_TX_SPARE_REGS_0 0xCC4 */

	Reg_Mac_Emu_Mode.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_MAC_EMU_MODE);
	Reg_Mac_Emu_Mode.bitFields.macEmuMode = 1;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_MAC_EMU_MODE, Reg_Mac_Emu_Mode.val);
	Reg_Activate_Mac_Emu.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG00 );
	Reg_Activate_Mac_Emu.bitFields.emuRun =1;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG00, Reg_Activate_Mac_Emu.val);
	
	// Give pulse to enable APCACC (and take digital gain)
	Reg_TPCACC_pulse.val = BBIC_Reg_Read(REG_PHY_TX_SPARE_REGS_0);
#if defined(ENET_INC_ARCH_WAVE600B) || defined(ENET_INC_ARCH_WAVE600D2)
	Reg_TPCACC_pulse.bitFields.spareGpr00 = 1;
#else
	Reg_TPCACC_pulse.bitFields.spareGpr0 = 1;
#endif
    BBIC_Reg_Write(REG_PHY_TX_SPARE_REGS_0, Reg_TPCACC_pulse.val);

	return;
}

void bbic_mac_emu_deactivate( )
/* deactivate mac emulator endless packect mode, (if sent finit number of packects, just wait for end)*/
{
	RegPhyTxMacEmuTxMacEmuReg00_u Reg_Activate_Mac_Emu;
	RegPhyTxMacEmuEndlessType_u Reg_Endless_Type;
	RegPhyTxMacEmuMacEmuMode_u    Reg_Mac_Emu_Mode;

	uint8_t maxTimeExceeded = 0;
	uint8_t txDone = 0;
	uint16_t Timeout = 5400;//5.4miliS
	uint32_t startTime = 0;
	uint16_t endTime = 0;

	Reg_Endless_Type.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_ENDLESS_TYPE  );
	Reg_Endless_Type.bitFields.endlessPackets = 0;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_ENDLESS_TYPE, Reg_Endless_Type.val);

	startTime = BBIC_Time_Stamp(0,startTime);
	while (!txDone && !maxTimeExceeded) // wait for txDone
	{
		Reg_Activate_Mac_Emu.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG00);
		txDone = Reg_Activate_Mac_Emu.bitFields.emuTxDone;
		endTime = BBIC_Time_Stamp(1,startTime);
		if (endTime > Timeout)
		{
			maxTimeExceeded = 1;
		}
	}
	if (txDone == 1)
	{
		Reg_Mac_Emu_Mode.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_MAC_EMU_MODE);
		Reg_Mac_Emu_Mode.bitFields.macEmuMode = 0;
		BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_MAC_EMU_MODE, Reg_Mac_Emu_Mode.val);
	}
	return;
}

void bbic_mac_emu_set_endless(BOOL IsEnabled)
{
	RegPhyTxTxBeReg11_u Reg_Set_Mac_Emu_Endless;
	RegPhyTxMacEmuEndlessType_u Reg_Endless_Data;
//HW asked to remove
	Reg_Set_Mac_Emu_Endless.val = BBIC_Reg_Read(REG_PHY_TX_TX_BE_REG_11);
	Reg_Set_Mac_Emu_Endless.bitFields.txEndlessMode = IsEnabled;
	BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_11 , Reg_Set_Mac_Emu_Endless.val);

	Reg_Endless_Data.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_ENDLESS_TYPE  );
	Reg_Endless_Data.bitFields.endlessData = IsEnabled;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_ENDLESS_TYPE, Reg_Endless_Data.val);

	return;
}

void bbic_mac_emu_set_scrambler(BOOL IsEnabled)
{
	RegPhyTxTxBeReg11_u Reg_Set_Mac_Emu_Scrambler;

	Reg_Set_Mac_Emu_Scrambler.val = BBIC_Reg_Read(REG_PHY_TX_TX_BE_REG_11);
	Reg_Set_Mac_Emu_Scrambler.bitFields.scrInitSel = IsEnabled;
	BBIC_Reg_Write(REG_PHY_TX_TX_BE_REG_11, Reg_Set_Mac_Emu_Scrambler.val);

	return;
}

//void SetMacEmulatorEndless(void)
//{
//	READ_MODIFY_WRITE(PHY_TX_BASE_ADDR, PHY_TX_TX_BE_REG_11_TX_ENDLESS_MODE_ADDR,PHY_TX_TX_BE_REG_11_TX_ENDLESS_MODE_SHIFT,PHY_TX_TX_BE_REG_11_TX_ENDLESS_MODE_MASK,1);
//
//}

void bbic_set_tcr_for_cal(uint8_t AntMsk, uint8_t RfPow, uint16_t BW, uint32_t DefaultTCRs[3][4], BOOL ReadFromPM)
{
	RegPhyTxMacEmuEmuPhyTxPhyTcr00_u Reg_Tcr_0_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcr01_u Reg_Tcr_0_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcr02_u Reg_Tcr_0_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcr03_u Reg_Tcr_0_3;
	RegPhyTxMacEmuEmuPhyTxPhyTcr10_u Reg_Tcr_1_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcr11_u Reg_Tcr_1_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcr12_u Reg_Tcr_1_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcr13_u Reg_Tcr_1_3;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU0_u Reg_Tcr_2_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU1_u Reg_Tcr_2_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU2_u Reg_Tcr_2_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU3_u Reg_Tcr_2_3;

	int i;
	uint16_t TwoBitAntMask = 0;

	if (ReadFromPM)
	{
		Reg_Tcr_0_0.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_0);
		Reg_Tcr_0_1.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_1);
		Reg_Tcr_0_1.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_2);
		Reg_Tcr_0_1.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_3);
		Reg_Tcr_1_0.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_0);
		Reg_Tcr_1_1.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_1);
		Reg_Tcr_1_2.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_2);
		Reg_Tcr_1_3.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_3);
		Reg_Tcr_2_0.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_0);
		Reg_Tcr_2_1.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_1);
		Reg_Tcr_2_2.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_2);
		Reg_Tcr_2_3.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_3);

	}
	else //only what's needed!
	{
		Reg_Tcr_0_0.val = DefaultTCRs[0][0];
		Reg_Tcr_0_1.val = DefaultTCRs[0][1];
		Reg_Tcr_0_2.val = DefaultTCRs[0][2];
		Reg_Tcr_0_3.val = DefaultTCRs[0][3];
		Reg_Tcr_1_0.val = DefaultTCRs[1][0];
		Reg_Tcr_1_1.val = DefaultTCRs[1][1];
		Reg_Tcr_1_2.val = DefaultTCRs[1][2];
		Reg_Tcr_1_3.val = DefaultTCRs[1][3];
		Reg_Tcr_2_0.val = DefaultTCRs[2][0];
		Reg_Tcr_2_1.val = DefaultTCRs[2][1];
		Reg_Tcr_2_2.val = DefaultTCRs[2][2];
		Reg_Tcr_2_3.val = DefaultTCRs[2][3];

	}
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			TwoBitAntMask = TwoBitAntMask + (i+1)*(i+1)*3;
		}
	}
	Reg_Tcr_0_0.val = TwoBitAntMask;
	Reg_Tcr_0_1.val = Reg_Tcr_0_1.val & 0xFFFFF300;
	Reg_Tcr_0_1.val = Reg_Tcr_0_1.val + (BW<<10) + RfPow;
	Reg_Tcr_1_2.val = Reg_Tcr_1_2.val |0x3F;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_0, Reg_Tcr_0_0.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_1, Reg_Tcr_0_1.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_2, Reg_Tcr_0_2.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_3, Reg_Tcr_0_3.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_0, Reg_Tcr_1_0.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_1, Reg_Tcr_1_1.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_2, Reg_Tcr_1_2.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_3, Reg_Tcr_1_3.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_0, Reg_Tcr_2_0.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_1, Reg_Tcr_2_1.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_2, Reg_Tcr_2_2.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_3, Reg_Tcr_2_3.val);

	return;

}


void bbic_mac_emu_set_tcr(uint8_t AntMsk, uint8_t RfPow, uint16_t BW, uint8_t packet_format, uint8_t mcs, uint8_t nss, uint32_t packet_length, uint8_t ldpc )
//num2str(packet_format),',',num2str(mcs),',',num2str(nss),',',num2str(hbmode_to_bbs_rep(hbmode+1)),',',num2str(bw),',',num2str(actual_length_bytes),',',num2str(he_ltf_gi),',',power,',',ant_select,',',cdd_ant1,',',cdd_ant2,',',cdd_ant3,',',bfmode,',',flags});
// packet format: 0 - legacy,1 - 11b,2 - HT,3 - VHT,4 - HE (SU),5 - HE (EX SU),6 - HE (MU – trigger based) (not supported by the Mac), 7- HE (MU – DL) 
// bw:0-20,1-40,2-80,3-160
{
	RegPhyTxMacEmuEmuPhyTxPhyTcr00_u Reg_Tcr_0_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcr01_u Reg_Tcr_0_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcr02_u Reg_Tcr_0_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcr03_u Reg_Tcr_0_3;
	RegPhyTxMacEmuEmuPhyTxPhyTcr10_u Reg_Tcr_1_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcr11_u Reg_Tcr_1_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcr12_u Reg_Tcr_1_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcr13_u Reg_Tcr_1_3;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU0_u Reg_Tcr_2_0;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU1_u Reg_Tcr_2_1;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU2_u Reg_Tcr_2_2;
	RegPhyTxMacEmuEmuPhyTxPhyTcrU3_u Reg_Tcr_2_3;

	int i;
	uint16_t TwoBitAntMask = 0;
	uint16_t rate = 0;

	// set antenna mask
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			TwoBitAntMask = TwoBitAntMask + (1<<(2*i))*3;
			//printf("TwoBitAntMask  %d AntMsk = %d\n", TwoBitAntMask, AntMsk);
		}
	}
	
// packet format: 0 - legacy,1 - 11b,2 - HT,3 - VHT,4 - HE (SU),5 - HE (EX SU),6 - HE (MU – trigger based) (not supported by the Mac), 7- HE (MU – DL) 
	// set  rate accrioding to phy mode and mcs and nss
switch (packet_format) {
	case 0 : rate=mcs;  break; // legacy [2:0] – mcs, [7:3] – 0
	case 1 : rate=mcs;  break;// 11b	[1:0] – rate, [2] – long preamble=0		[7:3] – 0
	case 2 : rate = mcs+nss*8 ; break;//ht rate is mcs[2:0]*nss is mcs[4:3], 	*mcs32 is when mcs[5:0]=32
	case 3 : rate = mcs+nss*16; break;//vht	[3:0] – mcs	[6:4] – Nss	[7] - 0
	default: rate = mcs+nss*16; //He	[3:0] – mcs	[6:4] – Nss	[7] - DCM
}
//	if (packet_format==0) rate=mcs;  // legacy [2:0] – mcs, [7:3] – 0
//	if (packet_format==1) rate=mcs;  // 11b	[1:0] – rate, [2] – long preamble=0		[7:3] – 0
//	if (packet_format==2) rate = mcs+nss*8 ; //ht rate is mcs[2:0]*nss is mcs[4:3], 	*mcs32 is when mcs[5:0]=32
//	if (packet_format==3) rate = mcs+nss*16; //vht	[3:0] – mcs	[6:4] – Nss	[7] - 0
//	if(packet_format>=4) rate = mcs+nss*16; //He	[3:0] – mcs	[6:4] – Nss	[7] - DCM
	    
//TCR0 (common) (bus state=1)			
//Antenna selection	16	[15:0]	2 bit per antenna up to 8 antennas
//Ant boost	16	[31:16]	2 bit per antenna up to 8 antennas 
//RF power	8	[39:32]	
//Tx loop mode	2	[41:40]	Enable to update TX power w/o changing the phase so BF is not impacted 0: closed loop 1: open loop 2..3: reserved
//CBW	2	[43:42]	0-20MHz 1-40MHz  2-80Mhz  3-160Mhz
//PHY MODE	3	[46:44]	Packet type: 0 - legacy  1 - 11b 2 - HT 3 - VHT  4 - HE (SU) 5 - HE (EX SU) 6 - HE (MU – trigger based) (not supported by the Mac)  7- HE (MU – DL) 
//HE_fullband_mu	1	[47]	Valid only in packet_type=7 If 1 then MU else OFDMA
//HE puncturing map	8	[55:48]	Valid only in HE MU DL, bit per band. Map according to 160MHz
//Force_tx	1	[56]	If force_tx then ignore disable tx instruction from mac 
//Reserved	7	[63:57]	TBD Reserved	32	[95:64]	TBD  Genrisc reserved	32	[127:96]	Genrisc reserved

//		TCR1 (common) (bus state=1)
//BSS color	6	[5:0]   Custom_BF	2	[7:6] N_HELTF	3	[10:8]  UL/DL	1	[11] TXOP PS NOT ALLOWED	1	[12]  Expect RCR	1	[13]  BW change	1	[14]  Txop_duration_from_mac	1	[15]
//Aggregate	1	[16] Dyn BW	1	[17]  Mu phy ndp	1	[18] Mu training	1	[19] Reserved	4	[23:20]  He CP	2	[25:24]  HE LTF	2	[27:26]  scp	1	[28] smoothing	1  [29] Not sounding	1	[30]  Stbc 	1	[31]
// Spatial reuse	16	[47:32] Trigger based data	16	[63:48] Group id	6	[69:64] reserved	2	[71:70]  HE SIGB RATE	8	[79:72] Mac_duration	15	[94:80]  Not in use	1	[95]GenRisc reserved	32	[127:96]
// TCR 2 (user 0)
//PSDU length	22	[21:0]
//Not in use	10	[31:22] //Sub band	3	[34:32] Start RU	4	[38:35] RU size	3	[41:39]  ldpc	1	[42]  TxBF	1	[43]
//PSDU rate	8	[51:44]
//Relative user power	4	[55:52] reserved	4	[59:56] Packet extension	2	[61:60] reserved	2	[63:62] Station AID	11	[74:64] reserved	5	[79:75] BF Index	8	[87:80 reserved	8	[95:88]


		
	Reg_Tcr_0_0.val = TwoBitAntMask;
	Reg_Tcr_0_1.val = 0x00FF0000+(packet_format<<12) + (BW<<10) + RfPow;
	Reg_Tcr_0_2.val = 0x0;
	Reg_Tcr_0_3.val = 0x0;
	Reg_Tcr_1_2.val = Reg_Tcr_1_2.val |0x3F;
	Reg_Tcr_1_0.val = 0x68008000;
	Reg_Tcr_1_1.val = 0xF3E80005;
	Reg_Tcr_1_2.val = 0x002A003F;
	Reg_Tcr_1_3.val = 0x0;
	Reg_Tcr_2_0.val = (packet_length);
	Reg_Tcr_2_1.val = (rate<<(44-32))+ (ldpc<<(42-32)) +((BW+3)<<(39-32));
	Reg_Tcr_2_2.val = (0x01);
	Reg_Tcr_2_3.val = 0;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_0, Reg_Tcr_0_0.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_1, Reg_Tcr_0_1.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_2, Reg_Tcr_0_2.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_3, Reg_Tcr_0_3.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_0, Reg_Tcr_1_0.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_1, Reg_Tcr_1_1.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_2, Reg_Tcr_1_2.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR1_3, Reg_Tcr_1_3.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_0, Reg_Tcr_2_0.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_1, Reg_Tcr_2_1.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_2, Reg_Tcr_2_2.val);
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR_U_3, Reg_Tcr_2_3.val);

	return;

}



void bbic_mac_emu_set_num_of_packets(uint16_t num_of_packets)
/* set number of packets, if 0 - endless packets.*/
{
	RegPhyTxMacEmuEndlessType_u Reg_Endless_Data;
	RegPhyTxMacEmuTxMacEmuReg01_u Reg_Nom_of_Packets;
if (num_of_packets==0) {
	// endless, write 1 to num of packets, enable endless mode
	Reg_Nom_of_Packets.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG01);
	Reg_Nom_of_Packets.bitFields.emuNumOfPackets = 1;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG01, Reg_Nom_of_Packets.val);
	Reg_Endless_Data.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_ENDLESS_TYPE  );
	Reg_Endless_Data.bitFields.endlessPackets = 1;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_ENDLESS_TYPE, Reg_Endless_Data.val);
}
else {
	Reg_Nom_of_Packets.val = BBIC_Reg_Read(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG01);
	Reg_Nom_of_Packets.bitFields.emuNumOfPackets = num_of_packets;
	BBIC_Reg_Write(REG_PHY_TX_MAC_EMU_TX_MAC_EMU_REG01, Reg_Nom_of_Packets.val);
}   

	return;

}


void bbic_set_difi2_gain(uint8_t AntMsk, uint8_t DifiGain[ANT_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg60_u Reg_Difi_Gain; // PHY_RXTD_ANT3_PHY_RXTD_ANT_REG60_DIFI2_OP_GAIN_DB
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Difi_Gain.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG60 + i*(ANT_OFFSET));
			Reg_Difi_Gain.bitFields.difi2OpGainDb = DifiGain[i];
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG60 + i*(ANT_OFFSET), Reg_Difi_Gain.val);
		}
	}
}

void bbic_read_difi2_gain(uint8_t AntMsk, uint8_t pDifiGain_rtrn[ANT_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg60_u Reg_Difi_Gain; // PHY_RXTD_ANT3_PHY_RXTD_ANT_REG60_DIFI2_OP_GAIN_DB
		int i;
		for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				Reg_Difi_Gain.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG60 + i*(ANT_OFFSET));
				pDifiGain_rtrn[i] = Reg_Difi_Gain.bitFields.difi2OpGainDb;
			}
		}
}


void bbic_set_dc_cancellation_mode(uint8_t AntMsk, BOOL DcCancelStatus)
{
	RegPhyRxtdAnt0PhyRxtdAntReg3B_u Reg_DC_Cancel; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_DC_Cancel.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B + i*(ANT_OFFSET));
			Reg_DC_Cancel.bitFields.rxDifi2DcCancelEnable = DcCancelStatus;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B + i*(ANT_OFFSET), Reg_DC_Cancel.val);
		}
	}
}

void bbic_enable_difi2_gain_agc(uint8_t AntMsk, BOOL EnableDifi2Control)
{
	RegPhyRxtdAnt0PhyRxtdAntReg3B_u Reg_Difi2_RSSI_Gain_Sel; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_Difi2_RSSI_Gain_Sel.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B + i*(ANT_OFFSET));
			Reg_Difi2_RSSI_Gain_Sel.bitFields.rxDifi2RssiGainHwSelect = EnableDifi2Control;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B + i*(ANT_OFFSET), Reg_Difi2_RSSI_Gain_Sel.val);
		}
	}
}

void bbic_enable_dc_accelerator(uint8_t AntMsk, BOOL EnableDcAccelerator)
{
	RegPhyRxtdAnt0PhyRxtdAntReg3B_u Reg_DC_HW_Accelerator; // REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B;
	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & 1<<i)
		{
			Reg_DC_HW_Accelerator.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B + i*(ANT_OFFSET));
			Reg_DC_HW_Accelerator.bitFields.rxDifi2DcCancelHwSelect = EnableDcAccelerator;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3B + i*(ANT_OFFSET), Reg_DC_HW_Accelerator.val);
		}
	}
}

 void accum_dc_to_uv(CorrResults_t pAccumResults[ANT_NUM], uint32_t NOS[ANT_NUM], int16_t pDCmV_I_rtrn[ANT_NUM], int16_t pDCmV_Q_rtrn[ANT_NUM])
 {
	 int i;

	 for (i=0; i<ANT_NUM; i++)
	 {

		 pDCmV_I_rtrn[i] = ((1000 * (pAccumResults[i].II.dcVal / NOS[i])) >> (ADC_BITS - 1)) * 1000 * ADC_FS / 4;// divided by 4 because ADC_FS is defined as 2 * (Actual ADC fullscale(1.5V)) and the correct formula includes another division by 2
		 pDCmV_Q_rtrn[i] = ((1000 * (pAccumResults[i].QQ.dcVal / NOS[i])) >> (ADC_BITS - 1)) * 1000 * ADC_FS / 4;

	 }

 }

void bbic_write_dc_values_to_pm(uint8_t AntMsk, int16_t pDcRegsI[ANT_NUM], int16_t pDcRegsQ[ANT_NUM])
{
	int i;
	RegPhyRxtdAnt0PhyRxtdAntReg3C_u Reg_Ant_dc; //REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3C


	for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				Reg_Ant_dc.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3C + i*(ANT_OFFSET));
				Reg_Ant_dc.bitFields.difi2DcCancelI = -pDcRegsI[i];
				Reg_Ant_dc.bitFields.difi2DcCancelQ = -pDcRegsQ[i];
				BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3C + i*(ANT_OFFSET), Reg_Ant_dc.val);
			}
		}
}

void bbic_read_dc_values_from_pm(uint8_t AntMsk, int16_t pDcRegsI_rtrn[ANT_NUM], int16_t pDcRegsQ_rtrn[ANT_NUM])
{
	int i;
	RegPhyRxtdAnt0PhyRxtdAntReg3C_u Reg_Ant_dc; //REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3C
	int Bits = 14;

	for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & (1<<i))
			{
				Reg_Ant_dc.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG3C + i * (ANT_OFFSET));
				pDcRegsI_rtrn[i] = Reg_Ant_dc.bitFields.difi2DcCancelI;
				pDcRegsQ_rtrn[i] = Reg_Ant_dc.bitFields.difi2DcCancelQ;
				
				if (Reg_Ant_dc.bitFields.difi2DcCancelI > (1<<(Bits-1)))
				{
					SIGN_EXTEND(pDcRegsI_rtrn[i], Bits);
				}
				if (Reg_Ant_dc.bitFields.difi2DcCancelQ > (1<<(Bits-1)))
				{
					SIGN_EXTEND(pDcRegsQ_rtrn[i], Bits);
				}
				
				pDcRegsI_rtrn[i] = -pDcRegsI_rtrn[i];
				pDcRegsQ_rtrn[i] = -pDcRegsQ_rtrn[i];

			}
		}
}



//DC verification
uint8_t  bbic_verify_DC_results(uint8_t AntMsk, int16_t pDcRegsI[ANT_NUM], int16_t pDcRegsQ[ANT_NUM], CorrOF_t pAccumOF[ANT_NUM], DCVerificationResults_t pDCVer_rtrn[ANT_NUM], uint16_t SuccessVal_uV)

{
	int i;
	int32_t SuccessVal;
	uint8_t ValidMask = 0;
	uint8_t ValidMask_I = 0;
	uint8_t ValidMask_Q = 0;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			SuccessVal = (((SuccessVal_uV * 4 / ADC_FS) << (ADC_BITS - 1)) / 1000000) ;
			if (((pDcRegsI[i] <  SuccessVal) && (pDcRegsI[i] > - SuccessVal)) && (pAccumOF[i].II == 0))
			{
				pDCVer_rtrn[i].DcSuccessI = TRUE;
				ValidMask_I |= (1 << i);
			}
			else
			{
				pDCVer_rtrn[i].DcSuccessI = FALSE;
			}

			if (((pDcRegsQ[i] <  SuccessVal) && (pDcRegsQ[i] > - SuccessVal)) && (pAccumOF[i].QQ == 0))
			{
				pDCVer_rtrn[i].DcSuccessQ = TRUE;
				ValidMask_Q |= (1 << i);

			}
			else
			{
				pDCVer_rtrn[i].DcSuccessQ = FALSE;

			}
		}
	}
	ValidMask = ValidMask_I & ValidMask_Q;
	return ValidMask;

}


void  bbic_write_ram_DC(uint8_t AntMsk, int16_t pDcRegsI[ANT_NUM], int16_t pDcRegsQ[ANT_NUM], DCVerificationResults_t pDCVer[ANT_NUM], uint8_t Pgc_Gain_dB, uint8_t lnagainidx, uint8_t iBW)
{
	int i;
	uint8_t Bits = 14;
	uint32_t ReadData;
	uint32_t Value =0;
	uint32_t I_Value = 0;
	uint32_t Q_Value = 0;

	for (i=0; i<ANT_NUM; i++)
		{
			if (AntMsk & 1<<i)
			{
				ReadData = BBIC_Reg_Read(RAM_RX_DC_BASE_ADDRESS + i*(ANT_OFFSET) + (TOTAL_GAINS_NUM * iBW + ((Pgc_Gain_dB / 2) * LNA_DC_RAM_GAIN_NUM) +(lnagainidx))*RXDC_RAM_REG_WIDTH);
				if (pDCVer[i].DcSuccessI) // I in MSBs, Q in LSBs
				{
					I_Value = (((int32_t)(-pDcRegsI[i])) << Bits); //negative values are added, thus DC is subtracted.
				}
				else
				{
					I_Value = ReadData & (((1<<Bits)-1)<<Bits);
				}

				if (pDCVer[i].DcSuccessQ)
				{
					Q_Value = (-pDcRegsQ[i]) & ((1<< Bits) - 1);
				}
				else
				{
					Q_Value = ReadData & ((1<< Bits) - 1);
				}
				Value = I_Value | Q_Value;
				BBIC_Reg_Write(RAM_RX_DC_BASE_ADDRESS + (i * ANT_OFFSET) + (TOTAL_GAINS_NUM * iBW + ((Pgc_Gain_dB / 2) * LNA_DC_RAM_GAIN_NUM) + (lnagainidx))*RXDC_RAM_REG_WIDTH, Value);


			}
		}
}

void  bbic_read_ram_DC(uint8_t AntMsk, int16_t pDcRegsI_rtrn[ANT_NUM], int16_t pDcRegsQ_rtrn[ANT_NUM], uint8_t Pgc_Gain_dB, uint8_t lnagainidx, uint8_t iBW)
{
	union
	{
		uint32_t val;
		struct
		{
			int32_t Q_DC:14;
			int32_t I_DC:14;
			uint32_t NA:4;
		} ramFields;
	} IQ_DC;

	int i;
	for (i=0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1 << i))
		{
			IQ_DC.val = BBIC_Reg_Read(RAM_RX_DC_BASE_ADDRESS + (i * ANT_OFFSET) + ((TOTAL_GAINS_NUM * iBW) + ((Pgc_Gain_dB / 2) * LNA_DC_RAM_GAIN_NUM) + (lnagainidx))*RXDC_RAM_REG_WIDTH);
			pDcRegsI_rtrn[i] = -(IQ_DC.ramFields.I_DC);
			pDcRegsQ_rtrn[i] = -(IQ_DC.ramFields.Q_DC);
		}

	}
}

void bbic_accum_avg_DC(uint8_t AntMsk, CorrResults_t pAccumResults[ANT_NUM], int16_t pDcRegsI_rtrn[ANT_NUM], int16_t pDcRegsQ_rtrn[ANT_NUM], uint32_t NOS[ANT_NUM])
{
	int i;
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			pDcRegsI_rtrn[i] = (int16_t)(pAccumResults[i].II.dcVal / NOS[i]);
			pDcRegsQ_rtrn[i] = (int16_t)(pAccumResults[i].QQ.dcVal / NOS[i]);
		}
	}
}


void bbic_clear_dc_values_ram(uint8_t AntMsk)
{
	int i, iBW, lnagainidx, pgcgainidx;
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			for (iBW = 0; iBW < BW_NUM; iBW++)
			{
				for (lnagainidx = 0; lnagainidx < LNA_DC_RAM_GAIN_NUM; lnagainidx++)
				{
					for (pgcgainidx = 0; pgcgainidx < PGC_GAIN_NUM; pgcgainidx++)
					{
						BBIC_Reg_Write(RAM_RX_DC_BASE_ADDRESS + i * (ANT_OFFSET)+(TOTAL_GAINS_NUM * iBW + (pgcgainidx * LNA_DC_RAM_GAIN_NUM) + (lnagainidx))*RXDC_RAM_REG_WIDTH, 0);
					}
				}
			}
		}
	}
}

void bbic_set_tx_dc_to_progmodel(uint8_t AntMsk, int16_t DcOffset_i[ANT_NUM], int16_t DcOffset_q[ANT_NUM], uint8_t offset_ser )
{

	RegPhyTxtdAnt0TxDcOffset0_u Reg_Tx_Dc_Offset;	//REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0 0x8
	uint32 diff_offset = REG_PHY_TXTD_ANT0_TX_DC_OFFSET_1 - REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0;

	int i;
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Tx_Dc_Offset.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0 + (diff_offset * offset_ser) + i*(ANT_OFFSET));
			Reg_Tx_Dc_Offset.bitFields.pmAfeOffsetI0 = DcOffset_i[i];
			Reg_Tx_Dc_Offset.bitFields.pmAfeOffsetQ0 = DcOffset_q[i];
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0 + (diff_offset * offset_ser) + i*(ANT_OFFSET),Reg_Tx_Dc_Offset.val);

		}
	}
}

void bbic_get_tx_dc_from_progmodel(uint8_t AntMsk, int16_t pDcOffset_i_rtrn[ANT_NUM], int16_t pDcOffset_q_rtrn[ANT_NUM], uint8_t offset_ser )
{
	RegPhyTxtdAnt0TxDcOffset0_u Reg_Tx_Dc_Offset;	//REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0 0x8
	uint32 diff_offset = REG_PHY_TXTD_ANT0_TX_DC_OFFSET_1 - REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0;

	int i;
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Tx_Dc_Offset.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_DC_OFFSET_0 + (diff_offset * offset_ser) +i*(ANT_OFFSET));
			pDcOffset_i_rtrn[i] = Reg_Tx_Dc_Offset.bitFields.pmAfeOffsetI0;
			pDcOffset_q_rtrn[i] = Reg_Tx_Dc_Offset.bitFields.pmAfeOffsetQ0;
			SIGN_EXTEND(pDcOffset_i_rtrn[i], 12);
			SIGN_EXTEND(pDcOffset_q_rtrn[i], 12);

		}
	}
}

void bbic_set_tx_dpd_bypass(uint8_t AntMsk, BOOL Dpd_bypass)
{
	RegPhyTxtdAnt0TxDpd_u Reg_Tx_Dpd;//REG_PHY_TXTD_ANT0_TX_DPD 0x4

	int i;
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Tx_Dpd.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_DPD + i*(ANT_OFFSET));
			Reg_Tx_Dpd.bitFields.pmDpdBypass = Dpd_bypass;
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TX_DPD + i*(ANT_OFFSET),Reg_Tx_Dpd.val);

		}
	}

}

void bbic_enable_tx_11b_mode(uint8_t AntMsk, BOOL Enable_11b)
{
	RegPhyTxtdAnt0TxMode11BEn_u Reg_Tx_11b_Mode; // REG_PHY_TXTD_ANT0_TX_MODE_11B_EN 0x0
	int i;
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Tx_11b_Mode.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_MODE_11B_EN + i*(ANT_OFFSET));
			Reg_Tx_11b_Mode.bitFields.pmMode11BEn = Enable_11b;
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TX_MODE_11B_EN + i*(ANT_OFFSET), Reg_Tx_11b_Mode.val);

		}
	}
}

void bbic_set_tx_iq_coefficients(uint8_t AntMsk, int16_t Bin, int16_t  a[ANT_NUM], int16_t  b[ANT_NUM] )
{
	int32_t ab_toset[ANT_NUM];
	uint8_t Bits = 10;
	int i;
	if (Bin<0)
	{
		Bin = Bin+512;
	}
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			if (a[i]<0)
			{
				a[i]+= 1024;
			}
			if (b[i]<0)
			{
				b[i]+= 1024;
			}
			ab_toset[i] = a[i] | (b[i]<< Bits);
			BBIC_Reg_Write(RAM_TX_IQ_BASE_ADDRESS + (TXIQ_BIN_BYTE_WIDTH * Bin) + (ANT_OFFSET*i), ab_toset[i] );
		}
	}


}
void bbic_get_tx_iq_coefficients(uint8_t AntMsk, int16_t Bin, int16_t  a_rtrn[ANT_NUM], int16_t  b_rtrn[ANT_NUM] )
{

	int i;
	int32_t txiq_reg;
	uint8_t Bits = 10;

	if (Bin<0)
	{
		Bin = Bin+512;
	}
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			txiq_reg = BBIC_Reg_Read(RAM_TX_IQ_BASE_ADDRESS + (TXIQ_BIN_BYTE_WIDTH * Bin) + (ANT_OFFSET*i));
			a_rtrn[i] = txiq_reg&((1<<Bits)-1);
			b_rtrn[i] = (txiq_reg&(((1 << Bits) - 1) << Bits))>>Bits;

			SIGN_EXTEND(a_rtrn[i], Bits);
			SIGN_EXTEND(b_rtrn[i], Bits);
		}
	}
}
void bbic_clear_all_tx_iq_coefficients(uint8_t AntMsk)
{

	int i, j;

	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			for (j = 0; j < TXIQ_BINS_NUM; j++)
			{
				BBIC_Reg_Write(RAM_TX_IQ_BASE_ADDRESS + (ANT_OFFSET * i) + (TXIQ_BIN_BYTE_WIDTH * j), 0);
			}
		}
	}

}




//void RunMacEmulator(void)
//{
//  UINT32 reg_val = 0;
//  UINT32 reg_val_1;
//  UINT32 packet_type;
//  UINT32 sifs_length;
//  //set ipg 16usec
//  READ_REG_FIELD (PHY_TX_BASE_ADDR, PHY_TX_MAC_EMU_EMU_PHY_TX_PHY_TCR0_0_EMU_PHY_TX_PHY_TCR0_0_ADDR,12,0x7,&packet_type);
//  sifs_length=0x2080;
//  if (packet_type==0x1)
//    {
//      sifs_length=0x1180;
//    }
//  READ_MODIFY_WRITE(PHY_TX_BASE_ADDR, PHY_TX_MAC_EMU_TX_MAC_EMU_REG0E_EMU_IPG_DLY_ADDR,
//                    PHY_TX_MAC_EMU_TX_MAC_EMU_REG0E_EMU_IPG_DLY_SHIFT, PHY_TX_MAC_EMU_TX_MAC_EMU_REG0E_EMU_IPG_DLY_MASK, sifs_length);
//  //READ_REG(PHY_TX_BASE_ADDR, PHY_TX_TX_IMPLICIT_CALIBRATION_TX_IMPLICIT_BF_CALIB_EN_ADDR, &reg_val);
//
//  //  READ_REG(SHARED_RAM_BASE_ADDR, BF_DATA_BASE, &reg_val);
//  //WRITE_REG(PHY_TX_BASE_ADDR,PHY_TX_MAC_EMU_TX_MAC_EMU_REG13_EMU_BF_HEADER10_ADDR, reg_val);
//  //READ_REG(SHARED_RAM_BASE_ADDR, BF_DATA_BASE + 0x4, &reg_val);
//  //WRITE_REG(PHY_TX_BASE_ADDR,PHY_TX_MAC_EMU_TX_MAC_EMU_REG14_EMU_BF_HEADER32_ADDR, reg_val);
//
// WRITE_REG(PHY_TX_BASE_ADDR, PHY_TX_MAC_EMU_MAC_EMU_MODE_REG_ADDR, 1);
//
//  READ_REG(PHY_TX_BASE_ADDR, PHY_TX_TONE_GEN_TONE_GEN_ON_ADDR, &reg_val);
//
//  PRINTF(Banner, "DEBUG %d", reg_val);
//
//  if(((reg_val & PHY_TX_TONE_GEN_TONE_GEN_ON_MASK) >> PHY_TX_TONE_GEN_TONE_GEN_ON_SHIFT) == 0x0)//mac emulator mode
//            {
//                    READ_MODIFY_WRITE(PHY_TX_BASE_ADDR, PHY_TX_MAC_EMU_TX_MAC_EMU_REG00_EMU_RUN_ADDR, PHY_TX_MAC_EMU_TX_MAC_EMU_REG00_EMU_RUN_SHIFT, PHY_TX_MAC_EMU_TX_MAC_EMU_REG00_EMU_RUN_MASK, 1);
//            }
//    else
//            {
//                    WRITE_REG_FIELD(PHY_TX_BASE_ADDR,PHY_TX_TONE_GEN_TONE_GEN_START_ADDR,PHY_TX_TONE_GEN_TONE_GEN_START_SHIFT,PHY_TX_TONE_GEN_TONE_GEN_START_MASK,0x1);
//            }
//
//  //check if tone gen is enabled
//  // if(reg_val) //tone_gen_on
//  //   {
//  //    READ_REG(PHY_TX_BASE_ADDR, PHY_TXB_IF_TR_TX_SWITCH_VECT_TX_ANTENNA_STBY_ADDR,  &reg_val_1);
//  //    WRITE_REG(PHY_RX_TD_BASE_ADDR,PHY_RX_TD_IF_AFE_DYN_CTRL_TX0_STBY_ADDR, reg_val_1);
//  //
//  // }
//
//}

//**** End of Lev's part
// ######################################################################################################
// Soc PVT temprature sensor
// ######################################################################################################

void bbic_get_soc_tempsens (uint16_t Timeout, uint16_t* tempsens_raw_rtrn)
{
#ifndef ENET_INC_ARCH_WAVE600D2
    RegSocRegsPvtSensorCtrl_u   PvtSensorCtrl;     /*REG_SOC_REGS_PVT_SENSOR_CTRL 0xD0 */
    RegSocRegsPvtSensorData_u   PvtSensorData;     /*REG_SOC_REGS_PVT_SENSOR_DATA 0xD4 */

	uint32_t startTime = 0;
	uint16_t endTime = 0;
	uint8_t maxTimeExceeded = 0;
	uint8_t isValid = 0;


 	// enable the sensor
	PvtSensorCtrl.val = BBIC_Reg_Read(REG_SOC_REGS_PVT_SENSOR_CTRL);
	PvtSensorCtrl.bitFields.pvtEna = 1;
	BBIC_Reg_Write(REG_SOC_REGS_PVT_SENSOR_CTRL, PvtSensorCtrl.val);
   // 
    // check Valid
	startTime = BBIC_Time_Stamp(0,startTime);

	PvtSensorData.val = BBIC_Reg_Read(REG_SOC_REGS_PVT_SENSOR_DATA);
	while (!isValid && !maxTimeExceeded) // wait for all antennas valid bit, while max time is not exceeded
	{
		PvtSensorData.val = BBIC_Reg_Read(REG_SOC_REGS_PVT_SENSOR_DATA);
		isValid = PvtSensorData.bitFields.pvtDataValid;
		endTime = BBIC_Time_Stamp(1,startTime);
		if (endTime > Timeout)
			maxTimeExceeded = 1;
	}

    *tempsens_raw_rtrn = PvtSensorData.bitFields.pvtData;
    
    //tempsens_data_rtrn = (a4*pow(temsens_data,2))*(a4*pow(tempsens_data,2))+(a3*pow(tempsens_data,2))*(a3*tempsens_data)+ a2*pow(tempsens_data,2)+a1*tempsens_data+a0;
		
	// disable the sensor
	PvtSensorCtrl.val = 0;
	BBIC_Reg_Write(REG_SOC_REGS_PVT_SENSOR_CTRL, PvtSensorCtrl.val);
#else
	UNUSED_PARAM(Timeout); 
	UNUSED_PARAM(tempsens_raw_rtrn);
#endif
}

// ######################################################################################################
// Soc PVT voltage sensor
// ######################################################################################################

void bbic_get_soc_voltsens (uint16_t Timeout, uint16_t* voltsens_raw_rtrn)
{
#ifndef ENET_INC_ARCH_WAVE600D2
    RegSocRegsPvtSensorCtrl_u   PvtSensorCtrl;     /*REG_SOC_REGS_PVT_SENSOR_CTRL 0xD0 */
    RegSocRegsPvtSensorData_u   PvtSensorData;     /*REG_SOC_REGS_PVT_SENSOR_DATA 0xD4 */

	uint32_t startTime = 0;
	uint16_t endTime = 0;
	uint8_t maxTimeExceeded = 0;
	uint8_t isValid = 0;


 	// enable the sensor

	PvtSensorCtrl.val = BBIC_Reg_Read(REG_SOC_REGS_PVT_SENSOR_CTRL);
	PvtSensorCtrl.bitFields.pvtVsample = 1;
	BBIC_Reg_Write(REG_SOC_REGS_PVT_SENSOR_CTRL, PvtSensorCtrl.val);
 	PvtSensorCtrl.bitFields.pvtEna = 1;
	BBIC_Reg_Write(REG_SOC_REGS_PVT_SENSOR_CTRL, PvtSensorCtrl.val);
  // 
    // check Valid
	startTime = BBIC_Time_Stamp(0,startTime);

	PvtSensorData.val = BBIC_Reg_Read(REG_SOC_REGS_PVT_SENSOR_DATA);
	while (!isValid && !maxTimeExceeded) // wait for all antennas valid bit, while max time is not exceeded
	{
		PvtSensorData.val = BBIC_Reg_Read(REG_SOC_REGS_PVT_SENSOR_DATA);
		isValid = PvtSensorData.bitFields.pvtDataValid;
		endTime = BBIC_Time_Stamp(1,startTime);
		if (endTime > Timeout)
			maxTimeExceeded = 1;
	}

    *voltsens_raw_rtrn = PvtSensorData.bitFields.pvtData;
    
    //tempsens_data_rtrn = (a4*pow(temsens_data,2))*(a4*pow(tempsens_data,2))+(a3*pow(tempsens_data,2))*(a3*tempsens_data)+ a2*pow(tempsens_data,2)+a1*tempsens_data+a0;
		
	// disable the sensor
	PvtSensorCtrl.val = 0;
	BBIC_Reg_Write(REG_SOC_REGS_PVT_SENSOR_CTRL, PvtSensorCtrl.val);
#else
	UNUSED_PARAM(Timeout); 
	UNUSED_PARAM(voltsens_raw_rtrn);
#endif
}


// ######################################################################################################
// TX Power Control 
// ######################################################################################################
// Offset setting

void bbic_set_tx_power_table_offset(uint8_t AntIdx, uint8_t BwIdx, int16_t rf_power_offset_val)
	// Set TX RF POWER offset used to shift the TPC table
	//AntIdx - antenna index, 
	//BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
	//rf_power_offset_val - offset value	
{
	RegPhyTxtdAnt0RfPowerOffset_u Reg_RF_POWER_OFFSET;  //*REG_PHY_TXTD_ANT0_RF_POWER_OFFSET 0xA4 */

	Reg_RF_POWER_OFFSET.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_RF_POWER_OFFSET+(AntIdx*(ANT_OFFSET)));
	
	switch (BwIdx) {
		case 0 : Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset20 = rf_power_offset_val;  break; // 20MHz
		case 1 : Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset40 = rf_power_offset_val;  break; // 40MHz
		case 2 : Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset80 = rf_power_offset_val;  break; // 80MHz
		case 3 : Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset16011B = rf_power_offset_val;  break; // 1600MHz
		default: break; 
	}
	BBIC_Reg_Write(REG_PHY_TXTD_ANT0_RF_POWER_OFFSET + AntIdx*(ANT_OFFSET) , Reg_RF_POWER_OFFSET.val);
}

void bbic_get_tx_power_table_offset(uint8_t AntIdx, uint8_t BwIdx, int16_t* rf_power_offset_val_rtrn)
{
	// Get TX RF POWER offset used to shift the TPC table
	//AntIdx - antenna index: 0 to 3
	//BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
	//rf_power_offset_val_rtrn - offset value
	RegPhyTxtdAnt0RfPowerOffset_u Reg_RF_POWER_OFFSET;  //*REG_PHY_TXTD_ANT0_RF_POWER_OFFSET 0xA4 */

	Reg_RF_POWER_OFFSET.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_RF_POWER_OFFSET+(AntIdx*(ANT_OFFSET)));		    
	switch (BwIdx) {
		case 0 :  *rf_power_offset_val_rtrn = Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset20;  break; // 20MHz
		case 1 :  *rf_power_offset_val_rtrn = Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset40;  break; // 40MHz
		case 2 :  *rf_power_offset_val_rtrn = Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset80;  break; // 80MHz
		case 3 :  *rf_power_offset_val_rtrn = Reg_RF_POWER_OFFSET.bitFields.rfPowerOffset16011B;  break; // 1600MHz
		default: break; 
	}
}

void bbic_set_tx_open_loop(uint8_t AntMsk, BOOL OpenLoopIsEnabled)
// Set TX  POWER Control Open Loop Mode = Correct gain to fix output power
//AntIdx - antenna index, 
//OpenLoopIsEnabled - 1 - Open Loop, 0 - Close loop 
{
	RegPhyTxtdAnt0TpcAccelerator_u Reg_TPC_ACCELERATOR;   /*REG_PHY_TXTD_ANT0_TPC_ACCELERATOR 0x88 */
	int AntIdx;
	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		if (AntMsk & ((1<<AntIdx)))
		{
			Reg_TPC_ACCELERATOR.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR+(ANT_OFFSET*AntIdx));
			Reg_TPC_ACCELERATOR.bitFields.tssiOpenLoopEn = OpenLoopIsEnabled;
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR+(ANT_OFFSET*AntIdx), Reg_TPC_ACCELERATOR.val);
		}
	}
	//return;
}
void bbic_get_tx_open_loop(uint8_t AntMsk, BOOL OpenLoopIsEnabled_rtrn[ANT_NUM])
// Set TX  POWER Control Open Loop Mode = Correct gain to fix output power
//AntIdx - antenna index, 
//OpenLoopIsEnabled - 1 - Open Loop, 0 - Close loop 
{
	RegPhyTxtdAnt0TpcAccelerator_u Reg_TPC_ACCELERATOR;   /*REG_PHY_TXTD_ANT0_TPC_ACCELERATOR 0x88 */
	int AntIdx;
	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		OpenLoopIsEnabled_rtrn[AntIdx]=0;
		if (AntMsk & (1 << AntIdx))
		{
			Reg_TPC_ACCELERATOR.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TPC_ACCELERATOR+(ANT_OFFSET*AntIdx));
			OpenLoopIsEnabled_rtrn[AntIdx] = Reg_TPC_ACCELERATOR.bitFields.tssiOpenLoopEn;
		}
	}
	//return;
}

void bbic_set_tx_power_coef(uint8_t AntIdx, uint8_t BWIdx, uint8_t RegionIdx, int16_t Coef_A, int16_t Coef_B)
// Set TX RF POWER Coefs: A,B,c
//AntIdx - antenna index, 
//BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
//RangeIdx	-Range Index 0-2
{
	uint8_t Bits = 16;
	BBIC_Reg_Write(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + 4*((TSSI_RAM_BW_SIZE * BWIdx) + (TSSI_RAM_REGINE_SIZE*RegionIdx)), (Coef_A & ((1<<Bits)-1)) + ((Coef_B& ((1<<Bits)-1))<<16) );
}

//void bbic_set_tx_power_s2dIdx(uint8_t AntIdx, uint8_t BWIdx, uint8_t RangeIdx, int16_t s2dIdx)
//// Set TX RF POWER Coefs: s2dIdx
////AntIdx - antenna index, 
////BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
////RangeIdx	-Range Index 0-2
//{
//	uint8_t Bits = 16;
//	uint32_t ReadData;
//	ReadData = BBIC_Reg_Read(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + (6 *4* BWIdx) + (2*4*RangeIdx) +4);
//	ReadData =ReadData & (((1<<3)-1)<<Bits);
//	BBIC_Reg_Write(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + (6 *4* BWIdx) + (2*4*RangeIdx) +4, (s2dIdx<<Bits) + ReadData );
//}
//
//void bbic_set_tx_power_PowerThreshold(uint8_t AntIdx, uint8_t BWIdx, uint8_t RegionIdx, int16_t PowerThreshold)
//// Set TX RF POWER Coefs: PowerThreshold
////AntIdx - antenna index, 
////BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
////RegionIdx	-Range Index 0-2
//{
//	uint8_t Bits = 24;
//	uint32_t ReadData;
//	ReadData = BBIC_Reg_Read(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + (TSSI_RAM_BW_SIZE *4* BWIdx) + (2*4*RegionIdx) +4);
//	ReadData =ReadData & (((1<<8)-1)<<Bits);
//	BBIC_Reg_Write(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + (TSSI_RAM_BW_SIZE *4* BWIdx) + (2*4*RegionIdx) +4, (PowerThreshold<<Bits) + ReadData );
//}
void bbic_get_tx_power_PowerThreshold(uint8_t AntIdx, uint8_t BWIdx, uint8_t RegionIdx, int16_t* PowerThreshold_rtrn)
{
//// Get TX RF POWER Coefs: PowerThreshold
////AntIdx - antenna index, 
////BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
////RegionIdx	-Range Index 0-2
	uint8_t Bits = 24;
	uint32_t ReadData;
	ReadData = BBIC_Reg_Read(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + 4*((TSSI_RAM_BW_SIZE * BWIdx) + (TSSI_RAM_REGINE_SIZE*RegionIdx) +1));
	*PowerThreshold_rtrn =(int16_t) (ReadData & (((1<<8)-1)<<Bits))>>Bits;	
}

void bbic_get_tx_power_Coef_C(uint8_t AntIdx, uint8_t BWIdx, uint8_t RegionIdx, int16_t* Coef_C_rtrn)
{
//// Get TX RF POWER Coefs: Coef_C
////AntIdx - antenna index, 
////BwIdx - BW index 0 -20, 1-40, 2 -80, 3- 160/ 11B
////RegionIdx	-Range Index 0-2
	uint8_t Bits = 16;
	uint32_t ReadData;
	ReadData = BBIC_Reg_Read(RAM_TX_TSSI_BASE_ADDRESS + AntIdx*(ANT_OFFSET) + 4*( (TSSI_RAM_BW_SIZE * BWIdx) + (TSSI_RAM_REGINE_SIZE*RegionIdx) +1));
	*Coef_C_rtrn =(int16_t) (ReadData & (1<<(Bits-1)));
}

// ######################################################################################################
// RSSI
// ######################################################################################################
void bbic_get_rssi(uint8_t AntMsk, int8_t rssiValues_rtrn[ANT_NUM] )
/// / Get RSSI values from PHY BackEnd - vector of 4.
////AntMsk - antenna Mask,
{
	RegPhyRxBeIfPhyRxbeIf184_u Reg_Rssi;   /*REG_PHY_RX_BE_IF_PHY_RXBE_IF184 0x1610 */

 	int AntIdx;
	uint8_t Bits = 8;

	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			Reg_Rssi.val = BBIC_Reg_Read(REG_PHY_RX_BE_IF_PHY_RXBE_IF184+ (AntIdx * 4));
			rssiValues_rtrn[AntIdx] = Reg_Rssi.bitFields.ant0AdRssi ;

			SIGN_EXTEND(rssiValues_rtrn[AntIdx], Bits);
			
		}
	}

}

void bbic_set_lna_gain_lut(uint8_t AntIdx, int16_t LnaGainValues[8] )
/// / Set LNA gain values , vector of 8 gains 0 - high, 7 lowest
////AnIdx - antenna Index ,(read one antenna at a time)
{
	RegPhyAgcAnt0PmLnaGainLut0_u Reg_LAN_GAIN_LUT;   /*REG_PHY_AGC_ANT0_PM_LNA_GAIN_LUT_0 0x14C */
	int gainIdx; //gain index
	for (gainIdx=0; gainIdx<8; gainIdx++) {
			Reg_LAN_GAIN_LUT.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_PM_LNA_GAIN_LUT_0+ (AntIdx * ANT_OFFSET)+gainIdx*4);
			Reg_LAN_GAIN_LUT.bitFields.pmLnaGainLut0 = LnaGainValues[gainIdx];
			BBIC_Reg_Write(REG_PHY_AGC_ANT0_PM_LNA_GAIN_LUT_0+ (AntIdx * ANT_OFFSET)+gainIdx*4,Reg_LAN_GAIN_LUT.val );
	}
}

void bbic_get_lna_gain_lut(uint8_t AntIdx, int16_t LnaGainValues_rtrn[8] )
/// / Get LNA gain values , vector of 8. 0 high, 7 lowest
////AnIdx - antenna Index ,(read one antenna at a time)
{
	RegPhyAgcAnt0PmLnaGainLut0_u Reg_LAN_GAIN_LUT;   /*REG_PHY_AGC_ANT0_PM_LNA_GAIN_LUT_0 0x14C */
	int gainIdx; //gain index
	int Bits = 9;
	for (gainIdx=0; gainIdx<8; gainIdx++) {
			Reg_LAN_GAIN_LUT.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_PM_LNA_GAIN_LUT_0+ (AntIdx * ANT_OFFSET)+gainIdx*4);
			LnaGainValues_rtrn[gainIdx] = Reg_LAN_GAIN_LUT.bitFields.pmLnaGainLut0;
			SIGN_EXTEND(LnaGainValues_rtrn[gainIdx], Bits);

	}

}

void bbic_set_rx_non_agg_test_mode(BOOL RxNonAggIsEnabled)
// Set RX Test mode: VHT / HE non aggregation mode 
// RxNonAggIsEnabled - 1 - Rx non agg, test mode 0 - RX regular mode 
{
	RegPhyRxBePhyRxbeReg67_u  Reg_Non_Agg;    /*REG_PHY_RX_BE_PHY_RXBE_REG67 0x19C */
	Reg_Non_Agg.val = BBIC_Reg_Read(REG_PHY_RX_BE_PHY_RXBE_REG67);
	Reg_Non_Agg.bitFields.vhtNonAggregate = RxNonAggIsEnabled;
	BBIC_Reg_Write(REG_PHY_RX_BE_PHY_RXBE_REG67, Reg_Non_Agg.val);
}
void bbic_get_rx_non_agg_test_mode(BOOL* RxNonAggIsEnabled_rtrn)
// Get RX Test mode: VHT / HE non aggregation mode 
// RxNonAggIsEnabled - 1 - Rx non agg, test mode 0 - RX regular mode 
{
	RegPhyRxBePhyRxbeReg67_u  Reg_Non_Agg;    /*REG_PHY_RX_BE_PHY_RXBE_REG67 0x19C */
	Reg_Non_Agg.val = BBIC_Reg_Read(REG_PHY_RX_BE_PHY_RXBE_REG67);
	*RxNonAggIsEnabled_rtrn = Reg_Non_Agg.bitFields.vhtNonAggregate;
}

void bbic_bypass_phy_frc(uint8_t AntMsk, BOOL BypassFRCMode)
{
	RegPhyRxtdAnt0PhyRxtdAntReg53_u Reg_Phy_FRC;	/*REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG53 */
	int i;
	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Phy_FRC.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG53 + i * (ANT_OFFSET));
			Reg_Phy_FRC.bitFields.frcBypass = BypassFRCMode;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG53 + i * (ANT_OFFSET), Reg_Phy_FRC.val);
		}
	}
}
void bbic_get_bypass_phy_frc_status(uint8_t AntMsk, BOOL Frc_status_rtrn[ANT_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg53_u Reg_Phy_FRC;	/*REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG53 */
	int i;
	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Phy_FRC.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG53 + i * (ANT_OFFSET));
			Frc_status_rtrn[i] = Reg_Phy_FRC.bitFields.frcBypass;
		}
	}
}
void bbic_bypass_phy_fdl(uint8_t AntMsk, BOOL BypassPhyFDLMode)
{
	RegPhyRxtdAnt0PhyRxtdAntReg50_u Reg_Phy_FDL;	/*REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 */
	int i;
	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Phy_FDL.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i * (ANT_OFFSET));
			Reg_Phy_FDL.bitFields.fdlBypass = BypassPhyFDLMode;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i * (ANT_OFFSET), Reg_Phy_FDL.val);
		}
	}
}
void bbic_get_bypass_phy_fdl_status(uint8_t AntMsk, BOOL PhyFDL_status_rtrn[ANT_NUM])
{
	RegPhyRxtdAnt0PhyRxtdAntReg50_u Reg_Phy_FDL;	/*REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 */
	int i;
	for (i = 0; i<ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
			Reg_Phy_FDL.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_PHY_RXTD_ANT_REG50 + i * (ANT_OFFSET));
			PhyFDL_status_rtrn[i] = Reg_Phy_FDL.bitFields.fdlBypass;
		}
	}
}
void bbic_set_detector_reset_th(uint8_t DetectorThreshold)
{
	RegPhyRxTdPhyRxtdReg062_u Reg_Phy_Detector;	/*REG_PHY_RX_TD_PHY_RXTD_REG062*/
	Reg_Phy_Detector.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG062 );
	Reg_Phy_Detector.bitFields.blockDetectionOnBands = DetectorThreshold;
	BBIC_Reg_Write(REG_PHY_RX_TD_PHY_RXTD_REG062, Reg_Phy_Detector.val);
}
uint8_t bbic_get_detector_reset_th(void)
{
	RegPhyRxTdPhyRxtdReg062_u Reg_Phy_Detector;	/*REG_PHY_RX_TD_PHY_RXTD_REG062*/
	uint8_t Threshold;
	Reg_Phy_Detector.val = BBIC_Reg_Read(REG_PHY_RX_TD_PHY_RXTD_REG062);
	Threshold = (uint8_t)Reg_Phy_Detector.bitFields.blockDetectionOnBands;
	return Threshold;
}
void bbic_enable_11b_detector(uint8_t Enable_Detector_11b)
{
	RegPhyTxbIfPhyTxbRiscReg1Cb_u Reg_phy_11b_Det;	/*REG_PHY_TXB_IF_PHY_TXB_RISC_REG1CB*/
	Reg_phy_11b_Det.val = BBIC_Reg_Read(REG_PHY_TXB_IF_PHY_TXB_RISC_REG1CB);
	Reg_phy_11b_Det.bitFields.dtRasChooseAnt = 1 - Enable_Detector_11b;
	BBIC_Reg_Write(REG_PHY_TXB_IF_PHY_TXB_RISC_REG1CB, Reg_phy_11b_Det.val);
}
uint8_t bbic_get_11b_detector_state(void)
{
	RegPhyTxbIfPhyTxbRiscReg1Cb_u Reg_phy_11b_Det;	/*REG_PHY_TXB_IF_PHY_TXB_RISC_REG1CB*/
	uint8_t Det_11b_enable;
	Reg_phy_11b_Det.val = BBIC_Reg_Read(REG_PHY_TXB_IF_PHY_TXB_RISC_REG1CB);
	Det_11b_enable = (uint8_t)(1 - Reg_phy_11b_Det.bitFields.dtRasChooseAnt);
	return Det_11b_enable;
}

void bbic_set_rx_Sc2FcPhase(int8_t AntMsk , uint8_t RxSc2FcPhase)
// Set RX digial Soc to ABB phase compensator: 0,1,2,3
{
	RegPhyRxtdAnt0RxAbbAntReg71_u  Reg_Rx_Sc2FcPhase;    /*REG_PHY_RXTD_ANT0_RX_ABB_ANT_REG71 0x1C4 */
	int AntIdx;
	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			Reg_Rx_Sc2FcPhase.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_RX_ABB_ANT_REG71+(4*AntIdx));
			Reg_Rx_Sc2FcPhase.bitFields.rxSc2FcPhaseRegfile = RxSc2FcPhase;
			BBIC_Reg_Write(REG_PHY_RXTD_ANT0_RX_ABB_ANT_REG71+(4*AntIdx), Reg_Rx_Sc2FcPhase.val);
		}
	}
}
void bbic_get_rx_Sc2FcPhase(int8_t AntMsk , uint8_t RxSc2FcPhase_rtrn[ANT_NUM])
// Get  RX digial Soc to ABB phase compensator: 0,1,2,3
{
	RegPhyRxtdAnt0RxAbbAntReg71_u  Reg_Rx_Sc2FcPhase;    /*REG_PHY_RXTD_ANT0_RX_ABB_ANT_REG71 0x1C4 */
	int AntIdx;
	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			Reg_Rx_Sc2FcPhase.val = BBIC_Reg_Read(REG_PHY_RXTD_ANT0_RX_ABB_ANT_REG71+(4*AntIdx));
			RxSc2FcPhase_rtrn[AntIdx] = Reg_Rx_Sc2FcPhase.bitFields.rxSc2FcPhaseRegfile;
		}
	}
}


void bbic_set_tx_Sc2FcPhase(int8_t AntMsk , uint8_t TxSc2FcPhase)
// SetTX digial Soc to ABB phase compensator: 0,1,2,3
{
	RegPhyTxtdAnt0TxAbb30_u  Reg_Tx_Sc2FcPhase;    /*REG_PHY_TXTD_ANT0_TX_ABB_30 0x78 */
	int AntIdx;
	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			Reg_Tx_Sc2FcPhase.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_ABB_30+(4*AntIdx));
			Reg_Tx_Sc2FcPhase.bitFields.txSc2FcPhaseRegfile = TxSc2FcPhase;
			BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TX_ABB_30+(4*AntIdx), Reg_Tx_Sc2FcPhase.val);
		}
	}
}
void bbic_get_tx_Sc2FcPhase(int8_t AntMsk , uint8_t TxSc2FcPhase_rtrn[ANT_NUM])
// Get  TX digial Soc to ABB phase compensator: 0,1,2,3
{
	RegPhyTxtdAnt0TxAbb30_u  Reg_Tx_Sc2FcPhase;    /*REG_PHY_TXTD_ANT0_TX_ABB_30 0x78 */
	int AntIdx;
	for (AntIdx = 0; AntIdx < ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx))
		{
			Reg_Tx_Sc2FcPhase.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TX_ABB_30+(4*AntIdx));
			TxSc2FcPhase_rtrn[AntIdx] = Reg_Tx_Sc2FcPhase.bitFields.txSc2FcPhaseRegfile;
		}
	}
}

// take RFIC PGC gain Filter Lut from Efuse, write into PHY RAM
//uint8_t		PgcFiltLUT[5][11] = {
//			{ 	0, 	0,	0,	0,	0,	0,	0,	0,	0,	3,	15}, /* 12.5MHz mode */
//			{ 	0, 	0,	0,	0,	0,	0,	3,	15,	25,	32,	38}, /* 25MHz mode */
//			{ 	0, 	0,	0,	3,	15,	25,	32,	38,	43,	47,	50}, /* 50MHz mode */
//			{ 	3, 	15,	25,	32,	38,	43,	47,	50,	52,	54,	56}, /* 100MHz mode */
//			{ 	23, 31,	37,	42,	46,	49,	52,	54,	55,	57,	58}, /* 150MHz (DPD) mode */
//	};
// PHY RAM:  (currently, all modes are the same)
    // modes: 'regular20','regular40','regular80','regular160','demo20','demo40','demo80','demo160','rp20','rp40','lp20','dpd160+')
	// in each mode, lines {pgc1,pgc0 }    {pgc2,pgc1 }  {pgc4,pgc3 }  ...   {pgc10,pgc9 } {spacer,pgc11 }
void bbic_set_PgcFiltLut(int8_t AntMsk , uint8_t PgcFiltLUT_local[5][11])
{
	int AntIdx, ModeIdx,BwIdx,GainIdx;
	uint32_t Value;
	
	//for (BwIdx=0; BwIdx<5;BwIdx++) {  // 20,40,80,160
	//	for (GainIdx=0; GainIdx<11;GainIdx++){
	//			printf("PgcFiltLut[%d][%d] = %d \n", BwIdx,GainIdx, PgcFiltLUT_local[BwIdx][GainIdx]);
	//	}
	//}
	for (AntIdx=0; AntIdx<ANT_NUM; AntIdx++)
	{
		if (AntMsk & (1<<AntIdx)) 
		{
			for (ModeIdx=0; ModeIdx<3;ModeIdx++) 
			{  // regular, demo, RP, LP, DPD
				for (BwIdx=0; BwIdx<BW_NUM;BwIdx++) 
				{  // 20,40,80,160
					for (GainIdx=0; GainIdx<((PGC_GAIN_NUM/2+1));GainIdx++) 
					{
						if (ModeIdx==2 & BwIdx==2) 
						{ // lp20
							Value = PgcFiltLUT_local[0][2*GainIdx];
							if (GainIdx<(PGC_GAIN_NUM/2))
							{
								Value +=((PgcFiltLUT_local[0][(2*GainIdx)+1])<<6);
							}
						}
						else 
						{	if (ModeIdx==2 & BwIdx==3) 
							{// dpd160
								Value = PgcFiltLUT_local[4][2*GainIdx];
								if (GainIdx<(PGC_GAIN_NUM/2))
								{
									Value +=((PgcFiltLUT_local[4][2*GainIdx+1])<<6);
								}
							}
							else	
							{
								Value = PgcFiltLUT_local[BwIdx][2*GainIdx];
								if (GainIdx<(PGC_GAIN_NUM/2))
								{
									Value += ((PgcFiltLUT_local[BwIdx][2*GainIdx+1])<<6);
								}
							}
						
						}
						BBIC_Reg_Write(RAM_RX_PGC_FILTER_BASE_ADDRESS + (AntIdx * ANT_OFFSET) + ( ((PGC_GAIN_NUM/2+1)*BW_NUM) * ModeIdx +((PGC_GAIN_NUM/2+1)*BwIdx) +GainIdx)*RXDC_RAM_REG_WIDTH, Value);
						//printf("value: %x, address: %x\n, %d %d %d %d", Value, ( (PGC_GAIN_NUM/2*BW_NUM) * ModeIdx +(PGC_GAIN_NUM/2*BwIdx) +GainIdx)*RXDC_RAM_REG_WIDTH,PgcFiltLUT_local[BwIdx][2*GainIdx],PgcFiltLUT_local[BwIdx][(2*GainIdx)+1], GainIdx,2*GainIdx+1);
					}  //for GainIdx
				} // fow BwIdx
			} // for ModeIdx
		} //if AntIdx&AntMsk
	} //for AntIdx
}

void bbic_set_rx_rssi_coef(uint8_t AntIdx, uint8_t RegionIdx, int16_t Coef_A, int16_t Coef_B)
// Set RX RF RSSI  Coefs: A,B,c  A -> nLog, B->C
//AntIdx - antenna index, 
//RangeIdx	-Range Index 0-2
{
	//RegPhyAgcAnt0RfRssiCoefCRange0Reg_u    RfRssiCoefCRange0Reg; /*REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG 0x7C */
	//RegPhyAgcAnt0RfRssiCoefNlogRange0Reg_u RfRssiCoefNlogRange0; /*REG_PHY_AGC_ANT0_RF_RSSI_COEF_NLOG_RANGE0_REG 0x84 */

	//RfRssiCoefCRange0Reg.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG);
	BBIC_Reg_Write(REG_PHY_AGC_ANT0_RF_RSSI_COEF_NLOG_RANGE0_REG + AntIdx*(ANT_OFFSET) + (4*RegionIdx), Coef_A  );
	BBIC_Reg_Write(REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG + AntIdx*(ANT_OFFSET) + (4*RegionIdx), Coef_B  );
}

void bbic_get_rx_rssi_coef(uint8_t AntIdx, uint8_t RegionIdx, int16_t* Coef_A_rtrn, int16_t* Coef_B_rtrn)
// Set RX RF RSSI  Coefs: A,B,c  A -> nLog, B->C
//AntIdx - antenna index, 
//RangeIdx	-Range Index 0-2
{
	RegPhyAgcAnt0RfRssiCoefCRange0Reg_u    RfRssiCoefCRange0Reg; /*REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG 0x7C */
	RegPhyAgcAnt0RfRssiCoefNlogRange0Reg_u RfRssiCoefNlogRange0; /*REG_PHY_AGC_ANT0_RF_RSSI_COEF_NLOG_RANGE0_REG 0x84 */

	RfRssiCoefCRange0Reg.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG + AntIdx*(ANT_OFFSET) + (4*RegionIdx));
	RfRssiCoefNlogRange0.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_RF_RSSI_COEF_NLOG_RANGE0_REG + AntIdx*(ANT_OFFSET) + (4*RegionIdx));
	*Coef_B_rtrn = RfRssiCoefCRange0Reg.bitFields.rfRssiCoefCRange0;
	*Coef_A_rtrn = RfRssiCoefNlogRange0.bitFields.rfRssiCoefNlogRange0;
}
void bbic_set_rx_rssi_offset1(uint8_t AntIdx, uint8_t RegionIdx, int16_t offset1)
// Set RX RF RSSI  offset1 - for DC 
//AntIdx - antenna index, 
//RangeIdx	-Range Index 0-1
{
	//RegPhyAgcAnt0RfRssiOffset1RxRange0_u  RfRssiOffset1Rx;	/*REG_PHY_AGC_ANT0_RF_RSSI_OFFSET1_RX_RANGE0 0x40 */
	//RfRssiCoefCRange0Reg.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG);
	BBIC_Reg_Write(REG_PHY_AGC_ANT0_RF_RSSI_OFFSET1_RX_RANGE0 + AntIdx*(ANT_OFFSET) + (4*RegionIdx), offset1  );
}
void bbic_get_rx_rssi_offset1(uint8_t AntIdx, uint8_t RegionIdx, int16_t* offset1_rtrn)
// Set RX RF RSSI  offset1 - for DC 
//AntIdx - antenna index, 
//RangeIdx	-Range Index 0-1
{
	RegPhyAgcAnt0RfRssiOffset1RxRange0_u  RfRssiOffset1Rx;	/*REG_PHY_AGC_ANT0_RF_RSSI_OFFSET1_RX_RANGE0 0x40 */
	//RfRssiCoefCRange0Reg.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_RF_RSSI_COEF_C_RANGE0_REG);
	RfRssiOffset1Rx.val = BBIC_Reg_Read(REG_PHY_AGC_ANT0_RF_RSSI_OFFSET1_RX_RANGE0 + AntIdx*(ANT_OFFSET) + (4*RegionIdx)  );
	*offset1_rtrn=RfRssiOffset1Rx.bitFields.rfRssiOffset1RxRange0;
}	

void bbic_api_chip_rev(uint8_t* Rev_rtrn)
{
#ifdef ENET_INC_ARCH_WAVE600B
	*Rev_rtrn = 2;
#else
	#ifdef ENET_INC_ARCH_WAVE600D2
		*Rev_rtrn = 3;
	#else
		*Rev_rtrn = 1;
	#endif
#endif
}

void bbic_tx_fft_in_scale(int8_t AntMsk, uint8_t ScaleFactor)

{
	RegPhyTxtdAnt0TxtdIfftInOutShift_u Reg_Tx_FFT_InOutScale;
	int i; 
	
	for (i = 0; i < ANT_NUM; i++)
	{
		if (AntMsk & (1<<i))
		{
				Reg_Tx_FFT_InOutScale.val = BBIC_Reg_Read(REG_PHY_TXTD_ANT0_TXTD_IFFT_IN_OUT_SHIFT + i * (ANT_OFFSET));
				Reg_Tx_FFT_InOutScale.bitFields.pmIfftInShiftToneGen = ScaleFactor;
				BBIC_Reg_Write(REG_PHY_TXTD_ANT0_TXTD_IFFT_IN_OUT_SHIFT + i * (ANT_OFFSET), Reg_Tx_FFT_InOutScale.val);
		}
	}
	

}



void bbic_set_cdd(uint8_t No_of_Ant, uint8_t offset1,uint8_t offset2,uint8_t offset3)
{
	RegPhyTxLegacyCdd0_u Reg_LEGACY_CDD_0 ; /*REG_PHY_TX_LEGACY_CDD_0 0xF4C */
	RegPhyTxLegacyCdd1_u Reg_LEGACY_CDD_1 ; /*REG_PHY_TX_LEGACY_CDD_1 0xF50 */
	
	Reg_LEGACY_CDD_0.val = BBIC_Reg_Read(REG_PHY_TX_LEGACY_CDD_0);
	Reg_LEGACY_CDD_1.val = BBIC_Reg_Read(REG_PHY_TX_LEGACY_CDD_1);



	switch (No_of_Ant) {
		case 2 : 
			Reg_LEGACY_CDD_0.bitFields.phyCdd2AntsOfst1=offset1;		
			break ; // 
		case 3 : 
			Reg_LEGACY_CDD_0.bitFields.phyCdd3AntsOfst1=offset1; 
			Reg_LEGACY_CDD_0.bitFields.phyCdd3AntsOfst2=offset2; 
			break;//  
		case 4 : 
			Reg_LEGACY_CDD_0.bitFields.phyCdd4AntsOfst1=offset1; 
			Reg_LEGACY_CDD_1.bitFields.phyCdd4AntsOfst2=offset2;
			Reg_LEGACY_CDD_1.bitFields.phyCdd4AntsOfst3=offset3;
			break;
		default: break;
	}
	
	BBIC_Reg_Write(REG_PHY_TX_LEGACY_CDD_0 , Reg_LEGACY_CDD_0.val);
	BBIC_Reg_Write(REG_PHY_TX_LEGACY_CDD_1 , Reg_LEGACY_CDD_1.val);
	
	return;
}


void bbic_get_cdd(uint8_t No_of_Ant, uint32_t* offset1_rtrn,uint32_t* offset2_rtrn,uint32_t* offset3_rtrn)
{

	RegPhyTxLegacyCdd0_u Reg_LEGACY_CDD_0 ; /*REG_PHY_TX_LEGACY_CDD_0 0xF4C */
	RegPhyTxLegacyCdd1_u Reg_LEGACY_CDD_1 ; /*REG_PHY_TX_LEGACY_CDD_1 0xF50 */
	
	Reg_LEGACY_CDD_0.val = BBIC_Reg_Read(REG_PHY_TX_LEGACY_CDD_0);
	Reg_LEGACY_CDD_1.val = BBIC_Reg_Read(REG_PHY_TX_LEGACY_CDD_1);
	
	switch (No_of_Ant) {
		case 2 : 
			*offset1_rtrn=Reg_LEGACY_CDD_0.bitFields.phyCdd2AntsOfst1;		
			break ; // 
		case 3 : 
			*offset1_rtrn=Reg_LEGACY_CDD_0.bitFields.phyCdd3AntsOfst1; 
			*offset2_rtrn=Reg_LEGACY_CDD_0.bitFields.phyCdd3AntsOfst2; 
			break;//  
		case 4 : 
			*offset1_rtrn=Reg_LEGACY_CDD_0.bitFields.phyCdd4AntsOfst1; 
			*offset2_rtrn=Reg_LEGACY_CDD_1.bitFields.phyCdd4AntsOfst2;
			*offset3_rtrn=Reg_LEGACY_CDD_1.bitFields.phyCdd4AntsOfst3;
			break;
		default: break;
	}
	return;
}



