function Matlab2SV(tModelName, cModelArgs)
% Usage
%   Matlab2SV(tModelName, cModelArgs)
%
% Parameters
%   tModelName  -   string containing the model/block/function name, eg. 
%                       tModelName = 'rx5_rf_lna';
%
%   cModelArgs  -   cell array containing the list of input types
%                       cModelArgs = cell(13,1);
%                       cModelArgs{1} = coder.typeof(fi(0,numerictype(0,48,0)));    % RFRX_ANTx
%                       cModelArgs{2} = coder.typeof(fi(0,numerictype(0,48,0)));    % RFRXGND_ANTx
%                       cModelArgs{3} = coder.typeof(fi(0,numerictype(0,6,0)));     % ctl_gain_lna0
%                       cModelArgs{4} = coder.typeof(fi(0,numerictype(0,3,0)));     % ctl_freq_lna0
%                       cModelArgs{5} = coder.typeof(false);                        % en_lna0
%                       cModelArgs{6} = coder.typeof(false);                        % stby_lna0
%                       cModelArgs{7} = coder.typeof(false);                        % en_noise_lna0
%                       cModelArgs{8} = coder.typeof(false);                        % en_clamp_lna0
%                       cModelArgs{9} = coder.typeof(false);                        % ib_pc_const_25u
%                       cModelArgs{10} = coder.typeof(false);                       % VDDRF2V5LDO
%                       cModelArgs{11} = coder.typeof(false);                       % VDDRF1V2RXLDO
%                       cModelArgs{12} = coder.typeof(false);                       % VSSAANTx
%                       cModelArgs{13} = coder.typeof(false);                       % VSSPSUB
%
% Description
%   Converts Matlab function to fixed-point arithmetic and then to
%   (System-)Verilog, copies resulting core function to release folder

% --- Authors -------------------------------------------------------------
% Steffen Trautmann
% -------------------------------------------------------------------------

% --- Copyright -----------------------------------------------------------
% Intel, Connected Home Division, Villach
% -------------------------------------------------------------------------

%% Fixed-point Configuration

rFixCfg = coder.config('fixpt');
rFixCfg.DesignFunctionName = tModelName;
rFixCfg.TestBenchName = { sprintf('%s_tb.m',tModelName) };
rFixCfg.fimath = sprintf('fimath(''RoundingMethod'', ''Nearest'', ''OverflowAction'', ''Saturate'', ''ProductMode'', ''FullPrecision'', ''SumMode'', ''FullPrecision'')');
% rFixCfg.fimath = sprintf('fimath(''RoundingMethod'', ''Floor'', ''OverflowAction'', ''Wrap'', ''ProductMode'', ''FullPrecision'', ''SumMode'', ''FullPrecision'')');
rFixCfg.FixPtFileNameSuffix = '_core';

%% HDL Coder Configuration
rHdlCfg = coder.config('hdl');
rHdlCfg.TargetLanguage = 'Verilog';
rHdlCfg.DesignFunctionName = sprintf('%s_core',tModelName);
rHdlCfg.TestBenchName = sprintf('%s_tb',tModelName);
rHdlCfg.NoResetInitializationMode = 'NoResetNone';
rHdlCfg.SimulationTool = 'No simulation tool available on system path';
rHdlCfg.SynthesisTool = 'No synthesis tool available on system path';
rHdlCfg.SynthesisToolChipFamily = '';
rHdlCfg.SynthesisToolDeviceName = '';
rHdlCfg.SynthesisToolPackageName = '';
rHdlCfg.SynthesisToolSpeedValue = '';
rHdlCfg.IPCoreName = sprintf('%s_core_ipcore',tModelName);
rHdlCfg.IPCoreVersion = '1.0';
rHdlCfg.GenerateHDLTestBench = true;
rHdlCfg.ResetAssertedLevel = 'ActiveLow';
rHdlCfg.ResetInputPort = 'reset_n';
rHdlCfg.InputType = 'SignedUnsigned';
rHdlCfg.UseFiAccelForTestBench = true;
rHdlCfg.SimulationIterationLimit = 0;
rHdlCfg.HDLSynthTool = 'ISE';
rHdlCfg.HDLSynthFilePostfix = '_ise.tcl';
rHdlCfg.HDLSynthInit = sprintf('set src_dir [pwd]\\nset prj_dir "synprj"\\nfile mkdir ../$prj_dir\\ncd ../$prj_dir\\nproject new %%s.xise\\nproject set family Virtex4\\nproject set device xc4vsx35\\nproject set package ff668\\nproject set speed -10\\n');
rHdlCfg.HDLSynthCmd = sprintf('xfile add $src_dir/%%s\\n');
rHdlCfg.HDLSynthTerm = 'process run "Synthesize - XST"';
rHdlCfg.RegisterInputs = true;
rHdlCfg.RegisterOutputs = true;
rHdlCfg.MinimizeClockEnables = true;
rHdlCfg.DistributedPipelining = true;
rHdlCfg.DistributedPipeliningPriority = 'Performance';


%% Invoke Fixed-point Conversion and SV Code Generation
codegen -float2fixed rFixCfg -d ../../run -config rHdlCfg -args cModelArgs

%% Copy Model Core File to Release Folder 
tPath = sprintf('../../run/%s/hdlsrc',tModelName);
tSourceName = sprintf('%s_core.v',tModelName);
tTargetName = sprintf('%s_core.v',tModelName);
if exist(sprintf('%s/%s',tPath,tSourceName), 'file')
    if ~exist(sprintf('../../release/%s',tModelName),'dir')
        bSuccessD = mkdir(sprintf('../../release/%s',tModelName));
    else
        bSuccessD = 1;
    end
    bSuccessF = copyfile(sprintf('%s/%s',tPath,tSourceName),sprintf('../../release/%s/%s',tModelName,tTargetName),'f');
    if ~bSuccessF || ~bSuccessD
        error('An error occurred when copying %s', tTargetName);
    else
        fprintf('\n### %s copied successfully to /release folder!\n', tTargetName);
    end
end

%% Completion Message
fprintf('\n### Finished System-Verilog code generation for %s model!\n', tModelName);

