#include <stdint.h>
#include "target.h"
#include "loggerAPI.h"

#include "interface_wrx654.h"
#include "lib_wrx654_api.h"

/* register packages */
#include "wrx654_fcsi0_addrmap.h"
#include "wrx654_fcsi0_regpkg.h"
#include "wrx654_rf_antenna_fcsi_fieldpkg.h"
#include "wrx654_rf_antenna_fcsi_regpkg.h"
#include "wrx654_fcsi1_addrmap.h"
#include "wrx654_fcsi1_regpkg.h"
#include "wrx654_fcsi2_addrmap.h"
#include "wrx654_fcsi2_regpkg.h"
#include "wrx654_fcsi3_addrmap.h"
#include "wrx654_fcsi3_regpkg.h"
#include "wrx654_central_addrmap.h"
//#include "wrx654_central_regpkg.h"
#include "wrx654_rf_central_fcsi_fieldpkg.h"
#include "wrx654_rf_central_fcsi_regpkg.h"

#define LOG_LOCAL_GID   GLOBAL_GID_RF_CODE
#define LOG_LOCAL_FID 	2

// intialize global variable PllConfig (defined as "extern" in lib_wrx654_api.h) to chip state after Reset
pll_config 	PllConfig[NUM_OF_PLLS];
rx_gain_cfg	RxGainConfig[NUM_OF_ANTS];

uint8_t		PgcFiltLUT[5][11] = {
			{ 	0, 	0,	0,	0,	0,	0,	0,	0,	0,	3,	15}, /* 12.5MHz mode */
			{ 	0, 	0,	0,	0,	0,	0,	3,	15,	25,	32,	38}, /* 25MHz mode */
			{ 	0, 	0,	0,	3,	15,	25,	32,	38,	43,	47,	50}, /* 50MHz mode */
			{ 	3, 	15,	25,	32,	38,	43,	47,	50,	52,	54,	56}, /* 100MHz mode */
			{ 	23, 31,	37,	42,	46,	49,	52,	54,	55,	57,	58}, /* 150MHz (DPD) mode */
	};

// initialize vector of corner frequencies for PGC modes
uint16_t	PgcFc100kHz[5] = {125, 250, 500, 1000, 1500}; // 10*{12.5, 25, 50, 100, 150} * 100kHz


uint16_t rfic_api_version(void)
// return 4-digit integer abcd representing current RFIC API release version Va.bcd
{
	return 2120; // v2.12 A0
}


void rfic_init_1(uint8_t AntMsk)
{
	UNUSED_PARAM(AntMsk);
	uint8_t iAnt, iCfg;
	// ########################################################################
	// #                 Enable Bandgap and Central Biasing                   #
	// ########################################################################
	//-------------------------------------------------------------------------
	// Register BIASCENT	CENTRAL BIAS					Central	0x68	(OK)
	//-------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BIASCENT,
			WRX654_RF_CENTRAL_FCSI_BIASCENT_EN_U2I_P2PR_1__SVAL					// enable U2I P2PR
			| WRX654_RF_CENTRAL_FCSI_BIASCENT_EN_U2I_CONST_1__SVAL				// enable U2I Const
			| WRX654_RF_CENTRAL_FCSI_BIASCENT_EN_BGP_1__SVAL 					// enable bandgap
			| 14 << WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_TRIM_BGP__POS);			// set bandgap trim value to -7%


	// ########################################################################
	// #                    Enable PLL LDOs for both PLL                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL0PUP0	PLL0 POWER UP CENTRAL			Central	0x40	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP0,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_VCO_1__SVAL					// power up VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_DIG_1__SVAL				// power up 1.2V Dig LDOs
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_XO_1__SVAL				// power up clock path in reference block
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FB_1__SVAL				// power up LDO for FB path - DIVN/FCAL
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FF_1__SVAL 				// power up LDO for FF path - PCCP, LPF
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_CPLP_FBIAS_1__SVAL			// power up PDCPLPF Bias circuitry
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_OPAMP_LPATH_1__SVAL 			// power up OpAmps in Int path of LPF block
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_LPATH_1__SVAL			// enable Int path in LPF
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_PPATH_1__SVAL			// enable direct path in LPF
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_REG_RESYNC_1__SVAL); 			// power up DIVN re-sync block regulator

	//--------------------------------------------------------------------------
	// Register PLL0PUP1	PLL1 POWER UP CENTRAL			Central	0x50	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PUP0,
			WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_VCO_1__SVAL					// power up VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_DIG_1__SVAL				// power up 1.2V Dig LDOs
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_XO_1__SVAL				// power up clock path in reference block
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_FB_1__SVAL				// power up LDO for FB path - DIVN/FCAL
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_FF_1__SVAL 				// power up LDO for FF path - PCCP, LPF
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_CPLP_FBIAS_1__SVAL			// power up PDCPLPF Bias circuitry
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_OPAMP_LPATH_1__SVAL 			// power up OpAmps in Int path of LPF block
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_VVBGEN_LPATH_1__SVAL			// enable Int path in LPF
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_VVBGEN_PPATH_1__SVAL			// enable direct path in LPF
			| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_REG_RESYNC_1__SVAL); 			// power up DIVN re-sync block regulator

	// ###################################################################################
	// # Explicitly write Reset values to POR connected registers (if POR did not reset) #
	// ###################################################################################
	//--------------------------------------------------------------------------
	// Register DCXO0		DCXO 0         					Central	0x60
	//--------------------------------------------------------------------------
	// RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_DCXO0, 38550);					// Reset value: 0x9696
	// DCXO register may be written before init_1() - do not overwrite!
	//--------------------------------------------------------------------------
	// Register DCXO1		DCXO 1         					Central	0x61
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_DCXO1, 512);					// Reset value: 0x0200
	//--------------------------------------------------------------------------
	// Register DCXO2		DCXO 2         					Central	0x62
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_DCXO2, 113);					// Reset value: 0x0071
	//--------------------------------------------------------------------------
	// Register DCXO3		DCXO 3         					Central	0x63
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_DCXO3, 6148);					// Reset value: 0x1804
	//--------------------------------------------------------------------------
	// Register FSYS		FSYS 1 + 2         				Central	0x64
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYS, 114);						// Reset value: 0x0072
	//--------------------------------------------------------------------------
	// Register FSYSLDO2V5	FSYS LDO2V5        				Central	0x65
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5, 0);					// Reset value: 0x0000
	//--------------------------------------------------------------------------
	// Register FCSILDO1V2	FSYS LDO1V2        				Central	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_FCSILDO1V2, 2048);				// Reset value: 0x0800

    // initial setup of PLL configuration
    PllConfig[0].Active     = 1;
    PllConfig[0].Reset      = 1;
    PllConfig[0].HbLb       = 0;
    PllConfig[0].LoFreqs[0] = 0;
    PllConfig[0].LoFreqs[1] = 0;
    PllConfig[0].LoFreqs[2] = 0;
    PllConfig[0].LoFreqs[3] = 0;
    PllConfig[0].LoFreqs[4] = 0;
    PllConfig[0].LoFreqs[5] = 0;
    PllConfig[0].LoFreqAct  = 0;
    PllConfig[0].RefClk     = 10;
    PllConfig[0].IntFrac    = 1;
    PllConfig[0].HiPwr      = 0;
    PllConfig[0].AntSet     = 0;
    PllConfig[0].FastHop    = 0;
    PllConfig[0].FhMaster   = 0;

    PllConfig[1].Active     = 1;
    PllConfig[1].Reset      = 1;
    PllConfig[1].HbLb       = 0;
    PllConfig[1].LoFreqs[0] = 0;
    PllConfig[1].LoFreqs[1] = 0;
    PllConfig[1].LoFreqs[2] = 0;
    PllConfig[1].LoFreqs[3] = 0;
    PllConfig[1].LoFreqs[4] = 0;
    PllConfig[1].LoFreqs[5] = 0;
    PllConfig[1].LoFreqAct  = 0;
    PllConfig[1].RefClk     = 10;
    PllConfig[1].IntFrac    = 1;
    PllConfig[1].HiPwr      = 0;
    PllConfig[1].AntSet     = 0;
    PllConfig[1].FastHop    = 0;
    PllConfig[1].FhMaster   = 0;


    // initial RX gain configuration after reset
    for (iAnt=0; iAnt<4; iAnt++)
    {
    	for (iCfg=0; iCfg<7; iCfg++)
    	{
    		RxGainConfig[iAnt].LnaGainCfg[iCfg] = 0;	// LNA gain configurations
    		RxGainConfig[iAnt].FectlCfg[iCfg] = 0;		// FECTL configurations
    	}
    	for (iCfg=0; iCfg<4; iCfg++)
    	{
    		RxGainConfig[iAnt].RfGainIdx[iCfg] = 0;		// RF gain constellation index (SR1 content and Direct Steps 1/2/3)
    		RxGainConfig[iAnt].RssiRange[iCfg] = 0;		// RSSI range (SR1 content and Direct Steps 1/2/3)
    		RxGainConfig[iAnt].PgcGainIdx[iCfg] = 0;	// PGC gain index (SR1 content and Direct Steps 1/2/3)
    		RxGainConfig[iAnt].PgcFiltIdx[iCfg] = 0;	// PGC filter index (SR1 content and Direct Steps 1/2/3)
    	}
    	RxGainConfig[iAnt].ShortReg3 = 0;				// copies SR3 content
    	RxGainConfig[iAnt].ForceOff = 0;				// ForceOff bit in SR1
    	RxGainConfig[iAnt].DirStepIdx = 0;				// equals SR3 content if ForceOff==0, equals 0 if ForceOff==1
    }

}


void rfic_init_2(uint8_t AntMsk)
{
	// ########################################################################
	// #                    Enable Antenna Bias and BB LDO                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_1__SVAL							// enable BB LDO
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_ANTBIAS_CURRENT_MIRROR_1__SVAL		// enable antenna biasing current mirror
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT_4__SVAL						// set Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL				// select LB set of doubled registers (should be set by default)
    		| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_LDO_RDY_1__SVAL					// LDO Weakness Fix: Toggle LDO RDY manually
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_FORCE_LDO_RDY_0__SVAL);				// LDO Weakness Fix: Toggle LDO RDY manually
}


void rfic_init_3(uint8_t AntMsk)
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_1__SVAL							// enable BB LDO
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_ANTBIAS_CURRENT_MIRROR_1__SVAL		// enable antenna biasing current mirror
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT_4__SVAL						// set Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL				// select LB set of doubled registers (should be set by default)
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_LDO_RDY_1__SVAL					// LDO Weakness Fix: Toggle LDO RDY manually
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_FORCE_LDO_RDY_1__SVAL);				// LDO Weakness Fix: Toggle LDO RDY manually

	// ########################################################################
	// #                 Enable BBO LDO and configure BBO                     #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BIAS_1__SVAL					// enable BBIO biasing
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP_1__SVAL				// set compensation cap to 4.5pF
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_4__SVAL				// set Vout to 1.26V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);				// enable BBIO LDO

	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_OLGAIN_NOP_7__SVAL				// normal operation OL gain to 41 dB
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_CM_GAIN_2__SVAL				// set CM loop gain to 31dB (changed default)
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I_1__SVAL				// enable In0 I input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q_1__SVAL				// enable In0 Q input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL				// enable BBO I
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q_1__SVAL);			// enable BBO Q

	//--------------------------------------------------------------------------
	// Register BBIOCONF1	BBIO CONFIG 1					Antx	0x43	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_BBIOCONF1,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_SOFT_PUP_1__SVAL				// enable soft power-up
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW160_0__SVAL			// 450fF feedback cap for 160 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW80_1__SVAL				// 900fF feedback cap for 80 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW2040_3__SVAL			// 1350fF feedback cap for 20 and 40 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_HP_0__SVAL			// HP power save mode to full power
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_RP_3__SVAL			// RP power save mode to -2 mA
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_LP_7__SVAL);		// LP power save mode to -3 mA


	// ########################################################################
	// #                      Configure PLL0 for LB mode                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL0PMU0	PLL0 PMU CENTRAL				Central	0x41	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU0,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU0_LDO_DIG_SEL_4__SVAL					// DigReg LDO to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_LDO_VCO_SEL_4__SVAL 				// VCO LDO to 1.417V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_REG_LO_SEL_2__SVAL  				// IQDIV LDO to 1.258V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_VCOREG_INTREF_SEL_1__SVAL			// internal reference for VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_DIAG_DISABLE_1__SVAL);			// disable diagnostic bus

	//--------------------------------------------------------------------------
	// Register PLL0PMU1	PLL0 PMU DIAG CONTROL			Central	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU1,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLL0PUP1	PLL0 POWER UP LOCAL				Central	0x43	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP1,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_FPD_LSBUFF_FB_1__SVAL); 			// force power-down 6G output buffers (==LB mode)

	//--------------------------------------------------------------------------
	// Register PLL0VCO0	PLL0 VCO CONTROL 0				Central	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0VCO0,
			WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V

	//--------------------------------------------------------------------------
	// Register PLL0SDM0	PLL0 SIGMA DELTA 0 				Central	0x46	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0,
			WRX654_RF_CENTRAL_FCSI_PLL0SDM0_SDM_EN_1__SVAL); 					// enable DSM (==Fractional Mode)

	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL0MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_REF_SEL_40M_1__SVAL);  		// set ref clock to 40MHz

	// ########################################################################
	// #                      Configure PLL1 for LB mode                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL1PMU0	PLL1 PMU CENTRAL				Central	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PMU0,
			WRX654_RF_CENTRAL_FCSI_PLL1PMU0_LDO_DIG_SEL_4__SVAL					// DigReg LDO to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_LDO_VCO_SEL_4__SVAL 				// VCO LDO to 1.417V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_REG_LO_SEL_2__SVAL  				// IQDIV LDO to 1.258V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_VCOREG_INTREF_SEL_1__SVAL			// internal reference for VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_DIAG_DISABLE_1__SVAL);			// disable diagnostic bus

	//--------------------------------------------------------------------------
	// Register PLL1PMU1	PLL1 PMU DIAG CONTROL			Central	0x52	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PMU1,
			WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLL1PUP1	PLL1 POWER UP LOCAL				Central	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PUP1,
			WRX654_RF_CENTRAL_FCSI_PLL1PUP1_FPD_LSBUFF_FB_1__SVAL); 			// force power-down 6G output buffers (==LB mode)

	//--------------------------------------------------------------------------
	// Register PLL1VCO0	PLL1 VCO CONTROL 0				Central	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1VCO0,
			WRX654_RF_CENTRAL_FCSI_PLL1VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL1VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V

	//--------------------------------------------------------------------------
	// Register PLL1SDM0	PLL1 SIGMA DELTA 0 				Central	0x56	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1SDM0,
			WRX654_RF_CENTRAL_FCSI_PLL1SDM0_SDM_EN_1__SVAL); 					// enable DSM (==Fractional Mode)

	//--------------------------------------------------------------------------
	// Register PLL1MISC	PLL1 MISCELLANEOUS				Central	0x5A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1MISC,
			WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL1MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_REF_SEL_40M_1__SVAL);  		// set ref clock to 40MHz


	// ########################################################################
	// #                             Disable FSYS2                            #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register FSYS		FSYS 1 + 2         				Central	0x64
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYS,
			WRX654_RF_CENTRAL_FCSI_FSYS_DISABLE__ISMSK,
			WRX654_RF_CENTRAL_FCSI_FSYS_DISABLE_1__SVAL);						// disable FSYS2


	// ########################################################################
	// #                     Enable VCM Buffer (0.6V/0.5V)                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_VCMBUFFER_1__SVAL 				// enable VCM buffer
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_VCMBUFFER_3__SVAL		// set control voltage output of VCM buffer to 0.5V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_I_H_VCMBUFFER_1__SVAL		// double current in output stage of VCM Opamp
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BIAS_1__SVAL					// keep BBIO biasing enabled
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP_1__SVAL				// keep compensation cap to 4.5pF
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_4__SVAL				// keep Vout to 1.26V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);				// keep BBIO LDO enabled

	//--------------------------------------------------------------------------
	// Register VCMBIAS		VCM CONTROL ANTENNA BIASING		Antx	0x65	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VCMBIAS,
			WRX654_RF_ANTENNA_FCSI_VCMBIAS_CTL_VOUT_VCMBUFFER_0V5_3__SVAL		// set 0.5V VCM buffer to 0.5V
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_EN_VCMBUFFER_0V5_1__SVAL 			// enable 0.5V VCM buffer
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_CTL_VOUT_VCMBUFFER_0V6_1__SVAL		// set 0.6V VCM buffer to 0.6V
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_EN_VCMBUFFER_0V6_1__SVAL);			// enable 0.6V VCM buffer


	// ########################################################################
	// #                           Enable RF 2V5 LDO                          #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC_1__SVAL						// enable loop compensation
			| WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE_1__SVAL);					// enable RF 2V5 LDO


	// ########################################################################
	// #                             Enable RX LDO                            #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RF LDO1V2 RX					Antx	0x46	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT_4__SVAL					// set Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE_1__SVAL);					// enable RX 1V2 LDO


	// ########################################################################
	// #                 Set Vout / Enable TX LDO HB and LB                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	RF LDO1V2 TX LB					Antx	0x45 	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT_6__SVAL					// set Vout to 1.28V
			| WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_1__SVAL);				// enable TX LB 1V2 LDO

	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	RF LDO1V2 TX HB					Antx	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT_8__SVAL					// set Vout to 1.3V
			| WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_0__SVAL);				// disable TX HB 1V2 LDO


	// ########################################################################
	// #                        Enable Mixer BB and PGC                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_MIXER_1__SVAL						// enable Mixer BB core
			| 3 << WRX654_RF_ANTENNA_FCSI_PGC_EN_CH_MIXER__POS					// activate I and Q branches
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW_1__SVAL			// close IQCAL GND switch (always closed in regular operation!)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);					// NEW: reduce PGC compensation cap in regular mode


	// ########################################################################
	// #                   Enable AnaMux, xSSI, Offset DAC                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register SSI			XSSI and ANAMUX					Antx	0x5d	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN_1__SVAL						// select xSSI path as AnaMux input
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX_1__SVAL					// enable AnaMux
			| WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE_1__SVAL 				// TSSI input divider ratio 0.9
			| WRX654_RF_ANTENNA_FCSI_SSI_CTL_SSI_ODAC_1__SVAL					// manually control S2D and ODAC enable
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_SSI_GAIN_1__SVAL				// xSSI S2D always on (needed?)
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_SSI_ODAC_1__SVAL); 				// xSSI ODAC always on (needed?)


	// ########################################################################
	// #                 Enable/Config PGC DC State Machine                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXDCOCTL	RX MIXER DC OFFSET CONTROL		Antx	0x66	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I_1__SVAL						// release reset for I branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TOGGLE_EN_DCO_I_1__SVAL				// enable toggling for I branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q_1__SVAL					// release reset for Q branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TOGGLE_EN_DCO_Q_1__SVAL);				// enable toggling for Q branch


	// ########################################################################
	// #          Central LO Configuration (all Antennas to PLL0 LB)          #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LOCENTLB	LO CENTRAL LB					Antx	0x40	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
			WRX654_RF_ANTENNA_FCSI_LOCENTLB_EN_PTRACK_1__SVAL						// enable process tracking
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT_2__SVAL						// set Vout to 1.234V (@@@ needs to be reviewed with Andriy)
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0_1__SVAL				// enable LB FOB for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0_1__SVAL			// enable LB driver for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1_0__SVAL				// disable LB FOB for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1_0__SVAL			// disable LB driver for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_0__SVAL);				// disable LB LDO to prevent activation without LO signal

	//--------------------------------------------------------------------------
	// Register LOCENTHB	LO CENTRAL HB					Antx	0x41	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
			WRX654_RF_ANTENNA_FCSI_LOCENTHB_EN_PTRACK_1__SVAL						// enable process tracking
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT_2__SVAL						// set Vout to 1.234V (@@@ needs to be reviewed with Andriy)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0_0__SVAL				// disable HB FOB for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0_0__SVAL				// disable HB buffer for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0_0__SVAL				// disable HB V2I for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1_1__SVAL				// enable HB FOB for PLL1 (not used as long HBLB is set to LB)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1_1__SVAL				// enable HB buffer for PLL1 (not used as long HBLB is set to LB)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1_1__SVAL);			// enable HB V2I for PLL1 (not used as long HBLB is set to LB)


	// ########################################################################
	// #                         Enable Temp Sensor                           #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central	0x6D	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_1__SVAL);


	// ########################################################################
	// # Set DC config, LO freq, BW, OpMode, PGC filter, default BS for TX/RX #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC_0__SVAL		// DC offset configuration #0
			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH_0__SVAL		// select PLL channel 0
			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW_0__SVAL		// set to 20MHz BW
			| WRX654_RF_ANTENNA_FCSI_SR0_MODE_0__SVAL		// set to High Performance mode
			| WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND_3__SVAL	// TX band selection 3d
			| WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND_5__SVAL);	// RX band selection 5d


	// ########################################################################
	// #                Set RX Filter, PGC Gain, RX RF Gain                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			42 << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS																	// set RX Filter Corner Freq to 25 MHz
			| WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN_10__SVAL							// set to 20dB PGC gain
			| WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG_0__SVAL);					// set to RX gain config #0 (requires to set up config #0 properly!)
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[0] = 10;
			RxGainConfig[iAnt].RfGainIdx[0] = 0;
		}
	}


	// ########################################################################
	// #                   Set Pre-Driver Gain, TPC Gain                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN_0__SVAL							// set Pre-Driver gain to max
			| WRX654_RF_ANTENNA_FCSI_SR2_TPC_2__SVAL);								// set to TPC gain config #2 (+2dB)


	// ========================================================================
	// #                 CONFIGURE HIGHBAND SET OF REGISTERS                  #
	// ========================================================================

	// ########################################################################
	// #      Switch to HB set of registers (but keep current HB/LB mode)     #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_1__SVAL							// keep BB LDO enabled
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_ANTBIAS_CURRENT_MIRROR_1__SVAL		// keep antenna biasing current mirror enabled
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT_4__SVAL						// keep Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL);				// select HB set of doubled registers (should be set by default)


	// ########################################################################
	// #                 Set LNA Mid Gain & RSSI Configuration                #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			10 << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS);			// set Mid Gain tuning to nominal value
	//		| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable (HB) LNA (leave HB blocks disabled)
	//		| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);					// enable (HB) RSSI (leave HB blocks disabled)

	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL01,
			WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0_2__SVAL);						// set RX gain config #0 to Mid LNA gain
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].LnaGainCfg[0] = 2;
			RxGainConfig[iAnt].FectlCfg[0] = 0;
			RxGainConfig[iAnt].LnaGainCfg[1] = 0;			// config 1 will be also reset
			RxGainConfig[iAnt].FectlCfg[1] = 0;				// config 1 will be also reset
		}
	}

	// ########################################################################
	// #                     RX Mixer RF Core Configuration                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL);					// enable I and Q branches
	//		| WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL); 				// enable (HB) RX Mixer (leave HB blocks disabled)


	// ########################################################################
	// #                       TPC Filter Default Tuning                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCFILTER0	TPC Filter Tuning 0		 		Antx	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCFILTER0,
			0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__POS		// set TPC FB Cap for 20MHz to 35d acc. Jan's proposal
			| 0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__POS);	// set TPC FB Cap for 40MHz to 35d acc. Jan's proposal

	//--------------------------------------------------------------------------
	// Register TPCFILTER1	TPC Filter Tuning 1		 		Antx	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCFILTER1,
			0x4f << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__POS		// set TPC FB Cap for 80MHz to 79d acc. Jan's proposal
			| 0x69 << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__POS);	// set TPC FB Cap for 160MHz to 105d acc. Jan's proposal


	// ########################################################################
	// #                    TPC and TX Mixer Configuration                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_ENABLE_ACT_MIRROR_1__SVAL				// enable Active Current Mirror for HB PAD
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_ENABLE_VC_BIAS_1__SVAL);				// enable VC biasing of pre-driver

	// ########################################################################
	// #                       Pre-Driver Configuration                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_VC_PAD_4__SVAL						// set control cascode voltage to 0.99V
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_PAD_BIAS_7__SVAL);				// set pre-driver bias to 90%


	// ========================================================================
	// #                  CONFIGURE LOWBAND SET OF REGISTERS                  #
	// ========================================================================

	// ########################################################################
	// #      REPEAT LB select (to make sure that LB set is selected)         #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_1__SVAL							// keep BB LDO enabled
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_ANTBIAS_CURRENT_MIRROR_1__SVAL		// keep antenna biasing current mirror enabled
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT_4__SVAL						// keep Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL);				// return to LB set of doubled registers (should be set by default)


	// ########################################################################
	// #                 Set LNA Mid Gain & RSSI Configuration                #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			10 << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS				// set Mid Gain tuning to nominal value
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable (LB) LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);					// enable (LB) RSSI

	//--------------------------------------------------------------------------
	// Register LNARSSI		RSSI Bias Configuration			Antx	0x5B
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RSSIBIAS,
			WRX654_RF_ANTENNA_FCSI_RSSIBIAS_ENABLE_RSSI_BIAS_1__SVAL				// enable RSSI Bias
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_BIAS_SELECT_1__SVAL				// select P2PR bias current
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_TEMP_COMP_1__SVAL);				// enable temperature compensation

	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL01,
			WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0_2__SVAL);						// set RX gain config #0 to Mid LNA gain
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].LnaGainCfg[0] = 2;
			RxGainConfig[iAnt].FectlCfg[0] = 0;
			RxGainConfig[iAnt].LnaGainCfg[1] = 0;			// config 1 will be also reset
			RxGainConfig[iAnt].FectlCfg[1] = 0;				// config 1 will be also reset
		}
	}


	// ########################################################################
	// #                     RX Mixer RF Core Configuration                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL						// enable I and Q branches
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL); 				// enable (LB) RX Mixer


	// ########################################################################
	// #                       TPC Filter Default Tuning                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCFILTER0	TPC Filter Tuning 0		 		Antx	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCFILTER0,
			0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__POS		// set TPC FB Cap for 20MHz to 35d acc. Jan's proposal
			| 0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__POS);	// set TPC FB Cap for 40MHz to 35d acc. Jan's proposal

	//--------------------------------------------------------------------------
	// Register TPCFILTER1	TPC Filter Tuning 1		 		Antx	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCFILTER1,
			0x4f << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__POS		// set TPC FB Cap for 80MHz to 79d acc. Jan's proposal
			| 0x69 << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__POS);	// set TPC FB Cap for 160MHz to 105d acc. Jan's proposal


	// ########################################################################
	// #                    TPC and TX Mixer Configuration                    #
	// ########################################################################
	// nothing to set so far in TPCMIXER register


	// ########################################################################
	// #                  TPC Gain / DC Offset Configuration                  #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCGAIN		TPC GAIN CONTROL		 		Antx	0x55	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCGAIN,
			WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0_0__SVAL					// set TPC Gain Config #0 to -6dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN1_2__SVAL					// set TPC Gain Config #1 to -2dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN2_4__SVAL					// set TPC Gain Config #2 to +2dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN3_6__SVAL);				// set TPC Gain Config #3 to +6dB


	// ########################################################################
	// #                       Pre-Driver Configuration                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_VC_PAD_5__SVAL						// set control cascode voltage to 0.93V
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_1__SVAL); 						// enable (LB) pre-driver

    // update PLL configuration
    PllConfig[0].RefClk     = 40;       // set ref clock to 40MHz
    PllConfig[0].IntFrac    = 0;        // enable DSM (==Fractional Mode)
    PllConfig[1].RefClk     = 40;       // set ref clock to 40MHz
    PllConfig[1].IntFrac    = 0;        // enable DSM (==Fractional Mode)
    PllConfig[0].AntSet     = AntMsk;   // all selectes antennas connected to PLL0 LB

}


void rfic_init_3_lb_only(void)
{
	uint8_t		iAnt;
	// ########################################################################
	// #                 Enable BBO LDO and configure BBO                     #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BIAS_1__SVAL					// enable BBIO biasing
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP_1__SVAL				// set compensation cap to 4.5pF
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_4__SVAL				// set Vout to 1.26V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);				// enable BBIO LDO

	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_OLGAIN_NOP_7__SVAL				// normal operation OL gain to 41 dB
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_CM_GAIN_2__SVAL				// set CM loop gain to 31dB (changed default)
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I_1__SVAL				// enable In0 I input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q_1__SVAL				// enable In0 Q input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL				// enable BBO I
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q_1__SVAL);			// enable BBO Q

	//--------------------------------------------------------------------------
	// Register BBIOCONF1	BBIO CONFIG 1					Antx	0x43	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_BBIOCONF1,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_SOFT_PUP_1__SVAL				// enable soft power-up
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW160_0__SVAL			// 450fF feedback cap for 160 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW80_1__SVAL				// 900fF feedback cap for 80 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW2040_3__SVAL			// 1350fF feedback cap for 20 and 40 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_HP_0__SVAL			// HP power save mode to full power
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_RP_3__SVAL			// RP power save mode to -2 mA
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_LP_7__SVAL);		// LP power save mode to -3 mA


	// ########################################################################
	// #                      Configure PLL0 for LB mode                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL0PMU0	PLL0 PMU CENTRAL				Central	0x41	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU0,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU0_LDO_DIG_SEL_4__SVAL					// DigReg LDO to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_LDO_VCO_SEL_4__SVAL 				// VCO LDO to 1.417V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_REG_LO_SEL_2__SVAL  				// IQDIV LDO to 1.258V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_VCOREG_INTREF_SEL_1__SVAL			// internal reference for VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_DIAG_DISABLE_1__SVAL);			// disable diagnostic bus

	//--------------------------------------------------------------------------
	// Register PLL0PMU1	PLL0 PMU DIAG CONTROL			Central	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU1,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLL0PUP1	PLL0 POWER UP LOCAL				Central	0x43	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP1,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_FPD_LSBUFF_FB_1__SVAL); 			// force power-down 6G output buffers (==LB mode)

	//--------------------------------------------------------------------------
	// Register PLL0VCO0	PLL0 VCO CONTROL 0				Central	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0VCO0,
			WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V

	//--------------------------------------------------------------------------
	// Register PLL0SDM0	PLL0 SIGMA DELTA 0 				Central	0x46	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0,
			WRX654_RF_CENTRAL_FCSI_PLL0SDM0_SDM_EN_1__SVAL); 					// enable DSM (==Fractional Mode)

	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL0MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_REF_SEL_40M_1__SVAL);  		// set ref clock to 40MHz


	// ########################################################################
	// #                      Configure PLL1 for LB mode                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL1PMU0	PLL1 PMU CENTRAL				Central	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PMU0,
			WRX654_RF_CENTRAL_FCSI_PLL1PMU0_LDO_DIG_SEL_4__SVAL					// DigReg LDO to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_LDO_VCO_SEL_4__SVAL 				// VCO LDO to 1.417V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_REG_LO_SEL_2__SVAL  				// IQDIV LDO to 1.258V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_VCOREG_INTREF_SEL_1__SVAL			// internal reference for VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_DIAG_DISABLE_1__SVAL);			// disable diagnostic bus

	//--------------------------------------------------------------------------
	// Register PLL1PMU1	PLL1 PMU DIAG CONTROL			Central	0x52	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PMU1,
			WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLL1PUP1	PLL1 POWER UP LOCAL				Central	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PUP1,
			WRX654_RF_CENTRAL_FCSI_PLL1PUP1_FPD_LSBUFF_FB_1__SVAL); 			// force power-down 6G output buffers (==LB mode)

	//--------------------------------------------------------------------------
	// Register PLL1VCO0	PLL1 VCO CONTROL 0				Central	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1VCO0,
			WRX654_RF_CENTRAL_FCSI_PLL1VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL1VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V

	//--------------------------------------------------------------------------
	// Register PLL1SDM0	PLL1 SIGMA DELTA 0 				Central	0x56	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1SDM0,
			WRX654_RF_CENTRAL_FCSI_PLL1SDM0_SDM_EN_1__SVAL); 					// enable DSM (==Fractional Mode)

	//--------------------------------------------------------------------------
	// Register PLL1MISC	PLL1 MISCELLANEOUS				Central	0x5A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1MISC,
			WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL1MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_REF_SEL_40M_1__SVAL);  		// set ref clock to 40MHz

	// ########################################################################
	// #                     Enable VCM Buffer (0.6V/0.5V)                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_VCMBUFFER_1__SVAL 				// enable VCM buffer
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_VCMBUFFER_3__SVAL		// set control voltage output of VCM buffer to 0.5V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_I_H_VCMBUFFER_1__SVAL		// double current in output stage of VCM Opamp
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BIAS_1__SVAL					// keep BBIO biasing enabled
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP_1__SVAL				// keep compensation cap to 4.5pF
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_4__SVAL				// keep Vout to 1.26V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);				// keep BBIO LDO enabled

	//--------------------------------------------------------------------------
	// Register VCMBIAS		VCM CONTROL ANTENNA BIASING		Antx	0x65	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_VCMBIAS,
			WRX654_RF_ANTENNA_FCSI_VCMBIAS_CTL_VOUT_VCMBUFFER_0V5_3__SVAL		// set 0.5V VCM buffer to 0.5V
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_EN_VCMBUFFER_0V5_1__SVAL 			// enable 0.5V VCM buffer
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_CTL_VOUT_VCMBUFFER_0V6_1__SVAL		// set 0.6V VCM buffer to 0.6V
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_EN_VCMBUFFER_0V6_1__SVAL);			// enable 0.6V VCM buffer


	// ########################################################################
	// #                           Enable RF 2V5 LDO                          #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC_1__SVAL						// enable loop compensation
			| WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE_1__SVAL);					// enable RF 2V5 LDO


	// ########################################################################
	// #                             Enable RX LDO                            #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RF LDO1V2 RX					Antx	0x46	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT_4__SVAL					// set Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE_1__SVAL);					// enable RX 1V2 LDO


	// ########################################################################
	// #                 Set Vout / Enable TX LDO HB and LB                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	RF LDO1V2 TX LB					Antx	0x45 	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT_6__SVAL					// set Vout to 1.28V
			| WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_1__SVAL);				// enable TX LB 1V2 LDO

	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	RF LDO1V2 TX HB					Antx	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT_8__SVAL					// set Vout to 1.3V
			| WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_0__SVAL);				// disable TX HB 1V2 LDO


	// ########################################################################
	// #                        Enable Mixer BB and PGC                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_MIXER_1__SVAL						// enable Mixer BB core
			| 3 << WRX654_RF_ANTENNA_FCSI_PGC_EN_CH_MIXER__POS					// activate I and Q branches
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW_1__SVAL			// close IQCAL GND switch (always closed in regular operation!)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);					// NEW: reduce PGC compensation cap in regular mode


	// ########################################################################
	// #                   Enable AnaMux, xSSI, Offset DAC                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register SSI			XSSI and ANAMUX					Antx	0x5d	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN_1__SVAL						// select xSSI path as AnaMux input
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX_1__SVAL					// enable AnaMux
			| WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE_1__SVAL 				// TSSI input divider ratio 0.9
			| WRX654_RF_ANTENNA_FCSI_SSI_CTL_SSI_ODAC_1__SVAL					// manually control S2D and ODAC enable
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_SSI_GAIN_1__SVAL				// xSSI S2D always on (needed?)
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_SSI_ODAC_1__SVAL); 				// xSSI ODAC always on (needed?)


	// ########################################################################
	// #                 Enable/Config PGC DC State Machine                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXDCOCTL	RX MIXER DC OFFSET CONTROL		Antx	0x66	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I_1__SVAL						// release reset for I branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TOGGLE_EN_DCO_I_1__SVAL				// enable toggling for I branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q_1__SVAL					// release reset for Q branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TOGGLE_EN_DCO_Q_1__SVAL);				// enable toggling for Q branch


	// ########################################################################
	// #          Central LO Configuration (all Antennas to PLL0 LB)          #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LOCENTLB	LO CENTRAL LB					Antx	0x40	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
			WRX654_RF_ANTENNA_FCSI_LOCENTLB_EN_PTRACK_1__SVAL						// enable process tracking
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT_2__SVAL						// set Vout to 1.234V (@@@ needs to be reviewed with Andriy)
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0_1__SVAL				// enable LB FOB for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0_1__SVAL			// enable LB driver for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1_0__SVAL				// disable LB FOB for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1_0__SVAL			// disable LB driver for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_0__SVAL);				// disable LB LDO to prevent activation without LO signal

	//--------------------------------------------------------------------------
	// Register LOCENTHB	LO CENTRAL HB					Antx	0x41	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
			WRX654_RF_ANTENNA_FCSI_LOCENTHB_EN_PTRACK_1__SVAL						// enable process tracking
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT_2__SVAL						// set Vout to 1.234V (@@@ needs to be reviewed with Andriy)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0_0__SVAL				// disable HB FOB for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0_0__SVAL				// disable HB buffer for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0_0__SVAL				// disable HB V2I for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1_1__SVAL				// enable HB FOB for PLL1 (not used as long HBLB is set to LB)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1_1__SVAL				// enable HB buffer for PLL1 (not used as long HBLB is set to LB)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1_1__SVAL);			// enable HB V2I for PLL1 (not used as long HBLB is set to LB)


	// ########################################################################
	// #                         Enable Temp Sensor                           #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central	0x6D	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_1__SVAL);


	// ########################################################################
	// # Set DC config, LO freq, BW, OpMode, PGC filter, default BS for TX/RX #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC_0__SVAL		// DC offset configuration #0
			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH_0__SVAL		// select PLL channel 0
			//			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW_0__SVAL		// set to 20MHz BW
			//			| WRX654_RF_ANTENNA_FCSI_SR0_MODE_0__SVAL		// set to High Performance mode
			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW_2__SVAL		// set to 80MHz BW
			| WRX654_RF_ANTENNA_FCSI_SR0_MODE_1__SVAL		// set to Reduced Performance mode
			| WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND_3__SVAL	// TX band selection 3d
			| WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND_5__SVAL);	// RX band selection 5d


	// ########################################################################
	// #                Set RX Filter, PGC Gain, RX RF Gain                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SR1,
			//			42 << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS							// set RX Filter Corner Freq to 25 MHz
			0 << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS							// set RX Filter Corner Freq to (>)50 MHz
			| WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN_10__SVAL							// set to 20dB PGC gain
			| WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG_0__SVAL);					// set to RX gain config #0 (requires to set up config #0 properly!)
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		RxGainConfig[iAnt].PgcGainIdx[0] = 10;
		RxGainConfig[iAnt].RfGainIdx[0] = 0;
	}

	// ########################################################################
	// #                   Set Pre-Driver Gain, TPC Gain                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN_0__SVAL							// set Pre-Driver gain to max
			| WRX654_RF_ANTENNA_FCSI_SR2_TPC_2__SVAL);								// set to TPC gain config #2 (+2dB)


	// ########################################################################
	// #                 Set LNA Mid Gain & RSSI Configuration                #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			10 << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS				// set Mid Gain tuning to nominal value
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable (LB) LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);					// enable (LB) RSSI

	//--------------------------------------------------------------------------
	// Register LNARSSI		RSSI Bias Configuration			Antx	0x5B
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RSSIBIAS,
			WRX654_RF_ANTENNA_FCSI_RSSIBIAS_ENABLE_RSSI_BIAS_1__SVAL				// enable RSSI Bias
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_BIAS_SELECT_1__SVAL				// select P2PR bias current
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_TEMP_COMP_1__SVAL);				// enable temperature compensation

	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_FECTL01,
			WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0_2__SVAL);						// set RX gain config #0 to Mid LNA gain
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		RxGainConfig[iAnt].LnaGainCfg[0] = 2;
		RxGainConfig[iAnt].FectlCfg[0] = 0;
		RxGainConfig[iAnt].LnaGainCfg[1] = 0;			// config 1 will be also reset
		RxGainConfig[iAnt].FectlCfg[1] = 0;				// config 1 will be also reset
	}

	// ########################################################################
	// #                     RX Mixer RF Core Configuration                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL						// enable I and Q branches
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL); 				// enable (LB) RX Mixer


	// ########################################################################
	// #                       TPC Filter Default Tuning                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCFILTER0	TPC Filter Tuning 0		 		Antx	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCFILTER0,
			0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__POS		// set TPC FB Cap for 20MHz to 35d acc. Jan's proposal
			| 0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__POS);	// set TPC FB Cap for 40MHz to 35d acc. Jan's proposal

	//--------------------------------------------------------------------------
	// Register TPCFILTER1	TPC Filter Tuning 1		 		Antx	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCFILTER1,
			0x4f << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__POS		// set TPC FB Cap for 80MHz to 79d acc. Jan's proposal
			| 0x69 << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__POS);	// set TPC FB Cap for 160MHz to 105d acc. Jan's proposal


	// ########################################################################
	// #                    TPC and TX Mixer Configuration                    #
	// ########################################################################
	// nothing to set so far in TPCMIXER register


	// ########################################################################
	// #                  TPC Gain / DC Offset Configuration                  #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCGAIN		TPC GAIN CONTROL		 		Antx	0x55	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCGAIN,
			WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0_0__SVAL					// set TPC Gain Config #0 to -6dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN1_2__SVAL					// set TPC Gain Config #1 to -2dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN2_4__SVAL					// set TPC Gain Config #2 to +2dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN3_6__SVAL);				// set TPC Gain Config #3 to +6dB


	// ########################################################################
	// #                       Pre-Driver Configuration                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_VC_PAD_5__SVAL						// set control cascode voltage to 0.93V
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_1__SVAL); 						// enable (LB) pre-driver

    // update PLL configuration
    PllConfig[0].RefClk     = 40;       // set ref clock to 40MHz
    PllConfig[0].IntFrac    = 0;        // enable DSM (==Fractional Mode)
    PllConfig[1].RefClk     = 40;       // set ref clock to 40MHz
    PllConfig[1].IntFrac    = 0;        // enable DSM (==Fractional Mode)
    PllConfig[0].AntSet     = ANTALL;   // all selectes antennas connected to PLL0 LB

}


void rfic_init_3_hb_only(void)
{
	uint8_t		iAnt;
	// ########################################################################
	// #                 Enable BBO LDO and configure BBO                     #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BIAS_1__SVAL					// enable BBIO biasing
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP_1__SVAL				// set compensation cap to 4.5pF
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_4__SVAL				// set Vout to 1.26V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);				// enable BBIO LDO

	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_OLGAIN_NOP_7__SVAL				// normal operation OL gain to 41 dB
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_CM_GAIN_2__SVAL				// set CM loop gain to 31dB (changed default)
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I_1__SVAL				// enable In0 I input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q_1__SVAL				// enable In0 Q input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL				// enable BBO I
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q_1__SVAL);			// enable BBO Q

	//--------------------------------------------------------------------------
	// Register BBIOCONF1	BBIO CONFIG 1					Antx	0x43	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_BBIOCONF1,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_SOFT_PUP_1__SVAL				// enable soft power-up
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW160_0__SVAL			// 450fF feedback cap for 160 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW80_1__SVAL				// 900fF feedback cap for 80 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_BW2040_3__SVAL			// 1350fF feedback cap for 20 and 40 MHz
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_HP_0__SVAL			// HP power save mode to full power
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_RP_3__SVAL			// RP power save mode to -2 mA
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF1_CTL_BBO_PSAVE_LP_7__SVAL);		// LP power save mode to -3 mA


	// ########################################################################
	// #                      Configure PLL0 for HB mode                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL0PMU0	PLL0 PMU CENTRAL				Central	0x41	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU0,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU0_LDO_DIG_SEL_4__SVAL					// DigReg LDO to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_LDO_VCO_SEL_0__SVAL 				// VCO LDO to 1.695V (HP HB)
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_REG_LO_SEL_5__SVAL  				// IQDIV LDO to 1.37V (HP HB)
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_VCOREG_INTREF_SEL_1__SVAL			// internal reference for VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU0_DIAG_DISABLE_1__SVAL);			// disable diagnostic bus

	//--------------------------------------------------------------------------
	// Register PLL0PMU1	PLL0 PMU DIAG CONTROL			Central	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU1,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLL0PUP1	PLL0 POWER UP LOCAL				Central	0x43	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP1,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_FPD_LSBUFF_OP_1__SVAL); 			// force power-down 3G output buffers (HB)

	//--------------------------------------------------------------------------
	// Register PLL0VCO0	PLL0 VCO CONTROL 0				Central	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0VCO0,
			WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V

	//--------------------------------------------------------------------------
	// Register PLL0SDM0	PLL0 SIGMA DELTA 0 				Central	0x46	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0,
			WRX654_RF_CENTRAL_FCSI_PLL0SDM0_SDM_EN_0__SVAL); 					// disable DSM (==Integer Mode)

	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL0MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_REF_SEL_20M_1__SVAL  		// set ref clock to 20MHz
			| WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_REF_SEL_40M_0__SVAL);


	// ########################################################################
	// #                      Configure PLL1 for HB mode                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLL1PMU0	PLL1 PMU CENTRAL				Central	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PMU0,
			WRX654_RF_CENTRAL_FCSI_PLL1PMU0_LDO_DIG_SEL_4__SVAL					// DigReg LDO to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_LDO_VCO_SEL_0__SVAL 				// VCO LDO to 1.695V (HP HB)
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_REG_LO_SEL_5__SVAL  				// IQDIV LDO to 1.37V (HP HB)
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_VCOREG_INTREF_SEL_1__SVAL			// internal reference for VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU0_DIAG_DISABLE_1__SVAL);			// disable diagnostic bus

	//--------------------------------------------------------------------------
	// Register PLL1PMU1	PLL1 PMU DIAG CONTROL			Central	0x52	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PMU1,
			WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL1PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLL1PUP1	PLL1 POWER UP LOCAL				Central	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PUP1,
			WRX654_RF_CENTRAL_FCSI_PLL1PUP1_FPD_LSBUFF_OP_1__SVAL); 			// force power-down 3G output buffers (HB)

	//--------------------------------------------------------------------------
	// Register PLL1VCO0	PLL1 VCO CONTROL 0				Central	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1VCO0,
			WRX654_RF_CENTRAL_FCSI_PLL1VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL1VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V

	//--------------------------------------------------------------------------
	// Register PLL1SDM0	PLL1 SIGMA DELTA 0 				Central	0x56	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1SDM0,
			WRX654_RF_CENTRAL_FCSI_PLL1SDM0_SDM_EN_0__SVAL); 					// disable DSM (==Integer Mode)

	//--------------------------------------------------------------------------
	// Register PLL1MISC	PLL1 MISCELLANEOUS				Central	0x5A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1MISC,
			WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL1MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_REF_SEL_20M_1__SVAL  		// set ref clock to 20MHz
			| WRX654_RF_CENTRAL_FCSI_PLL1MISC_RFPLL_REF_SEL_40M_0__SVAL);


	// ########################################################################
	// #                     Enable VCM Buffer (0.6V/0.5V)                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_VCMBUFFER_1__SVAL 				// enable VCM buffer
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_VCMBUFFER_3__SVAL		// set control voltage output of VCM buffer to 0.5V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_I_H_VCMBUFFER_1__SVAL		// double current in output stage of VCM Opamp
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BIAS_1__SVAL					// keep BBIO biasing enabled
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP_1__SVAL				// keep compensation cap to 4.5pF
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_4__SVAL				// keep Vout to 1.26V
			| WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);				// keep BBIO LDO enabled

	//--------------------------------------------------------------------------
	// Register VCMBIAS		VCM CONTROL ANTENNA BIASING		Antx	0x65	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_VCMBIAS,
			WRX654_RF_ANTENNA_FCSI_VCMBIAS_CTL_VOUT_VCMBUFFER_0V5_3__SVAL		// set 0.5V VCM buffer to 0.5V
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_EN_VCMBUFFER_0V5_1__SVAL 			// enable 0.5V VCM buffer
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_CTL_VOUT_VCMBUFFER_0V6_1__SVAL		// set 0.6V VCM buffer to 0.6V
			| WRX654_RF_ANTENNA_FCSI_VCMBIAS_EN_VCMBUFFER_0V6_1__SVAL);			// enable 0.6V VCM buffer


	// ########################################################################
	// #                           Enable RF 2V5 LDO                          #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC_1__SVAL						// enable loop compensation
			| WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE_1__SVAL);					// enable RF 2V5 LDO


	// ########################################################################
	// #                             Enable RX LDO                            #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RF LDO1V2 RX					Antx	0x46	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT_4__SVAL					// set Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE_1__SVAL);					// enable RX 1V2 LDO


	// ########################################################################
	// #                 Set Vout / Enable TX LDO HB and LB                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	RF LDO1V2 TX LB					Antx	0x45 	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT_6__SVAL					// set Vout to 1.28V
			| WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_0__SVAL);				// disable TX LB 1V2 LDO

	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	RF LDO1V2 TX HB					Antx	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT_8__SVAL					// set Vout to 1.3V
			| WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_1__SVAL);				// enable TX HB 1V2 LDO


	// ########################################################################
	// #                        Enable Mixer BB and PGC                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_MIXER_1__SVAL						// enable Mixer BB core
			| 3 << WRX654_RF_ANTENNA_FCSI_PGC_EN_CH_MIXER__POS					// activate I and Q branches
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW_1__SVAL			// close IQCAL GND switch (always closed in regular operation!)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);					// NEW: reduce PGC compensation cap in regular mode


	// ########################################################################
	// #                   Enable AnaMux, xSSI, Offset DAC                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register SSI			XSSI and ANAMUX					Antx	0x5d	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN_1__SVAL						// select xSSI path as AnaMux input
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX_1__SVAL					// enable AnaMux
			| WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE_1__SVAL 				// TSSI input divider ratio 0.9
			| WRX654_RF_ANTENNA_FCSI_SSI_CTL_SSI_ODAC_1__SVAL					// manually control S2D and ODAC enable
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_SSI_GAIN_1__SVAL				// xSSI S2D always on (needed?)
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_SSI_ODAC_1__SVAL); 				// xSSI ODAC always on (needed?)


	// ########################################################################
	// #                 Enable/Config PGC DC State Machine                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXDCOCTL	RX MIXER DC OFFSET CONTROL		Antx	0x66	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I_1__SVAL						// release reset for I branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TOGGLE_EN_DCO_I_1__SVAL				// enable toggling for I branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q_1__SVAL					// release reset for Q branch
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TOGGLE_EN_DCO_Q_1__SVAL);				// enable toggling for Q branch


	// ########################################################################
	// #          Central LO Configuration (all Antennas to PLL0 HB)          #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LOCENTLB	LO CENTRAL LB					Antx	0x40	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
			WRX654_RF_ANTENNA_FCSI_LOCENTLB_EN_PTRACK_1__SVAL						// enable process tracking
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT_2__SVAL						// set Vout to 1.234V (@@@ needs to be reviewed with Andriy)
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0_0__SVAL				// disable LB FOB for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0_0__SVAL			// disable LB driver for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1_1__SVAL				// enable LB FOB for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1_1__SVAL);			// enable LB driver for PLL1

	//--------------------------------------------------------------------------
	// Register LOCENTHB	LO CENTRAL HB					Antx	0x41	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
			WRX654_RF_ANTENNA_FCSI_LOCENTHB_EN_PTRACK_1__SVAL						// enable process tracking
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT_2__SVAL						// set Vout to 1.234V (@@@ needs to be reviewed with Andriy)
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0_1__SVAL				// enable HB FOB for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0_1__SVAL				// enable HB buffer for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0_1__SVAL				// enable HB V2I for PLL0
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1_0__SVAL				// disable HB FOB for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1_0__SVAL				// disable HB buffer for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1_0__SVAL				// disable HB V2I for PLL1
			| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_0__SVAL);				// disable HB LDO to prevent activation without LO signal



	// ########################################################################
	// #                         Enable Temp Sensor                           #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central	0x6D	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_1__SVAL);


	// ########################################################################
	// # Set DC config, LO freq, BW, OpMode, PGC filter, default BS for TX/RX #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC_0__SVAL		// DC offset configuration #0
			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH_0__SVAL		// select PLL channel 0
			| WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW_3__SVAL		// set to 160MHz BW
			| WRX654_RF_ANTENNA_FCSI_SR0_MODE_0__SVAL		// set to High Performance mode
			| WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND_3__SVAL	// TX band selection 3d
			| WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND_5__SVAL);	// RX band selection 5d


	// ########################################################################
	// #                Set RX Filter, PGC Gain, RX RF Gain                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SR1,
			56 << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS							// set RX Filter Corner Freq to 100 MHz
			| WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN_10__SVAL							// set to 20dB PGC gain
			| WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG_0__SVAL);					// set to RX gain config #0 (requires to set up config #0 properly!)
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		RxGainConfig[iAnt].PgcGainIdx[0] = 10;
		RxGainConfig[iAnt].RfGainIdx[0] = 0;
	}

	// ########################################################################
	// #                   Set Pre-Driver Gain, TPC Gain                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN_0__SVAL							// set Pre-Driver gain to max
			| WRX654_RF_ANTENNA_FCSI_SR2_TPC_2__SVAL);								// set to TPC gain config #2 (+2dB)


	// ========================================================================
	// #                 CONFIGURE HIGHBAND SET OF REGISTERS                  #
	// ========================================================================

	// ########################################################################
	// #      Switch to HB set of registers (but keep current HB/LB mode)     #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_1__SVAL							// keep BB LDO enabled
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_ANTBIAS_CURRENT_MIRROR_1__SVAL		// keep antenna biasing current mirror enabled
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT_4__SVAL						// keep Vout to 1.26V
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL				// select HB set of doubled registers
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_1__SVAL);					// switch to HB


	// ########################################################################
	// #                 Set LNA Mid Gain & RSSI Configuration                #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			10 << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS				// set Mid Gain tuning to nominal value
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable (HB) LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);					// enable (HB) RSSI

	//--------------------------------------------------------------------------
	// Register LNARSSI		RSSI Bias Configuration			Antx	0x5B
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RSSIBIAS,
			WRX654_RF_ANTENNA_FCSI_RSSIBIAS_ENABLE_RSSI_BIAS_1__SVAL				// enable RSSI Bias
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_BIAS_SELECT_1__SVAL				// select P2PR bias current
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_TEMP_COMP_1__SVAL);				// enable temperature compensation

	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_FECTL01,
			WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0_2__SVAL);						// set RX gain config #0 to Mid LNA gain
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		RxGainConfig[iAnt].LnaGainCfg[0] = 2;
		RxGainConfig[iAnt].FectlCfg[0] = 0;
		RxGainConfig[iAnt].LnaGainCfg[1] = 0;			// config 1 will be also reset
		RxGainConfig[iAnt].FectlCfg[1] = 0;				// config 1 will be also reset
	}


	// ########################################################################
	// #                     RX Mixer RF Core Configuration                   #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL						// enable I and Q branches
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL); 				// enable (HB) RX Mixer


	// ########################################################################
	// #                       TPC Filter Default Tuning                      #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCFILTER0	TPC Filter Tuning 0		 		Antx	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCFILTER0,
			0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__POS		// set TPC FB Cap for 20MHz to 35d acc. Jan's proposal
			| 0x23 << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__POS);	// set TPC FB Cap for 40MHz to 35d acc. Jan's proposal

	//--------------------------------------------------------------------------
	// Register TPCFILTER1	TPC Filter Tuning 1		 		Antx	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCFILTER1,
			0x4f << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__POS		// set TPC FB Cap for 80MHz to 79d acc. Jan's proposal
			| 0x69 << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__POS);	// set TPC FB Cap for 160MHz to 105d acc. Jan's proposal


	// ########################################################################
	// #                  TPC Gain / DC Offset Configuration                  #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCGAIN		TPC GAIN CONTROL		 		Antx	0x55	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCGAIN,
			WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0_0__SVAL					// set TPC Gain Config #0 to -6dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN1_2__SVAL					// set TPC Gain Config #1 to -2dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN2_4__SVAL					// set TPC Gain Config #2 to +2dB
			| WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN3_6__SVAL);				// set TPC Gain Config #3 to +6dB


	// ########################################################################
	// #                    TPC and TX Mixer Configuration                    #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_ENABLE_ACT_MIRROR_1__SVAL				// enable Active Current Mirror for HB PAD
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_ENABLE_VC_BIAS_1__SVAL);				// enable VC biasing of pre-driver

	// ########################################################################
	// #                       Pre-Driver Configuration                       #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_VC_PAD_4__SVAL						// set control cascode voltage to 0.99V
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_PAD_BIAS_7__SVAL					// set pre-driver bias to 90%
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_1__SVAL); 						// enable (HB) pre-driver

    // update PLL configuration
    PllConfig[0].HbLb       = 1;        // PLL0 in HB mode
    PllConfig[0].RefClk     = 20;       // set ref clock to 40MHz
    PllConfig[0].IntFrac    = 1;        // disable DSM (==Integer Mode)
    PllConfig[1].HbLb       = 1;        // PLL1 in HB mode
    PllConfig[1].RefClk     = 20;       // set ref clock to 40MHz
    PllConfig[1].IntFrac    = 1;        // disable DSM (==Integer Mode)
    PllConfig[0].AntSet     = ANTALL;   // all selectes antennas connected to PLL0 HB
}


void rfic_rxon(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SR1			Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RXON__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR1_RXON_1__SVAL);								// set RXON = 1
}


void rfic_rxoff(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SR1			Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RXON__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR1_RXON_0__SVAL);								// set RXON = 0
}


void rfic_txon(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SR2			Short Register 2				Antx	0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_TXON__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR2_TXON_1__SVAL);								// set TXON = 1
}


void rfic_txoff(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SR2			Short Register 2				Antx	0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_TXON__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR2_TXON_0__SVAL);								// set TXON = 0
}


void rfic_enable_fsys2 (void)
// enable FSYS2 clock output
{
	//--------------------------------------------------------------------------
	// Register FSYS 		FSYS 1 + 2         				Central	0x64
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYS,
			WRX654_RF_CENTRAL_FCSI_FSYS_DISABLE__ISMSK,
			WRX654_RF_CENTRAL_FCSI_FSYS_DISABLE_0__SVAL);
}


void rfic_disable_fsys2 (void)
// disable FSYS2 clock output
{
	//--------------------------------------------------------------------------
	// Register FSYS		FSYS 1 + 2         				Central	0x64
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYS,
			WRX654_RF_CENTRAL_FCSI_FSYS_DISABLE__ISMSK,
			WRX654_RF_CENTRAL_FCSI_FSYS_DISABLE_1__SVAL);
}

void rfic_trim_dcxo (uint8_t CapTrimX1,uint8_t CapTrimX2)
// trim DCXO
{
	//--------------------------------------------------------------------------
	// Register DCXO0		DCXO0 Trimming		Ant1	0x40
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_DCXO0,
			WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X1__ISMSK
			& WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X2__ISMSK,
			(CapTrimX1 & WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X1__MSK) << WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X1__POS
			| (CapTrimX2 & WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X2__MSK) << WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X2__POS);

}



void rfic_antenna_off (uint8_t AntMsk, uint16_t Twait_us)
// switch off/on antennas (sequentially and with optional waits in between)
{
	uint16_t	data[5];
	uint8_t		AntMskOff, AntSel, ant, Wait_needed;
	//--------------------------------------------------------------------------
	// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
	//--------------------------------------------------------------------------
	RFIC_FCSI_Read(ANTALL, WRX654_RF_ANTENNA_FCSI_LOCENTLB, data);

	// currently switched-off antennas (as a 4-bit mask)
	AntMskOff = (data[0] & WRX654_RF_ANTENNA_FCSI_LOCENTLB_ANT_OFF__SMSK) +
			((data[1] & WRX654_RF_ANTENNA_FCSI_LOCENTLB_ANT_OFF__SMSK) << 1) +
			((data[2] & WRX654_RF_ANTENNA_FCSI_LOCENTLB_ANT_OFF__SMSK) << 2) +
			((data[3] & WRX654_RF_ANTENNA_FCSI_LOCENTLB_ANT_OFF__SMSK) << 3);

	Wait_needed = 0;
	for (ant=0; ant < 4; ant++)
	{
		AntSel = 1 << ant;
		if ((AntMsk & AntSel) != (AntMskOff & AntSel))
		{
			if (Wait_needed)
				RFIC_Wait(Twait_us); // only add a wait if needed
			AntMskOff = ((AntMskOff & (ANTALL-AntSel)) + (AntMsk & AntSel));
			RFIC_FCSI_RMW(AntSel, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
					WRX654_RF_ANTENNA_FCSI_LOCENTLB_ANT_OFF__ISMSK,
					((AntMsk & AntSel) > 0) << WRX654_RF_ANTENNA_FCSI_LOCENTLB_ANT_OFF__POS);
			Wait_needed = 1;
		}
	}
}

void rfic_set_softreset(void)
// activate soft-reset of the RFIC
{
	//--------------------------------------------------------------------------
	// Register CONFIG1	CONFIG1								Central	0x6a
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CONFIG1,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_SOFT_RESET__ISMSK,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_SOFT_RESET_1__SVAL);				// set soft-reset
}

void rfic_release_softreset(void)
// release soft-reset of the RFIC
{
	//--------------------------------------------------------------------------
	// Register CONFIG1	CONFIG1								Central	0x6a
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CONFIG1,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_SOFT_RESET__ISMSK,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_SOFT_RESET_0__SVAL);				// release soft-reset
}



void rfic_activate_lb(uint8_t AntMsk)
{
	// Register operations:
	// - select HB set of doubled registers
	// - disable all HB versions of following blocks: LNA, RSSI, RXMIXER, PAD
	// - select LB set of doubled registers and switch to LB
	// - enable all LB versions of following blocks: LNA, RSSI, RXMIXER, PAD
	// - disable TX HB LDO, enable TX LB LDO

	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL);				// select HB set of doubled registers

	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_0__SVAL						// disable HB LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_0__SVAL);					// disable HB RSSI

	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_0__SVAL); 				// disable HB RX Mixer

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_0__SVAL); 						// disable HB Pre-Driver

	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL					// select LB set of doubled registers
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_0__SVAL);					// switch to LB

	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable LB LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);					// enable LB RSSI

	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL); 				// enable LB RX Mixer

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_1__SVAL); 						// enable LB Pre-Driver

	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	RF LDO1V2 TX HB					Antx	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_0__SVAL);					// disable TX HB 1V2 LDO

	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	RF LDO1V2 TX LB					Antx	0x45 	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_1__SVAL);					// enable TX LB 1V2 LDO

}


void rfic_activate_hb(uint8_t AntMsk)
{
	// Register operations:
	// - select LB set of doubled registers
	// - disable all LB versions of following blocks: LNA, RSSI, RXMIXER, PAD
	// - select HB set of doubled registers and switch to HB
	// - enable all HB versions of following blocks: LNA, RSSI, RXMIXER, PAD
	// - disable TX LB LDO, enable TX HB LDO

	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL);				// select LB set of doubled registers

	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_0__SVAL						// disable LB LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_0__SVAL);					// disable LB RSSI

	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_0__SVAL); 				// disable LB RX Mixer

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_0__SVAL); 						// disable LB Pre-Driver

	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL					// select HB set of doubled registers
			| WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_1__SVAL);					// switch to HB

	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable HB LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);					// enable HB RSSI

	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL); 				// enable HB RX Mixer

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_EN_PAD_1__SVAL); 						// enable HB Pre-Driver

	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	RF LDO1V2 TX LB					Antx	0x45 	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_0__SVAL);					// disable TX LB 1V2 LDO

	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	RF LDO1V2 TX HB					Antx	0x44	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_1__SVAL);					// enable TX HB 1V2 LDO
}

void rfic_set_hblb(uint8_t AntMsk, uint8_t HbLb)
// set HBLB flag exclusively without touching any other register, HB/LB doubled register block is NOT changed!
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB__ISMSK,
			(HbLb & WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB__MSK) << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB__POS);				// switch to HB/LB

}

void rfic_enable_pll(uint8_t Pll)
{
	if (Pll == 0)
		//--------------------------------------------------------------------------
		// Register PLL0PUP0	PLL0 POWER UP CENTRAL			Central	0x40	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP0,
				WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_VCO_1__SVAL					// power up VCO LDO
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_DIG_1__SVAL				// power up 1.2V Dig LDOs
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_XO_1__SVAL				// power up clock path in reference block
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FB_1__SVAL				// power up LDO for FB path - DIVN/FCAL
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FF_1__SVAL 				// power up LDO for FF path - PCCP, LPF
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_CPLP_FBIAS_1__SVAL			// power up PDCPLPF Bias circuitry
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_OPAMP_LPATH_1__SVAL 			// power up OpAmps in Int path of LPF block
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_LPATH_1__SVAL			// enable Int path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_PPATH_1__SVAL			// enable direct path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_REG_RESYNC_1__SVAL); 			// power up DIVN re-sync block regulator
	else
		//--------------------------------------------------------------------------
		// Register PLL0PUP1	PLL1 POWER UP CENTRAL			Central	0x50	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PUP0,
				WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_VCO_1__SVAL					// power up VCO LDO
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_DIG_1__SVAL				// power up 1.2V Dig LDOs
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_XO_1__SVAL				// power up clock path in reference block
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_FB_1__SVAL				// power up LDO for FB path - DIVN/FCAL
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_FF_1__SVAL 				// power up LDO for FF path - PCCP, LPF
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_CPLP_FBIAS_1__SVAL			// power up PDCPLPF Bias circuitry
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_OPAMP_LPATH_1__SVAL 			// power up OpAmps in Int path of LPF block
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_VVBGEN_LPATH_1__SVAL			// enable Int path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_VVBGEN_PPATH_1__SVAL			// enable direct path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_REG_RESYNC_1__SVAL); 			// power up DIVN re-sync block regulator

    // update PLL configuration
    PllConfig[Pll].Active = 1;
}

void rfic_disable_pll(uint8_t Pll)
{
	if (Pll == 0)
		//--------------------------------------------------------------------------
		// Register PLL0PUP0	PLL0 POWER UP CENTRAL			Central	0x40	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP0,
				WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_VCO_0__SVAL					// power down VCO LDO
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_DIG_0__SVAL				// power down 1.2V Dig LDOs
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_XO_0__SVAL				// power down clock path in reference block
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FB_0__SVAL				// power down LDO for FB path - DIVN/FCAL
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FF_0__SVAL 				// power down LDO for FF path - PCCP, LPF
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_CPLP_FBIAS_0__SVAL			// power down PDCPLPF Bias circuitry
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_OPAMP_LPATH_0__SVAL 			// power down OpAmps in Int path of LPF block
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_LPATH_0__SVAL			// disable Int path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_PPATH_0__SVAL			// disable direct path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_REG_RESYNC_0__SVAL); 			// power down DIVN re-sync block regulator
	else
		//--------------------------------------------------------------------------
		// Register PLL1PUP0	PLL1 POWER UP CENTRAL			Central	0x50	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL1PUP0,
				WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_VCO_0__SVAL					// power down VCO LDO
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_DIG_0__SVAL				// power down 1.2V Dig LDOs
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_XO_0__SVAL				// power down clock path in reference block
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_FB_0__SVAL				// power down LDO for FB path - DIVN/FCAL
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_LDO_FF_0__SVAL 				// power down LDO for FF path - PCCP, LPF
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_CPLP_FBIAS_0__SVAL			// power down PDCPLPF Bias circuitry
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_OPAMP_LPATH_0__SVAL 			// power down OpAmps in Int path of LPF block
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_VVBGEN_LPATH_0__SVAL			// disable Int path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_VVBGEN_PPATH_0__SVAL			// disable direct path in LPF
				| WRX654_RF_CENTRAL_FCSI_PLL1PUP0_PUP_REG_RESYNC_0__SVAL); 			// power down DIVN re-sync block regulator

    // update PLL configuration
    PllConfig[Pll].Active = 0;
}

void rfic_set_pllreset(uint8_t Pll)
{
	//--------------------------------------------------------------------------
	// Register PLLxPUP1	PLLx POWER UP LOCAL				Central	0x43/0x53
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP1 + 0x10 * (Pll != 0),
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_RFPLL_RESETN__ISMSK,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_RFPLL_RESETN_1__SVAL);					// force PLL into Reset mode

    // update PLL configuration
    PllConfig[Pll].Reset = 1;
}

void rfic_release_pllreset(uint8_t Pll)
{
	//--------------------------------------------------------------------------
	// Register PLLxPUP1	PLLx POWER UP LOCAL				Central	0x43/0x53
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP1 + 0x10 * (Pll != 0),
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_RFPLL_RESETN__ISMSK,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP1_RFPLL_RESETN_0__SVAL);					// release PLL Reset mode

    // update PLL configuration
    PllConfig[Pll].Reset = 0;
}

void rfic_set_pll_mode(uint8_t Pll, uint8_t HbLb, uint8_t RefClk, uint8_t HiPwr)
{
	// Operational steps:
	// - check for selected PLL, HB/LB, reference clock, power mode and set some internal variables
	// - bring PLL into Reset state
	// - re-program all PLL registers of selected
	// - keep PLL in Reset mode (assuming that a re-programming of PLL freq(s) will follow)

	uint16_t	AddrOffs, RegPMU0;
	uint8_t		Sel20M, Sel40M;

	// ########################################################################
	// #                             Check Inputs                             #
	// ########################################################################
	if (Pll != 0)
		AddrOffs = 0x10;
	else
		AddrOffs = 0;

	if (HbLb == 0)
		RegPMU0 = 4235;		// 0b0001000010001011 - PMU1 content for 2G Mode
	else
	{
		if (HiPwr == 0)
			RegPMU0 = 4247; 	// 0b0001000010010111 - PMU1 content for 5G Reduced Power Mode
		else
			RegPMU0 = 151; 		// 0b0000000010010111 - PMU1 content for 5G High Power Mode
	}

	if (RefClk == 40)
	{
		// set RefClk to 40MHz
		Sel20M = 0;
		Sel40M = 1;
	}
	else if (RefClk == 20)
	{
		// set RefClk to 20MHz
		Sel20M = 1;
		Sel40M = 0;
	}
	else
	{
		// default: set RefClk to 10MHz
		Sel20M = 0;
		Sel40M = 0;
	}

	// ########################################################################
	// #                               Reset PLL                              #
	// ########################################################################
	//--------------------------------------------------------------------------
	// Register PLLxPUP1	PLLx POWER UP LOCAL				Central	0x43/0x53
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP1 + AddrOffs,
			(HbLb == 0) << WRX654_RF_CENTRAL_FCSI_PLL0PUP1_FPD_LSBUFF_FB__POS 			// power-down 6G output buffers
			| (HbLb != 0) << WRX654_RF_CENTRAL_FCSI_PLL0PUP1_FPD_LSBUFF_OP__POS 		// power-down 3G output buffers
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP1_RFPLL_RESETN_1__SVAL);					// force PLL into Reset mode

	// ########################################################################
	// #                       Write other PLL Registers                      #
	// ########################################################################
	/*	//--------------------------------------------------------------------------
	// Register PLLxPUP0	PLLx POWER UP CENTRAL			Central	0x40/0x50
	//--------------------------------------------------------------------------
	// (@@@ SIMILAR TO INIT - REALLY NEEDED?)
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PUP0 + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_VCO_1__SVAL					// power up VCO LDO
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_DIG_1__SVAL				// power up 1.2V Dig LDOs
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_XO_1__SVAL				// power up clock path in reference block
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FB_1__SVAL				// power up LDO for FB path - DIVN/FCAL
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_LDO_FF_1__SVAL 				// power up LDO for FF path - PCCP, LPF
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_CPLP_FBIAS_1__SVAL			// power up PDCPLPF Bias circuitry
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_OPAMP_LPATH_1__SVAL 			// power up OpAmps in Int path of LPF block
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_LPATH_1__SVAL			// enable Int path in LPF
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_VVBGEN_PPATH_1__SVAL			// enable direct path in LPF
			| WRX654_RF_CENTRAL_FCSI_PLL0PUP0_PUP_REG_RESYNC_1__SVAL); 			// power up DIVN re-sync block regulator
	 */
	//--------------------------------------------------------------------------
	// Register PLLxPMU0	PLLx PMU CENTRAL				Central	0x41/0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU0 + AddrOffs,
			RegPMU0);															// write pre-defined content according mode
	/*
	//--------------------------------------------------------------------------
	// Register PLLxPMU1	PLLx PMU DIAG CONTROL			Central	0x42/0x52
	//--------------------------------------------------------------------------
	// (@@@ SIMILAR TO INIT - REALLY NEEDED?)
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0PMU1 + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_FF_SEL_4__SVAL					// LPF FF LDO to 1.704V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_RESYNC_SEL_4__SVAL			// DIVN/FCAL/RESYNC voltage to 1.293V
			| WRX654_RF_CENTRAL_FCSI_PLL0PMU1_LDO_XO_SEL_4__SVAL);				// XO LDO to 1.293V

	//--------------------------------------------------------------------------
	// Register PLLxVCO0	PLLx VCO CONTROL 0				Central	0x44/0x54
	//--------------------------------------------------------------------------
	// (@@@ SIMILAR TO INIT - REALLY NEEDED?)
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0VCO0 + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_OFF_REF_1__SVAL 			// ref voltage for coarse cal OFF cap to 1.0V
			| WRX654_RF_CENTRAL_FCSI_PLL0VCO0_BUMP_DCAP_ON_REF_1__SVAL);		// ref voltage for coarse cal ON cap to 1.55V
	 */
	//--------------------------------------------------------------------------
	// Register PLLxSDM0	PLLx SIGMA DELTA 0 				Central	0x46/0x56
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0 + AddrOffs,
			(HbLb == 0) << WRX654_RF_CENTRAL_FCSI_PLL0SDM0_SDM_EN__POS); 		// enable Frac-N for 2G, disable for 5G

	//--------------------------------------------------------------------------
	// Register PLLxMISC	PLLx MISCELLANEOUS				Central	0x4A/0x5A
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL	// enable fast channel hopping
			| TESTER << WRX654_RF_CENTRAL_FCSI_PLL0MISC_SEL_CLK__POS			// select internal clock for Lab(0), or external for Tester(1)
			| Sel40M << WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_REF_SEL_40M__POS
			| Sel20M << WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_REF_SEL_20M__POS);  // set ref clock to 10/20/40MHz

    // update PLL configuration
	if (PllConfig[Pll].HbLb != HbLb)
		PllConfig[Pll].AntSet = 0;		// if band changes from LB to HB or vice versa already connected antennas are considered disconnected
    PllConfig[Pll].HbLb         = HbLb;
    PllConfig[Pll].RefClk       = RefClk;
    PllConfig[Pll].IntFrac      = HbLb;
    PllConfig[Pll].HiPwr        = HiPwr;
}

void rfic_set_pll_5G_10M_ax(uint8_t Pll)
// short-cut to 5G/10MHz/high-power mode
{
	rfic_set_pll_mode(Pll, 1, 10, 1);
}

void rfic_set_pll_5G_20M_ax(uint8_t Pll)
// short-cut to 5G/20MHz/high-power mode
{
	rfic_set_pll_mode(Pll, 1, 20, 1);
}

void rfic_set_pll_5G_40M_ax(uint8_t Pll)
// short-cut to 5G/40MHz/high-power mode
{
	rfic_set_pll_mode(Pll, 1, 40, 1);
}

void rfic_set_pll_5G_10M_rp(uint8_t Pll)
// short-cut to 5G/10MHz/reduced-power mode
{
	rfic_set_pll_mode(Pll, 1, 10, 0);
}

void rfic_set_pll_5G_20M_rp(uint8_t Pll)
// short-cut to 5G/20MHz/reduced-power mode
{
	rfic_set_pll_mode(Pll, 1, 20, 0);
}

void rfic_set_pll_5G_40M_rp(uint8_t Pll)
// short-cut to 5G/40MHz/reduced-power mode
{
	rfic_set_pll_mode(Pll, 1, 40, 0);
}

void rfic_set_pll_2G(uint8_t Pll)
// short-cut to 2G/40MHz/Frac-N mode
{
	rfic_set_pll_mode(Pll, 0, 40, 0);
}


void rfic_set_pll_intnmode(uint8_t Pll)
{
	//--------------------------------------------------------------------------
	// Register PLLxSDM0	PLLx SIGMA DELTA 0 				Central	0x46/0x56
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0 + 0x10 * (Pll != 0),
			WRX654_RF_CENTRAL_FCSI_PLL0SDM0_SDM_EN_0__SVAL);					// disable DSM (== Integer-N mode)

    // update PLL configuration
    PllConfig[Pll].IntFrac      = 1;
}

void rfic_set_pll_fracnmode(uint8_t Pll)
{
	//--------------------------------------------------------------------------
	// Register PLLxSDM0	PLLx SIGMA DELTA 0 				Central	0x46/0x56
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0 + 0x10 * (Pll != 0),
			WRX654_RF_CENTRAL_FCSI_PLL0SDM0_SDM_EN_1__SVAL);					// enable DSM (== Fractional-N mode)

    // update PLL configuration
    PllConfig[Pll].IntFrac      = 0;
}

void rfic_select_pll(uint8_t AntMsk, uint8_t Pll, uint8_t HbLb)
// switch selected Antenna(s) to selected PLL in selected mode
{
	uint8_t Pll0 = (uint8_t) (Pll==0);
	uint8_t Pll1 = (uint8_t) (Pll!=0);

	if (HbLb == 0) // switching from HB to LB
	{
		// first switch off HB LDO
		//--------------------------------------------------------------------------
		// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_0__SVAL);							// switch off HB LDO

		// then switch off all HB buffers
		//--------------------------------------------------------------------------
		// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0_0__SVAL						// switch off HB FOB for PLL0
				| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0_0__SVAL						// switch off HB buffer for PLL0
				| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0_0__SVAL						// switch off HB V2I for PLL0
				| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1_0__SVAL						// switch off HB FOB for PLL1
				| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1_0__SVAL						// switch off HB buffer for PLL1
				| WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1_0__SVAL);					// switch off HB V2I for PLL1

		// then switch on LB buffers of PLL0 or PLL1
		//--------------------------------------------------------------------------
		// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1__ISMSK,
				Pll0 << WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0__POS					// LB FOB for PLL0
				| Pll0 << WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0__POS				// LB driver for PLL0
				| Pll1 << WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1__POS				// LB FOB for PLL1
				| Pll1 << WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1__POS);				// LB driver for PLL1

		// then switch on LB LDO
		//--------------------------------------------------------------------------
		// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_1__SVAL);							// activate LB LDO
	}
	else		// switching from LB to HB
	{
		// first switch off LB LDO
		//--------------------------------------------------------------------------
		// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_0__SVAL);							// switch off LB LDO

		// then switch off all LB buffers
		//--------------------------------------------------------------------------
		// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL0_0__SVAL						// switch off LB FOB for PLL0
				| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL0_0__SVAL					// switch off LB driver for PLL0
				| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBSW1_PLL1_0__SVAL						// switch off LB FOB for PLL1
				| WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBDRV1_PLL1_0__SVAL);					// switch off LB driver for PLL1

		// then switch on HB buffers of PLL0 or PLL1
		//--------------------------------------------------------------------------
		// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1__ISMSK,
				Pll0 << WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL0__POS					// HB FOB for PLL0
				| Pll0 << WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL0__POS				// HB buffer for PLL0
				| Pll0 << WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL0__POS				// HB V2I for PLL0
				| Pll1 << WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW1_PLL1__POS				// HB FOB for PLL1
				| Pll1 << WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBSW2_PLL1__POS				// HB buffer for PLL1
				| Pll1 << WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBV2I_PLL1__POS);				// HB V2I for PLL1

		// then switch on HB LDO
		//--------------------------------------------------------------------------
		// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_1__SVAL);							// activate HB LDO
	}

    // update PLL configuration
    if (PllConfig[Pll].HbLb != HbLb)
    {
        // PLL Pll not in same HbLb mode. This would be an error. How to report?
    }
    PllConfig[Pll].AntSet = (PllConfig[Pll].AntSet | AntMsk); // connect additional antennas to PLL Pll
    PllConfig[Pll==0].AntSet = (PllConfig[Pll==0].AntSet & (15-AntMsk)); // eventually remove selected  antennas from other PLL

}

void rfic_select_fasthop_master(uint8_t Pll, uint8_t Ant)
// select master antenna for fast-hop control for selected PLL
{
	if (Pll != 0)
		RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_SR0,
				WRX654_RF_CENTRAL_FCSI_SR0_SEL_ANT_PLL1__ISMSK,
				Ant <<  WRX654_RF_CENTRAL_FCSI_SR0_SEL_ANT_PLL1__POS);	// set Master antenna for PLL1 in Central SR0
	else
		RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_SR0,
				WRX654_RF_CENTRAL_FCSI_SR0_SEL_ANT_PLL0__ISMSK,
				Ant <<  WRX654_RF_CENTRAL_FCSI_SR0_SEL_ANT_PLL0__POS);	// set Master antenna for PLL0 in Central SR0

    // update PLL configuration
    if ((PllConfig[Pll].AntSet & 1<<Ant) == 0)
    {
        // Antenna Ant is not connected to PLL Pll. This would be an error. How to report?
    }
    PllConfig[Pll].FhMaster = Ant; // set Ant as Fasthop Master
}

void rfic_select_fasthop_channel(uint8_t MasterAnt, uint8_t Channel)
// select fast-hop channel via master antenna
{
	uint8_t AntMsk[4] = {ANT0, ANT1, ANT2, ANT3};			// array with single antenna masks

	RFIC_FCSI_RMW(AntMsk[MasterAnt], WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH__ISMSK,
			Channel <<  WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH__POS);	// set channel in SR0 of Master antenna

    // update PLL configuration
    if ((PllConfig[0].FastHop) && (PllConfig[0].FhMaster == MasterAnt))
        PllConfig[0].LoFreqAct = PllConfig[0].LoFreqs[Channel];     // change active fasthop frequency
    if ((PllConfig[1].FastHop) && (PllConfig[1].FhMaster == MasterAnt))
        PllConfig[1].LoFreqAct = PllConfig[1].LoFreqs[Channel];     // change active fasthop frequency
}

void rfic_set_lo_frequency_intn(uint8_t Pll, uint16_t LoFreq, uint8_t HbLb, uint8_t RefClk)
// set single LO frequency in Int-N mode
{
	uint16_t	AddrOffs, ClkDiv;

	if (HbLb != 0)
		ClkDiv = 2;						// HB mode
	else
		ClkDiv = 4;						// LB mode

	if (Pll != 0)
		AddrOffs = 0x10;				// select PLL1
	else
		AddrOffs = 0;					// select PLL0

	// disable fast-channel hopping
	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN__ISMSK,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_0__SVAL);	// disable fast channel hopping

	// write frequency preset
	//--------------------------------------------------------------------------
	// Register PLLxSDM2	PLLx SIGMA DELTA 2				Central	0x48/0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM2 + AddrOffs,
			LoFreq*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0SDM2_DIVN_INT_CH1__POS);		// set LoFreq

    // update PLL configuration
    if ((PllConfig[Pll].IntFrac != 1) | (PllConfig[Pll].HbLb != HbLb) | (PllConfig[Pll].RefClk != RefClk))
    {
        // PLL mode, RefClk, and/or HbLb do not match. This would be an error! How to report?
    }
    PllConfig[Pll].FastHop = 0;
    PllConfig[Pll].LoFreqs[0] = LoFreq;
    PllConfig[Pll].LoFreqs[1] = 0;
    PllConfig[Pll].LoFreqs[2] = 0;
    PllConfig[Pll].LoFreqs[3] = 0;
    PllConfig[Pll].LoFreqs[4] = 0;
    PllConfig[Pll].LoFreqs[5] = 0;
    PllConfig[Pll].LoFreqAct = LoFreq;
}

void rfic_set_lo_frequency_fracn(uint8_t Pll, uint16_t LoFreq, uint8_t HbLb)
// set single LO frequency in Frac-N mode (only 40MHz ref clock allowed)
{
	uint16_t	F0_int, X, AddrOffs, FpShift;
	uint32_t	F0_frac, XY;
	int32_t		Y;

	if (Pll != 0)
		AddrOffs = 0x10;
	else
		AddrOffs = 0;

	if (HbLb != 0)
		FpShift = 19;
	else
		FpShift = 20;


	XY = (uint32_t) (LoFreq << FpShift)/5 - (8 << 21);;
	X = (uint16_t) (XY >> 22);
	Y = (int32_t) XY - (int32_t)(X << 22);

	if (Y-((Y >> 21) << 21) >= (1 << 20))
	{
		X++;
		Y -= (2 << 21);
	}

	F0_int = X;
	if (Y < 0)
		F0_frac = (Y + (8 << 21));
	else
		F0_frac = Y;

	// disable fast-channel hopping
	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN__ISMSK,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_0__SVAL);	// disable fast channel hopping

	// write frequency preset
	//--------------------------------------------------------------------------
	// Register PLLxSDM1	PLLx SIGMA DELTA 1				Central	0x47/0x57
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0SDM1 + AddrOffs,
			F0_frac >> 8);															// set fractional part for LoFreq

	//--------------------------------------------------------------------------
	// Register PLLxSDM2	PLLx SIGMA DELTA 2				Central	0x48/0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0SDM2 + AddrOffs,
			F0_int + ((F0_frac & 255) << 8));										// set integer part for LoFreq

    // update PLL configuration
    if ((PllConfig[Pll].IntFrac != 0) | (PllConfig[Pll].HbLb != HbLb) | (PllConfig[Pll].RefClk != 40))
    {
        // PLL mode, RefClk, and/or HbLb do not match. This would be an error! How to report?
    }
    PllConfig[Pll].FastHop = 0;
    PllConfig[Pll].LoFreqs[0] = LoFreq;
    PllConfig[Pll].LoFreqs[1] = 0;
    PllConfig[Pll].LoFreqs[2] = 0;
    PllConfig[Pll].LoFreqs[3] = 0;
    PllConfig[Pll].LoFreqs[4] = 0;
    PllConfig[Pll].LoFreqs[5] = 0;
    PllConfig[Pll].LoFreqAct = LoFreq;
}

void rfic_set_lo_frequencies_intn(uint8_t Pll, uint16_t LoFreq0, uint16_t LoFreq1, uint16_t LoFreq2, uint16_t LoFreq3, uint16_t LoFreq4, uint16_t LoFreq5, uint8_t HbLb, uint8_t RefClk)
// set 6 fast-hopping frequencies in Int-N mode
{
	uint16_t	AddrOffs, ClkDiv;

	if (HbLb != 0)
		ClkDiv = 2;						// HB mode
	else
		ClkDiv = 4;						// LB mode

	if (Pll != 0)
		AddrOffs = 0x10;					// select PLL1
	else
		AddrOffs = 0;					// select PLL0

	// enable fast-channel hopping
	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN__ISMSK,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL);	// enable fast channel hopping

	// write frequency presets
	//--------------------------------------------------------------------------
	// Register PLLxSDM2	PLLx SIGMA DELTA 2				Central	0x48/0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM2 + AddrOffs,
			LoFreq0*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0SDM2_DIVN_INT_CH1__POS);		// set LoFreq0

	//--------------------------------------------------------------------------
	// Register PLLxCH2		PLLx FAST CHANNEL HOP 2			Central	0x4b/0x5b
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0CH2 + AddrOffs,
			LoFreq1*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0CH2_DIV_INT_CH2__POS);			// set LoFreq1

	//--------------------------------------------------------------------------
	// Register PLLxCH3		PLLx FAST CHANNEL HOP 3			Central	0x4c/0x5c
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0CH3 + AddrOffs,
			LoFreq2*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0CH3_DIV_INT_CH3__POS);			// set LoFreq2

	//--------------------------------------------------------------------------
	// Register PLLxCH4		PLLx FAST CHANNEL HOP 4			Central	0x4d/0x5d
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0CH4 + AddrOffs,
			LoFreq3*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0CH4_DIV_INT_CH4__POS);			// set LoFreq3

	//--------------------------------------------------------------------------
	// Register PLLxCH5		PLLx FAST CHANNEL HOP 5			Central	0x4e/0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0CH5 + AddrOffs,
			LoFreq4*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0CH5_DIV_INT_CH5__POS);			// set LoFreq4

	//--------------------------------------------------------------------------
	// Register PLLxCH6		PLLx FAST CHANNEL HOP 6			Central	0x4f/0x5f
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0CH6 + AddrOffs,
			LoFreq5*ClkDiv/RefClk << WRX654_RF_CENTRAL_FCSI_PLL0CH6_DIV_INT_CH6__POS);			// set LoFreq5

    // update PLL configuration
    if ((PllConfig[Pll].IntFrac != 1) | (PllConfig[Pll].HbLb != HbLb) | (PllConfig[Pll].RefClk != RefClk))
    {
        // PLL mode, RefClk, and/or HbLb do not match. This would be an error! How to report?
    }
    PllConfig[Pll].FastHop = 1;
    PllConfig[Pll].LoFreqs[0] = LoFreq0;
    PllConfig[Pll].LoFreqs[1] = LoFreq1;
    PllConfig[Pll].LoFreqs[2] = LoFreq2;
    PllConfig[Pll].LoFreqs[3] = LoFreq3;
    PllConfig[Pll].LoFreqs[4] = LoFreq4;
    PllConfig[Pll].LoFreqs[5] = LoFreq5;
    PllConfig[Pll].LoFreqAct = LoFreq0; // not sure if this is correct --> read-out SR0 or add further Index field to PllConfig
}

void rfic_set_lo_frequencies_fracn(uint8_t Pll, uint16_t LoFreq0, uint16_t LoFreq1, uint16_t LoFreq2, uint16_t LoFreq3, uint16_t LoFreq4, uint16_t LoFreq5, uint8_t HbLb)
// set 6 fast-hopping frequencies in Frac-N mode (only 40MHz ref clock allowed, max freq in 5G mode 5180MHz!)
{
	uint16_t	k, F0_int, X[6], F_lo_MHz[6], Ftmp, Fp, F_preset[5], AddrOffs, FpShift;
	uint32_t	F0_frac, XY;
	int16_t		Yint;
	int32_t		Y[6], Yfrac;


	if (Pll != 0)
		AddrOffs = 0x10;
	else
		AddrOffs = 0;

	if (HbLb != 0)
		FpShift = 19;
	else
		FpShift = 20;


	// copy input freqs to internal array
	F_lo_MHz[0] = LoFreq0;
	F_lo_MHz[1] = LoFreq1;
	F_lo_MHz[2] = LoFreq2;
	F_lo_MHz[3] = LoFreq3;
	F_lo_MHz[4] = LoFreq4;
	F_lo_MHz[5] = LoFreq5;

	for (k=0;k<6;k++)
	{
		if (F_lo_MHz[k] > 0)
		{
			XY = (uint32_t) (F_lo_MHz[k] << FpShift)/5 - (8 << 21);
			X[k] = (uint16_t) (XY >> 22);
			Y[k] = (int32_t) XY - (int32_t)(X[k] << 22);

			if (Y[k]-((Y[k] >> 21) << 21) >= (1 << 20))
			{
				X[k]++;
				Y[k] -= (2 << 21);
			}
		}
		else
		{
			X[k] = 0; Y[k] = 0.0;
		}
	}

	F0_int = X[0];
	if (Y[0] < 0)
		F0_frac = (Y[0] + (8 << 21));
	else
		F0_frac = Y[0];

	for (k=1;k<6;k++)
	{
		Yint = (int16_t) (Y[k] >> 21);
		Yfrac = Y[k] - (Yint << 21);
		Fp = (X[k]-(16+32+64)) << 7;
		if (Yint < 0)
			Ftmp = (8 + Yint) << 4;
		else
			Ftmp = Yint << 4;
		Fp += Ftmp;
		if (Yfrac >= (1 << 20))
		{
			Fp += 8;
			Fp += (uint16_t) ((10*(Yfrac - (1 << 20)) + (1 << 20)) >> 21); // add 0.5 for correct rounding
		}
		else
		{
			Fp += (uint16_t) ((10*Yfrac + (1 << 20)) >> 21); // add 0.5 for correct rounding
		}
		F_preset[k-1] = Fp;
	}

	// enable fast-channel hopping
	//--------------------------------------------------------------------------
	// Register PLL0MISC	PLL0 MISCELLANEOUS				Central	0x4A	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0MISC + AddrOffs,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN__ISMSK,
			WRX654_RF_CENTRAL_FCSI_PLL0MISC_RFPLL_FAST_CHANNEL_HOP_EN_1__SVAL);	// enable fast channel hopping

	// write frequency presets
	//--------------------------------------------------------------------------
	// Register PLLxSDM1	PLLx SIGMA DELTA 1				Central	0x47/0x57
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0SDM1 + AddrOffs,
			F0_frac >> 8);															// set fractional part for LoFreq0

	//--------------------------------------------------------------------------
	// Register PLLxSDM2	PLLx SIGMA DELTA 2				Central	0x48/0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0SDM2 + AddrOffs,
			F0_int + ((F0_frac & 255) << 8));										// set integer part for LoFreq0

	//--------------------------------------------------------------------------
	// Register PLLxCH2		PLLx FAST CHANNEL HOP 2			Central	0x4b/0x5b
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0CH2 + AddrOffs,
			F_preset[0] << WRX654_RF_CENTRAL_FCSI_PLL0CH2_DIV_INT_CH2__POS);		// set LoFreq1

	//--------------------------------------------------------------------------
	// Register PLLxCH3		PLLx FAST CHANNEL HOP 3			Central	0x4c/0x5c
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0CH3 + AddrOffs,
			F_preset[1] << WRX654_RF_CENTRAL_FCSI_PLL0CH3_DIV_INT_CH3__POS);		// set LoFreq2

	//--------------------------------------------------------------------------
	// Register PLLxCH4		PLLx FAST CHANNEL HOP 4			Central	0x4d/0x5d
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0CH4 + AddrOffs,
			F_preset[2] << WRX654_RF_CENTRAL_FCSI_PLL0CH4_DIV_INT_CH4__POS);		// set LoFreq3

	//--------------------------------------------------------------------------
	// Register PLLxCH5		PLLx FAST CHANNEL HOP 5			Central	0x4e/0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0CH5 + AddrOffs,
			F_preset[3] << WRX654_RF_CENTRAL_FCSI_PLL0CH5_DIV_INT_CH5__POS);		// set LoFreq4

	//--------------------------------------------------------------------------
	// Register PLLxCH6		PLLx FAST CHANNEL HOP 6			Central	0x4f/0x5f
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL,WRX654_RF_CENTRAL_FCSI_PLL0CH6 + AddrOffs,
			F_preset[4] << WRX654_RF_CENTRAL_FCSI_PLL0CH6_DIV_INT_CH6__POS);		// set LoFreq5

    // update PLL configuration
    if ((PllConfig[Pll].IntFrac != 0) | (PllConfig[Pll].HbLb != HbLb) | (PllConfig[Pll].RefClk != 40))
    {
        // PLL mode, RefClk, and/or HbLb do not match. This would be an error! How to report?
    }
    PllConfig[Pll].FastHop = 1;
    PllConfig[Pll].LoFreqs[0] = LoFreq0;
    PllConfig[Pll].LoFreqs[1] = LoFreq1;
    PllConfig[Pll].LoFreqs[2] = LoFreq2;
    PllConfig[Pll].LoFreqs[3] = LoFreq3;
    PllConfig[Pll].LoFreqs[4] = LoFreq4;
    PllConfig[Pll].LoFreqs[5] = LoFreq5;
    PllConfig[Pll].LoFreqAct = LoFreq0; // not sure if this is correct --> read-out SR0 or add further Index field to PllConfig
}

uint8_t rfic_check_pll_lock(uint8_t Pll)
// retrieve PLL lock information
{

    uint8_t     PllLocked, AddrOffs;
	uint16_t    Data[5];

    if (Pll != 0)
        AddrOffs = 0x10;                // select PLL1
    else
        AddrOffs = 0;					// select PLL0

	//--------------------------------------------------------------------------
    // Register PLLxSDM0	PLLx SIGMA DELTA 0				Central	0x46/0x56
    //--------------------------------------------------------------------------
	RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_PLL0SDM0 + AddrOffs, Data);

    PllLocked = (Data[4] & WRX654_RF_CENTRAL_FCSI_PLL0SDM0_LOCK_O__SMSK) >> WRX654_RF_CENTRAL_FCSI_PLL0SDM0_LOCK_O__POS;

    return PllLocked;

}


void rfic_set_lnagain(uint8_t AntMsk, uint8_t gainidx)
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG_0__SVAL);	// choose RF RX gain configuration 0

	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL01,
			WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0__ISMSK,
			(gainidx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0__POS); // set LNA gain index #0
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].LnaGainCfg[0] = (gainidx & 3);
			RxGainConfig[iAnt].RfGainIdx[0] = 0;
		}
	}
}


void rfic_set_pgcgain(uint8_t AntMsk, int8_t gaindB)
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__ISMSK,
			((gaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__POS); // set PGC gain
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[0] = (gaindB >> 1);
		}
	}
}

void rfic_set_tpcgain(uint8_t AntMsk, int8_t gaindB)
{
	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_TPC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR2_TPC_0__SVAL); // choose TPC gain configuration 0

	//--------------------------------------------------------------------------
	// Register TPCGAIN - TPC GAIN CONTROL					Antx	0x55
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCGAIN,
			WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0__ISMSK,
			(((gaindB+6) >> 1) & WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0__MSK) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0__POS); // set TPC gain #0)
}

void rfic_set_padatt(uint8_t AntMsk, uint8_t attdB)
{
	uint8_t		PadAttIdx;

		// extra handling of 30dB and 36 dB attenuation setting
		if (attdB > 27)
			if (attdB > 32)
				PadAttIdx = 15;				// pick 36dB setting for values > 32 dB
			else
				PadAttIdx = 14;				// pick 30dB setting for values between 27..32 dB
		else
			PadAttIdx = attdB >> 1;

		//--------------------------------------------------------------------------
		// Short Reg	SR2		Short Register 2				Antx	0x02
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
				WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN__ISMSK,
				(PadAttIdx & WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN__POS);	// set PAD
}

void rfic_set_txfectl(uint8_t AntMsk, uint8_t txfectl)
// set TX front-end controls
{
		//--------------------------------------------------------------------------
		// Short Reg	SR2		Short Register 2				Antx	0x02
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
				WRX654_RF_ANTENNA_FCSI_SR2_FECTL0__ISMSK &
				WRX654_RF_ANTENNA_FCSI_SR2_FECTL1__ISMSK &
				WRX654_RF_ANTENNA_FCSI_SR2_FECTL2__ISMSK &
				WRX654_RF_ANTENNA_FCSI_SR2_FECTL3__ISMSK &
				WRX654_RF_ANTENNA_FCSI_SR2_FECTL4__ISMSK &
				WRX654_RF_ANTENNA_FCSI_SR2_FECTL5__ISMSK,
				(txfectl & 63) << WRX654_RF_ANTENNA_FCSI_SR2_FECTL0__POS);	// set TX FECTLs
}

void rfic_config_tpcgain (uint8_t AntMsk, int8_t gain0dB, int8_t gain1dB, int8_t gain2dB, int8_t gain3dB)
// set TPC gains for all 4 TX gain constellations (gain values -6,-4,..,+6 dB)
{
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCGAIN,
			(((gain0dB+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0__POS		// set TPC gain #0
			| (((gain1dB+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN1__POS		// set TPC gain #1
			| (((gain2dB+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN2__POS		// set TPC gain #2
			| (((gain3dB+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN3__POS);	// set TPC gain #3
}


void rfic_config_tpcfiltidx (uint8_t AntMsk, int8_t tpcfiltidx160, int8_t tpcfiltidx80, int8_t tpcfiltidx40, int8_t tpcfiltidx20)
// set TPC filter indices for all 4 bandwidth modes
{
	//--------------------------------------------------------------------------
	// Register TPCFILTER0	TPC Filter Tuning 0		 		Antx	0x53	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCFILTER0,
			(tpcfiltidx20 & WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__MSK) << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__POS		// set TPC FB Cap for 20MHz
			| (tpcfiltidx40 & WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__MSK) << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__POS);	// set TPC FB Cap for 40MHz

	//--------------------------------------------------------------------------
	// Register TPCFILTER1	TPC Filter Tuning 1		 		Antx	0x54	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCFILTER1,
			(tpcfiltidx80 & WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__MSK) << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__POS		// set TPC FB Cap for 80MHz
			| (tpcfiltidx160 & WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__MSK) << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__POS);// set TPC FB Cap for 160MHz

}


void rfic_config_lnagain (uint8_t AntMsk, int8_t gain0idx, int8_t gain1idx, int8_t gain2idx, int8_t gain3idx, int8_t gain4idx, int8_t gain5idx, int8_t gain6idx)
// set LNA gain indices for all 7 RX gain constellations (gain index values 0,1,2,3)
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL01,
			WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0__ISMSK
			& WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN1__ISMSK,
			(gain0idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0__POS			// set LNA gain index #0
			| (gain1idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN1__POS);		// set LNA gain index #1

	//--------------------------------------------------------------------------
	// Register FECTL23		FECTL CONFIG RX GAIN23			Antx	0x4a
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL23,
			WRX654_RF_ANTENNA_FCSI_FECTL23_LNA_GAIN2__ISMSK
			& WRX654_RF_ANTENNA_FCSI_FECTL23_LNA_GAIN3__ISMSK,
			(gain2idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL23_LNA_GAIN2__POS			// set LNA gain index #2
			| (gain3idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL23_LNA_GAIN3__POS);		// set LNA gain index #3

	//--------------------------------------------------------------------------
	// Register FECTL45		FECTL CONFIG RX GAIN45			Antx	0x4b
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL45,
			WRX654_RF_ANTENNA_FCSI_FECTL45_LNA_GAIN4__ISMSK
			& WRX654_RF_ANTENNA_FCSI_FECTL45_LNA_GAIN5__ISMSK,
			(gain4idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL45_LNA_GAIN4__POS			// set LNA gain index #4
			| (gain5idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL45_LNA_GAIN5__POS);		// set LNA gain index #5

	//--------------------------------------------------------------------------
	// Register FECTL6		FECTL CONFIG RX GAIN6			Antx	0x4d
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL6,
			WRX654_RF_ANTENNA_FCSI_FECTL6_LNA_GAIN6__ISMSK,
			(gain6idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL6_LNA_GAIN6__POS);		// set LNA gain index #6
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].LnaGainCfg[0] = (gain0idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[1] = (gain1idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[2] = (gain2idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[3] = (gain3idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[4] = (gain4idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[5] = (gain5idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[6] = (gain6idx & 3);
		}
	}

}

void rfic_config_rfrxgain (uint8_t AntMsk, uint8_t fectl0, int8_t gain0idx, uint8_t fectl1, int8_t gain1idx,
		uint8_t fectl2, int8_t gain2idx, uint8_t fectl3, int8_t gain3idx, uint8_t fectl4, int8_t gain4idx,
		uint8_t fectl5, int8_t gain5idx, uint8_t fectl6, int8_t gain6idx)
// set FECTL outputs and LNA gain indices for all 7 RX gain constellations (gain index values 0,1,2,3)
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Register FECTL01		FECTL CONFIG RX GAIN01			Antx	0x49
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL01,
			(gain0idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN0__POS			// set LNA gain index #0
			| (fectl0 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL01_FECTL_GAIN0__POS		// set FECTL output #0
			| (gain1idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL01_LNA_GAIN1__POS		// set LNA gain index #1
			| (fectl1 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL01_FECTL_GAIN1__POS);	// set FECTL output #1

	//--------------------------------------------------------------------------
	// Register FECTL23		FECTL CONFIG RX GAIN23			Antx	0x4a
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL23,
			(gain2idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL23_LNA_GAIN2__POS			// set LNA gain index #2
			| (fectl2 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL23_FECTL_GAIN2__POS		// set FECTL output #2
			| (gain3idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL23_LNA_GAIN3__POS		// set LNA gain index #3
			| (fectl3 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL23_FECTL_GAIN3__POS);	// set FECTL output #3

	//--------------------------------------------------------------------------
	// Register FECTL45		FECTL CONFIG RX GAIN45			Antx	0x4b
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL45,
			(gain4idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL45_LNA_GAIN4__POS			// set LNA gain index #4
			| (fectl4 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL45_FECTL_GAIN4__POS		// set FECTL output #4
			| (gain5idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL45_LNA_GAIN5__POS		// set LNA gain index #5
			| (fectl5 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL45_FECTL_GAIN5__POS);	// set FECTL output #5

	//--------------------------------------------------------------------------
	// Register FECTL6		FECTL CONFIG RX GAIN6			Antx	0x4d
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_FECTL6,
			(gain6idx & 3) << WRX654_RF_ANTENNA_FCSI_FECTL6_LNA_GAIN6__POS			// set LNA gain index #6
			| (fectl6 & 63) << WRX654_RF_ANTENNA_FCSI_FECTL6_FECTL_GAIN6__POS);		// set FECTL output #6
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].LnaGainCfg[0] = (gain0idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[1] = (gain1idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[2] = (gain2idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[3] = (gain3idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[4] = (gain4idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[5] = (gain5idx & 3);
			RxGainConfig[iAnt].LnaGainCfg[6] = (gain6idx & 3);
			RxGainConfig[iAnt].FectlCfg[0] = (fectl0 & 63);
			RxGainConfig[iAnt].FectlCfg[1] = (fectl1 & 63);
			RxGainConfig[iAnt].FectlCfg[2] = (fectl2 & 63);
			RxGainConfig[iAnt].FectlCfg[3] = (fectl3 & 63);
			RxGainConfig[iAnt].FectlCfg[4] = (fectl4 & 63);
			RxGainConfig[iAnt].FectlCfg[5] = (fectl5 & 63);
			RxGainConfig[iAnt].FectlCfg[6] = (fectl6 & 63);
		}
	}

}


void rfic_config_rssi(uint8_t AntMsk, uint8_t cfg_nr, int8_t gaindB, uint8_t coarse_offset_idx, uint8_t fine_offset_idx)
// set RSSI configuration
{
	//--------------------------------------------------------------------------
	// Register RSSI0/1		RSSI0/1 CONFIG					Antx	0x60/0x61
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RSSI0 + (cfg_nr & 1),
			(fine_offset_idx & WRX654_RF_ANTENNA_FCSI_RSSI0_S2D_OFFSET__MSK) << WRX654_RF_ANTENNA_FCSI_RSSI0_S2D_OFFSET__POS
			| (((gaindB+10) >> 1) & WRX654_RF_ANTENNA_FCSI_RSSI0_S2D_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_RSSI0_S2D_GAIN__POS
			| (coarse_offset_idx & WRX654_RF_ANTENNA_FCSI_RSSI0_RSSI_DET_RANGE__MSK) << WRX654_RF_ANTENNA_FCSI_RSSI0_RSSI_DET_RANGE__POS);
}


void rfic_select_rssi_bias(uint8_t AntMsk, uint8_t bias_type)
// set RSSI bias current type for coarse offset (1 = p2pr, 0 = const.)
{
	//--------------------------------------------------------------------------
	// Register LNARSSI		RSSI Bias Configuration			Antx	0x5B
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_RSSIBIAS,
			WRX654_RF_ANTENNA_FCSI_RSSIBIAS_ENABLE_RSSI_BIAS_1__SVAL				// enable RSSI Bias
			| (bias_type & WRX654_RF_ANTENNA_FCSI_RSSI0_S2D_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_BIAS_SELECT__POS				// set bias current
			| WRX654_RF_ANTENNA_FCSI_RSSIBIAS_CTL_TEMP_COMP_1__SVAL);				// enable temperature compensation

}

void rfic_select_rssi(uint8_t AntMsk, uint8_t cfg_nr)
// select RSSI configuration
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__ISMSK,
			(cfg_nr & WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__POS);	// set RSSI configuration #0 or #1
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].RssiRange[0] = (cfg_nr & WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__MSK);
		}
	}
}


void rfic_config_tssi(uint8_t AntMsk, uint8_t cfg_nr, int8_t gaindB, uint8_t fine_offset_idx)
// set TSSI configuration  (cfg_nr=0/1/2/3)
{
	//--------------------------------------------------------------------------
	// Register TSSI0/1/2	TSSI0/1/2 CONFIG				Antx	0x62/0x63/0x64
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TSSI0 + (cfg_nr & 3),
			(fine_offset_idx & WRX654_RF_ANTENNA_FCSI_TSSI0_S2D_OFFSET__MSK) << WRX654_RF_ANTENNA_FCSI_TSSI0_S2D_OFFSET__POS
			| (((gaindB+10) >> 1) & WRX654_RF_ANTENNA_FCSI_TSSI0_S2D_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_TSSI0_S2D_GAIN__POS);

}


void rfic_select_tssi(uint8_t AntMsk, uint8_t cfg_nr)
// select TSSI configuration
{
	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_SEL_S2D_TX__ISMSK,
			(cfg_nr & 3) << WRX654_RF_ANTENNA_FCSI_SR2_SEL_S2D_TX__POS);			// set TSSI configuration #0,1,2,3
}


void rfic_set_tssi_rdiv (uint8_t AntMsk, uint8_t rdiv)
// set TSSI resistive input divider, default: 9|1 (==0.9), fall-back: 6|0 (==0.6)
{
	if ((rdiv == 9) | (rdiv == 1))
	{
		//--------------------------------------------------------------------------
		// Register SSI			XSSI and ANAMUX					Antx	0x5d
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
				WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE__ISMSK,
				WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE_1__SVAL);		// set divider ratio to 0.9
	}
	else
	{
		//--------------------------------------------------------------------------
		// Register SSI			XSSI and ANAMUX					Antx	0x5d
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
				WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE__ISMSK,
				WRX654_RF_ANTENNA_FCSI_SSI_CTL_TSSI_RANGE_0__SVAL);		// set divider ratio to 0.9
	}
}


void rfic_set_rxgain (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB)
// choose RF RX gain constellation, set BB RX gain
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__ISMSK,
			(rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__POS	// choose RF RX gain constellation
			| ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__POS);			// set PGC gain
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[0] = ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK);
			RxGainConfig[iAnt].RfGainIdx[0] = (rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK);
		}
	}
}


void rfic_set_rfrxgainidx (uint8_t AntMsk, uint8_t rfrxgainidx)
// select RF RX gain constellation only
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__ISMSK,
			(rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__POS);	// choose RF RX gain constellation
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].RfGainIdx[0] = (rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK);
		}
	}
}

void rfic_set_pgcbwidth (uint8_t AntMsk, uint8_t rfbandwidth)
// set RX filter bandwidth only (PGC gain a-priori required)
{
    uint16_t	Values[5];
	uint8_t		iAnt, BwIdx;
	if (rfbandwidth==20)
		BwIdx = 0;					// 20MHz RF bandwidth
	else if (rfbandwidth==40)
		BwIdx = 1;					// 40MHz RF bandwidth
	else if (rfbandwidth == 80)
		BwIdx = 2;					// 80MHz RF bandwidth
	else if (rfbandwidth == 160)
		BwIdx = 3;					// 160MHz RF bandwidth
	else
		BwIdx = 4;					// for all other values maximally open == DPD mode (~150MHz)

	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcFiltIdx[0] = (PgcFiltLUT[BwIdx][RxGainConfig[iAnt].PgcGainIdx[0]] & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK);
			Values[iAnt] = RxGainConfig[iAnt].PgcFiltIdx[0] << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS;
		}
	}

	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------

	RFIC_FCSI_ParRMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__ISMSK,
			Values);	// set PGC filter index, potentially different for each antenna, depending on PGC gain
}


void rfic_set_rxgain_filter_rssi (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB, uint8_t rxfiltidx, uint8_t rssi)
// choose RF RX gain constellation, set BB RX gain, set PGC filter index, select RSSI config
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__ISMSK,
			(rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__POS	// choose RF RX gain constellation
			| ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__POS				// set PGC gain
			| (rxfiltidx & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS					// set PGC filter index
			| (rssi & WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__POS);					// select RSSI range
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[0] = ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK);
			RxGainConfig[iAnt].RfGainIdx[0] = (rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK);
			RxGainConfig[iAnt].RssiRange[0] = (rssi & WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__MSK);
			RxGainConfig[iAnt].PgcFiltIdx[0] = (rxfiltidx & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK);
		}
	}
}

void rfic_set_rxgain_bwidth (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB, uint8_t rfbandwidth)
// choose RF RX gain constellation, set BB RX gain, set RX filter bandwidth
{
	uint8_t	iAnt, PgcGainIdx, BwIdx;
	if (rfbandwidth==20)
		BwIdx = 0;					// 20MHz RF bandwidth
	else if (rfbandwidth==40)
		BwIdx = 1;					// 40MHz RF bandwidth
	else if (rfbandwidth == 80)
		BwIdx = 2;					// 80MHz RF bandwidth
	else if (rfbandwidth == 160)
		BwIdx = 3;					// 160MHz RF bandwidth
	else
		BwIdx = 4;					// for all other values maximally open == DPD mode (~150MHz)

	PgcGainIdx = ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK);
	ASSERT(PgcGainIdx < 11);

	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__ISMSK,
			(rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__POS			// choose RF RX gain constellation
			| PgcGainIdx << WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__POS																		// set PGC gain according LUT
			| (PgcFiltLUT[BwIdx][PgcGainIdx] & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS);	// set PGC filter index
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[0] = PgcGainIdx;
			RxGainConfig[iAnt].RfGainIdx[0] = (rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK);
			RxGainConfig[iAnt].PgcFiltIdx[0] = (PgcFiltLUT[BwIdx][PgcGainIdx] & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK);
		}
	}
}

void rfic_set_rxgain_bwidth_rssi (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB, uint8_t rfbandwidth, uint8_t rssi)
// choose RF RX gain constellation, set BB RX gain, set RX filter bandwidth, select RSSI config
{
	uint8_t	iAnt, PgcGainIdx, BwIdx;
	if (rfbandwidth==20)
		BwIdx = 0;					// 20MHz RF bandwidth
	else if (rfbandwidth==40)
		BwIdx = 1;					// 40MHz RF bandwidth
	else if (rfbandwidth == 80)
		BwIdx = 2;					// 80MHz RF bandwidth
	else if (rfbandwidth == 160)
		BwIdx = 3;					// 160MHz RF bandwidth
	else
		BwIdx = 4;					// for all other values maximally open == DPD mode (~150MHz)

	PgcGainIdx = ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__MSK);
	ASSERT(PgcGainIdx < 11);

	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__ISMSK,
			(rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__POS		// choose RF RX gain constellation
			| PgcGainIdx << WRX654_RF_ANTENNA_FCSI_SR1_PGC_GAIN__POS																	// set PGC gain
			| (PgcFiltLUT[BwIdx][PgcGainIdx] & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__POS	// set PGC filter index according LUT
			| (rssi & WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__MSK) << WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__POS);						// select RSSI range
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[0] = PgcGainIdx;
			RxGainConfig[iAnt].RfGainIdx[0] = (rfrxgainidx & WRX654_RF_ANTENNA_FCSI_SR1_RFRX_GAIN_CONFIG__MSK);
			RxGainConfig[iAnt].PgcFiltIdx[0] = (PgcFiltLUT[BwIdx][PgcGainIdx] & WRX654_RF_ANTENNA_FCSI_SR1_RX_FILTER__MSK);
			RxGainConfig[iAnt].RssiRange[0] = (rssi & WRX654_RF_ANTENNA_FCSI_SR1_SEL_S2D_RX__MSK);
		}
	}
}


void rfic_set_pgcfc100khz(uint16_t *vPgcFc100kHzNew)
// re-define PGC corner frequencies for all modes
{
    uint8_t		iF;

    for (iF=0; iF<5; iF++)
    {
        // only replace the frequencies different from 0
        if (vPgcFc100kHzNew[iF] > 0)
            PgcFc100kHz[iF] = vPgcFc100kHzNew[iF];
    }
}

void rfic_create_pgcfiltlut(uint16_t sA, int16_t sB)
// create internal PGC Filter Tuning LUT from eFuse or given sA/sB parameters
{
	uint16_t    vPgcGainInvFp14[11] = {16384, 13014, 10338, 8211, 6523, 5181, 4115, 3269, 2597, 2063, 1638};	// PGC Gain in FP with 14 FracBits: round(1./10.^([0:2:20]/20)*16384)
	uint8_t		iF, iP, Method;
	int32_t		TmpFrac;
    int16_t     TmpRnd;
	uint16_t	vData[5];
    uint16_t    kA, kB;

	if (sA == 0)
	{
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE14, &vData[0]);
		Method = ((vData[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE14_METHOD__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE14_METHOD__MSK);
        if (Method == 0)
        {
            kA = ((vData[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE14_FACTOR_KA__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE14_FACTOR_KA__MSK);
            kB = ((vData[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE14_FACTOR_KB__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE14_FACTOR_KB__MSK);
            if (kA == 0)
            {
                sA = 5830;
                sB = -151;
            }
            else
            {
                sA = 5000 + (kA << 3);
                TmpRnd = -kB * 100;
                sB = (TmpRnd >> 5) + ((TmpRnd >> 4) & 0x01);
//                printf("sA: %d, sB: %d\n", sA, sB);
            }
        }
        else if (Method == 1)
        {
            // currently a placeholder
            sA = 5830;
            sB = -151;
        }
        else if (Method == 2)
        {
            // currently a placeholder
            sA = 5830;
            sB = -151;
        }
        else if (Method == 3)
        {
            // currently a placeholder
            sA = 5830;
            sB = -151;
        }
	}

	for (iF=0; iF < 5; iF++)
	{
//		printf("Mode %d:    ", iF);
		for (iP=0; iP < 11; iP++)
		{
			TmpFrac = sA * vPgcGainInvFp14[iP] * 10 / PgcFc100kHz[iF] - (sB << 14)/100;
			TmpRnd = 63 - ((TmpFrac >> 14) + ((TmpFrac >> 13) & 0x01));
            if (TmpRnd < 0)
                PgcFiltLUT[iF][iP] = 0;
            else
                PgcFiltLUT[iF][iP] = (uint16_t) TmpRnd;
// 			printf("%d  ", PgcFiltLUT[iF][iP]);

		}
//		printf("\n");
	}
}

void rfic_get_pgcfiltlut (uint8_t mLut[5][11])
// read the current PGC Filter Tuning LUT from memory
{
	uint8_t			iM,iG;

	for (iM=0; iM < 5; iM++)
	{
		for (iG=0; iG < 11; iG++)
		{
			mLut[iM][iG] = PgcFiltLUT[iM][iG];
		}
	}
}


void rfic_set_txgain(uint8_t AntMsk, uint8_t tpcgainidx, uint8_t padattdB)
// choose TPC gain constellation, set PAD attenuation
{
	uint8_t		PadAttIdx;

	// extra handling of 30dB and 36 dB attenuation setting
	if (padattdB > 27)
		if (padattdB > 32)
			PadAttIdx = 15;				// pick 36dB setting for values > 32 dB
		else
			PadAttIdx = 14;				// pick 30dB setting for values between 27..32 dB
	else
		PadAttIdx = padattdB >> 1;

	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN__ISMSK,
			(tpcgainidx & WRX654_RF_ANTENNA_FCSI_SR2_TPC__MSK) << WRX654_RF_ANTENNA_FCSI_SR2_TPC__POS
			| (PadAttIdx & WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_SR2_PA_DRV_GAIN__POS);	// set PAD
}


void rfic_set_tpcgainidx(uint8_t AntMsk, uint8_t tpcgainidx)
// select TPC gain constellation only
{
	//--------------------------------------------------------------------------
	// Short Reg	SR2		Short Register 2				Antx	0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR2,
			WRX654_RF_ANTENNA_FCSI_SR2_TPC__ISMSK,
			(tpcgainidx & WRX654_RF_ANTENNA_FCSI_SR2_TPC__MSK) << WRX654_RF_ANTENNA_FCSI_SR2_TPC__POS);	// set TPC gain constellation
}


void rfic_set_txband(uint8_t AntMsk, uint8_t bandsel)
// set TX band selection
{
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND__ISMSK,
			bandsel << WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND__POS);		// set TX band selection
}

void rfic_read_txband(uint8_t AntMsk, uint16_t *tx_band)
// read the current TX band selection from selected antennas
{
	uint16_t	data[5], iK;
	// read from SR0
	RFIC_FCSI_Read(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0, &data[0]);
	for (iK = 0; iK < 4; iK++)
	{
		if (AntMsk & (1 << iK))		// only return in-range number for selected antennas
		{
			tx_band[iK] = ((data[iK] >> WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND__POS) & WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND__MSK);
		}
		else						// for not-selected antennas return out-of-range number
		{
			tx_band[iK] = 0xf;
		}
	}
}


uint16_t rfic_get_txband(uint16_t LoFreq)
// return best-guess TX HB or LB band selection
{
	int16_t 	XFreq[7];
	uint16_t	data[5];
	uint8_t		Version, ChipFused, HbLb;
	int8_t		TxBand = 0, iX, TxBandDelta, TxBandOffs;

	TxBandDelta = 0;
	TxBandOffs = 0;

	if (LoFreq > 2500)
		HbLb = 1;
	else
		HbLb = 0;

	if (HbLb)
	// HB band selection
	{
		// check if WRX654 A0 or A1
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_VERSION, &data[0]);
		Version = ((data[4] >> WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__POS) & WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__MSK);
		if (Version == 64) // = 0b0100000 --> A0: 01=A 00000=0
		{
			// Crossing Point frequencies, Nominal Centerband is 4, best guess from a single A0 Level1 measurement on Ant1
			XFreq[0] = 5000;
			XFreq[1] = 5200;
			XFreq[2] = 5375;
			XFreq[3] = 5580;
			XFreq[4] = 5770;
			XFreq[5] = 6000;
			XFreq[6] = 6250;
		}
		else if (Version == 65) // = 0b0100001 --> A1: 01=A 00001=1
		{
			// Crossing Point frequencies, Nominal Centerband is 4, best guess from a single A0 Level1 measurement on Ant1
			XFreq[0] = 5000;
			XFreq[1] = 5200;
			XFreq[2] = 5375;
			XFreq[3] = 5580;
			XFreq[4] = 5770;
			XFreq[5] = 6000;
			XFreq[6] = 6250;
		}
        else // only for debug
		{
			// Crossing Point frequencies, Nominal Centerband is 4, best guess from a single A0 Level1 measurement on Ant1
			XFreq[0] = 5000;
			XFreq[1] = 5200;
			XFreq[2] = 5375;
			XFreq[3] = 5580;
			XFreq[4] = 5770;
			XFreq[5] = 6000;
			XFreq[6] = 6250;
		}

		// Check if chip is fused
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE15, &data[0]);
		if ((data[4] & (7 << WRX654_RF_CENTRAL_FCSI_EFUSE15_PRODUCTCODE__POS)) == 0)
			ChipFused = 0;
		else
			ChipFused = 1;

		// Read eFuse register #12
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE12, &data[0]);
		// extract band selection offset fuse information for CH0/1/2
		if (ChipFused)
		{
			if (LoFreq < 5410) // for LO freqs up to 5410 MHz use fuse information from CH0 (5250 MHz) band selection offset
			{
				TxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH0__MSK);
				TxBandOffs = -1; // expected band -1 smaller than Nominal band
			}
			else if ((LoFreq >= 5410) && (LoFreq < 5700))  // for LO freqs between 5410 and 5700 MHz use fuse information from CH1 (5570 MHz) band selection offset
			{
				TxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH1__MSK);
				TxBandOffs = 0; // expected band fits to Nominal band
			}
			if (LoFreq >= 5700) // for LO freqs above 5700 MHz use fuse information from CH2 (5825 MHz) band selection offset
			{
				TxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH2__MSK);
				TxBandOffs = 1; // expected band +1 larger than Nominal band
			}
			if (TxBandDelta > 1) // two's complement
				TxBandDelta = TxBandDelta - 4;
			TxBandDelta = TxBandDelta - TxBandOffs;
		}
		else
			TxBandDelta = 0; // no fuse information available -> assume nominal behavior

		for (iX=6; iX>=0; iX--)
		{
			if (LoFreq >= XFreq[iX])
			{
                TxBand = iX+1;
				break;
			}
		}
		// correct TX Band according fused offset information and check limits
		TxBand = TxBand + TxBandDelta;
		if (TxBand < 0)
			TxBand = 0;
		if (TxBand > 7)
			TxBand = 7;
	}
	else
	// LB band selection
	{
		// Read eFuse register #12
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE12, &data[0]);
		TxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXLBBANDSELECTION__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXLBBANDSELECTION__MSK);
		if (TxBandDelta > 1) // two's complement
			TxBandDelta = TxBandDelta - 4;

		TxBand = TX_LB_BAND_NOM + TxBandDelta; // band index = nominal index (#2) + delta
	}

	return (uint16_t) TxBand;
}


void rfic_set_rxband(uint8_t AntMsk, uint8_t bandsel)
// set RX band selection
{
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND__ISMSK,
			bandsel << WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND__POS);		// set RX band selection
}

void rfic_read_rxband(uint8_t AntMsk, uint16_t *rx_band)
// read the current TX and RX band selection from selected antennas
{
	uint16_t	data[5], iK;
	// read from SR0
	RFIC_FCSI_Read(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0, &data[0]);
	for (iK = 0; iK < 4; iK++)
	{
		if (AntMsk & (1 << iK))		// only return in-range number for selected antennas
		{
			rx_band[iK] = ((data[iK] >> WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND__POS) & WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND__MSK);
		}
		else						// for not-selected antennas return out-of-range number
		{
			rx_band[iK] = 0xf;
		}
	}
}


uint16_t rfic_get_rxband(uint16_t LoFreq)
// return best-guess RX HB or LB band selection
{
	int16_t 	XFreq[7];
	uint16_t	data[5];
	uint8_t		Version, ChipFused, HbLb;
	int8_t		RxBand = 0, iX, RxBandDelta, RxBandOffs;

	RxBandDelta = 0;
	RxBandOffs = 0;

	if (LoFreq > 2500)
		HbLb = 1;
	else
		HbLb = 0;

	if (HbLb)
	// HB band selection
	{
		// check if WRX654 A0 or A1
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_VERSION, &data[0]);
		Version = ((data[4] >> WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__POS) & WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__MSK);
		if (Version == 64) // = 0b0100000 --> A0: 01=A 00000=0
		{
			// Crossing Point frequencies, Nominal Centerband is 6, best guess from a single A0 Level1 measurement on Ant1
			XFreq[0] = 4610;
			XFreq[1] = 4760;
			XFreq[2] = 4930;
			XFreq[3] = 5120;
			XFreq[4] = 5330;
			XFreq[5] = 5570;
			XFreq[6] = 5830;
		}
		else if (Version == 65) // = 0b0100001 --> A1: 01=A 00001=1
		{
			// Crossing Point frequencies, Nominal Centerband is 6, best guess from a single A0 Level1 measurement on Ant1
			XFreq[0] = 4610;
			XFreq[1] = 4760;
			XFreq[2] = 4930;
			XFreq[3] = 5120;
			XFreq[4] = 5330;
			XFreq[5] = 5570;
			XFreq[6] = 5830;
		}
        else // only for debug
		{
			// Crossing Point frequencies, Nominal Centerband is 6, best guess from a single A0 Level1 measurement on Ant1
			XFreq[0] = 4610;
			XFreq[1] = 4760;
			XFreq[2] = 4930;
			XFreq[3] = 5120;
			XFreq[4] = 5330;
			XFreq[5] = 5570;
			XFreq[6] = 5830;
		}

		// Check if chip is fused
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE15, &data[0]);
		if ((data[4] & (7 << WRX654_RF_CENTRAL_FCSI_EFUSE15_PRODUCTCODE__POS)) == 0)
			ChipFused = 0;
		else
			ChipFused = 1;

		// Read eFuse register #12
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE12, &data[0]);
		// extract band selection offset fuse information for CH0/1/2
		if (ChipFused)
		{
			if (LoFreq < 5410) // for LO freqs up to 5410 MHz use fuse information from CH0 (5250 MHz) band selection offset
			{
				RxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH0__MSK);
				RxBandOffs = -1; // expected band -1 smaller than Nominal band
			}
			else if ((LoFreq >= 5410) && (LoFreq < 5700))  // for LO freqs between 5410 and 5700 MHz use fuse information from CH1 (5570 MHz) band selection offset
			{
				RxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH1__MSK);
				RxBandOffs = 0; // expected band fits to Nominal band
			}
			if (LoFreq >= 5700) // for LO freqs above 5700 MHz use fuse information from CH2 (5825 MHz) band selection offset
			{
				RxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH2__MSK);
				RxBandOffs = 1; // expected band +1 larger than Nominal band
			}
			if (RxBandDelta > 1) // two's complement
				RxBandDelta = RxBandDelta - 4;
			RxBandDelta = RxBandDelta - RxBandOffs;
		}
		else
			RxBandDelta = 0; // no fuse information available -> assume nominal behavior

		for (iX=6; iX>=0; iX--)
		{
			if (LoFreq >= XFreq[iX])
			{
                RxBand = iX+1;
				break;
			}
		}
		// correct TX Band according fused offset information and check limits
		RxBand = RxBand + RxBandDelta;
		if (RxBand < 0)
			RxBand = 0;
		if (RxBand > 7)
			RxBand = 7;
	}
	else
	// LB band selection
	{
		// Read eFuse register #12
		RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE12, &data[0]);
		RxBandDelta = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXLBBANDSELECTION__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXLBBANDSELECTION__MSK);
		if (RxBandDelta > 1) // two's complement
			RxBandDelta = RxBandDelta - 4;

		RxBand = RX_LB_BAND_NOM + RxBandDelta; // band index = nominal index (#6) + delta
	}

	return (uint16_t) RxBand;
}

void rfic_set_lnamidgain(uint8_t AntMsk, uint8_t lnamidgain)
// set LNA Mid gain fine-tuning value (in the currently active Band - LB or HB!!!)
{
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__ISMSK,
			lnamidgain << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS);			// set Mid Gain tuning value
}

uint16_t rfic_get_lnamidgain(uint8_t Ant, uint8_t HbLb)
// return RX LNA HB/LB mid gain tuning index, either from eFuse or best-guess
{
	uint16_t	data[5], MidTuneIdx;
	uint8_t		Version, ChipFused, bAnt01;

	// Chip fused?
	RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE15, &data[0]);
	if ((data[4] & (7 << WRX654_RF_CENTRAL_FCSI_EFUSE15_PRODUCTCODE__POS)) == 0)
		ChipFused = 0;
	else
		ChipFused = 1;
//    printf("Chip fused: %d\n", ChipFused);

	// Chip version
	RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_VERSION, &data[0]);
	Version = ((data[4] >> WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__POS) & WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__MSK);

	// Antenna 0/1 or 2/3?
	if (Ant > 1)
		bAnt01 = 0;
	else
		bAnt01 = 1;

	if (HbLb)
	// HB band mid gain tuning
	{
		if (ChipFused)
		{
			// Read eFuse register #12
			RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE10, &data[0]);
			if (bAnt01)
			{
				MidTuneIdx = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT01__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT01__MSK) + MID2HIGHSTEP_HB_NOM;
				if (MidTuneIdx > 1 + MID2HIGHSTEP_HB_NOM)
					MidTuneIdx = MidTuneIdx - 4;
			}
			else
			{
				MidTuneIdx = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT23__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT23__MSK) + MID2HIGHSTEP_HB_NOM;
				if (MidTuneIdx > 1 + MID2HIGHSTEP_HB_NOM)
					MidTuneIdx = MidTuneIdx - 4;
			}
		}
		else
		{
			if ((Version == 64) | (Version == 65))	// A0 or A1
				MidTuneIdx = (uint16_t) (2 + MID2HIGHSTEP_HB_NOM);	// Mean according A1 TTT lot: 2.0
			else	// placeholder for unknown versions
				MidTuneIdx = (uint16_t) (2 + MID2HIGHSTEP_HB_NOM);	// Mean according A1 TTT lot: 2.0
		}
	}
	else
	// LB band mid gain tuning
	{
		if (ChipFused)
		{
			// Read eFuse register #12
			RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE9, &data[0]);
			if (bAnt01)
			{
				MidTuneIdx = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__MSK) + MID2HIGHSTEP_LB_NOM;
				if (MidTuneIdx > 1 + MID2HIGHSTEP_LB_NOM)
					MidTuneIdx = MidTuneIdx - 4;
			}
			else
			{
				MidTuneIdx = ((data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__MSK) + MID2HIGHSTEP_LB_NOM;
				if (MidTuneIdx > 1 + MID2HIGHSTEP_LB_NOM)
					MidTuneIdx = MidTuneIdx - 4;
			}
		}
		else
		{
			if ((Version == 64) | (Version == 65))	// A0 or A1
				MidTuneIdx = (uint16_t) (1 + MID2HIGHSTEP_LB_NOM);	// Mean according A1 TTT lot: 1.0
			else	// placeholder for unknown versions
				MidTuneIdx = (uint16_t) (1 + MID2HIGHSTEP_LB_NOM);	// Mean according A1 TTT lot: 1.0
		}
	}

	return (uint16_t) MidTuneIdx;
}


void rfic_set_bandwidth(uint8_t AntMsk, uint8_t rfbandwidth)
// select (RF) bandwidth (in MHz: 20, 40, 80, or 160)
{
	uint8_t BwIdx;

	if (rfbandwidth == 160)
		BwIdx = 3;
	else if (rfbandwidth == 80)
		BwIdx = 2;
	else if (rfbandwidth == 40)
		BwIdx = 1;
	else
		BwIdx = 0;

	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW__ISMSK,
			BwIdx <<  WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW__POS);	// select bandwidth
}


void rfic_set_powermode(uint8_t AntMsk, uint8_t modeidx)
// select power mode: 0:HP, 1:RP, 2:LP, 3:DPD
{
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_MODE__ISMSK,
			modeidx <<  WRX654_RF_ANTENNA_FCSI_SR0_MODE__POS);	// select power mode
}

void rfic_set_rxdc(uint8_t AntMsk, uint8_t rxdcidx)
// select RX DC offset constellation
{
	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC__ISMSK,
			rxdcidx <<  WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC__POS);	// select RX DC offset
}

void rfic_set_rfch_dc_bw(uint8_t AntMsk, uint8_t Channel, uint8_t rxdcidx, uint8_t rfbandwidth)
// select fasthop channel, RX DC Offset, and RF bandwidth
{
	uint8_t BwIdx;

	if (rfbandwidth == 160)
		BwIdx = 3;
	else if (rfbandwidth == 80)
		BwIdx = 2;
	else if (rfbandwidth == 40)
		BwIdx = 1;
	else
		BwIdx = 0;

	//--------------------------------------------------------------------------
	// Short Reg	SR0		Short Register 0				Antx	0x00
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR0,
			WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC__ISMSK,
			BwIdx <<  WRX654_RF_ANTENNA_FCSI_SR0_SEL_BW__POS
			| Channel <<  WRX654_RF_ANTENNA_FCSI_SR0_SEL_CH__POS
			| rxdcidx <<  WRX654_RF_ANTENNA_FCSI_SR0_SEL_RXDC__POS);	// select RX DC offset
}

void rfic_enable_lnanoise(uint8_t AntMsk)
// enable LNA noise source
{
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_EN_LNA_NOISE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_0__SVAL				// disable LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_EN_LNA_NOISE_1__SVAL);			// enable noise source
}

void rfic_disable_lnanoise(uint8_t AntMsk)
// disable LNA noise source
{
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_EN_LNA_NOISE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL				// enable LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_EN_LNA_NOISE_0__SVAL);			// disable noise source
}

void rfic_enable_lna(uint8_t AntMsk)
// enable full LNA
{
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_1__SVAL);			// enable LNA
}

void rfic_disable_lna(uint8_t AntMsk)
// disable full LNA
{
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_RSSI_0__SVAL);			// disable LNA
}


void rfic_config_dirstep(uint8_t AntMsk, uint8_t dstepidx, uint8_t lnagainidx, uint8_t pgcgaindB, uint8_t rssirange, uint8_t rxfiltidx)
// configure direct steps 1, 2, 3
{
	uint8_t		iAnt;
	uint16_t	RegAddr;

	if (dstepidx == 2)
		RegAddr = WRX654_RF_ANTENNA_FCSI_DIRSTEP2;	// select direct step 2
	else if (dstepidx == 3)
		RegAddr = WRX654_RF_ANTENNA_FCSI_DIRSTEP3;	// select direct step 3
	else
	{
		dstepidx = 1;								// make sure that index doesn't exceed range
		RegAddr = WRX654_RF_ANTENNA_FCSI_DIRSTEP1;	// default: select direct step 1
	}

	//--------------------------------------------------------------------------
	// Register DIRSTEP1/2/3	DIRECTSTEP 1/2/3			Antx	0x4f/0x4e/0x5a
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, RegAddr,
			(lnagainidx & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_LNA__MSK) << WRX654_RF_ANTENNA_FCSI_DIRSTEP1_LNA__POS						// RF RX gain index
			| (rssirange & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_SEL_S2D_RX__MSK) << WRX654_RF_ANTENNA_FCSI_DIRSTEP1_SEL_S2D_RX__POS		// RSSI range
			| ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_PGC_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_DIRSTEP1_PGC_GAIN__POS	// PGC gain
			| (rxfiltidx & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_RX_FILTER__MSK) << WRX654_RF_ANTENNA_FCSI_DIRSTEP1_RX_FILTER__POS);		// RX filter cap setting
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].PgcGainIdx[dstepidx] = ((pgcgaindB >> 1) & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_PGC_GAIN__MSK);
			RxGainConfig[iAnt].RfGainIdx[dstepidx] = (lnagainidx & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_LNA__MSK);
			RxGainConfig[iAnt].RssiRange[dstepidx] = (rssirange & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_SEL_S2D_RX__MSK);
			RxGainConfig[iAnt].PgcFiltIdx[dstepidx] = (rxfiltidx & WRX654_RF_ANTENNA_FCSI_DIRSTEP1_RX_FILTER__MSK);
		}
	}
}


void rfic_select_dirstep(uint8_t AntMsk, uint8_t dstepidx)
// select direct steps 1, 2, 3, and 0 for deactivation
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR3		Short Register 3				Antx	0x03
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_SR3,
			(dstepidx & WRX654_RF_ANTENNA_FCSI_SR3_STEP__MSK) << 14);		// select direct step (@@@ Hack: shift to MSB position of 16-bit data)
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].ShortReg3 = (dstepidx & WRX654_RF_ANTENNA_FCSI_SR3_STEP__MSK);
			if (RxGainConfig[iAnt].ForceOff == 0)
				RxGainConfig[iAnt].DirStepIdx = RxGainConfig[iAnt].ShortReg3;
			else
				RxGainConfig[iAnt].DirStepIdx = 0;
		}
	}
}


void rfic_dirstep_force_on(uint8_t AntMsk)
// clear FORCE_OFF bit in SR1
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_FORCE_OFF__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR1_FORCE_OFF_0__SVAL);		// clear FORCE_OFF bit (== FORCE ON)
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].ForceOff = 0;
			RxGainConfig[iAnt].DirStepIdx = RxGainConfig[iAnt].ShortReg3;
		}
	}
}


void rfic_dirstep_force_off(uint8_t AntMsk)
// set FORCE_OFF bit in SR1
{
	uint8_t		iAnt;
	//--------------------------------------------------------------------------
	// Short Reg	SR1		Short Register 1				Antx	0x01
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SR1,
			WRX654_RF_ANTENNA_FCSI_SR1_FORCE_OFF__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SR1_FORCE_OFF_1__SVAL);		// set FORCE_OFF bit
	// update RxGainConfig accordingly
	for (iAnt=0; iAnt<NUM_OF_ANTS; iAnt++)
	{
		if (AntMsk & (1<<iAnt))
		{
			RxGainConfig[iAnt].ForceOff = 1;
			RxGainConfig[iAnt].DirStepIdx = 0;
		}
	}
}


void rfic_enable_bbloop(uint8_t AntMsk, uint8_t HbLb, uint8_t LoopType, uint8_t IqSel)
// enable baseband loop for calibration, specify HB or LB, specify LoopType (0==SSBCAL, 1==IQCAL) and choose I, or Q, or both as output
// - antennas in antenna mask are in the correct band HbLb, and TXOn is set
// - disable or enable IQCAL switch, depending on loop type
// - disable IQCAL and IQCAL GND switch
// - enable PGCCAL switches for I, or Q, or both
// - increase PGC bias current and reduce comp cap to improve matching of loop and main mode
// - manually enable according PGC branch(es)
// - manually enable PGC
// - enable opposite HB or LB Mixer Core for Offset DAC
// - manually enable according BBO branch(es)
// - enable or disable SSBCAL, depending on loop type
{
	UNUSED_PARAM(HbLb);
	// disable IQCAL and IQCAL GND switch, select PGCCAL paths, enable PGC
	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_LOOP__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_EN_PGC_CAL__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_PGC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW_0__SVAL					// open IQCAL GND switch
			| (LoopType & 1) << WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_LOOP__POS	// open (SSBCAL) or close (IQCAL) IQCAL connection to TPC
			| (IqSel & WRX654_RF_ANTENNA_FCSI_PGC_EN_PGC_CAL__MSK) << WRX654_RF_ANTENNA_FCSI_PGC_EN_PGC_CAL__POS
			// activate selected PGC calibration path(s)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE_1__SVAL							// activate manual control of PGC enable
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_PGC_1__SVAL							// manually enable the PGC
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY_1__SVAL							// manual STBY control
			| WRX654_RF_ANTENNA_FCSI_PGC_STBY_0__SVAL 								// release Mixer core from standby
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT_1__SVAL					// NEW: increase bias current by 25% to improve IQMM matching in loop mode
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);						// NEW: reduce compensation cap to improve IQMM matching in loop mode

//	// change HB_LB_FCSI flag to other band
//	//--------------------------------------------------------------------------
//	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
//	//--------------------------------------------------------------------------
//	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
//			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
//			(HbLb == 0) << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__POS); // select other band's set of doubled registers

	// activate Mixer core in other band
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL			// enable other band's mixer core
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER_1__SVAL			// manual standby control
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_STBY_MIXER_0__SVAL				// release from standby
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL); 			// always enable both mixer channels

//	// reset HB_LB_FCSI flag to active band
//	//--------------------------------------------------------------------------
//	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
//	//--------------------------------------------------------------------------
//	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
//			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
//			HbLb << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__POS); 	// return to active band set of doubled registers

	// manually enable BBO branch(es) (BBO I is always enabled)
	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I_1__SVAL			// enable In0 I input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q_1__SVAL			// enable In0 Q input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL			// enable BBO I
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q_1__SVAL);		// enable BBO Q

	if (LoopType == 0)
		// enable SSBCAL in currently active band
		//--------------------------------------------------------------------------
		// Register SSBDET	SSB DETECTOR						Antx	0x50
		//--------------------------------------------------------------------------
		RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_SSBDET,
				WRX654_RF_ANTENNA_FCSI_SSBDET_CTL_BIAS_SELECT_0__SVAL						// currently, use default bias setting
				| WRX654_RF_ANTENNA_FCSI_SSBDET_CTL_ISS_0__SVAL								// currently, use default bias type
				| WRX654_RF_ANTENNA_FCSI_SSBDET_ENABLE_1__SVAL);							// enable SSB detector
}


void rfic_disable_bbloop(uint8_t AntMsk, uint8_t HbLb)
// disable baseband loop for calibration
// - it is assumed that antenna is set to TXON in either band according to HbLb setting
// - disable SSBCAL (always, even if IQCAL was active)
// - remove manual control from BBO
// - remove manual control from PGC, disable PGC cal branches
// - return PGC to default bias and comp cap settings
// - open IQCAL switch (always, even if it was already open)
// - close IQCAL GND switch
// - disable other band's mixer core and remove manual control
{
	UNUSED_PARAM(HbLb);
	// disable SSBCAL in currently active band (always)
	//--------------------------------------------------------------------------
	// Register SSBDET	SSB DETECTOR						Antx	0x50
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSBDET,
			WRX654_RF_ANTENNA_FCSI_SSBDET_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SSBDET_ENABLE_0__SVAL);							// disable SSB detector

	// remove manual control from BBO (BBO I remains enabled)
	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO_0__SVAL						// remove manual control
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL);				// switch EN_BBO_I back on, needed for regular operation

	// deactivate PGCCAL paths, disable manual PGC control
	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_EN_PGC_CAL__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_LOOP__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PGC_EN_PGC_CAL_0__SVAL						// deactivate all PGC calibration paths
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE_0__SVAL						// remove manual PGC control
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY_0__SVAL						// remove manual STBY control
			| WRX654_RF_ANTENNA_FCSI_PGC_STBY_0__SVAL							// release Mixer core from standby
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_GND_SW_1__SVAL			// close IQCAL GND switch (@@@ is this preferable? @@@)
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_IQCAL_LOOP_0__SVAL				// open IQCAL connection to TPC (always)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT_0__SVAL				// NEW: change bias current back to default
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);					// NEW: change compensation cap back to default (now different from reset value)

//	// change HB_LB_FCSI flag to other band
//	//--------------------------------------------------------------------------
//	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
//	//--------------------------------------------------------------------------
//	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
//			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
//			(HbLb == 0) << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__POS); // select other band's set of doubled registers

	// deactivate Mixer core in other band
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
//	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
//			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK
//			& WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER__ISMSK
//			& WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER__ISMSK,
//			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_0__SVAL			// disable other band's mixer core
//			| WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER_0__SVAL			// remove manual standby control
//			| WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL);			// re-enable I+Q mixer channels

	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER_0__SVAL			// remove manual standby control
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL);			// re-enable I+Q mixer channels

//	// reset HB_LB_FCSI flag to active band
//	//--------------------------------------------------------------------------
//	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
//	//--------------------------------------------------------------------------
//	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
//			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
//			HbLb << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__POS); // return to active set of doubled registers
}

void rfic_enable_ssbcal_hb(uint8_t AntMsk)
// enable SSBCAL calibration in HB mode, output to I and Q
// incl. manual TXON setting
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL); // select HB set of doubled registers

	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC_1__SVAL); // enable manually TPC

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD_0__SVAL); // enable manually PAD (via STBY)

	//--------------------------------------------------------------------------
	// Register LOLOCAL - LO LOCAL							Antx	0x5f
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2_1__SVAL); // enable I2V and TX HB buffers

	rfic_enable_bbloop(AntMsk, 1, 0, 3);
}


void rfic_disable_ssbcal_hb(uint8_t AntMsk)
// disable SSBCAL calibration in HB mode, output to I and Q
// incl. manual TXON setting
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL); // select HB set of doubled registers

	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC_0__SVAL); // disable manually TPC

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD_0__SVAL); // disable manually PAD (via STBY)

	//--------------------------------------------------------------------------
	// Register LOLOCAL - LO LOCAL							Antx	0x5f
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2_0__SVAL); // disable I2V and TX HB buffers

	rfic_disable_bbloop(AntMsk, 1);
}

void rfic_enable_ssbcal_lb(uint8_t AntMsk)
// enable SSBCAL calibration in LB mode, output to I and Q
// incl. manual TXON setting
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL); // select LB set of doubled registers

	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC_1__SVAL); // enable manually TPC

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD_0__SVAL); // enable manually PAD (via STBY)

	//--------------------------------------------------------------------------
	// Register LOLOCAL - LO LOCAL							Antx	0x5f
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB_1__SVAL); // enable TX LB buffer


	rfic_enable_bbloop(AntMsk, 0, 0, 3);
}

void rfic_disable_ssbcal_lb(uint8_t AntMsk)
// disable SSBCAL calibration in LB mode, output to I and Q
// incl. manual TXON setting
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL); // select LB set of doubled registers

	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC_0__SVAL); // disable manually TPC

	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD_0__SVAL); // disable manually PAD (via STBY)

	//--------------------------------------------------------------------------
	// Register LOLOCAL - LO LOCAL							Antx	0x5f
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB_0__SVAL); // disable TX LB buffer


	rfic_disable_bbloop(AntMsk, 0);
}

void rfic_enable_iqcal_hb(uint8_t AntMsk, uint8_t IqSel)
// enable IQ calibration in HB mode, output to I, to Q, or to both I+Q
{
	rfic_enable_bbloop(AntMsk, 1, 1, IqSel);
}

void rfic_enable_iqcal_lb(uint8_t AntMsk, uint8_t IqSel)
// enable IQ calibration in HB mode, output to I, to Q, or to both I+Q
{
	rfic_enable_bbloop(AntMsk, 0, 1, IqSel);
}

void rfic_enable_filtercal_hb(uint8_t AntMsk)
// enable RX filter calibration in HB mode, output to I and Q
{
	rfic_enable_bbloop(AntMsk, 1, 1, 3);
}

void rfic_enable_filtercal_lb(uint8_t AntMsk)
// enable RX filter calibration in LB mode, output to I and Q
{
	rfic_enable_bbloop(AntMsk, 0, 1, 3);
}

void rfic_disable_iqcal_hb(uint8_t AntMsk)
// disable IQ calibration in HB mode
{
	rfic_disable_bbloop(AntMsk, 1);
}

void rfic_disable_iqcal_lb(uint8_t AntMsk)
// disable IQ calibration in LB mode
{
	rfic_disable_bbloop(AntMsk, 0);
}

void rfic_disable_filtercal_hb(uint8_t AntMsk)
// disable RX filter calibration in HB mode
{
	rfic_disable_bbloop(AntMsk, 1);
}

void rfic_disable_filtercal_lb(uint8_t AntMsk)
// disable RX filter calibration in LB mode
{
	rfic_disable_bbloop(AntMsk, 0);
}

void rfic_enable_dpdcal(uint8_t AntMsk, uint8_t HbLb)
// manually activate TX and RX path in DPD mode (maximum bandwidth and linearity) (>NO< RFIC TXON or RXON required!)
// - pre-condition: antennas in antenna mask are in the correct band HbLb, TPC and PGC gain already set, TX power mode and bandwidth is set
// - switch off TXON and RXON
// - select set of doubled registers according HbLb
// - (optionally) activate LB or HB according HbLb
// - manually enable TPC
// - prevent Vcm connection to TXMIXER coil center tap
// - manually enable Pre-Driver
// - manually enable LNA
// - manually enable RX Mixer
// - manually enable PGC
// - increase PGC bias current and reduce comp cap for maximum bandwidth
// - set PGC filter to DPD mode for maximum bandwidth
// - manually enable BBO
// - manually enable TX and RX LO according HbLb
{
	// *************************************************************************
	// *              TXOFF, RXOFF, explicitly switch HB or LB                 *
	// *************************************************************************
	rfic_txoff(AntMsk);
	rfic_rxoff(AntMsk);
	// *************************************************************************
	// *             select set of doubled registers according HbLb            *
	// *************************************************************************
	if (HbLb > 0)
	{
		rfic_activate_hb(AntMsk);
		//--------------------------------------------------------------------------
		// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL); // select HB set of doubled registers
	}
	else
	{
		rfic_activate_lb(AntMsk);
		//--------------------------------------------------------------------------
		// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL); // select LB set of doubled registers
	}

	// *************************************************************************
	// *                          Activate TX Path                             *
	// *************************************************************************
	// manually activate TPC and TXMIXER
	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_DISABLE_RXON_CONTROL__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC_1__SVAL 					// enable manually TPC
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_DISABLE_RXON_CONTROL_1__SVAL); 	// prevent TXMIXER VCM connection to center tap (may not be needed since RXON should be off!)

	// manually activate Pre-Driver
	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD_0__SVAL); // enable manually PAD (via STBY)


	// *************************************************************************
	// *                          Activate RX Path                             *
	// *************************************************************************
	// activate LNA
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_STBY_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_STBY_LNA__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_ENABLE_LNA_1__SVAL						// enable LNA
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_STBY_LNA_1__SVAL					// enable manual STBY control
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_STBY_LNA_0__SVAL);						// release LNA from STBY

	// activate RX Mixer
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_ENABLE_RXMIXER_1__SVAL			// enable mixer core
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER_1__SVAL			// manual standby control
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_STBY_MIXER_0__SVAL				// release from standby
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL); 			// always enable both mixer channels

	// activate PGC
	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_PGC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE_1__SVAL							// activate manual control of PGC enable
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_PGC_1__SVAL							// manually enable the PGC
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY_1__SVAL							// manual STBY control
			| WRX654_RF_ANTENNA_FCSI_PGC_STBY_0__SVAL								// release Mixer core from standby
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT_1__SVAL					// increase bias current by 25% for maximum bandwidth
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);						// reduce compensation cap for maximum bandwidth

	// rfic_set_pgcbwidth(AntMsk, 250);												// set PGC to maximum bandwidth

	// manually enable BBO
	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_I_1__SVAL			// enable In0 I input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_IN0_Q_1__SVAL			// enable In0 Q input
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL			// enable BBO I
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_Q_1__SVAL);		// enable BBO Q

	// *************************************************************************
	// *                       Activate TX and RX LO                           *
	// *************************************************************************
	if (HbLb > 0)
	{
		//--------------------------------------------------------------------------
		// Register LOLOCAL - LO LOCAL							Antx	0x5f
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF2__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2_1__SVAL 		// enable I2V and TX HB buffers
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF1_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF2_1__SVAL); 	// enable RX HB buffers
	}
	else
	{
		//--------------------------------------------------------------------------
		// Register LOLOCAL - LO LOCAL							Antx	0x5f
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF_LB__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF_LB__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB_1__SVAL 	// enable TX LB buffer
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF_LB_1__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF_LB_1__SVAL); 	// enable RX LB buffer
	}
}

void rfic_disable_dpdcal(uint8_t AntMsk, uint8_t HbLb)
// remove manually activation of RX and TX path in DPD mode
// - select set of doubled registers according HbLb
// - deactivate TX and RX LOs
// - return BBO to auto mode
// - set PGC filter to 160MHz mode
// - nominal PGC bias current and comp cap
// - return PGC to automatic mode
// - return RX Mixer to automatic mode
// - return LNA to automatic mode
// - return Pre-Driver to regular mode
// - return TX MIXER and TPC to regular mode
{
	// *************************************************************************
	// *             select set of doubled registers according HbLb            *
	// *************************************************************************
	if (HbLb > 0)
	{
		//--------------------------------------------------------------------------
		// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_1__SVAL); // select HB set of doubled registers
	}
	else
	{
		//--------------------------------------------------------------------------
		// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47	(OK)
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI__ISMSK,
				WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_HB_LB_FCSI_0__SVAL); // select LB set of doubled registers
	}

	// *************************************************************************
	// *                       Deactivate TX and RX LO                           *
	// *************************************************************************
	if (HbLb > 0)
	{
		//--------------------------------------------------------------------------
		// Register LOLOCAL - LO LOCAL							Antx	0x5f
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF2__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_I2V_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_I2V_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF1_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF2_0__SVAL 		// disable I2V and TX HB buffers
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF1_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF2_0__SVAL); 	// disable RX HB buffers
	}
	else
	{
		//--------------------------------------------------------------------------
		// Register LOLOCAL - LO LOCAL							Antx	0x5f
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOLOCAL,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF_LB__ISMSK
				& WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF_LB__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_TXBUF_LB_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_TXBUF_LB_0__SVAL 	// disable TX LB buffer
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_CTL_RXBUF_LB_0__SVAL
				| WRX654_RF_ANTENNA_FCSI_LOLOCAL_ENABLE_RXBUF_LB_0__SVAL); 	// disable RX LB buffer
	}

	// *************************************************************************
	// *                         Deactivate RX Path                            *
	// *************************************************************************
	// remove manual control from BBO (BBO I remains enabled)
	//--------------------------------------------------------------------------
	// Register BBIOCONF0	BBIO CONFIG 0					Antx	0x42	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBIOCONF0,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO__ISMSK
			& WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBIOCONF0_CTL_EN_BBO_0__SVAL						// remove manual control
			| WRX654_RF_ANTENNA_FCSI_BBIOCONF0_ENABLE_BBO_I_1__SVAL);				// switch EN_BBO_I back on, needed for regular operation

	// PGC back to regular mode
//	rfic_set_pgcbwidth(AntMsk, 160);												// set PGC to 160MHz

	//--------------------------------------------------------------------------
	// Register PGC			MIXER BB and PGC				Antx	0x5c	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PGC,
			WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_PGC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PGC_CTL_ENABLE_0__SVAL							// disable manual control of PGC enable
			| WRX654_RF_ANTENNA_FCSI_PGC_ENABLE_PGC_0__SVAL							// manually disable the PGC (optional)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_STBY_0__SVAL							// disable manual STBY control
			| WRX654_RF_ANTENNA_FCSI_PGC_STBY_1__SVAL								// put Mixer core to standby (optional)
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_BIAS_CURRENT_0__SVAL					// NEW: change bias current back to default
			| WRX654_RF_ANTENNA_FCSI_PGC_CTL_COMP_CAP_2__SVAL);						// NEW: change compensation cap back to default (now different from reset value)

	// RX Mixer back to regular mode
	//--------------------------------------------------------------------------
	// Register RXMIXER		RX MIXER CONTROL		 		Antx	0x59	(OK)
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXMIXER,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_STBY_MIXER__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXMIXER_CTL_STBY_MIXER_0__SVAL			// disable manual standby control
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_STBY_MIXER_1__SVAL				// put to standby (optional)
			| WRX654_RF_ANTENNA_FCSI_RXMIXER_EN_CH_MIXER_3__SVAL); 			// always enable both mixer channels

	// LNA back to regular mode
	//--------------------------------------------------------------------------
	// Register LNARSSI		LNA and RSSI					Antx	0x58
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LNARSSI,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_STBY_LNA__ISMSK
			& WRX654_RF_ANTENNA_FCSI_LNARSSI_STBY_LNA__ISMSK,
			WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_STBY_LNA_0__SVAL					// disable manual STBY control
			| WRX654_RF_ANTENNA_FCSI_LNARSSI_STBY_LNA_1__SVAL);					// put LNA to STBY (optional)


	// *************************************************************************
	// *                         Deactivate RX Path                            *
	// *************************************************************************
	// Pre-Driver back to regular mode
	//--------------------------------------------------------------------------
	// Register PADRIVER	PA DRIVER GENERAL				Antx	0x51
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_PADRIVER,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY__ISMSK
			& WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_PADRIVER_CTL_STBY_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_PADRIVER_STBY_PAD_0__SVAL); 				// disable manually PAD (via STBY)

	// TPC and TX MIXER back to regular mode
	//--------------------------------------------------------------------------
	// Register TPCMIXER	TPC MIXER						Antx	0x52
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCMIXER,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC__ISMSK
			& WRX654_RF_ANTENNA_FCSI_TPCMIXER_DISABLE_RXON_CONTROL__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TPCMIXER_CTL_EN_TPC_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_EN_TPC_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_TPCMIXER_DISABLE_RXON_CONTROL_0__SVAL); 	// return to RXON-controlled TXMIXER VCM connection
}


void rfic_start_dc_clock (uint8_t AntMsk)
// start clock of DC state machine
{
	//--------------------------------------------------------------------------
	// Register          CMOSCLKTREE - CMOSCLKTREE + ABBLDO		Central	0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE,
			WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_DCO_CLK_ANT0__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_DCO_CLK_ANT1__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_DCO_CLK_ANT2__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_DCO_CLK_ANT3__ISMSK,
			(AntMsk & 15) << WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_DCO_CLK_ANT0__POS);
}

void rfic_stop_dc_clock (uint8_t AntMsk)
// stop clock of DC state machine
{
	//--------------------------------------------------------------------------
	// Register          CMOSCLKTREE - CMOSCLKTREE + ABBLDO		Central 0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE,
			0xffff - (AntMsk << WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_DCO_CLK_ANT0__POS),
			0x0);
}

void rfic_start_dcramp (uint8_t AntMsk)
// start DC ramp
{
	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_I_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_Q_1__SVAL);
}

void rfic_stop_dcramp (uint8_t AntMsk)
// stop DC ramp
{
	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_I_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_TEST_DCO_Q_0__SVAL);
}

void rfic_start_dccancel (uint8_t AntMsk)
// start DC offset cancellation
{
	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q_1__SVAL);
}

void rfic_stop_dccancel (uint8_t AntMsk)
// stop DC offset cancellation
{
	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_I_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_EN_DCO_Q_0__SVAL);
}

uint8_t rfic_check_dccancel (uint8_t AntMsk)
// check if DC offset cancellation has finished (for all selected antennas, both I and Q)
{
	uint8_t 	DcStatus;
	uint8_t 	Ant;
	uint16_t    Data[5];

	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_Read(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL, Data);

	DcStatus = 1;
	for (Ant=0; Ant < 4; Ant++)
	{
		if (AntMsk & 1 << Ant)
		{
			DcStatus = DcStatus & ((Data[Ant] & 1 << WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RDY2START_DCO_I__POS) > 0)
					& ((Data[Ant] & 1 << WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RDY2START_DCO_Q__POS) > 0);
		}
	}

	return DcStatus;
}

void rfic_reset_dccancel (uint8_t AntMsk)
// reset DC offset cancellation
{
	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I_0__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q_0__SVAL);

	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_I_1__SVAL
			| WRX654_RF_ANTENNA_FCSI_RXDCOCTL_RSTN_DCO_Q_1__SVAL);
}

void rfic_set_dcintegrator(uint8_t AntMsk, uint8_t IntBW)
// set DC state machine integration bandwidth
{
	//--------------------------------------------------------------------------
	// Register RXDCOCTL - RX MIXER DC OFFSET CONTROL 			Antx	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOCTL,
			WRX654_RF_ANTENNA_FCSI_RXDCOCTL_INT_BWIDTH_DCO_I__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOCTL_INT_BWIDTH_DCO_Q__ISMSK,
			((IntBW-10) & WRX654_RF_ANTENNA_FCSI_RXDCOCTL_INT_BWIDTH_DCO_I__MSK) << WRX654_RF_ANTENNA_FCSI_RXDCOCTL_INT_BWIDTH_DCO_I__POS
			| ((IntBW-10) & WRX654_RF_ANTENNA_FCSI_RXDCOCTL_INT_BWIDTH_DCO_Q__MSK) << WRX654_RF_ANTENNA_FCSI_RXDCOCTL_INT_BWIDTH_DCO_Q__POS);
}


void rfic_enable_rxdc_man (uint8_t AntMsk)
// enable manual RX DC offset DAC setting
{
	//--------------------------------------------------------------------------
	// Register RXDCOMAN - RX MIXER DC OFFSET CONTROL MANUAL 	Antx	0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOMAN,
			WRX654_RF_ANTENNA_FCSI_RXDCOMAN_FORCE_DAC0__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOMAN_FORCE_DAC0_1__SVAL);
}


void rfic_disable_rxdc_man (uint8_t AntMsk)
// enable manual RX DC offset DAC setting
{
	//--------------------------------------------------------------------------
	// Register RXDCOMAN - RX MIXER DC OFFSET CONTROL MANUAL	Antx	0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOMAN,
			WRX654_RF_ANTENNA_FCSI_RXDCOMAN_FORCE_DAC0__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXDCOMAN_FORCE_DAC0_0__SVAL);
}


void rfic_set_rxdcoffset (uint8_t AntMsk, int8_t dacI, int8_t dacQ)
// set RX DC offset DAC values
{
	if (dacI<0)
	{
		dacI = dacI+64;
	}
	if (dacQ<0)
	{
		dacQ = dacQ+64;
	}
	//--------------------------------------------------------------------------
	// Register RXDCOMAN - RX MIXER DC OFFSET CONTROL MANUAL	Antx	0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXDCOMAN,
			WRX654_RF_ANTENNA_FCSI_RXDCOMAN_I_OFFSET_DAC0__ISMSK
			& WRX654_RF_ANTENNA_FCSI_RXDCOMAN_Q_OFFSET_DAC0__ISMSK,
			(dacI & WRX654_RF_ANTENNA_FCSI_RXDCOMAN_I_OFFSET_DAC0__MSK) << WRX654_RF_ANTENNA_FCSI_RXDCOMAN_I_OFFSET_DAC0__POS
			| (dacQ & WRX654_RF_ANTENNA_FCSI_RXDCOMAN_Q_OFFSET_DAC0__MSK) << WRX654_RF_ANTENNA_FCSI_RXDCOMAN_Q_OFFSET_DAC0__POS);
}

void rfic_set_rxdcoffset_reg (uint8_t AntMsk, uint8_t rfrxgainidx, uint8_t rxdcidx, uint16_t dcReg)
// set RX DC offset DAC values for specific antenna, RF gain, and RXDC Channel(unprocessed, i.e. cascaded I+Q and success bits)
{

               if (rfrxgainidx > 5)
                              rfrxgainidx = 5;
               if (rxdcidx > 3)
                              rxdcidx = 3;

               RFIC_FCSI_Write(AntMsk, rxdcidx + 4*rfrxgainidx + WRX654_RF_ANTENNA_FCSI_RXMIXDC00, dcReg);

}

uint16_t rfic_get_rxdcoffset (uint8_t Ant, uint8_t rfrxgainidx, uint8_t rxdcidx)
// get RX DC offset register entries for specific antenna, RF gain, and RXDC Channel(unprocessed, i.e. cascaded I+Q and success bits)
{
	uint16_t	data[5];
	uint16_t	RegVal;

	if (Ant > 3)
		Ant = 3;
	if (rfrxgainidx > 5)
		rfrxgainidx = 5;
	if (rxdcidx > 3)
		rxdcidx = 3;

		RFIC_FCSI_Read(CENTRAL, rxdcidx + 4*rfrxgainidx + WRX654_RF_ANTENNA_FCSI_RXMIXDC00, &data[0]);
		RegVal = data[Ant];

		return RegVal;
}


void rfic_set_txdcoffset (uint8_t AntMsk, int8_t dacI, int8_t dacQ)
// set TX DC offset DAC values (for all configurations the same)
{
	if (dacI<0)
	{
		dacI = dacI+16;
	}
	if (dacQ<0)
	{
		dacQ = dacQ+16;
	}
	//--------------------------------------------------------------------------
	// Register TPCOFF01 - TPC Offset 0/1					Antx	0x56
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCOFF01,
			(dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__POS
			| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__POS
			| (dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__POS
			| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__POS);

	//--------------------------------------------------------------------------
	// Register TPCOFF23 - TPC Offset 2/3					Antx	0x57
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCOFF23,
			(dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__POS
			| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__POS
			| (dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__POS
			| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__POS);
}

void rfic_set_txdc_tpccfg (uint8_t AntMsk, int8_t dacI, int8_t dacQ, uint8_t TpcCfg)
// set TX DC offset DAC values (for one specific TPC configuration)
{
	if (dacI<0)
	{
		dacI = dacI+16;
	}
	if (dacQ<0)
	{
		dacQ = dacQ+16;
	}
	if (TpcCfg == 0)
		//--------------------------------------------------------------------------
		// Register TPCOFF01 - TPC Offset 0/1					Antx	0x56
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCOFF01,
				WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__ISMSK
				& WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__ISMSK,
				(dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__POS
				| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__POS);
	else if (TpcCfg == 1)
		//--------------------------------------------------------------------------
		// Register TPCOFF01 - TPC Offset 0/1					Antx	0x56
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCOFF01,
				WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__ISMSK
				& WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__ISMSK,
				(dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__POS
				| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__POS);
	else if (TpcCfg == 2)
		//--------------------------------------------------------------------------
		// Register TPCOFF23 - TPC Offset 2/3					Antx	0x57
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCOFF23,
				WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__ISMSK
				& WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__ISMSK,
				(dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__POS
				| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__POS);
	else if (TpcCfg == 3)
		//--------------------------------------------------------------------------
		// Register TPCOFF23 - TPC Offset 2/3					Antx	0x57
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCOFF23,
				WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__ISMSK
				& WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__ISMSK,
				(dacI & WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__POS
				| (dacQ & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__MSK) << WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__POS);

}


// ########################################################################
// #                                BBO LDO                               #
// ########################################################################

void rfic_enable_1v2bboldo(void)
{
        //--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
        RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
		               WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO__ISMSK,
		               WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_1__SVAL);
}

void rfic_disable_1v2bboldo(void)
{
        //--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
        RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
		               WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO__ISMSK,
		               WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_LDO_0__SVAL);
}

void rfic_enable_bypass_1v2bboldo(void)
{
        //--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
	                 WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BYPASS__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BYPASS_1__SVAL);

}

void rfic_disable_bypass_1v2bboldo(void)
{
        //--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
	                 WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BYPASS__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_BYPASS_0__SVAL);

}

void rfic_set_bbioldo_ctl_vout (int step)
{
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT__POS);
}


void rfic_set_bbioldo_vout_vcmbuffer (int step)
{
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_VCMBUFFER__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT_VCMBUFFER__POS);
}

void rfic_set_bbioldo_ctl_comp (int step)
{
	//--------------------------------------------------------------------------
	// Register BBIOLDO1V2	BBIO LDO1V2 + BIAS				Central	0x6b
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
			WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_COMP__POS);
}


// ########################################################################
// #                               FCSI LDO                               #
// ########################################################################
void rfic_set_fcsildo_ctl_vout (int step)
{
	//--------------------------------------------------------------------------
	// Register FCSILDO1V2	FCSI LDO1V2						Central	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FCSILDO1V2,
			WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_CTL_VOUT__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_CTL_VOUT__POS);
}


// ########################################################################
// #                                ABB LDO                               #
// ########################################################################
void rfic_set_abbldo_ctl_rdg_prog(int step)
{
	//--------------------------------------------------------------------------
	// Register FCSILDO1V2	FCSI LDO1V2						Central	0x66
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FCSILDO1V2,
			WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_CTL_RDG_PROG__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_CTL_RDG_PROG__POS);
}


// ########################################################################
// #                             LO TREE LDO                              #
// ########################################################################
void rfic_set_locenthb_ctl_vout(uint8_t AntMsk ,int step)
{
	//--------------------------------------------------------------------------
	// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
			WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT__POS);
}

void rfic_set_locentlb_ctl_vout(uint8_t AntMsk ,int step)
{
	//--------------------------------------------------------------------------
	// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
			WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT__POS);
}

void rfic_enable_lotreeldo(uint8_t AntMsk, uint8_t HbLb)
{
	if (HbLb == 0)
		//--------------------------------------------------------------------------
		// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_1__SVAL);
	else
		//--------------------------------------------------------------------------
		// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_1__SVAL);
}

void rfic_disable_lotreeldo(uint8_t AntMsk, uint8_t HbLb)
{
	if (HbLb == 0)
		//--------------------------------------------------------------------------
		// Register LOCENTLB	LO CENTRAL LB					Antx	0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_0__SVAL);
	else
		//--------------------------------------------------------------------------
		// Register LOCENTHB	LO CENTRAL HB					Antx	0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_0__SVAL);
}

// ########################################################################
// #                               FSYS LDO                               #
// ########################################################################
void rfic_set_fsysldo_ctl_vout (int step)
{
	//--------------------------------------------------------------------------
	// Register FSYSLDO2V5	FSYSLDO2V5						Central	0x65
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5,
			WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_CTL_VOUT__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_CTL_VOUT__POS);
}

void rfic_set_fsysldo_ctl_iref (int step)
{
	//--------------------------------------------------------------------------
	// Register FSYSLDO2V5	FSYSLDO2V5						Central	0x65
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5,
			WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_CTL_IREF__ISMSK,
			step << WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_CTL_IREF__POS);
}


// ########################################################################
// #                                BB LDO                                #
// ########################################################################
void rfic_enable_1v2bbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_1__SVAL);
}

void rfic_disable_1v2bbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_ENABLE_0__SVAL);
}

void rfic_enable_bypass_1v2bbldo (uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_BYPASS_1__SVAL);
}

void rfic_disable_bypass_1v2bbldo (uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_BYPASS_0__SVAL);
}


void rfic_enable_rload_1v2bbldo (uint8_t AntMsk)
// RLOAD Activation
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_RLOAD_1__SVAL);
}

void rfic_disable_rload_1v2bbldo (uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_EN_RLOAD_0__SVAL);
}

void rfic_force_ldo_ready_1v2bbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_FORCE_LDO_RDY__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_FORCE_LDO_RDY_1__SVAL);
}

void rfic_ctl_ldo_ready_1v2bbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_LDO_RDY__ISMSK,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_LDO_RDY_1__SVAL);
}


void rfic_set_ctl_vout_1v2bbldo(uint8_t AntMsk, int8_t step)
// Output Voltage Control
{
	//--------------------------------------------------------------------------
	// Register BBLDO1V2	RF LDO1V2 BB					Antx	0x47
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT__POS);
}


// ########################################################################
// #                               2V5 LDO                                #
// ########################################################################
void rfic_enable_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE_1__SVAL);
}

void rfic_disable_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_ENABLE_0__SVAL);
}

void rfic_enable_rload_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_RLOAD_1__SVAL);
}

void rfic_disable_rload_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_RLOAD_0__SVAL);
}

void rfic_enable_bypass_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_BYPASS_1__SVAL);
}

void rfic_disable_bypass_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_BYPASS_0__SVAL);
}


void rfic_set_ctl_vout_2v5ldo(uint8_t AntMsk, int8_t step)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_VOUT__POS);
}

void rfic_set_ctl_lpgain_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_LPGAIN__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_LPGAIN_1__SVAL);
}

void rfic_set_ctl_lpgain_off_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_LPGAIN__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_LPGAIN_0__SVAL);
}

void rfic_disable_lc_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC_0__SVAL);
}

void rfic_enable_lc_2v5ldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RFLDO2V5	RF LDO2V5						Antx	0x48
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RFLDO2V5_EN_LC_1__SVAL);
}


// ########################################################################
// #                                VDDMON CENTRAL                        #
// ########################################################################

void rfic_enable_vddmon_central (void)
// ENABLE CENTRAL VDDMON
{
        //--------------------------------------------------------------------------
	// Register CONFIG0	   CONFIG 0     			Central	0x69
	//--------------------------------------------------------------------------
 RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CONFIG0,
		        WRX654_RF_CENTRAL_FCSI_CONFIG0_EN_LDOMON__ISMSK,
	                WRX654_RF_CENTRAL_FCSI_CONFIG0_EN_LDOMON_1__SVAL);


}

void rfic_disable_vddmon_central (void)
// DISABLE CENTRAL VDDMON
{
        //--------------------------------------------------------------------------
	// Register CONFIG0	   CONFIG 0     			Central	0x69
	//--------------------------------------------------------------------------
 RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CONFIG0,
		        WRX654_RF_CENTRAL_FCSI_CONFIG0_EN_LDOMON__ISMSK,
	                WRX654_RF_CENTRAL_FCSI_CONFIG0_EN_LDOMON_0__SVAL);


}

void rfic_enable_vddmon_fsysldo (void)
// ENABLE FSYSLDO TEST
{
        //--------------------------------------------------------------------------
	// Register FSYSLDO2V5	   FSYS LDO 2V5 			Central	0x65
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5,
		         WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_EN_TEST__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_EN_TEST_1__SVAL);
}

void rfic_disable_vddmon_fsysldo (void)
// DISABLE FSYSLDO TESTING
{
        //--------------------------------------------------------------------------
	// Register FSYSLDO2V5	   FSYS LDO 2V5 			Central	0x65
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5,
		         WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_EN_TEST__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_EN_TEST_0__SVAL);
}

void rfic_enable_vddmon_bbioldo (void)
// ENABLE BBIO LDO TESTING
{
        //--------------------------------------------------------------------------
	// Register BBIOLDO1V2	   BBIO LDO 1V2 			Central	0x6B
	//--------------------------------------------------------------------------
 RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
                        WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_TEST__ISMSK,
	                WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_TEST_1__SVAL);
}

void rfic_disable_vddmon_bbioldo (void)
// DISABLE BBIO LDO TESTING
{
        //--------------------------------------------------------------------------
	// Register BBIOLDO1V2	   BBIO LDO 1V2 			Central	0x6B
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
                         WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_TEST__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_EN_TEST_0__SVAL);
}


void rfic_enable_vddmon_fcsildo (void)
// ENABLE FCSI LDO TESTING
{
        //--------------------------------------------------------------------------
	// Register FCSILDO1V2	   FCSI LDO 1V2 			Central	0x66
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FCSILDO1V2,
                         WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_EN_TEST__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_EN_TEST_1__SVAL);

}

void rfic_disable_vddmon_fcsildo (void)
// DISABLE FCSI LDO TESTING
{
        //--------------------------------------------------------------------------
	// Register FCSILDO1V2	   FCSI LDO 1V2 			Central	0x66
	//--------------------------------------------------------------------------
  RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FCSILDO1V2,
                         WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_EN_TEST__ISMSK,
	                 WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_EN_TEST_0__SVAL);
}

void rfic_enable_vddmon_lotreeldo (uint8_t AntMsk, uint8_t HbLb)
{
	if (HbLb == 0)
		//--------------------------------------------------------------------------
		// Register LOCENTLB	   LOCENT LB 			        Antenna  0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_TEST__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_TEST_1__SVAL);
	else
		//--------------------------------------------------------------------------
		// Register LOCENTHB	   LOCENT HB 			        Antenna  0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_TEST__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_TEST_1__SVAL);
}

void rfic_disable_vddmon_lotreeldo (uint8_t AntMsk, uint8_t HbLb)
{
	if (HbLb == 0)
		//--------------------------------------------------------------------------
		// Register LOCENTLB	   LOCENT LB 			        Antenna  0x40
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_TEST__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTLB_ENABLE_LBLDO_TEST_0__SVAL);
	else
		//--------------------------------------------------------------------------
		// Register LOCENTHB	   LOCENT HB 			        Antenna  0x41
		//--------------------------------------------------------------------------
		RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_TEST__ISMSK,
				WRX654_RF_ANTENNA_FCSI_LOCENTHB_ENABLE_HBLDO_TEST_0__SVAL);
}


// ########################################################################
// #                                VDDMON ANTENNA                        #
// ########################################################################
void rfic_disable_vddmon(uint8_t AntMsk)
// Disable VDDMON
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			0x0000);
}

void rfic_enable_vddmon_1v2bbldo(uint8_t AntMsk)
// VDDMON Configuration f.1V2  BBLDO Test (ldo_1v5to1v26_bb)
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_BBLDO_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL);
}

void rfic_enable_vddmon_2v5ldo(uint8_t AntMsk)
// VDDMON Configuration f. 2V5LDO Test (ldo_3v3to2v5_top)
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_2V5LDO_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL);
}

void rfic_enable_vddmon_hbldo(uint8_t AntMsk)
// VDDMON Configuration f. HIGH Band LDO Test (ldo_1v5to1v26_5G)
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_HBLDO_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL);
}

void rfic_enable_vddmon_lbldo(uint8_t AntMsk)
// VDDMON Configuration f. LOW Band LDO Test (ldo_1v5to1v26_2G)
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_LBLDO_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL);
}

void rfic_enable_vddmon_1v2rxldo(uint8_t AntMsk)
// VDDMON Configuration f. RX LDO Test  (ldo_1v5to1v26_rx)
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_RXLDO_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL);
}

void rfic_enable_vddmon_1v5ldoin(uint8_t AntMsk)
// VDDMON Configuration f. 1V5 LDO in Test
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_1V5LDO_IN_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_1V5LDO_CURRENT_1__SVAL);
}

void rfic_enable_vddmon_3v3ldoin(uint8_t AntMsk)
// VDDMON Configuration f. 3V3 LDO in Test
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_3V3LDO_IN_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_3V3LDO_CURRENT_1__SVAL);
}

void rfic_enable_vddmon_tssiin(uint8_t AntMsk)
// VDDMON Configuration f. TSSI in
{
	//--------------------------------------------------------------------------
	// Register VDDMON		VDD MONITOR   					Antx	0x5e
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(AntMsk, WRX654_RF_ANTENNA_FCSI_VDDMON,
			WRX654_RF_ANTENNA_FCSI_VDDMON_CTL_TSSI_IN_1__SVAL
			|WRX654_RF_ANTENNA_FCSI_VDDMON_EN_OPAMPS_1__SVAL);
}


// ########################################################################
// #                                ANAMUX                                #
// ########################################################################
void rfic_enable_anamux(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SSI 		SSI   							Antx	0x5d
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX_1__SVAL);

}

void rfic_disable_anamux(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SSI 		SSI   							Antx	0x5d
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX_0__SVAL);

}

void rfic_enable_anamux_vddmonin(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SSI 		SSI   							Antx	0x5d
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SUPPLYMON_IN__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX__ISMSK
            & WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SUPPLYMON_IN_1__SVAL                  // activate VDDMON input
			| WRX654_RF_ANTENNA_FCSI_SSI_ENABLE_ANAMUX_1__SVAL                  // activate ANAMUX (if not already on)
            | WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN_0__SVAL);                    // deactivate SSI input
}

void rfic_disable_anamux_vddmonin(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register SSI 		SSI   							Antx	0x5d
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_SSI,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SUPPLYMON_IN__ISMSK
			& WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN__ISMSK,
			WRX654_RF_ANTENNA_FCSI_SSI_EN_SUPPLYMON_IN_0__SVAL                  // deactivate VDDMON input
			| WRX654_RF_ANTENNA_FCSI_SSI_EN_SSI_IN_1__SVAL);                    // re-activate SSI input
}


// ########################################################################
// #                                RX LDO                                #
// ########################################################################
void rfic_enable_rxldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE_1__SVAL);
}

void rfic_disable_rxldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_ENABLE_0__SVAL);
}

void rfic_enable_rload_rxldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_RLOAD_1__SVAL);
}

void rfic_disable_rload_rxldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_RLOAD_0__SVAL);
}

void rfic_enable_bypass_rxldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_BYPASS_1__SVAL);
}

void rfic_disable_bypass_rxldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_EN_BYPASS_0__SVAL);
}

void rfic_set_ctl_vout_rxldo(uint8_t AntMsk, int8_t step)
{
	//--------------------------------------------------------------------------
	// Register RXLDO1V2	RXLDO1V2   						Antx	0x46
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
			WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT__POS);
}


// ########################################################################
// #                              TX HB LDO                               #
// ########################################################################
void rfic_enable_hbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_1__SVAL);
}

void rfic_disable_hbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_ENABLE_0__SVAL);
}

void rfic_enable_rload_hbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_RLOAD_1__SVAL);
}

void rfic_disable_rload_hbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_RLOAD_0__SVAL);
}

void rfic_enable_bypass_hbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_BYPASS_1__SVAL);
}

void rfic_disable_bypass_hbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_EN_BYPASS_0__SVAL);
}

void rfic_set_ctl_vout_hbldo(uint8_t AntMsk, int8_t step)
{
	//--------------------------------------------------------------------------
	// Register TXHBLDO1V2	TXHBLDO1V2   					Antx	0x44
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT__POS);
}


// ########################################################################
// #                              TX LB LDO                               #
// ########################################################################
void rfic_enable_lbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_1__SVAL);
}

void rfic_disable_lbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_ENABLE_0__SVAL);
}

void rfic_enable_rload_lbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_RLOAD_1__SVAL);
}

void rfic_disable_rload_lbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_RLOAD__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_RLOAD_0__SVAL);
}

void rfic_enable_bypass_lbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_BYPASS_1__SVAL);
}

void rfic_disable_bypass_lbldo(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_BYPASS__ISMSK,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_EN_BYPASS_0__SVAL);
}

void rfic_set_ctl_vout_lbldo(uint8_t AntMsk, int8_t step)
{
	//--------------------------------------------------------------------------
	// Register TXLBLDO1V2	TXLBLDO1V2   					Antx	0x45
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
			WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT__ISMSK,
			step << WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT__POS);
}


// ########################################################################
// #                          TEMPERATURE SENSOR                          #
// ########################################################################
void rfic_enable_tempsensor (void)
{
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central 0x6D
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_1__SVAL);
}

void rfic_disable_tempsensor (void)
{
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central 0x6D
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_0__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_0__SVAL);
}

void rfic_start_tempsensor_clock (void)
{
	//--------------------------------------------------------------------------
	// Register	CMOSCLKTREE	CMOSCLKTREE + ABBLDO			Central	0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE,
			WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_TEMPSENSOR_CLK__ISMSK,
			WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_TEMPSENSOR_CLK_1__SVAL);
}

void rfic_stop_tempsensor_clock (void)
{
	//--------------------------------------------------------------------------
	// Register CMOSCLKTREE CMOSCLKTREE + ABBLDO			Central	0x67
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE,
			WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_TEMPSENSOR_CLK__ISMSK,
			WRX654_RF_CENTRAL_FCSI_CMOSCLKTREE_EN_TEMPSENSOR_CLK_0__SVAL);
}

void rfic_start_tempsensor_cal (void)
{
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central 0x6D
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_VCAL_1__SVAL);
}

void rfic_stop_tempsensor_cal (void)
{
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central 0x6D
	//--------------------------------------------------------------------------
	RFIC_FCSI_Write(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_FLP_CMP_SWAP_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_DTS_1__SVAL
			| WRX654_RF_CENTRAL_FCSI_TEMPSENS1_EN_VCAL_0__SVAL);
}

void rfic_start_tempsensor (void)
{
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 1					Central 0x6D
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_START__ISMSK,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_START_0__SVAL);

	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_START__ISMSK,
			WRX654_RF_CENTRAL_FCSI_TEMPSENS1_START_1__SVAL);
}

uint16_t rfic_read_tempsensor (void)
{
	//--------------------------------------------------------------------------
	// Register	TEMPSENS1	TEMP SENSOR 0					Central 0x6C
	//--------------------------------------------------------------------------
	uint16_t               data[5];
	RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS0, data);

	return (data[4] & WRX654_RF_CENTRAL_FCSI_TEMPSENS0_RESULT__MSK);
}

void rfic_set_bgp_trim (int TuneVal)
{
	if (TuneVal<0)
	{
		TuneVal = TuneVal+16;
	}
	//--------------------------------------------------------------------------
	// Register	BIASCENT	CENTRAL BIAS					Central 0x68
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BIASCENT,
			WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_TRIM_BGP__ISMSK,
			(TuneVal & WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_TRIM_BGP__MSK) << WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_TRIM_BGP__POS);
}

void rfic_set_iconst_trim (int TuneVal)
{
	//--------------------------------------------------------------------------
	// Register	BIASCENT	CENTRAL BIAS					Central 0x68
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BIASCENT,
			WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_CAL_ICONST__ISMSK,
			(TuneVal & WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_CAL_ICONST__MSK) << WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_CAL_ICONST__POS);
}

void rfic_set_antenna_fectl_man(uint8_t AntMsk, uint8_t fectl)
// set manual antenna FECTLs
{
	//--------------------------------------------------------------------------
	// Register	MANFECTL	MANFECTL PADCTL					Antx	0x4C
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_MANFECTL,
			WRX654_RF_ANTENNA_FCSI_MANFECTL_CTL_MANUEL_FECTL__ISMSK
			& WRX654_RF_ANTENNA_FCSI_MANFECTL_FECTL_GAIN__ISMSK,
			WRX654_RF_ANTENNA_FCSI_MANFECTL_CTL_MANUEL_FECTL_1__SVAL          // enable manual FECTL setting
			| (fectl & WRX654_RF_ANTENNA_FCSI_MANFECTL_FECTL_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_MANFECTL_FECTL_GAIN__POS);
}

void rfic_disable_antenna_fectl(uint8_t AntMsk)
{
	//--------------------------------------------------------------------------
	// Register	MANFECTL	MANFECTL PADCTL					Antx	0x4C
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(AntMsk, WRX654_RF_ANTENNA_FCSI_MANFECTL,
			WRX654_RF_ANTENNA_FCSI_MANFECTL_CTL_MANUEL_FECTL__ISMSK,
			WRX654_RF_ANTENNA_FCSI_MANFECTL_CTL_MANUEL_FECTL_0__SVAL);       // disable manual FECTL setting

}

void rfic_set_central_fectl_man(uint8_t fectl)
{
	//--------------------------------------------------------------------------
	// Register	SR2			SHORT REGISTER 2				Central 0x02
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_SR2,
			WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C0__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C1__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C2__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C3__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C4__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C5__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C6__ISMSK
			& WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C7__ISMSK,
			fectl << WRX654_RF_CENTRAL_FCSI_SR2_FECTL_C0__POS);
}

// eFuse Programming
void rfic_enable_efuse (void)														// enable eFuse programming
{
	//--------------------------------------------------------------------------
	// Register CONFIG1	CONFIG1								Central	0x6a
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CONFIG1,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_INPUT_FECTL_C01__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_RES_N__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_CLK_WRITE__ISMSK,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_INPUT_FECTL_C01_1__SVAL				// Enable FECTL_C0 and FECTL_C1 as input pad to drive CLK and RES_N from outside
			| WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_RES_N_1__SVAL						// Enable reset_n coming from pad FECTL_1
			| WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_CLK_WRITE_1__SVAL);					// Enable clock coming from pad FECTL_0
}

void rfic_disable_efuse (void)														// stop eFuse programming
{
	//--------------------------------------------------------------------------
	// Register CONFIG1	CONFIG1								Central	0x6a
	//--------------------------------------------------------------------------
	RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_CONFIG1,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_INPUT_FECTL_C01__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_RES_N__ISMSK
			& WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_CLK_WRITE__ISMSK,
			WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_INPUT_FECTL_C01_0__SVAL				// Disable FECTL_C0 and FECTL_C1 as input pad to drive CLK and RES_N from outside
			| WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_RES_N_0__SVAL						// Disable reset_n coming from pad FECTL_1
			| WRX654_RF_CENTRAL_FCSI_CONFIG1_EN_CLK_WRITE_0__SVAL);					// Disable clock coming from pad FECTL_0
}

// eFuse Readout
uint16_t rfic_read_efuse_values(uint16_t SectMsk, uint16_t *softfuse_data)
// read eFuse data and distribute trimming values (still keeping softfuse option)
// SectMsk --> Section Mask (1 - Central registers only, 2 - Antenna registers only, 3 - both)
{

	uint16_t	data[5];
	uint16_t    fuse_data[16];
	uint16_t	efuse_vref;
	uint16_t	efuse_txhbldo1v2[4], efuse_txlbldo1v2[4], efuse_rxldo1v2[4], efuse_bbldo1v2[4];
	uint16_t	efuse_rfldo2v5[4], efuse_hblotreeldo[4], efuse_lblotreeldo[4];
	uint16_t    efuse_bboldo1v2, efuse_fsysldo2v5, efuse_fcsildo1v2;
	uint16_t	efuse_rx_hb_bandsel[3],efuse_rx_lb_bandsel,efuse_tx_hb_bandsel[3],efuse_tx_lb_bandsel;
	uint16_t	efuse_rx_lb_mid2high_gain_ant01, efuse_rx_lb_mid2high_gain_ant23, efuse_rx_hb_mid2high_gain_ant01, efuse_rx_hb_mid2high_gain_ant23;
	uint16_t	efuse_vcal_rsit, efuse_bias;
	uint16_t	efuse_tpc_filter_40MHz, efuse_tpc_filter_80MHz, efuse_tpc_filter_160MHz;
	uint16_t	efuse_mcm;
	uint16_t    reg, ant, version;
	uint16_t	chip_fused = 1;

	efuse_tpc_filter_40MHz = 0;
	efuse_tpc_filter_80MHz = 0;
	efuse_tpc_filter_160MHz = 0;
	efuse_rx_lb_mid2high_gain_ant01 = 0;
	efuse_rx_lb_mid2high_gain_ant23 = 0;
	efuse_rx_lb_bandsel = 0;
	efuse_tx_lb_bandsel = 0;
	efuse_vref = 0;
	efuse_bboldo1v2 = 0;
	efuse_fsysldo2v5 = 0;
	efuse_fcsildo1v2 = 0;
	efuse_vcal_rsit = 0;
	efuse_bias = 0;

	// force to read from softfuse data as long as softfuse_data[15] != 0
	if (softfuse_data[15] == 0)
	{
		for (reg=0; reg<16; reg++)
		{
			RFIC_FCSI_Read(CENTRAL, reg+WRX654_RF_CENTRAL_FCSI_EFUSE0, &data[0]);
			fuse_data[reg] = data[4]; // read from CENTRAL FCSI branch
		}

		if ((fuse_data[15] & (7 << WRX654_RF_CENTRAL_FCSI_EFUSE15_PRODUCTCODE__POS)) == 0)
		{
			chip_fused = 0;
			// check for design step
			RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_VERSION, &data[0]);
			version = ((data[4] >> WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__POS) & WRX654_RF_CENTRAL_FCSI_VERSION_DESIGN_STEP__MSK);
			if ((version == 32) || (version == 33)) // = 0b0100000 or 0b0100001  --> A0 or A1: 01=A 00000=0|00001=1
				chip_fused = 2;
		}
	}
	else // or use external softfuse_data
	{
		for (reg=0; reg<16; reg++)
		{
			fuse_data[reg] = softfuse_data[reg];
// 			fuse_data[reg] = *(softfuse_data+reg);
		}
	}

	if (chip_fused > 0)
	{
		if (chip_fused == 1)
		{
			efuse_vref     		= ((fuse_data[1] >> WRX654_RF_CENTRAL_FCSI_EFUSE1_CTL_TRIM_BGP__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE1_CTL_TRIM_BGP__MSK);

			efuse_txhbldo1v2[0] = ((fuse_data[2] >> WRX654_RF_CENTRAL_FCSI_EFUSE2_HBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE2_HBLDO1V2__MSK);
			efuse_txlbldo1v2[0] = ((fuse_data[2] >> WRX654_RF_CENTRAL_FCSI_EFUSE2_LBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE2_LBLDO1V2__MSK);
			efuse_rxldo1v2[0] 	= ((fuse_data[2] >> WRX654_RF_CENTRAL_FCSI_EFUSE2_RXLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE2_RXLDO1V2__MSK);
			efuse_bbldo1v2[0] 	= ((fuse_data[2] >> WRX654_RF_CENTRAL_FCSI_EFUSE2_BBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE2_BBLDO1V2__MSK);

			efuse_txhbldo1v2[1] = ((fuse_data[3] >> WRX654_RF_CENTRAL_FCSI_EFUSE3_HBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE3_HBLDO1V2__MSK);
			efuse_txlbldo1v2[1] = ((fuse_data[3] >> WRX654_RF_CENTRAL_FCSI_EFUSE3_LBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE3_LBLDO1V2__MSK);
			efuse_rxldo1v2[1] 	= ((fuse_data[3] >> WRX654_RF_CENTRAL_FCSI_EFUSE3_RXLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE3_RXLDO1V2__MSK);
			efuse_bbldo1v2[1] 	= ((fuse_data[3] >> WRX654_RF_CENTRAL_FCSI_EFUSE3_BBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE3_BBLDO1V2__MSK);

			efuse_txhbldo1v2[2] = ((fuse_data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE4_HBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE4_HBLDO1V2__MSK);
			efuse_txlbldo1v2[2] = ((fuse_data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE4_LBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE4_LBLDO1V2__MSK);
			efuse_rxldo1v2[2] 	= ((fuse_data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE4_RXLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE4_RXLDO1V2__MSK);
			efuse_bbldo1v2[2] 	= ((fuse_data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE4_BBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE4_BBLDO1V2__MSK);

			efuse_txhbldo1v2[3] = ((fuse_data[5] >> WRX654_RF_CENTRAL_FCSI_EFUSE5_HBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE5_HBLDO1V2__MSK);
			efuse_txlbldo1v2[3] = ((fuse_data[5] >> WRX654_RF_CENTRAL_FCSI_EFUSE5_LBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE5_LBLDO1V2__MSK);
			efuse_rxldo1v2[3] 	= ((fuse_data[5] >> WRX654_RF_CENTRAL_FCSI_EFUSE5_RXLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE5_RXLDO1V2__MSK);
			efuse_bbldo1v2[3] 	= ((fuse_data[5] >> WRX654_RF_CENTRAL_FCSI_EFUSE5_BBLDO1V2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE5_BBLDO1V2__MSK);

			efuse_rfldo2v5[0] 	= ((fuse_data[6] >> WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT0__MSK);
			efuse_rfldo2v5[1] 	= ((fuse_data[6] >> WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT1__MSK);
			efuse_rfldo2v5[2] 	= ((fuse_data[6] >> WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT2__MSK);
			efuse_rfldo2v5[3] 	= ((fuse_data[6] >> WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT3__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE6_LDO2V5ANT3__MSK);

			efuse_hblotreeldo[0] = ((fuse_data[7] >> WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT0__MSK);
			efuse_hblotreeldo[1] = ((fuse_data[7] >> WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT1__MSK);
			efuse_hblotreeldo[2] = ((fuse_data[7] >> WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT2__MSK);
			efuse_hblotreeldo[3] = ((fuse_data[7] >> WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT3__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE7_LOTREELDOANT3__MSK);

			efuse_lblotreeldo[0] = ((fuse_data[8] >> WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT0__MSK);
			efuse_lblotreeldo[1] = ((fuse_data[8] >> WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT1__MSK);
			efuse_lblotreeldo[2] = ((fuse_data[8] >> WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT2__MSK);
			efuse_lblotreeldo[3] = ((fuse_data[8] >> WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT3__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE8_LOTREELDOANT3__MSK);

			efuse_bboldo1v2 	= ((fuse_data[9] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_CTL_VOUT_BBOLDO__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_CTL_VOUT_BBOLDO__MSK);
			efuse_fsysldo2v5 	= ((fuse_data[9] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_CTL_VOUT_FSYSLDO__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_CTL_VOUT_FSYSLDO__MSK);
			efuse_fcsildo1v2 	= ((fuse_data[9] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_CTL_VOUT_FCSILDO__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_CTL_VOUT_FCSILDO__MSK);

			efuse_rx_lb_mid2high_gain_ant01 = ((fuse_data[9] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__MSK) + MID2HIGHSTEP_LB_NOM;
			if (efuse_rx_lb_mid2high_gain_ant01 > 1 + MID2HIGHSTEP_LB_NOM)
				efuse_rx_lb_mid2high_gain_ant01 = efuse_rx_lb_mid2high_gain_ant01 - 4;
			efuse_rx_lb_mid2high_gain_ant23 = ((fuse_data[9] >> WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE9_LB_LNA_MID2HIGH_ANT01__MSK) + MID2HIGHSTEP_LB_NOM;
			if (efuse_rx_lb_mid2high_gain_ant23 > 1 + MID2HIGHSTEP_LB_NOM)
				efuse_rx_lb_mid2high_gain_ant23 = efuse_rx_lb_mid2high_gain_ant23 - 4;
			efuse_rx_hb_mid2high_gain_ant01 = ((fuse_data[10] >> WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT01__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT01__MSK) + MID2HIGHSTEP_HB_NOM;
			if (efuse_rx_hb_mid2high_gain_ant01 > 1 + MID2HIGHSTEP_HB_NOM)
				efuse_rx_hb_mid2high_gain_ant01 = efuse_rx_hb_mid2high_gain_ant01 - 4;
			efuse_rx_hb_mid2high_gain_ant23 = ((fuse_data[10] >> WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT23__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE10_HB_LNA_MID2HIGH_ANT23__MSK) + MID2HIGHSTEP_HB_NOM;
			if (efuse_rx_hb_mid2high_gain_ant23 > 1 + MID2HIGHSTEP_HB_NOM)
				efuse_rx_hb_mid2high_gain_ant23 = efuse_rx_hb_mid2high_gain_ant23 - 4;

			efuse_bias = ((fuse_data[10] >> WRX654_RF_CENTRAL_FCSI_EFUSE10_CTL_CAL_ICONST__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE10_CTL_CAL_ICONST__MSK);

			efuse_tpc_filter_40MHz = ((fuse_data[10] >> WRX654_RF_CENTRAL_FCSI_EFUSE10_CTL_TPC_FB_CSEL_40MHZ__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE10_CTL_TPC_FB_CSEL_40MHZ__MSK);

			efuse_vcal_rsit = ((fuse_data[11] >> WRX654_RF_CENTRAL_FCSI_EFUSE11_PROCESSCAL__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE11_PROCESSCAL__MSK);

			efuse_rx_hb_bandsel[0] = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH0__MSK) + RX_HB_BAND_CH0_NOM;
			if (efuse_rx_hb_bandsel[0] > 1 + RX_HB_BAND_CH0_NOM)
				efuse_rx_hb_bandsel[0] = efuse_rx_hb_bandsel[0] - 4;
			efuse_rx_hb_bandsel[1] = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH1__MSK) + RX_HB_BAND_CH1_NOM;
			if (efuse_rx_hb_bandsel[1] > 1 + RX_HB_BAND_CH1_NOM)
				efuse_rx_hb_bandsel[1] = efuse_rx_hb_bandsel[1] - 4;
			efuse_rx_hb_bandsel[2] = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXHBBANDSELECTION_CH2__MSK) + RX_HB_BAND_CH2_NOM;
			if (efuse_rx_hb_bandsel[2] > 1 + RX_HB_BAND_CH2_NOM)
				efuse_rx_hb_bandsel[2] = efuse_rx_hb_bandsel[2] - 4;
			efuse_tx_hb_bandsel[0] = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH0__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH0__MSK) + TX_HB_BAND_CH0_NOM;
			if (efuse_tx_hb_bandsel[0] > 1 + TX_HB_BAND_CH0_NOM)
				efuse_tx_hb_bandsel[0] = efuse_tx_hb_bandsel[0] - 4;
			efuse_tx_hb_bandsel[1] = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH1__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH1__MSK) + TX_HB_BAND_CH1_NOM;
			if (efuse_tx_hb_bandsel[1] > 1 + TX_HB_BAND_CH1_NOM)
				efuse_tx_hb_bandsel[1] = efuse_tx_hb_bandsel[1] - 4;
			efuse_tx_hb_bandsel[2] = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH2__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXHBBANDSELECTION_CH2__MSK) + TX_HB_BAND_CH2_NOM;
			if (efuse_tx_hb_bandsel[2] > 1 + TX_HB_BAND_CH2_NOM)
				efuse_tx_hb_bandsel[2] = efuse_tx_hb_bandsel[2] - 4;
			efuse_rx_lb_bandsel = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_RXLBBANDSELECTION__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_RXLBBANDSELECTION__MSK) + RX_LB_BAND_NOM;
			if (efuse_rx_lb_bandsel > 1 + RX_LB_BAND_NOM)
				efuse_rx_lb_bandsel = efuse_rx_lb_bandsel - 4;
			efuse_tx_lb_bandsel = ((fuse_data[12] >> WRX654_RF_CENTRAL_FCSI_EFUSE12_TXLBBANDSELECTION__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE12_TXLBBANDSELECTION__MSK) + TX_LB_BAND_NOM;
			if (efuse_tx_lb_bandsel > 1 + TX_LB_BAND_NOM)
				efuse_tx_lb_bandsel = efuse_tx_lb_bandsel - 4;

			efuse_tpc_filter_80MHz = ((fuse_data[13] >> WRX654_RF_CENTRAL_FCSI_EFUSE13_CTL_TPC_FB_CSEL_80MHZ__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE13_CTL_TPC_FB_CSEL_80MHZ__MSK);
			efuse_tpc_filter_160MHz = ((fuse_data[13] >> WRX654_RF_CENTRAL_FCSI_EFUSE13_CTL_TPC_FB_CSEL_160MHZ__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE13_CTL_TPC_FB_CSEL_160MHZ__MSK);

			// fall-back to TPC best-guess values in case TPC filter indexes are not fused
			if (efuse_tpc_filter_40MHz == 0)
				efuse_tpc_filter_40MHz 		= 35;	// no data available, take nominal 0d35
			if (efuse_tpc_filter_80MHz == 0)
				efuse_tpc_filter_80MHz 		= 57;	// mean values over antennas: 59, (74.3?), 54.9, (70.3?)
			if (efuse_tpc_filter_160MHz == 0)
				efuse_tpc_filter_160MHz		= 72;	// mean values over antennas: 72,4, 77, 66.9, 71.8

			efuse_mcm = ((fuse_data[15] >> WRX654_RF_CENTRAL_FCSI_EFUSE15_MCM__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE15_MCM__MSK);
		}
		// replacing with A0/A1 default values
		else if (chip_fused == 2)
		{
			// EFUSE1
			efuse_vref             		= 15;	// two's complement: majority at -1 (==15)

			// EFUSE2
			efuse_txhbldo1v2[0] 		= 5;	// Mean according A1 TTT lot: 4.57
			efuse_txlbldo1v2[0] 		= 4;	// Mean according A1 TTT lot: 3.66
			efuse_rxldo1v2[0] 			= 5;	// Mean according A1 TTT lot: 4.48
			efuse_bbldo1v2[0] 			= 4;	// Mean according A1 TTT lot: 3.38

			// EFUSE3
			efuse_txhbldo1v2[1] 		= 5;	// Mean according A1 TTT lot: 4.84
			efuse_txlbldo1v2[1] 		= 4;	// Mean according A1 TTT lot: 4.06
			efuse_rxldo1v2[1] 			= 5;	// Mean according A1 TTT lot: 4.56
			efuse_bbldo1v2[1] 			= 4;	// Mean according A1 TTT lot: 3.48

			// EFUSE4
			efuse_txhbldo1v2[2] 		= 5;	// Mean according A1 TTT lot: 4.67
			efuse_txlbldo1v2[2] 		= 4;	// Mean according A1 TTT lot: 3.77
			efuse_rxldo1v2[2] 			= 5;	// Mean according A1 TTT lot: 4.55
			efuse_bbldo1v2[2] 			= 4;	// Mean according A1 TTT lot: 3.63

			// EFUSE5
			efuse_txhbldo1v2[3] 		= 5;	// Mean according A1 TTT lot: 4.7
			efuse_txlbldo1v2[3] 		= 4;	// Mean according A1 TTT lot: 3.92
			efuse_rxldo1v2[3] 			= 5;	// Mean according A1 TTT lot: 4.32
			efuse_bbldo1v2[3] 			= 4;	// Mean according A1 TTT lot: 3.48

			// EFUSE6
			efuse_rfldo2v5[0] 			= 0;	// two's complement: majority at 0
			efuse_rfldo2v5[1] 			= 0;	// two's complement: majority at 0
			efuse_rfldo2v5[2] 			= 0;	// two's complement: majority at 0
			efuse_rfldo2v5[3] 			= 0;	// two's complement: majority at 0

			// EFUSE7
			efuse_hblotreeldo[0] 		= 0;	// pre-defined
			efuse_hblotreeldo[1] 		= 1;	// pre-defined
			efuse_hblotreeldo[2] 		= 1;	// pre-defined
			efuse_hblotreeldo[3] 		= 2;	// pre-defined

			// EFUSE8
			efuse_lblotreeldo[0] 		= 1;	// pre-defined
			efuse_lblotreeldo[1] 		= 1;	// pre-defined
			efuse_lblotreeldo[2] 		= 1;	// pre-defined
			efuse_lblotreeldo[3] 		= 1;	// pre-defined

			// EFUSE9
			efuse_bboldo1v2 			= 1;	// Mean according A1 TTT lot: 1.15
			efuse_fsysldo2v5 			= 2;	// Mean according A1 TTT lot: 2.45
			efuse_fcsildo1v2 			= 0;	// two's complement: majority at 0

			// EFUSE9 & 10
			efuse_rx_lb_mid2high_gain_ant01 = (uint16_t) (1 + MID2HIGHSTEP_LB_NOM);	// Mean according A1 TTT lot: 1.0
			efuse_rx_lb_mid2high_gain_ant23 = (uint16_t) (1 + MID2HIGHSTEP_LB_NOM);	// Mean according A1 TTT lot: 1.0
			efuse_rx_hb_mid2high_gain_ant01 = (uint16_t) (2 + MID2HIGHSTEP_HB_NOM);	// Mean according A1 TTT lot: 1.99
			efuse_rx_hb_mid2high_gain_ant23 = (uint16_t) (2 + MID2HIGHSTEP_HB_NOM);	// Mean according A1 TTT lot: 2.0

			// EFUSE10 & 11
			efuse_bias 					= 14;	// Mean according A1 TTT lot: 14.3
			efuse_vcal_rsit 			= 20;	// Mean according A1 TTT lot: 20.9 (only even values allowed? --> ask Harald)

			// EFUSE10 & 13
			efuse_tpc_filter_40MHz 		= 35;	// no data available, take nominal 0d35
			efuse_tpc_filter_80MHz 		= 57;	// mean values over antennas: 59, (74.3?), 54.9, (70.3?)
			efuse_tpc_filter_160MHz		= 72;	// mean values over antennas: 72,4, 77, 66.9, 71.8

			// EFUSE12
			efuse_rx_hb_bandsel[0] 		= (uint16_t) (-1 + RX_HB_BAND_CH0_NOM);	// absolute mean values over antennas: 4.0, 5.69, 5.11, 4.27 (== -1 average delta from nom 6)
			efuse_rx_hb_bandsel[1] 		= (uint16_t) (-1 + RX_HB_BAND_CH1_NOM);	// absolute mean values over antennas: 5.09, 5.25, 5.14, 5.15 (== -1 average delta from nom 6)
			efuse_rx_hb_bandsel[2] 		= (uint16_t) (0 + RX_HB_BAND_CH2_NOM);	// absolute mean values over antennas: 6.09, 6.1, 6.03, 6.06 (== 0 average delta from nom 6)
			efuse_tx_hb_bandsel[0] 		= (uint16_t) (-2 + TX_HB_BAND_CH0_NOM);	// absolute mean values over antennas: 2.0, 2.06, 2.04, 2.49 (== -2 average delta from nom 4)
			efuse_tx_hb_bandsel[1] 		= (uint16_t) (-1 + TX_HB_BAND_CH1_NOM);	// absolute mean values over antennas: 3.08, 3.83, 3.08, 3.91 (== -1 average delta from nom 4)
			efuse_tx_hb_bandsel[2] 		= (uint16_t) (+1 + TX_HB_BAND_CH2_NOM);	// absolute mean values over antennas: 4.99, 4.79, 4.43, 4.90 (== +1 average delta from nom 4)
			efuse_rx_lb_bandsel 		= (uint16_t) (-1 + RX_LB_BAND_NOM);	// absolute mean values over antennas: 5 over all antennas (== -1 average delta from nom 6)
			efuse_tx_lb_bandsel 		= (uint16_t) (-1 + TX_LB_BAND_NOM);	// absolute mean values over antennas: 1.39, 0.99, 1.37, 1.0 (== -1 average delta from nom 2)

			// EFUSE15
			efuse_mcm 					= 1;
		}

		if (SectMsk & 1)
		// Fuse values written to Central registers
		{
			//--------------------------------------------------------------------------
			// write efuse_vref and efuse_bias to Register BIASCENT CTL_TRIM_BGP - Central 0x68
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BIASCENT,
					WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_TRIM_BGP__ISMSK
					& WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_CAL_ICONST__ISMSK,
					efuse_vref << WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_TRIM_BGP__POS
					| efuse_bias << WRX654_RF_CENTRAL_FCSI_BIASCENT_CTL_CAL_ICONST__POS);

			//--------------------------------------------------------------------------
			// write efuse_bbioldo1v2 to Register BBIOLDO1V2 CTL_VOUT - Central 0x6b
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2,
					WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT__ISMSK,
					efuse_bboldo1v2 << WRX654_RF_CENTRAL_FCSI_BBIOLDO1V2_CTL_VOUT__POS);

			//--------------------------------------------------------------------------
			// write efuse_bbioldo1v2 to Register FSYSLDO2V5 CTL_VOUT - Central 0x65
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5,
					WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_CTL_VOUT__ISMSK,
					efuse_fsysldo2v5 << WRX654_RF_CENTRAL_FCSI_FSYSLDO2V5_CTL_VOUT__POS);

			//--------------------------------------------------------------------------
			// write efuse_fcsildo1v2 to Register FCSILDO1V2 CTL_VOUT - Central 0x66
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_FCSILDO1V2,
					WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_CTL_VOUT__ISMSK,
					efuse_fcsildo1v2 << WRX654_RF_CENTRAL_FCSI_FCSILDO1V2_CTL_VOUT__POS);

			//--------------------------------------------------------------------------
			// write efuse_vcal_rsit to Register TEMPSENS1 - Central 0x6d
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(CENTRAL, WRX654_RF_CENTRAL_FCSI_TEMPSENS1,
					WRX654_RF_CENTRAL_FCSI_TEMPSENS1_VCAL_VAL__ISMSK,
					efuse_vcal_rsit << WRX654_RF_CENTRAL_FCSI_TEMPSENS1_VCAL_VAL__POS);

		}

		if (SectMsk & 2)
		// Fuse values written to Antenna registers
		{
			for (ant=0; ant < 4; ant++)
			{
				//--------------------------------------------------------------------------
				// write efuse_txhbldo1v2[x] to Register TXHBLDO1V2 CTL_VOUT - Antx 0x44
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2,
						WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT__ISMSK,
						efuse_txhbldo1v2[ant] << WRX654_RF_ANTENNA_FCSI_TXHBLDO1V2_CTL_VOUT__POS);

				//--------------------------------------------------------------------------
				// write efuse_txlbldo1v2[x] to Register TXLBLDO1V2 CTL_VOUT - Antx 0x45
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2,
						WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT__ISMSK,
						efuse_txlbldo1v2[ant] << WRX654_RF_ANTENNA_FCSI_TXLBLDO1V2_CTL_VOUT__POS);

				//--------------------------------------------------------------------------
				// write efuse_rxldo1v2[x] to Register RXLDO1V2 CTL_VOUT - Antx 0x46
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_RXLDO1V2,
						WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT__ISMSK,
						efuse_rxldo1v2[ant] << WRX654_RF_ANTENNA_FCSI_RXLDO1V2_CTL_VOUT__POS);

				//--------------------------------------------------------------------------
				// write efuse_bbldo1v2[x] to Register BBLDO1V2 CTL_VOUT - Antx 0x47
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_BBLDO1V2,
						WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT__ISMSK,
						efuse_bbldo1v2[ant] << WRX654_RF_ANTENNA_FCSI_BBLDO1V2_CTL_VOUT__POS);

				//--------------------------------------------------------------------------
				// write efuse_rfldo2v5[x] to Register RFLDO2V5 CTL_VOUT - Antx 0x48
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_RFLDO2V5,
						WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_VOUT__ISMSK,
						efuse_rfldo2v5[ant] << WRX654_RF_ANTENNA_FCSI_RFLDO2V5_CTL_VOUT__POS);

				//--------------------------------------------------------------------------
				// write efuse_lblotreeldo[x] to Register LOCENTLB CTL_VOUT - Antx 0x40
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_LOCENTLB,
						WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT__ISMSK,
						efuse_lblotreeldo[ant] << WRX654_RF_ANTENNA_FCSI_LOCENTLB_CTL_VOUT__POS);

				//--------------------------------------------------------------------------
				// write efuse_hblotreeldo[x] to Register LOCENTHB CTL_VOUT - Antx 0x41
				//--------------------------------------------------------------------------
				RFIC_FCSI_RMW(1<<ant, WRX654_RF_ANTENNA_FCSI_LOCENTHB,
						WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT__ISMSK,
						efuse_hblotreeldo[ant] << WRX654_RF_ANTENNA_FCSI_LOCENTHB_CTL_VOUT__POS);

			}

			// write TPC filter values to registers
			//--------------------------------------------------------------------------
			// Register TPCFILTER0	TPC Filter Tuning 0		 		Antx	0x53	(OK)
			//--------------------------------------------------------------------------
			RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCFILTER0,
					efuse_tpc_filter_40MHz << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_20MHZ__POS		// set TPC FB Cap for 20MHz
					| efuse_tpc_filter_40MHz << WRX654_RF_ANTENNA_FCSI_TPCFILTER0_CTL_TPC_FB_CSEL_40MHZ__POS);	// set TPC FB Cap for 40MHz

			//--------------------------------------------------------------------------
			// Register TPCFILTER1	TPC Filter Tuning 1		 		Antx	0x54	(OK)
			//--------------------------------------------------------------------------
			RFIC_FCSI_Write(ANTALL, WRX654_RF_ANTENNA_FCSI_TPCFILTER1,
					efuse_tpc_filter_80MHz << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_80MHZ__POS		// set TPC FB Cap for 80MHz
					| efuse_tpc_filter_160MHz << WRX654_RF_ANTENNA_FCSI_TPCFILTER1_CTL_TPC_FB_CSEL_160MHZ__POS);	// set TPC FB Cap for 160MHz


			//--------------------------------------------------------------------------
			// write LB efuse_rx_mid_gain to Register LNARSSI 0x68 in each antenna
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(ANT0+ANT1, WRX654_RF_ANTENNA_FCSI_LNARSSI,
					WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__ISMSK,
					efuse_rx_lb_mid2high_gain_ant01 << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS);

			//--------------------------------------------------------------------------
			// write LB efuse_rx_mid_gain to Register LNARSSI 0x68 in each antenna
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(ANT2+ANT3, WRX654_RF_ANTENNA_FCSI_LNARSSI,
					WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__ISMSK,
					efuse_rx_lb_mid2high_gain_ant23 << WRX654_RF_ANTENNA_FCSI_LNARSSI_CTL_LNA_FINEGAIN__POS);

			//--------------------------------------------------------------------------
			// Correct LB TX/RX Band Selection for each Antenna - Antx SR0
			//--------------------------------------------------------------------------
			RFIC_FCSI_RMW(ANTALL, WRX654_RF_ANTENNA_FCSI_SR0,
					WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND__ISMSK
					& WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND__ISMSK,
					efuse_tx_lb_bandsel << WRX654_RF_ANTENNA_FCSI_SR0_TX_BAND__POS			// use TX LB band selection by default
					| efuse_rx_lb_bandsel << WRX654_RF_ANTENNA_FCSI_SR0_RX_BAND__POS);		// use RX LB band selection by default

		}



	}
	return chip_fused;
}


uint16_t rfic_get_efuse_values(uint16_t *fuse_data)
// read eFuse data from CENTRAL fuse registers
{
	uint16_t	data[5];
	uint16_t    reg;
	uint16_t	chip_fused = 1;

	for (reg=0; reg<16; reg++)
	{
		RFIC_FCSI_Read(CENTRAL, reg+WRX654_RF_CENTRAL_FCSI_EFUSE0, &data[0]);
		fuse_data[reg] = data[4]; // read from CENTRAL FCSI branch
	}

	// check if chip is fused
	if ((fuse_data[15] & (7 << WRX654_RF_CENTRAL_FCSI_EFUSE15_PRODUCTCODE__POS)) == 0)
	{
		chip_fused = 0;
		fuse_data[15] = 0; // if chip is not fused, also set the hard-coded bits in efuse[15] to zero (for re-use as softfuse data)
	}

	return chip_fused;
}


uint16_t rfic_read_efuse_handlertemp(uint16_t *softfuse_data)
// read handler temperature from eFuse data
{
	uint16_t handlertemp;
	uint16_t read_data[5];
    // force to read from softfuse data as long as softfuse_data[15] != 0
    if (softfuse_data[15] == 0)
    {
    	//--------------------------------------------------------------------------
    	// read EFUSE2 register (read on all antennas, but only Ant3 reading is used)
    	//--------------------------------------------------------------------------
    	RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE1, read_data);
    }
    else // or use external softfuse_data
    {
    	//--------------------------------------------------------------------------
    	// write softfuse data for fuse_reg[1] to read_data[4] to mimic on-chip reading
    	//--------------------------------------------------------------------------
    	read_data[4] = softfuse_data[1];
    }
	handlertemp = ((read_data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE1_HANDLERTEMP__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE1_HANDLERTEMP__MSK);

	return handlertemp;
}


uint16_t rfic_read_efuse_tempsensorresult(uint16_t *softfuse_data)
// read temperature sensor calibration result from eFuse data
{
	uint16_t tempsensorresult;
	uint16_t read_data[5];

    // force to read from softfuse data as long softfuse_data[15] != 0
    if (softfuse_data[15] == 0)
    {
    	//--------------------------------------------------------------------------
    	// read EFUSE11 register (read on all antennas, but only Ant3 reading is used)
    	//--------------------------------------------------------------------------
    	RFIC_FCSI_Read(CENTRAL, WRX654_RF_CENTRAL_FCSI_EFUSE11, read_data);
    }
    else // or use external softfuse_data
    {
    	//--------------------------------------------------------------------------
    	// write softfuse data for fuse_reg[11] to read_data[4] to mimic on-chip reading
    	//--------------------------------------------------------------------------
    	read_data[4] = softfuse_data[11];
    }
	tempsensorresult = ((read_data[4] >> WRX654_RF_CENTRAL_FCSI_EFUSE11_TEMPSENSORRESULT__POS) & WRX654_RF_CENTRAL_FCSI_EFUSE11_TEMPSENSORRESULT__MSK);

	return tempsensorresult;
}


// ########################################################################
// #                  APIs with parallel antenna access                   #
// ########################################################################
void rfic_config_tpcgain_par(uint8_t AntMsk, int8_t *gain0dB, int8_t *gain1dB, int8_t *gain2dB, int8_t *gain3dB)
// set TPC gains for all 4 TX gain constellations (gain values -6,-4,..,+6 dB) - parallel write of different gain values per antenna
{
    uint16_t	Values[5];
    uint8_t     iAnt;

    for(iAnt=0; iAnt < 4; iAnt++)
    {
        Values[iAnt] = (((gain0dB[iAnt]+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN0__POS		// set TPC gain #0
                | (((gain1dB[iAnt]+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN1__POS		// set TPC gain #1
                | (((gain2dB[iAnt]+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN2__POS		// set TPC gain #2
                | (((gain3dB[iAnt]+6) >> 1) & 7) << WRX654_RF_ANTENNA_FCSI_TPCGAIN_CTL_TPC_GAIN3__POS;		// set TPC gain #3
    }

    RFIC_FCSI_ParWrite(AntMsk, WRX654_RF_ANTENNA_FCSI_TPCGAIN, Values);
}

void rfic_set_txdc_tpccfg_par (uint8_t AntMsk, int8_t *dacI, int8_t *dacQ, uint8_t TpcCfg)
// set TX DC offset DAC values (for one specific TPC configuration) - parallel write of different DAC values per antenna
{
    uint16_t	Values[5], IMsk=0, QMsk=0, ISMsk=0, RegAdr=WRX654_RF_ANTENNA_FCSI_TPCOFF01;
    uint8_t		IPos=0, QPos=0, iAnt;

	if (TpcCfg == 0)
	{
		IMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__MSK;
		QMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__MSK;
		IPos = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__POS;
		QPos = WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__POS;
		ISMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__ISMSK & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__ISMSK;
		RegAdr = WRX654_RF_ANTENNA_FCSI_TPCOFF01;
	}
	else if (TpcCfg == 1)
	{
		IMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__MSK;
		QMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__MSK;
		IPos = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__POS;
		QPos = WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__POS;
		ISMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__ISMSK & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__ISMSK;
		RegAdr = WRX654_RF_ANTENNA_FCSI_TPCOFF01;
	}
	else if (TpcCfg == 2)
	{
		IMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__MSK;
		QMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__MSK;
		IPos = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__POS;
		QPos = WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__POS;
		ISMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__ISMSK & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__ISMSK;
		RegAdr = WRX654_RF_ANTENNA_FCSI_TPCOFF23;
	}
	else if (TpcCfg == 3)
	{
		IMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__MSK;
		QMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__MSK;
		IPos = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__POS;
		QPos = WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__POS;
		ISMsk = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__ISMSK & WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__ISMSK;
		RegAdr = WRX654_RF_ANTENNA_FCSI_TPCOFF23;
	}

    for(iAnt=0; iAnt < 4; iAnt++)
    {
    	if (dacI[iAnt]<0)
    	{
    		dacI[iAnt] = dacI[iAnt]+16;
    	}
    	if (dacQ[iAnt]<0)
    	{
    		dacQ[iAnt] = dacQ[iAnt]+16;
    	}

    	Values[iAnt] = (dacI[iAnt] & IMsk) << IPos
				| (dacQ[iAnt] & QMsk) << QPos;
    }
	//--------------------------------------------------------------------------
	// Register TPCOFF01/23 - TPC Offset 0/1/2/3			Antx	0x56/0x57
	//--------------------------------------------------------------------------
	RFIC_FCSI_ParRMW(AntMsk, RegAdr, ISMsk, Values);

}

void rfic_set_antenna_fectl_man_par(uint8_t AntMsk, uint8_t *fectl)
// set manual antenna FECTLs, different for each selected antenna
{
	uint16_t	Values[5];
    uint8_t		iAnt;
			
	for(iAnt=0; iAnt < 4; iAnt++)
    {
    	Values[iAnt] =  ((fectl[iAnt] & WRX654_RF_ANTENNA_FCSI_MANFECTL_FECTL_GAIN__MSK) << WRX654_RF_ANTENNA_FCSI_MANFECTL_FECTL_GAIN__POS ) | WRX654_RF_ANTENNA_FCSI_MANFECTL_CTL_MANUEL_FECTL_1__SVAL;
    }
	//--------------------------------------------------------------------------
	// Register	MANFECTL	MANFECTL PADCTL					Antx	0x4C
	//--------------------------------------------------------------------------
	RFIC_FCSI_ParRMW(AntMsk, WRX654_RF_ANTENNA_FCSI_MANFECTL,
			WRX654_RF_ANTENNA_FCSI_MANFECTL_CTL_MANUEL_FECTL__ISMSK
			& WRX654_RF_ANTENNA_FCSI_MANFECTL_FECTL_GAIN__ISMSK,
			Values);
}

