#include <stdint.h>
#include "target.h"

#include "interface_wrx654.h"
#include "lib_wrx654_api.h"
#include "lib_wrx654_sequences.h"


/* register packages */
#include "wrx654_fcsi0_addrmap.h"
#include "wrx654_fcsi0_regpkg.h"
#include "wrx654_rf_antenna_fcsi_fieldpkg.h"
#include "wrx654_rf_antenna_fcsi_regpkg.h"
#include "wrx654_fcsi1_addrmap.h"
#include "wrx654_fcsi1_regpkg.h"
#include "wrx654_fcsi2_addrmap.h"
#include "wrx654_fcsi2_regpkg.h"
#include "wrx654_fcsi3_addrmap.h"
#include "wrx654_fcsi3_regpkg.h"
#include "wrx654_central_addrmap.h"
/* #include "wrx654_central_regpkg.h" */
#include "wrx654_rf_central_fcsi_fieldpkg.h"
#include "wrx654_rf_central_fcsi_regpkg.h"

//-------------------------------------------------------------------------
// Init Sequence (rfic_init_1/2/3/4 and required waits in between)
//-------------------------------------------------------------------------
void rfic_init(void)
{
	// init_1
	rfic_init_1(ANTALL);
	// wait 50ms for band gap settling
	RFIC_Wait(50000);
	// init_2
	rfic_init_2(ANTALL);
	// wait 10us
	RFIC_Wait(10);
	// init_3
	rfic_init_3(ANTALL);

}

void rfic_central_init(void)
{
	// init_1
	rfic_init_1(ANTALL);
	// wait 50ms for band gap settling
	RFIC_Wait(50000);
}

void rfic_band_init(uint8_t AntMsk)
{
	// init_2
	rfic_init_2(AntMsk);
	// wait 10us
	RFIC_Wait(10);
	// init_3
	rfic_init_3(AntMsk);
}


//-------------------------------------------------------------------------
// Read out PLL config struct
//-------------------------------------------------------------------------
pll_config rfic_get_pll_status(uint8_t Pll)
{
    pll_config PllCfg;

    // make sure that Pll index is in allowed range
    if (Pll > NUM_OF_PLLS-1)
        Pll = 0;

    PllCfg = PllConfig[Pll];

    return PllCfg;
}

//-------------------------------------------------------------------------
// Read out RX gain config struct
//-------------------------------------------------------------------------
rx_gain_cfg rfic_get_rxgain_status(uint8_t Ant)
{
    rx_gain_cfg RxGainCfg;

    // make sure that Ant index is in allowed range
    if (Ant > NUM_OF_ANTS-1)
        Ant = 0;

    RxGainCfg = RxGainConfig[Ant];

    return RxGainCfg;
}

//-------------------------------------------------------------------------
// complete sequence to start PLL in single LO mode
//-------------------------------------------------------------------------
uint8_t rfic_start_pll(uint8_t Pll, uint8_t AntSet, uint16_t LoFreq)
//  - initialize PLL according chosen LO freq in HB or LB
//  - for HB: if LO freq multiple of 10 then RefClk = 20MHz, otherwise RefClk = 10MHz
//  - program single LO frequency,
//  - release PLL reset (i.e. start calibration) and wait for PLL lock (timeout after 1ms)
//  - connect selected antennas and switch them to according HB/LB mode
//  - return PLL lock information
{
    uint8_t     HbLb, RefClk, HiPwr, iK;
    uint16_t    PllLocked;

    PllLocked = 0;

    // make sure that Pll index is in allowed range
    if (Pll > NUM_OF_PLLS-1)
        Pll = 0;

    if (LoFreq > 2500)
    {
        HbLb = 1;
        HiPwr = 1;
        if (LoFreq % 10 == 0)
            RefClk = 20;
        else
            RefClk = 10;
    }
    else
    {
        HbLb = 0;
        HiPwr = 0;
        RefClk = 40;
    }

    // program PLL to according mode with RefClk ref clk (PLL resets automatically)
    rfic_set_pll_mode(Pll, HbLb, RefClk, HiPwr);

    if (HbLb)
        rfic_activate_hb(AntSet);       // activate HB for AntSet
    else
        rfic_activate_lb(AntSet);       // activate LB for AntSet

    // set PLL single frequency (HbLb, RefClk, IntFrac already set in rfic_set_pll_mode)
    rfic_set_lo_frequency(Pll, LoFreq);

    // release PLL (start calibration)
    rfic_release_pllreset(Pll);

    // wait maximum 1ms (100 x 10us) to detect successful PLL lock
    for (iK=0; iK<100; iK++)
    {
        RFIC_Wait(10);      // wait 10us
        PllLocked = rfic_check_pll_lock(Pll);
        if (PllLocked)
            break;          // leave for() loop if PLL has locked
    }

    // set Central LO switches for AntSet to PLL in according mode
    rfic_select_pll(AntSet, Pll, PllConfig[Pll].HbLb);

    // return PLL lock information
    return PllLocked;
}

void rfic_lock_pll(uint8_t Pll, uint16_t LoFreq)
//  - initialize PLL according chosen LO freq in HB or LB
//  - for HB: if LO freq multiple of 10 then RefClk = 20MHz, otherwise RefClk = 10MHz
//  - program single LO frequency,
//  - release PLL reset (i.e. start calibration)
{
    uint8_t     HbLb, RefClk, HiPwr;

    // make sure that Pll index is in allowed range
    if (Pll > NUM_OF_PLLS-1)
        Pll = 0;

    if (LoFreq > 2500)
    {
        HbLb = 1;
        HiPwr = 1;
        if (LoFreq % 10 == 0)
            RefClk = 20;
        else
            RefClk = 10;
    }
    else
    {
        HbLb = 0;
        HiPwr = 0;
        RefClk = 40;
    }

    // program PLL to according mode with RefClk ref clk (PLL resets automatically)
    rfic_set_pll_mode(Pll, HbLb, RefClk, HiPwr);

    // set PLL single frequency (HbLb, RefClk, IntFrac already set in rfic_set_pll_mode)
    rfic_set_lo_frequency(Pll, LoFreq);

    // release PLL (start calibration)
    rfic_release_pllreset(Pll);
}
uint16_t rfic_poll_pll_lock(uint8_t Pll)
//  - wait for PLL lock (timeout after 1ms)
//  - return PLL lock information
{
	uint16_t iK;
    uint16_t PllLocked;

    PllLocked = 0;

	// wait maximum 1ms (100 x 10us) to detect successful PLL lock
	for (iK=0; iK<100; iK++)
	{
		RFIC_Wait(10);		// wait 10us
		PllLocked = rfic_check_pll_lock(Pll);
		if (PllLocked)
			break;			// leave for() loop if PLL has locked
	}

	// return PLL lock information
	return PllLocked;
}


//-------------------------------------------------------------------------
// complete sequence to start PLL in fasthop LO mode
//-------------------------------------------------------------------------
uint8_t rfic_start_pll_fasthop(uint8_t Pll, uint8_t AntSet, uint16_t LoFreq0, uint16_t LoFreq1, uint16_t LoFreq2, uint16_t LoFreq3, uint16_t LoFreq4, uint16_t LoFreq5)
//  - initialize PLL according chosen LO freqs in HB or LB
//  - for HB: if all LO freqs multiples of 10 then RefClk = 20MHz, otherwise RefClk = 10MHz
//  - program LO frequencies and activate fasthop mode,
//  - release PLL reset (i.e. start calibration) and wait for PLL lock (timeout after 1ms)
//  - connect selected antennas and switch them to according HB/LB mode
//  - return PLL lock information
{
    uint8_t     HbLb, RefClk, HiPwr, iK, MasterAnt;
    uint16_t    PllLocked, LastNonZeroFreq;

    PllLocked = 0;

    // make sure that Pll index is in allowed range
    if (Pll > NUM_OF_PLLS-1)
        Pll = 0;

    if (LoFreq0 > 2500)
    {
        HbLb = 1;
        HiPwr = 1;
        if ((LoFreq0 % 10 == 0) & (LoFreq1 % 10 == 0) & (LoFreq2 % 10 == 0) & (LoFreq3 % 10 == 0) & (LoFreq4 % 10 == 0) & (LoFreq5 % 10 == 0))
            RefClk = 20;
        else
            RefClk = 10;
        LastNonZeroFreq = 5250;
    }
    else
    {
        HbLb = 0;
        HiPwr = 0;
        RefClk = 40;
        LastNonZeroFreq = 2412;
    }

    // fill up zeros in the list of 6 fasthop channels with a finite number (otherwise calibration would fail)
    if (LoFreq0 == 0)
        LoFreq0 = LastNonZeroFreq;
    else
        LastNonZeroFreq = LoFreq0;
    if (LoFreq1 == 0)
        LoFreq1 = LastNonZeroFreq;
    else
        LastNonZeroFreq = LoFreq1;
    if (LoFreq2 == 0)
        LoFreq2 = LastNonZeroFreq;
    else
        LastNonZeroFreq = LoFreq2;
    if (LoFreq3 == 0)
        LoFreq3 = LastNonZeroFreq;
    else
        LastNonZeroFreq = LoFreq3;
    if (LoFreq4 == 0)
        LoFreq4 = LastNonZeroFreq;
    else
        LastNonZeroFreq = LoFreq4;
    if (LoFreq5 == 0)
        LoFreq5 = LastNonZeroFreq;
    else
        LastNonZeroFreq = LoFreq5;

    // program PLL to according mode with RefClk ref clk (PLL resets automatically)
    rfic_set_pll_mode(Pll, HbLb, RefClk, HiPwr);

    if (HbLb)
        rfic_activate_hb(AntSet);       // activate HB for AntSet
    else
        rfic_activate_lb(AntSet);       // activate LB for AntSet

    // set PLL fasthop frequencies (HbLb, RefClk, IntFrac already set in rfic_set_pll_mode)
    rfic_set_lo_frequencies(Pll, LoFreq0, LoFreq1, LoFreq2, LoFreq3, LoFreq4, LoFreq5);

    // release PLL (start calibration)
    rfic_release_pllreset(Pll);

    // wait maximum 1ms (100 x 10us) to detect successful PLL lock
    for (iK=0; iK<100; iK++)
    {
        RFIC_Wait(10);      // wait 10us
        PllLocked = rfic_check_pll_lock(Pll);
        if (PllLocked)
            break;          // leave for() loop if PLL has locked
    }

    if (AntSet > 0)
    {
        // set Central LO switches for AntSet to PLL in according mode
    	rfic_select_pll(AntSet, Pll, PllConfig[Pll].HbLb);

        // find antenna with lowest index belonging to AntSet and define it as fasthop master
        MasterAnt = 0;
        while ((AntSet & 1 << MasterAnt) == 0)
            MasterAnt++;
        rfic_select_fasthop_master(Pll, MasterAnt);
    }

    // return PLL lock information
    return PllLocked;
}


//-------------------------------------------------------------------------
// Some PLL shortcuts (partly reading PLL status from PLL config struct)
//-------------------------------------------------------------------------
// Initialize PLL (including explicit choice of IntN/FracN mode)
void rfic_init_pll(uint8_t Pll, uint8_t IntFracMode, uint8_t HbLb, uint8_t RefClk, uint8_t HiPwr)
{
	rfic_set_pll_mode(Pll, HbLb, RefClk, HiPwr);
	rfic_set_pll_intfrac(Pll, IntFracMode);        // IntN/FracN mode normally already set in rfic_set_pll_mode but can be changed here
}

// set PLL to HB or LB
void rfic_set_pll_hblb(uint8_t Pll, uint8_t HbLb)
{
	rfic_set_pll_mode(Pll, HbLb, PllConfig[Pll].RefClk, PllConfig[Pll].HiPwr);
}

// set PLL power mode to High Power or Reduced Power
void rfic_set_pll_pwrmode(uint8_t Pll, uint8_t HiPwr)
{
	rfic_set_pll_mode(Pll, PllConfig[Pll].HbLb, PllConfig[Pll].RefClk, HiPwr);
}

// set PLL to IntN or FracN mode
void rfic_set_pll_intfrac(uint8_t Pll, uint8_t IntFracMode)
{
	PllConfig[Pll].IntFrac = IntFracMode;

	if (PllConfig[Pll].IntFrac)
	{
		rfic_set_pll_intnmode(Pll);
	}
	else
	{
		rfic_set_pll_fracnmode(Pll);
	}
}

// change PLL reference clock
void rfic_set_pll_refclk(uint8_t Pll, uint8_t RefClk)
{
	rfic_set_pll_mode( Pll,  PllConfig[Pll].HbLb,  RefClk,  PllConfig[Pll].HiPwr);
}

// program LO single frequency (IntN/FracN choice according PLL configuration)
void rfic_set_lo_frequency(uint8_t Pll, uint16_t LoFreq)
{
	if (PllConfig[Pll].IntFrac)
	{
		rfic_set_lo_frequency_intn( Pll, LoFreq, PllConfig[Pll].HbLb, PllConfig[Pll].RefClk);
	}
	else
	{
		rfic_set_lo_frequency_fracn(Pll, LoFreq,  PllConfig[Pll].HbLb);
	}

}

// program LO fasthop frequencies (IntN/FracN choice according PLL configuration)
void rfic_set_lo_frequencies(uint8_t Pll, uint16_t LoFreq0, uint16_t LoFreq1, uint16_t LoFreq2, uint16_t LoFreq3, uint16_t LoFreq4, uint16_t LoFreq5)
{
	if (PllConfig[Pll].IntFrac)
	{
		rfic_set_lo_frequencies_intn(Pll, LoFreq0, LoFreq1, LoFreq2, LoFreq3, LoFreq4, LoFreq5, PllConfig[Pll].HbLb, PllConfig[Pll].RefClk);
	}
	else
	{
		rfic_set_lo_frequencies_fracn(Pll, LoFreq0, LoFreq1, LoFreq2, LoFreq3, LoFreq4, LoFreq5, PllConfig[Pll].HbLb);
	}
}

// connect set of antennas to specific PLL (HB or LB chosen according PLL configuration), optionally activate HB/LB
void rfic_connect_to_pll(uint8_t AntSet, uint8_t Pll, uint8_t bActHbLb)
{
	if (bActHbLb)
	{
		if (PllConfig[Pll].HbLb)
			rfic_activate_hb(AntSet);       // activate HB for AntSet
		else
			rfic_activate_lb(AntSet);       // activate LB for AntSet
	}

	rfic_select_pll(AntSet, Pll, PllConfig[Pll].HbLb);
}

