function [vRead, rStatus] = FCSI_READ(sAntMsk, sRegAdr, bDebug)
% FCSI_FCSI_READ - USB Read function for WRX6xx Level1 Test Board
    
% --- Authors -------------------------------------------------------------
% Steffen Trautmann, Denis Matveev
% -------------------------------------------------------------------------
    
% --- Copyright -----------------------------------------------------------
% Intel Austria GmbH, Connected Home Division, Villach
% -------------------------------------------------------------------------

% debugging on/off
if nargin < 3
    bDebug = 0;
end

global bbs
if (prod(size(bbs))>0)
    X = sprintf('AntMsk:  b%s reg: 0x%04x ', dec2bin(sAntMsk,4), sRegAdr);
    disp(X);
    %read_par=param2;
    cmdstrs={['@RFIC_FCSI_Read(b' dec2bin(sAntMsk) ',h' dec2hex(sRegAdr) ')']};
    vRead = readvar(cmdstrs);
    disp(cmdstrs)    
    return
end

% check for address range
if sRegAdr < 4
    bShort = 1;
    if sRegAdr == 3
        bUltraShort = 1;
    else
        bUltraShort = 0;
    end
elseif sRegAdr >= 64
    bShort = 0;
    bUltraShort = 0;
else
    error('Address 0x%x is not defined!',sRegAdr);
end

% determine address and data length
nAdrLen = 7 - bShort*4;
nDataLen = 16 - bUltraShort*14;

% calculate number of clock half cycles
nWait = 1.5;
nCycles = (nAdrLen + nDataLen + 6 + nWait+0.5) * 2 - 1;

% prepare variables and mask vector 
nAnt = 5;
vClk = zeros(nAnt, nCycles);
vEnable = zeros(nAnt, nCycles);
vData = zeros(nAnt, nCycles);

vAntMsk = zeros(nAnt, 1);
for iA = 1:nAnt
    vAntMsk(iA) = bitget(sAntMsk,iA);
end

% generate clock and enables for each antenna 
vClk(:,2:2:end) = repmat(vAntMsk, 1, floor(nCycles/2));
vEnable(:,4:nCycles-4) = repmat(vAntMsk, 1, nCycles-7);

% generate address sequence 
for iAl = 1:nAdrLen
    vData(:, 2+2*iAl:3+2*iAl) = repmat(vAntMsk * bitget(sRegAdr, nAdrLen+1-iAl),1,2);
end
% ... set read bit for each antenna
vData(:, 4+2*nAdrLen:5+2*nAdrLen) = repmat(vAntMsk,1,2);
if bDebug
    xRegData = [123, 456, 789, 1011, 1213]';
    for iDl = 1:nDataLen
        vData(:, 4+2*nAdrLen+2*iDl+2*(nWait+.5):5+2*nAdrLen+2*iDl+2*(nWait+.5)) = repmat(vAntMsk .* bitget(xRegData, nDataLen+1-iDl),1,2);
    end
end

% construct the parallel output to the USB interface
mUsbOut = zeros(nAnt*3, nCycles);
mUsbOut(1:3:end,:) = vEnable;
mUsbOut(2:3:end,:) = vClk;
mUsbOut(3:3:end,:) = vData;

% combine bit streams to parallel words
vUsbOutPar_A = uint8(2.^([0:7])*mUsbOut(1:8,:)); 
vUsbOutPar_B = uint8(2.^([0:6])*mUsbOut(9:15,:)); 

vRead = zeros(nAnt,1);
if ~bDebug
    % init interface
    h1=FTD2XX_NET.FTDI; h2=FTD2XX_NET.FTDI;
    rc = OpenByIndex(h1,0);rc = OpenByIndex(h2,1);
    rc = SetBaudRate(h1, 115200); rc = SetBaudRate(h2, 115200); 
    rc = SetBitMode(h1,255,1); rc = SetBitMode(h2,255,1);
    % write start bits and address
    for i=1:(3+2*nAdrLen+2) 
        rc = Write(h1, vUsbOutPar_A(i), 1, 0); 
        rc = Write(h2, vUsbOutPar_B(i), 1, 0);
    end;
    % read delay
    rc = SetBitMode(h1,219,1); rc = SetBitMode(h2,182,1);
    for i=(3+2*nAdrLen+3):(3+2*nAdrLen+2+(nWait+0.5)*2)
        rc = Write(h1, vUsbOutPar_A(i), 1, 0); 
        rc = Write(h2, vUsbOutPar_B(i), 1, 0);
    end;
    % read data
    for iD=1:nDataLen
        rc = Write(h1, vUsbOutPar_A(3+2*nAdrLen+1+(nWait+0.5)*2+2*iD), 1, 0); 
        rc = Write(h2, vUsbOutPar_B(3+2*nAdrLen+1+(nWait+0.5)*2+2*iD), 1, 0);
        [rc, sData_A] = GetPinStates(h1, 1);
        [rc, sData_B] = GetPinStates(h2, 1);
        vRead(1) = vRead(1)*2 + double(bitget(sData_A,3));
        vRead(2) = vRead(2)*2 + double(bitget(sData_A,6));
        vRead(3) = vRead(3)*2 + double(bitget(sData_B,1));
        vRead(4) = vRead(4)*2 + double(bitget(sData_B,4));
        vRead(5) = vRead(5)*2 + double(bitget(sData_B,7));
        rc = Write(h1, vUsbOutPar_A(3+2*nAdrLen+2+(nWait+0.5)*2+2*iD), 1, 0); 
        rc = Write(h2, vUsbOutPar_B(3+2*nAdrLen+2+(nWait+0.5)*2+2*iD), 1, 0);
    end;
    % write stop bits
    rc = SetBitMode(h1,255,1); rc = SetBitMode(h2,255,1);
    for i=nCycles-5:nCycles 
        rc = Write(h1, vUsbOutPar_A(i), 1, 0); 
        rc = Write(h2, vUsbOutPar_B(i), 1, 0);
    end;
    % close interface
    rc = Close(h1);rc = Close(h2);
    rStatus = 0;
else
    h1 = 0; h2 = 0;
    rStatus = Write_Debug(h1, vUsbOutPar_A(1:3+2*nAdrLen+2), 3+2*nAdrLen+2, 0);
    rStatus = Write_Debug(h2, vUsbOutPar_B(1:3+2*nAdrLen+2), 3+2*nAdrLen+2, 0);
    % wait read delay
    rStatus = Write_Debug(h1, vUsbOutPar_A(3+2*nAdrLen+3:3+2*nAdrLen+2+(nWait+0.5)*2), (nWait+0.5)*2, 0);
    rStatus = Write_Debug(h2, vUsbOutPar_B(3+2*nAdrLen+3:3+2*nAdrLen+2+(nWait+0.5)*2), (nWait+0.5)*2, 0);
    for iD=1:nDataLen
        rStatus = Write_Debug(h1, vUsbOutPar_A(3+2*nAdrLen+1+(nWait+0.5)*2+2*iD), 1, 0);
        rStatus = Write_Debug(h2, vUsbOutPar_B(3+2*nAdrLen+1+(nWait+0.5)*2+2*iD), 1, 0);
        [rStatus, sData_A] = GetPinStates_Debug(h1, 1, vUsbOutPar_A(3+2*nAdrLen+1+(nWait+0.5)*2+2*iD));
        [rStatus, sData_B] = GetPinStates_Debug(h2, 1, vUsbOutPar_B(3+2*nAdrLen+1+(nWait+0.5)*2+2*iD));
        vRead(1) = vRead(1)*2 + double(bitget(sData_A,3));
        vRead(2) = vRead(2)*2 + double(bitget(sData_A,6));
        vRead(3) = vRead(3)*2 + double(bitget(sData_B,1));
        vRead(4) = vRead(4)*2 + double(bitget(sData_B,4));
        vRead(5) = vRead(5)*2 + double(bitget(sData_B,7));
        rStatus = Write_Debug(h1, vUsbOutPar_A(3+2*nAdrLen+2+(nWait+0.5)*2+2*iD), 1, 0);
        rStatus = Write_Debug(h2, vUsbOutPar_B(3+2*nAdrLen+2+(nWait+0.5)*2+2*iD), 1, 0);
    end;
    rStatus = Write_Debug(h1, vUsbOutPar_A(nCycles-3:nCycles), 4, 0);
    rStatus = Write_Debug(h2, vUsbOutPar_B(nCycles-3:nCycles), 4, 0);
    
end

% status output
vAntNo = {'0' '1' '2' '3' 'Central'};
tAnt = sprintf('%s+',vAntNo{logical(bitget(sAntMsk,[1:5]))}); tAnt = tAnt(1:end-1);
tAdr = sprintf('0x%02x', sRegAdr);
tData = sprintf('0x%04x,', vRead(logical(bitget(sAntMsk,[1:5])))); tData = tData(1:end-1);
fprintf('Reading from Ant %s: Address %s - %s ...\n', tAnt, tAdr, tData);


% inline functions for Debug mode
function rStatus = Write_Debug(~, vX, nX, ~)

for iX = 1:nX
    vX(iX);
end
rStatus = 0;

function [rStatus, xData] = GetPinStates_Debug(~, ~, vY)
xData = vY;
rStatus = 0;
