#include <stdint.h>

#ifndef _LIB_WRX654_API_H
#define _LIB_WRX654_API_H

// pre-processor constants for Tester compiling
#define TESTER      0       // 0 = Lab Measurement using internal clock
                            // 1 = Tester measurement using external clock
#define MCM         0		// 0 = Stand-alone WRX
                            // 1 = MCM

// Antenna mask bits
#define CENTRAL     16		// Set bit 4 for Central FCSI branch
#define ANT0        1 		// Set bit 0 for Antenna0 FCSI branch
#define ANT1        2 		// Set bit 1 for Antenna1 FCSI branch
#define ANT2        4 		// Set bit 2 for Antenna2 FCSI branch
#define ANT3        8 		// Set bit 3 for Antenna3 FCSI branch
#define ANTALL      15 		// Set all antenna FCSI bits

// Nominal reference values for fusing
#define	RX_HB_BAND_CH0_NOM	6
#define	RX_HB_BAND_CH1_NOM	6
#define	RX_HB_BAND_CH2_NOM 	6
#define	RX_LB_BAND_NOM		6
#define	TX_HB_BAND_CH0_NOM	4
#define	TX_HB_BAND_CH1_NOM	4
#define	TX_HB_BAND_CH2_NOM	4
#define	TX_LB_BAND_NOM		2
#define	MID2HIGHSTEP_LB_NOM	6
#define	MID2HIGHSTEP_HB_NOM	9

// other useful constants
#define NUM_OF_ANTS         4
#define NUM_OF_RXGFIELDS    9
#define NUM_OF_PLLS         2
#define NUM_OF_PLLFIELDS    11
#define LOBAND              0
#define HIBAND              1
#define FRACN               0
#define INTN                1
#define MHZ10               10
#define MHZ20               20
#define MHZ40               40

// definitions for global struct to store chip configuration
// PLL Configuration
typedef struct
{
    uint8_t     Active;         // 1 - PLL is active, 0 - PLL is powered down
    uint8_t     Reset;          // 1 - PLL in Reset (Standby), 0 - PLL is active
	uint8_t     HbLb;           // 1 - Hb, 0 - Lb
    uint16_t    LoFreqs[6];     // currently programmed LO frequencies in MHz (==0 means unused or not initialized)
    uint16_t    LoFreqAct;      // currently selected LO frequency
	uint8_t     RefClk;         // 10, 20, or 40 MHz
	uint8_t     IntFrac;        // 1 - IntN Mode, 0 - FracN Mode
	uint8_t     HiPwr;          // 1 - High Power, 0 - Reduced Power
    uint8_t     AntSet;         // currently connected set of antennas (as bit mask from 0 to 15)
    uint8_t     FastHop;        // 0 - Single Freq Mode, 1 - Fasthop Mode
    uint8_t     FhMaster;       // Fasthop Master antenna 0, 1, 2, or 3;
} pll_config;

// RX Gain Configuration
typedef struct
{
	uint8_t		LnaGainCfg[7];	// LNA gain configurations
	uint8_t		FectlCfg[7];	// FECTL configurations
	uint8_t		RfGainIdx[4];	// RF gain constellation index (SR1 content and Direct Steps 1/2/3)
	uint8_t		RssiRange[4];	// RSSI range (SR1 content and Direct Steps 1/2/3)
	uint8_t		PgcGainIdx[4];	// PGC gain index (SR1 content and Direct Steps 1/2/3)
	uint8_t		PgcFiltIdx[4];	// PGC filter index (SR1 content and Direct Steps 1/2/3)
	uint8_t		ShortReg3;		// copies SR3 content
	uint8_t		ForceOff;		// ForceOff bit in SR1
	uint8_t		DirStepIdx;		// equals SR3 content if ForceOff==0, equals 0 if ForceOff==1
} rx_gain_cfg;

// extern declarations of global structs and arrays
extern	pll_config 	PllConfig[NUM_OF_PLLS];
extern	rx_gain_cfg	RxGainConfig[NUM_OF_ANTS];

// LUT for PGC filter tuning
extern	uint8_t		PgcFiltLUT[5][11];

// vector of PGC corner frequencies vs. mode (in 8.8 unsigned FP format)
extern	uint16_t	PgcFc100kHz[5];

// API version info
uint16_t rfic_api_version(void);													// return 4-digit integer abcd representing current RFIC API release version Va.bcd

// Initialization
void rfic_init_1(uint8_t AntMsk);
void rfic_init_2(uint8_t AntMsk);
void rfic_init_3(uint8_t AntMsk);
void rfic_init_3_lb_only(void);
void rfic_init_3_hb_only(void);

// RX/TX switching
void rfic_rxon(uint8_t AntMsk);
void rfic_rxoff(uint8_t AntMsk);
void rfic_txon(uint8_t AntMsk);
void rfic_txoff(uint8_t AntMsk);

// FSYS2 clock
void rfic_enable_fsys2 (void); 														// enable FSYS2 clock output
void rfic_disable_fsys2 (void);														// disable FSYS2 clock output

// DCXO
void rfic_trim_dcxo (uint8_t CapTrimX1,uint8_t CapTrimX2);							// trim DCXO

// Antenna on/off, Soft Reset
void rfic_antenna_off (uint8_t AntMsk, uint16_t Twait_us);							// switch off/on antennas (sequentially and with optional waits in between)
void rfic_set_softreset(void); 														// activate soft-reset of the RFIC
void rfic_release_softreset(void); 													// release soft-reset of the RFIC

// LB/HB switching
void rfic_activate_lb(uint8_t AntMsk);												// activate LB blocks for selected antenna(s), deactivate HB blocks
void rfic_activate_hb(uint8_t AntMsk);												// activate HB blocks for selected antenna(s), deactivate LB blocks
void rfic_set_hblb(uint8_t AntMsk, uint8_t HbLb);									// set HBLB flag exclusively without touching any other register, HB/LB doubled register block is NOT changed!

// PLL programming
void rfic_enable_pll(uint8_t Pll);													// (re-)enable PLL0/1
void rfic_disable_pll(uint8_t Pll);													// disable PLL0/1
void rfic_set_pllreset(uint8_t Pll);												// bring PLL0/1 into Reset
void rfic_release_pllreset(uint8_t Pll);											// release PLL0/1 from Reset
void rfic_set_pll_mode(uint8_t Pll, uint8_t HbLb, uint8_t RefClk, uint8_t HiPwr);	// configure PLL0/1 for specific mode
void rfic_set_pll_5G_10M_ax(uint8_t Pll); 											// short-cut to 5G/10MHz/high-power mode
void rfic_set_pll_5G_20M_ax(uint8_t Pll); 											// short-cut to 5G/20MHz/high-power mode
void rfic_set_pll_5G_40M_ax(uint8_t Pll); 											// short-cut to 5G/40MHz/high-power mode
void rfic_set_pll_5G_10M_rp(uint8_t Pll); 											// short-cut to 5G/10MHz/reduced-power mode
void rfic_set_pll_5G_20M_rp(uint8_t Pll); 											// short-cut to 5G/20MHz/reduced-power mode
void rfic_set_pll_5G_40M_rp(uint8_t Pll); 											// short-cut to 5G/40MHz/reduced-power mode
void rfic_set_pll_2G(uint8_t Pll); 													// short-cut to 2G/40MHz/Frac-N mode
void rfic_set_pll_intnmode(uint8_t Pll);											// switch PLL0/1 to Integer-N mode
void rfic_set_pll_fracnmode(uint8_t Pll);											// switch PLL0/1 to Fractional-N mode
void rfic_select_pll(uint8_t AntMsk, uint8_t Pll, uint8_t HbLb);	  				// switch selected Antenna(s) to selected PLL in selected mode
void rfic_select_fasthop_master(uint8_t Pll, uint8_t Ant);							// select master antenna for fast-hop control for selected PLL
void rfic_select_fasthop_channel(uint8_t MasterAnt, uint8_t Channel);				// select fast-hop channel via master antenna
void rfic_set_lo_frequency_intn(uint8_t Pll, uint16_t LoFreq, uint8_t HbLb, uint8_t RefClk);
																					// set single LO frequency in Int-N mode
void rfic_set_lo_frequency_fracn(uint8_t Pll, uint16_t LoFreq, uint8_t HbLb);
																					// set single LO frequency in Frac-N mode
void rfic_set_lo_frequencies_intn(uint8_t Pll, uint16_t LoFreq0, uint16_t LoFreq1, uint16_t LoFreq2, uint16_t LoFreq3, uint16_t LoFreq4, uint16_t LoFreq5, uint8_t HbLb, uint8_t RefClk);
																					// set 6 fast-hopping frequencies in Int-N mode
void rfic_set_lo_frequencies_fracn(uint8_t Pll, uint16_t LoFreq0, uint16_t LoFreq1, uint16_t LoFreq2, uint16_t LoFreq3, uint16_t LoFreq4, uint16_t LoFreq5, uint8_t HbLb);
																					// set 6 fast-hopping frequencies in Frac-N mode
uint8_t rfic_check_pll_lock(uint8_t Pll);                                           // retrieve PLL lock information

// Manual gain setting
void rfic_set_lnagain(uint8_t AntMsk, uint8_t gainidx);
void rfic_set_pgcgain(uint8_t AntMsk, int8_t gaindB);
void rfic_set_tpcgain(uint8_t AntMsk, int8_t gaindB);
void rfic_set_padatt(uint8_t AntMsk, uint8_t attdB);

// Gain configuration
void rfic_set_txfectl(uint8_t AntMsk, uint8_t txfectl);								// set TX front-end controls
void rfic_config_tpcgain (uint8_t AntMsk, int8_t gain0dB, int8_t gain1dB, int8_t gain2dB, int8_t gain3dB);
																					// set TPC gains for all 4 TX gain constellations
void rfic_config_tpcfiltidx (uint8_t AntMsk, int8_t tpcfiltidx160, int8_t tpcfiltidx80, int8_t tpcfiltidx40, int8_t tpcfiltidx20);
																					// set TPC filter indices for all 4 bandwidth modes
void rfic_config_lnagain (uint8_t AntMsk, int8_t gain0idx, int8_t gain1idx, int8_t gain2idx, int8_t gain3idx, int8_t gain4idx, int8_t gain5idx, int8_t gain6idx);
																					// set LNA gain indices for all 7 RX gain constellations (gain index values 0,1,2,3)
void rfic_config_rfrxgain (uint8_t AntMsk, uint8_t fectl0, int8_t gain0idx, uint8_t fectl1, int8_t gain1idx,
		uint8_t fectl2, int8_t gain2idx, uint8_t fectl3, int8_t gain3idx, uint8_t fectl4, int8_t gain4idx,
		uint8_t fectl5, int8_t gain5idx, uint8_t fectl6, int8_t gain6idx);
																					// set FECTL outputs and LNA gain indices for all 7 RX gain constellations (gain index values 0,1,2,3)
void rfic_config_rssi(uint8_t AntMsk, uint8_t cfg_nr, int8_t gaindB, uint8_t coarse_offset_idx, uint8_t fine_offset_idx);
																					// set RSSI configuration (cfg_nr=0/1)
void rfic_select_rssi_bias(uint8_t AntMsk, uint8_t bias_type);						// select RSSI bias current type for coarse offset (1 = p2pr, 0 = const.)
void rfic_select_rssi(uint8_t AntMsk, uint8_t cfg_nr);								// select RSSI configuration
void rfic_config_tssi(uint8_t AntMsk, uint8_t cfg_nr, int8_t gaindB, uint8_t fine_offset_idx);
																					// set TSSI configuration  (cfg_nr=0/1/2/3)
void rfic_select_tssi(uint8_t AntMsk, uint8_t cfg_nr);								// select TSSI configuration
void rfic_set_tssi_rdiv (uint8_t AntMsk, uint8_t rdiv);								// set TSSI resistive input divider, default: 9 (==0.9), fall-back: 6 (==0.6)
void rfic_set_rxgain (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB);		// choose RF RX gain constellation, set BB RX gain
void rfic_set_rfrxgainidx (uint8_t AntMsk, uint8_t rfrxgainidx);					// select RF RX gain constellation only
void rfic_set_pgcbwidth (uint8_t AntMsk, uint8_t rfbandwidth);						// set RX filter bandwidth only
void rfic_set_rxgain_filter_rssi (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB, uint8_t rxfiltidx, uint8_t rssi);
																					// choose RF RX gain constellation, set BB RX gain, set PGC filter index, select RSSI config
void rfic_set_rxgain_bwidth (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB, uint8_t rfbandwidth);
																					// choose RF RX gain constellation, set BB RX gain, set RX filter bandwidth
void rfic_set_rxgain_bwidth_rssi (uint8_t AntMsk, uint8_t rfrxgainidx, int8_t pgcgaindB, uint8_t rfbandwidth, uint8_t rssi);
																					// choose RF RX gain constellation, set BB RX gain, set RX filter bandwidth, select RSSI config
void rfic_set_pgcfc100khz(uint16_t *vPgcFc100kHzNew);                               // re-define vector of PGC corner frequencies for all modes
void rfic_create_pgcfiltlut(uint16_t sA, int16_t sB);								// create internal PGC Filter Tuning LUT from eFuse or given sA/sB parameters
void rfic_get_pgcfiltlut (uint8_t mLut[5][11]);										// read the current PGC Filter Tuning LUT from memory


void rfic_set_txgain(uint8_t AntMsk, uint8_t tpcgainidx, uint8_t padattdB);			// choose TPC gain constellation, set PAD attenuation
void rfic_set_tpcgainidx(uint8_t AntMsk, uint8_t tpcgainidx);						// select TPC gain constellation only
void rfic_set_txband(uint8_t AntMsk, uint8_t bandsel);								// set TX band selection
void rfic_read_txband(uint8_t AntMsk, uint16_t *tx_band);							// read the current TX band selection from selected antennas
uint16_t rfic_get_txband(uint16_t LoFreq);											// return best-guess TX HB or LB band selection
void rfic_set_rxband(uint8_t AntMsk, uint8_t bandsel);								// set RX band selection
void rfic_read_rxband(uint8_t AntMsk, uint16_t *rx_band);							// read the current RX band selection from selected antennas
uint16_t rfic_get_rxband(uint16_t LoFreq);											// return best-guess RX HB or LB band selection
void rfic_set_lnamidgain(uint8_t AntMsk, uint8_t lnamidgain);						// set LNA Mid gain fine-tuning value
uint16_t rfic_get_lnamidgain(uint8_t Ant, uint8_t HbLb); 							// return RX LNA HB/LB mid gain tuning index, either from eFuse or best-guess

// Power Mode and Bandwidth selection
void rfic_set_bandwidth(uint8_t AntMsk, uint8_t rfbandwidth);						// select (RF) bandwidth (in MHz: 20, 40, 80, or 160)
void rfic_set_powermode(uint8_t AntMsk, uint8_t modeidx);							// select power mode: 0:HP, 1:RP, 2:LP, 3:DPD
void rfic_set_rxdc(uint8_t AntMsk, uint8_t rxdcidx);								// select RX DC offset constellation
void rfic_set_rfch_dc_bw(uint8_t AntMsk, uint8_t Channel, uint8_t rxdcidx, uint8_t rfbandwidth);
																					// select fasthop channel, RX DC Offset, and RF bandwidth
// LNA Noise Source on/off
void rfic_enable_lnanoise(uint8_t AntMsk);											// enable LNA noise source in new mode with higher level (B0 only)
void rfic_enable_lnanoise_old(uint8_t AntMsk);										// enable LNA noise source in legacy mode (like A0/A1 and Gen5)
void rfic_disable_lnanoise(uint8_t AntMsk);											// disable LNA noise source
void rfic_enable_lna(uint8_t AntMsk);												// enable LNA
void rfic_disable_lna(uint8_t AntMsk);												// disable LNA

// Direct Step RX Gains
void rfic_config_dirstep(uint8_t AntMsk, uint8_t dstepidx, uint8_t lnagainidx, uint8_t pgcgaindB, uint8_t rssirange, uint8_t rxfiltidx);
																					// configure direct steps 1, 2, 3
void rfic_select_dirstep(uint8_t AntMsk, uint8_t dstepidx);							// select direct steps 1, 2, 3, and 0 for deactivation
void rfic_dirstep_force_on(uint8_t AntMsk);											// clear FORCE_OFF bit in SR1
void rfic_dirstep_force_off(uint8_t AntMsk);										// set FORCE_OFF bit in SR1

// Baseband Calibration Loops
void rfic_enable_bbloop (uint8_t AntMsk, uint8_t HbLb, uint8_t LoopType, uint8_t IqSel);
																					// enable baseband loop for calibration, specify HB or LB,
																					// specify LoopType (0==SSBCAL, 1==IQCAL) and choose I, or Q, or both as output
void rfic_disable_bbloop (uint8_t AntMsk, uint8_t HbLb);							// disable SSB Calibration, in HB or LB mode (shortcut to rfic_enable_bbloop() )
void rfic_set_pgc_inputres(uint8_t AntMsk, uint16_t Rin, uint8_t IqSel);			// select input resistor in BB loop mode (250, 830 or 360 Ohms) (B0 only!)
void rfic_enable_ssbcal_hb(uint8_t AntMsk);											// enable SSBCAL calibration in HB mode, output to I and Q (shortcut to rfic_enable_bbloop() )
void rfic_enable_ssbcal_lb(uint8_t AntMsk);											// enable SSBCAL calibration in LB mode, output to I and Q (shortcut to rfic_enable_bbloop() )
void rfic_enable_iqcal_hb(uint8_t AntMsk, uint8_t IqSel);							// enable IQ calibration in HB mode, output to I, to Q, or to both I+Q (shortcut to rfic_enable_bbloop() )
void rfic_enable_iqcal_lb(uint8_t AntMsk, uint8_t IqSel);							// enable IQ calibration in HB mode, output to I, to Q, or to both I+Q (shortcut to rfic_enable_bbloop() )
void rfic_enable_filtercal_hb(uint8_t AntMsk);										// enable RX filter calibration in HB mode, output to I and Q (shortcut to rfic_enable_bbloop() )
void rfic_enable_filtercal_lb(uint8_t AntMsk);										// enable RX filter calibration in LB mode, output to I and Q (shortcut to rfic_enable_bbloop() )
void rfic_disable_ssbcal_hb(uint8_t AntMsk);										// disable SSBCAL calibration in HB mode (shortcut to rfic_disable_bbloop() )
void rfic_disable_ssbcal_lb(uint8_t AntMsk);										// disable SSBCAL calibration in LB mode (shortcut to rfic_disable_bbloop() )
void rfic_disable_iqcal_hb(uint8_t AntMsk);											// disable IQ calibration in HB mode (shortcut to rfic_disable_bbloop() )
void rfic_disable_iqcal_lb(uint8_t AntMsk);											// disable IQ calibration in LB mode (shortcut to rfic_disable_bbloop() )
void rfic_disable_filtercal_hb(uint8_t AntMsk);										// disable RX filter calibration in HB mode (shortcut to rfic_disable_bbloop() )
void rfic_disable_filtercal_lb(uint8_t AntMsk);										// disable RX filter calibration in LB mode (shortcut to rfic_disable_bbloop() )
void rfic_enable_dpdcal(uint8_t AntMsk, uint8_t HbLb);								// manually activate TX and RX path in DPD mode (maximum bandwidth and linearity) (>NO< RFIC TXON or RXON required!)
void rfic_disable_dpdcal(uint8_t AntMsk, uint8_t HbLb);								// remove manually activation of RX and TX path in DPD mode

// enable/disable BBO output
void rfic_enable_bbo(uint8_t AntMsk);												// (re-)enable BBO
void rfic_disable_bbo(uint8_t AntMsk);												// disable BBO (tie output to VSS)

// RX DC Offset Cancellation
void rfic_start_dc_clock (uint8_t AntMsk);											// start clock of DC state machine
void rfic_stop_dc_clock (uint8_t AntMsk);											// stop clock of DC state machine

void rfic_start_dcramp (uint8_t AntMsk);											// start DC ramp
void rfic_stop_dcramp (uint8_t AntMsk);												// stop DC ramp

void rfic_start_dccancel (uint8_t AntMsk);											// start DC offset cancellation
void rfic_stop_dccancel (uint8_t AntMsk);											// stop DC offset cancellation
uint8_t rfic_check_dccancel (uint8_t AntMsk);										// check if DC offset cancellation has finished

void rfic_reset_dccancel (uint8_t AntMsk);											// reset DC offset cancellation

void rfic_set_dcintegrator(uint8_t AntMsk, uint8_t IntBW);							// set DC state machine integration bandwidth

void rfic_enable_rxdc_man (uint8_t AntMsk);											// enable manual RX DC offset DAC setting
void rfic_disable_rxdc_man (uint8_t AntMsk);										// enable manual RX DC offset DAC setting

void rfic_set_rxdcoffset (uint8_t AntMsk, int8_t dacI, int8_t dacQ);				// set RX DC offset DAC values (-31...31)
void rfic_set_rxdcoffset_reg (uint8_t AntMsk, uint8_t rfrxgainidx, uint8_t rxdcidx, uint16_t dcReg);
																					// set RX DC offset DAC values for specific antenna,
																					// RF gain, and RXDC Channel(unprocessed, i.e. cascaded I+Q and success bits)
uint16_t rfic_get_rxdcoffset (uint8_t Ant, uint8_t rfrxgainidx, uint8_t rxdcidx);	// get RX DC offset register entry for specific antenna, RF gain, and RXDC Channel(unprocessed, i.e. cascaded I+Q and success bits)


// TX DC Offset Cancellation
void rfic_set_txdcoffset (uint8_t AntMsk, int8_t dacI, int8_t dacQ);				// set TX DC offset DAC values (for all configurations the same)
void rfic_set_txdc_tpccfg (uint8_t AntMsk, int8_t dacI, int8_t dacQ, uint8_t TpcCfg);
																					// set TX DC offset DAC values (for one specific TPC configuration)

// LDO Programming

// BBO LDO
void rfic_enable_1v2bboldo(void); // NEW
void rfic_disable_1v2bboldo(void);// NEW
void rfic_enable_bypass_1v2bboldo(void); // NEW
void rfic_disable_bypass_1v2bboldo(void);// NEW

void rfic_set_bbioldo_ctl_vout(int step);
void rfic_set_bbioldo_vout_vcmbuffer (int step);
void rfic_set_bbioldo_ctl_comp(int step);

// FCSI LDO
void rfic_set_fcsildo_ctl_vout(int step);

// ABB LDO
void rfic_set_abbldo_ctl_rdg_prog(int step);

// LO Central
void rfic_set_locenthb_ctl_vout(uint8_t AntMsk ,int step);
void rfic_set_locentlb_ctl_vout(uint8_t AntMsk ,int step);
void rfic_enable_lotreeldo(uint8_t AntMsk, uint8_t HbLb);
void rfic_disable_lotreeldo(uint8_t AntMsk, uint8_t HbLb);

// FSYS LDO
void rfic_set_fsysldo_ctl_vout(int step);
void rfic_set_fsysldo_ctl_iref (int step);

// BB LDO
void rfic_set_ctl_vout_1v2bbldo(uint8_t AntMsk, int8_t step);
void rfic_enable_1v2bbldo(uint8_t AntMsk);
void rfic_disable_1v2bbldo(uint8_t AntMsk);
void rfic_enable_bypass_1v2bbldo(uint8_t AntMsk);
void rfic_disable_bypass_1v2bbldo(uint8_t AntMsk);
void rfic_enable_rload_1v2bbldo(uint8_t AntMsk);
void rfic_disable_rload_1v2bbldo(uint8_t AntMsk);
void rfic_force_ldo_ready_1v2bbldo(uint8_t AntMsk);
void rfic_ctl_ldo_ready_1v2bbldo(uint8_t AntMsk);

// 2V5 LDO
void rfic_enable_2v5ldo(uint8_t AntMsk);
void rfic_disable_2v5ldo(uint8_t AntMsk);
void rfic_disable_rload_2v5ldo(uint8_t AntMsk);
void rfic_enable_rload_2v5ldo(uint8_t AntMsk);
void rfic_enable_bypass_2v5ldo(uint8_t AntMsk);
void rfic_disable_bypass_2v5ldo(uint8_t AntMsk);
void rfic_set_ctl_lpgain_2v5ldo(uint8_t AntMsk);
void rfic_set_ctl_lpgain_off_2v5ldo(uint8_t AntMsk);
void rfic_set_ctl_vout_2v5ldo(uint8_t AntMsk, int8_t step);
void rfic_enable_lc_2v5ldo(uint8_t AntMsk);
void rfic_disable_lc_2v5ldo(uint8_t AntMsk);

// RX LDO
void rfic_enable_rxldo(uint8_t AntMsk);
void rfic_disable_rxldo(uint8_t AntMsk);
void rfic_enable_rload_rxldo(uint8_t AntMsk);
void rfic_disable_rload_rxldo(uint8_t AntMsk);
void rfic_enable_bypass_rxldo(uint8_t AntMsk);
void rfic_disable_bypass_rxldo(uint8_t AntMsk);
void rfic_set_ctl_vout_rxldo(uint8_t AntMsk, int8_t step);

// TX HB LDO
void rfic_enable_hbldo(uint8_t AntMsk);
void rfic_disable_hbldo(uint8_t AntMsk);
void rfic_enable_rload_hbldo(uint8_t AntMsk);
void rfic_disable_rload_hbldo(uint8_t AntMsk);
void rfic_enable_bypass_hbldo(uint8_t AntMsk);
void rfic_disable_bypass_hbldo(uint8_t AntMsk);
void rfic_set_ctl_vout_hbldo(uint8_t AntMsk, int8_t step);

// TX LB LDO
void rfic_enable_lbldo(uint8_t AntMsk);
void rfic_disable_lbldo(uint8_t AntMsk);
void rfic_enable_rload_lbldo(uint8_t AntMsk);
void rfic_disable_rload_lbldo(uint8_t AntMsk);
void rfic_enable_bypass_lbldo(uint8_t AntMsk);
void rfic_disable_bypass_lbldo(uint8_t AntMsk);
void rfic_set_ctl_vout_lbldo(uint8_t AntMsk, int8_t step);

// CENTRAL VDDMON
void rfic_enable_vddmon_central (void);
void rfic_disable_vddmon_central (void);

void rfic_enable_vddmon_fcsildo (void);
void rfic_disable_vddmon_fcsildo (void);
void rfic_enable_vddmon_fsysldo (void);
void rfic_disable_vddmon_fsysldo (void);
void rfic_enable_vddmon_bbioldo (void);
void rfic_disable_vddmon_bbioldo (void);
void rfic_enable_vddmon_lotreeldo (uint8_t AntMsk, uint8_t HbLb);
void rfic_disable_vddmon_lotreeldo (uint8_t AntMsk, uint8_t HbLb);

// Supply Monitor (VDDMON w/i ANTENNA)
void rfic_disable_vddmon(uint8_t AntMsk);
void rfic_enable_vddmon_1v2bbldo(uint8_t AntMsk);
void rfic_enable_vddmon_2v5ldo(uint8_t AntMsk);
void rfic_enable_vddmon_hbldo(uint8_t AntMsk);
void rfic_enable_vddmon_lbldo(uint8_t AntMsk);
void rfic_enable_vddmon_1v2rxldo(uint8_t AntMsk);
void rfic_enable_vddmon_1v5ldoin(uint8_t AntMsk);
void rfic_enable_vddmon_3v3ldoin(uint8_t AntMsk);
void rfic_enable_vddmon_tssiin(uint8_t AntMsk);

// Temperature Sensor
void rfic_enable_tempsensor (void);													// enable Temp sensor
void rfic_disable_tempsensor (void);												// disable Temp sensor
void rfic_start_tempsensor_clock (void);											// start clock for Temp sensor
void rfic_stop_tempsensor_clock (void);												// stop clock for Temp sensor
void rfic_start_tempsensor_cal (void);												// start calibration
void rfic_stop_tempsensor_cal (void);												// stop calibration
void rfic_start_tempsensor (void);													// start measurement
uint16_t rfic_read_tempsensor (void);												// read calibration result

// Central bias
void rfic_set_bgp_trim (int TuneVal);												// bandgap tuning (-8...7) [%]
void rfic_set_iconst_trim (int TuneVal);											// Icost tuning (0...31)

// FECTL
void rfic_set_antenna_fectl_man(uint8_t AntMsk, uint8_t fectl);						// set manual antenna FECTLs
void rfic_disable_antenna_fectl(uint8_t AntMsk);									// disable manual FECTL control
void rfic_set_central_fectl_man(uint8_t fectl);										// set manual central FECTLs (0...255)

// ANAMUX
void rfic_enable_anamux(uint8_t AntMsk);											// enable AnaMux
void rfic_disable_anamux(uint8_t AntMsk);											// disable AnaMux
void rfic_enable_anamux_vddmonin(uint8_t AntMsk);                                   // enable Supply Monitor Input at AnaMux
void rfic_disable_anamux_vddmonin(uint8_t AntMsk);									// disable Supply Monitor Input at AnaMux

// eFuse Programming
void rfic_enable_efuse (void);														// enable eFuse programming
void rfic_disable_efuse (void);														// stop eFuse programming

// eFuse Readout
uint16_t rfic_read_efuse_values(uint16_t sections, uint16_t *softfuse_data);		// read eFuse data and distribute trimming values (still keeping softfuse option)
uint16_t rfic_get_efuse_values(uint16_t *fuse_data);								// read eFuse data from CENTRAL fuse registers
uint16_t rfic_read_efuse_handlertemp(uint16_t *softfuse_data);						// read handler temperature from eFuse data
uint16_t rfic_read_efuse_tempsensorresult(uint16_t *softfuse_data);					// read temperature sensor calibration result from eFuse data

// New B0 Features
void rfic_select_lnabiasmode(uint8_t AntMsk, uint8_t BiasMode);						// select LNA Bias mode: 0==replica biasing (default), 1==diode biasing
void rfic_select_tpcdacrange(uint8_t AntMsk, uint8_t DacRange);						// select TPC DAC range: 0 or 6 == 6mV (default), 1 or 9 == 9mV
void rfic_select_bbovcminput(uint8_t VcmInput);										// select Vcm input:  0==P2PR resistor (default), 1==VCM buffer


// APIs with parallel antenna access
void rfic_config_tpcgain_par (uint8_t AntMsk, int8_t *gain0dB, int8_t *gain1dB, int8_t *gain2dB, int8_t *gain3dB);
 																					// set TPC gains for all 4 TX gain constellations, different for each antenna
void rfic_set_txdc_tpccfg_par (uint8_t AntMsk, int8_t *dacI, int8_t *dacQ, uint8_t TpcCfg);
																					// set TX DC offset DAC values (for one specific TPC configuration) - parallel write of different DAC values per antenna
void rfic_set_antenna_fectl_man_par(uint8_t AntMsk, uint8_t *fectl);				// set manual antenna FECTLs, different for each selected antenna


#endif
